/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Kai Ahrens <kai.ahrens@open-xchange.com>
 */

define('io.ox/office/tk/dialogs',
    ['io.ox/core/tk/dialogs',
     'io.ox/office/tk/utils',
     'gettext!io.ox/office/main'
    ], function (CoreDialogs, Utils, gt) {

    'use strict';

    // private helper functions ===============================================

    /**
     * Adds OK and Cancel buttons to the passed dialog.
     */
    function addDialogButtons(dialog, options) {
        dialog.addPrimaryButton('ok', Utils.getStringOption(options, 'okLabel', gt('OK')), 'ok', { tabIndex: 1 })
            .addButton('cancel', Utils.getStringOption(options, 'cancelLabel', gt('Cancel')), 'cancel', { tabIndex: 1 });
    }

    /**
     * Returns a promise for the passed Deferred object that contains an
     * additional method 'close()' which when invoked will close the specified
     * dialog and will reject the Deferred object.
     */
    function getExtendedPromise(def, dialog) {
        return _(def.promise()).extend({
            close: function () {
                if (def.state() === 'pending') {
                    dialog.close();
                    def.reject();
                }
            },
            promise: function () {
                return this;
            }
        });
    }

    // static class Dialogs ===================================================

    var Dialogs = {};

    // methods ----------------------------------------------------------------

    /**
     * Creates and returns an empty modal dialog.
     *
     * @param {Object} [options]
     *  Additional options that control the appearance and behavior of the
     *  dialog. The following options are supported:
     *  @param {String} [options.title]
     *      The title of the dialog window that will be shown in a larger font.
     *  @param {Boolean} [options.async]
     *      Whether the dialog will be opened in asynchronous mode.
     *
     * @returns {CoreDialogs.ModalDialog}
     *  A modal dialog object initialized with options.
     */
    Dialogs.createDialog = function (options) {

        var // create the dialog instance
            dialog = new CoreDialogs.ModalDialog({
                width: Utils.getIntegerOption(options, 'width', 400),
                async: Utils.getBooleanOption(options, 'async', false),
                tabTrap: true,
                enter: Utils.getFunctionOption(options, 'enter', undefined)
            }),
            // the title text
            title = Utils.getStringOption(options, 'title'),
            // the dummy input to catch the cursor
            focusCatcher = $('<input>').css({position: 'absolute', top: '-10000px', left: '-10000px', width: '1px', height: '1px'});

        // add title
        if (_.isString(title)) {
            dialog.header($('<h4>').text(title).attr('id', 'dialog-title'));
        }

        // add dummy input to catch the cursor on dialog open
        dialog.getFooter().append(focusCatcher);

        // set focus to dummy input
        dialog.getPopup().on('show', function () {
            focusCatcher.focus();
        });

        // remove dummy input
        dialog.getPopup().on('shown', function () {
            focusCatcher.remove();
        });

        // The dialog is opened with nodes.popup.show() instead of nodes.popup.modal('show')
        // which prevents the 'show' and 'shown' events being triggered.
        // We call tab('show') to trigger the events.
        dialog.on('show', function () {
            dialog.getPopup().tab('show');
        });

        return dialog;
    };

    /**
     * Shows a simple OK/Cancel dialog with customizable button texts.
     *
     * @param {Object} [options]
     *  Additional options that control the appearance and behavior of the
     *  dialog. The following options are supported:
     *  @param {String} [options.title]
     *      The title of the dialog window that will be shown in a larger font.
     *  @param {String} [options.message]
     *      If specified, the message shown in the dialog body.
     *  @param {String} [options.okLabel=gt('OK')]
     *      The label of the primary OK button that resolves the promise object
     *      returned by this method.
     *  @param {String} [options.cancelLabel=gt('Cancel')]
     *      The label of the Cancel button that rejects the promise object
     *      returned by this method.
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the dialog
     *  has been confirmed, or rejected if the dialog has been canceled.
     *  Additionally, this object will contain a method 'close()' that will
     *  close the dialog immediately and will reject this promise.
     */
    Dialogs.showOkCancelDialog = function (options) {

        var // the dialog object
            dialog = Dialogs.createDialog(options),

            // the message text
            message = Utils.getStringOption(options, 'message'),

            // the result deferred
            def = $.Deferred();

        // add the message text
        if (_.isString(message)) {
            dialog.text(message);
        }

        // add OK and Cancel buttons
        addDialogButtons(dialog, options);

        // show the dialog and register listeners for the results
        dialog.show().done(function (action) {
            def[(action === 'ok') ? 'resolve' : 'reject']();
        });

        return getExtendedPromise(def, dialog);
    };

    /**
     * Shows a simple Yes/No dialog with customizable button texts. This is a
     * slightly modified version of the Dialogs.showOkCancelDialog() method
     * with modified defaults for the dialog buttons.
     *
     * @param {Object} [options]
     *  Additional options that control the appearance and behavior of the
     *  dialog. The following options are supported:
     *  @param {String} [options.title]
     *      If specified, the title of the dialog window that will be shown in
     *      a larger font.
     *  @param {String} [options.message]
     *      If specified, the message shown in the dialog body.
     *  @param {String} [options.okLabel=gt('Yes')]
     *      The label of the primary Yes button that resolves the promise
     *      object returned by this method.
     *  @param {String} [options.cancelLabel=gt('No')]
     *      The label of the No button that rejects the promise object returned
     *      by this method.
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the dialog
     *  has been confirmed, or rejected if the dialog has been canceled.
     *  Additionally, this object will contain a method 'close()' that will
     *  close the dialog immediately and will reject this promise.
     */
    Dialogs.showYesNoDialog = function (options) {
        return Dialogs.showOkCancelDialog(Utils.extendOptions({ okLabel: gt('Yes'), cancelLabel: gt('No') }, options));
    };

    /**
     * Shows a simple text input dialog.
     *
     * @param {Object} [options]
     *  Additional options that control the appearance and behavior of the
     *  dialog. The following options are supported:
     *  @param {String} [options.title]
     *      If specified, the title of the dialog window that will be shown in
     *      a larger font.
     *  @param {String} [options.value='']
     *      The initial value of the text field.
     *  @param {String} [options.placeholder='']
     *      The place-holder text that will be shown in the empty text field.
     *  @param {String} [options.okLabel=gt('OK')]
     *      The label of the primary button that triggers the intended action
     *      by resolving the promise object returned by this method.
     *  @param {String} [options.cancelLabel=gt('Cancel')]
     *      The label of the Cancel button that rejects the promise object
     *      returned by this method.
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the primary
     *  button has been activated, or rejected if the dialog has been canceled.
     *  The done handlers registered at the promise object will receive the
     *  entered text. Additionally, this object will contain a method 'close()'
     *  that will close the dialog immediately and will reject this promise.
     */
    Dialogs.showTextDialog = function (options) {

        var // the text input field
            input = $('<input>', {
                type: 'text',
                placeholder: Utils.getStringOption(options, 'placeholder', ''),
                value: Utils.getStringOption(options, 'value', ''),
                tabindex: 1,
                'data-default-enter': true
            }).addClass('span12'),

            // the dialog object
            dialog = Dialogs.createDialog(Utils.extendOptions({ enter: Dialogs.defaultKeyEnterHandler }, options))
                .append($('<div>').addClass('row-fluid').append(input)),

            // the result deferred
            def = $.Deferred();

        // add OK and Cancel buttons
        addDialogButtons(dialog, options);

        // set focus to text field, after dialog has been shown
        dialog.getPopup().on('shown', function () {
            input.focus().select();
        });

        // show the dialog and register listeners for the results
        dialog.show()
        .done(function (action) {
            if (action === 'ok') {
                def.resolve(input.val());
            } else {
                def.reject();
            }
        });

        return getExtendedPromise(def, dialog);
    };

    /**
     * Shows a generic file selector dialog.
     *
     * @param {Object} [options]
     *  Additional options that control the appearance and behavior of the
     *  dialog. The following options are supported:
     *  @param {String} [options.title]
     *      If specified, the title of the dialog window that will be shown in
     *      a larger font.
     *  @param {String} [options.filter='*']
     *      The filter string restricting the type of files that will be able
     *      to select.
     *  @param {String} [options.placeholder='']
     *      The place-holder text that will be shown in the empty text field.
     *  @param {String} [options.okLabel=gt('OK')]
     *      The label of the primary button that triggers the intended action
     *      by resolving the promise object returned by this method.
     *  @param {String} [options.cancelLabel=gt('Cancel')]
     *      The label of the Cancel button that rejects the promise object
     *      returned by this method.
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the primary
     *  button has been activated, or rejected if the dialog has been canceled.
     *  The done handlers registered at the promise object will receive the
     *  file descriptor object of the selected file.
     */
    Dialogs.showFileDialog = function (options) {

        var // the text input field
            input = $('<input>', {
                    placeholder: Utils.getStringOption(options, 'placeholder', ''),
                    value: '',
                    name: 'files[]',
                    type: 'file',
                    accept: Utils.getStringOption(options, 'filter', '*')
                }),

            // the dialog object
            dialog = Dialogs.createDialog(options).append(input.addClass('nice-input')),

            // the file descriptor of the file currently selected
            file = null,

            // the result deferred
            def = $.Deferred();

        // add OK and Cancel buttons
        addDialogButtons(dialog, options);

        // register a change handler at the input field that extracts the file descriptor
        input.change(function (event) {
            file = (event.target && event.target.files && event.target.files[0]) || null;  // requires IE 10+
            if ((file === null) && (event.target.value)) {
                file = event.target.value;
            }
        });

        // show the dialog and register listeners for the results
        dialog.show(function () { input.focus(); })
        .done(function (action) {
            if ((action === 'ok') && (_.isObject(file) || _.isString(file))) {
                def.resolve(file);
            } else {
                def.reject();
            }
        });

        return def.promise();
    };

    /**
     * Default ENTER handler for dialogs. Must be set via 'enter' options at
     * the dialog. Checks if the primary button is not disabled and invokes the
     * stored function on controls which have the 'data-default-enter'
     * attribute set to true.
     *
     * @returns {Boolean}
     *  Whether the event should be processed by bubbling.
     */
    Dialogs.defaultKeyEnterHandler = function () {
        var button = null, focus = $(document.activeElement);

        // handler for the ENTER key on the dialog
        if (this.getFooter().children(':focus').length === 0 && focus.attr('data-default-enter')) {
            button = this.getFooter().find('[disabled!="disabled"][data-action].btn-primary').first();
            if (button.length) {
                this.invoke(button.attr('data-action'));
                return false;
            }
        }

        // let the event bubble
        return true;
    };

    // exports ================================================================

    return Dialogs;

});
