/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/tk/control/boundradiolists',
    ['io.ox/office/tk/control/radiolist'], function (RadioList) {

    'use strict';

    // class BoundRadioLists ========================================================

    /**
     * BoundRadioLists offer a tool-object with two Radiolists bound on each other,
     * the category-RadioList and the entry-RadioList.
     * if a value is chosen in the category-RL, it chooses the correct entries for the
     * entry-RL
     *
     * the content is set in constructor and by calling setSourceData
     *
     * @param {Array} sources
     *      [{value, label, entries: [{value, label}, {value, label}]]
     * @param {Object} catOptions
     *      options for the category-RadioList
     *      @see RadioList.constructor
     * @param {Object} entryOptions
     *      options for the entry-RadioList
     *      @see RadioList.constructor
     */
    function BoundRadioLists(sources, catOptions, entryOptions) {
        var self = this,
            catList = new RadioList(catOptions),
            entList = new RadioList(entryOptions),
            mysources = null,
            activeCategory = null,
            activeEntry = null;

        // private methods ----------------------------------------------------

        function catHandler(category) {
            activeCategory = category;
            entList.clearMenu();

            if (!category) {
                return;
            }

            var entryHolder = getEntryHolder(category);

            if (!entryHolder) {
                console.warn('no EntryList found for Category:', category);
            } else {
                _(entryHolder.entries).each(function (entry) {
                    entList.createOptionButton('', entry.value, { label: entry.label, dataValue: entry.value });
                });
            }
        }

        function entHandler(entry) {
            activeEntry = entry;
        }

        function getEntryHolder(category) {
            return _.find(mysources, function (elem) {
                return elem.value === category;
            });
        }

        catList.registerUpdateHandler(catHandler);
        entList.registerUpdateHandler(entHandler);


        // methods ------------------------------------------------------------

        /**
         * changes the sources and refresh the category-RL
         * @param {Array} newSources
         *      [{value, label, entries: [{value, label}, {value, label}]]
         */
        this.setSourceData = function (newSources) {
            mysources = newSources;

            catList.clearMenu();

            _(mysources).each(function (category) {
                catList.createOptionButton('', category.value, { label: category.label, dataValue: category.value });
            });
        };

        this.getCatRadioList = function () {
            return catList;
        };

        this.getEntRadioList = function () {
            return entList;
        };

        /**
         * @param {String} category
         */
        this.getFirstEntry = function (category) {
            var entryHolder = getEntryHolder(category);
            if (!entryHolder) {
                console.warn('no EntryList found for Category:', category);
            } else {
                return entryHolder.entries[0];
            }
        };

        /**
         * Could be called before changing the category, if the new category has
         * a entry with the same value as the current value, the entry is returned.
         * Otherwise the first entry of the new category is returned
         * @param {String} category
         */
        this.getCompatibleEntry = function (category) {
            var entryHolder = getEntryHolder(category);
            if (!entryHolder) {
                console.warn('no EntryList found for Category:', category);
            } else {
                for (var i = 0; i < entryHolder.entries.length; i++) {
                    var entry = entryHolder.entries[i];
                    if (entry.value === activeEntry) {
                        return entry;
                    }
                }
                return entryHolder.entries[0];
            }
        };

        this.getEntryCount = function () {
            var entryHolder = getEntryHolder(activeCategory);

            if (!entryHolder) {
                console.warn('no EntryList found for Category:', activeCategory);
            } else {
                return entryHolder.entries.length;
            }
        };
        // init code ----------------------------------------------------------

        self.setSourceData(sources);


    } // class BoundRadioLists

    // exports ================================================================

    return _.makeExtendable(BoundRadioLists);

});
