/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Marko Benigar <marko.benigar@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/validationcollection',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject',
     'io.ox/office/editframework/model/format/attributedmodel',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/model/tokenarray'
    ], function (Utils, ModelObject, AttributedModel, SheetUtils, TokenArray) {

    'use strict';

    // class EntryModel =======================================================

    /**
     * Stores data validation settings for a single range in a specific sheet.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this data validation range.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this data validation range.
     *
     * @param {Array} ranges
     *  The logical addresses of all cell ranges that contain the data
     *  validation settings.
     *
     * @param {Object} [attrs]
     *  The data validation attribute set.
     */
    var EntryModel = AttributedModel.extend({ constructor: function (app, sheetModel, ranges, attrs) {

        var // self reference
            self = this,

            // the spreadsheet document model
            model = app.getModel(),

            // token array representing the validation attribute 'value1'
            tokenArray1 = new TokenArray(app, sheetModel),

            // token array representing the validation attribute 'value2'
            tokenArray2 = new TokenArray(app, sheetModel);

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, app, attrs, { silent: true, additionalFamilies: 'validation' });

        // methods ------------------------------------------------------------

        /**
         * Returns the addresses of all cell ranges of this collection entry.
         *
         * @returns {Array}
         *  The logical addresses of all cell ranges.
         */
        this.getRanges = function () {
            return _.copy(ranges, true);
        };

        /**
         * Returns whether the passed cell address is contained by the cell
         * ranges of this collection entry.
         *
         * @param {Number[]} address
         *  The logical address of the cell to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified cell is contained in this collection entry.
         */
        this.containsCell = function (address) {
            return SheetUtils.rangesContainCell(ranges, address);
        };

        /**
         * Returns the reference addresses of this collection entry. The
         * reference address is the top-left cell of the bounding range of all
         * ranges of this collection entry.
         *
         * @returns {Number[]}
         *  The logical address of the reference cell of this collection entry.
         */
        this.getRefAddress = function () {
            return SheetUtils.getBoundingRange(ranges).start;
        };

        /**
         * Returns the data source ranges for validations of type 'source'.
         * Relative references in the formula expression of the attribute
         * 'value1' will be resolved according to the passed target address.
         *
         * @param {Number[]} targetAddress
         *  The target address used to resolve relative references.
         *
         * @returns {Array|Null}
         *  A list of logical range addresses, if this collection entry is a
         *  data validation of type 'source', and the formula expression
         *  contained in the attribute 'value1' could be resolved successfully
         *  to range addresses; otherwise null. Each range address object
         *  contains the additional property 'sheet' with the zero-based sheet
         *  index of the range.
         */
        this.getSourceRanges = function (targetAddress) {
            if (this.getMergedAttributes().validation.type !== 'source') { return null; }
            return tokenArray1.resolveRangeList({ refAddress: this.getRefAddress(), targetAddress: targetAddress });
        };

        /**
         * Transforms the addresses of the cell ranges containing the data
         * validation settings, after columns or rows have been inserted into
         * or removed from the sheet.
         *
         * @returns {Boolean}
         *  Whether the transformed ranges are still valid (not all deleted).
         *  If false is returned, this collection entry must be deleted from
         *  the validation collection.
         */
        this.transformRanges = function (interval, insert, columns) {
            ranges = _.chain(ranges)
                .map(function (range) { return model.transformRange(range, interval, insert, columns, true); })
                .filter(_.isObject)
                .value();
            return ranges.length > 0;
        };

        /**
         * Recalculates the formula expressions stored in the validation
         * attributes 'value1' and 'value2', after columns or rows have been
         * inserted into or deleted from any sheet in the document.
         */
        this.transformValues = function (sheet, interval, insert, columns) {
            tokenArray1.transformFormula(sheet, interval, insert, columns);
            tokenArray2.transformFormula(sheet, interval, insert, columns);
        };

        // initialization -----------------------------------------------------

        // parse the source formulas
        (function () {
            var attributes = self.getMergedAttributes().validation;
            tokenArray1.parseFormula(attributes.value1);
            tokenArray2.parseFormula(attributes.value2);
        }());

        // clone the range addresses to prevent accidental changes from external code
        ranges = _.copy(ranges, true);

        // update source values after the token array has changed
        tokenArray1.on('triggered', function () { self.setAttributes({ validation: { value1: tokenArray1.getFormula() } }); });
        tokenArray2.on('triggered', function () { self.setAttributes({ validation: { value2: tokenArray2.getFormula() } }); });

        // constructor called from BaseObject.clone()
        this.registerCloneConstructor(function (newSheetModel, newSheet) {
            // pass own entries as hidden argument to the constructor
            return new EntryModel(app, newSheetModel, ranges, this.getExplicitAttributes(), { oldSheet: sheetModel.getIndex(), newSheet: newSheet });
        });

        // adjust formulas according to sheet indexes passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[EntryModel.length];
            if (_.isObject(cloneData)) {
                tokenArray1.relocateSheet(cloneData.oldSheet, cloneData.newSheet);
                tokenArray2.relocateSheet(cloneData.oldSheet, cloneData.newSheet);
            }
        }(arguments));

        // destroy all class members
        this.registerDestructor(function () {
            tokenArray1.destroy();
            tokenArray2.destroy();
            model = tokenArray1 = tokenArray2 = null;
            sheetModel = ranges = null;
        });

    }}); // class EntryModel

    // class ValidationCollection =============================================

    /**
     * Stores data validation settings in a specific sheet. Data validation can
     * be used to restrict the values allowed to be entered in cell ranges, and
     * to show tooltips and error messages.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this collection instance.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     */
    function ValidationCollection(app, sheetModel) {

        var // the spreadsheet document model
            model = app.getModel(),

            // all collection entries with validation settings
            entries = [];

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Returns the collection entry that covers the specified cell.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {EntryModel|Null}
         *  The collection entry for the specified cell. If the cell is not
         *  validated at all, returns null.
         */
        function getEntry(address) {
            var entry = _(entries).find(function (entry2) {
                return entry2.containsCell(address);
            });
            return entry ? entry : null;
        }

        /**
         * Recalculates the position of all validation ranges, after columns or
         * rows have been inserted into or deleted from the sheet.
         */
        function transformRanges(interval, insert, columns) {
            Utils.iterateArray(entries, function (entry, index) {
                if (!entry.transformRanges(interval, insert, columns)) {
                    entries[index].destroy();
                    entries.splice(index, 1);
                }
            }, { reverse: true }); // reverse to be able to delete in-place
        }

        /**
         * Recalculates the formula expressions stored in the validation
         * attributes 'value1' and 'value2', after columns or rows have been
         * inserted into or deleted from any sheet in the document.
         */
        function transformValues(sheet, interval, insert, columns) {
            _(entries).invoke('transformValues', sheet, interval, insert, columns);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the validation settings for the specified cell.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {Object|Null}
         *  The validation settings of the specified cell, in the following
         *  properties:
         *  - {Array} ranges
         *      The logical addresses of all related cell ranges containing the
         *      same validation settings.
         *  - {Object} attributes
         *      All validation attributes.
         *  If the cell is not validated at all, returns null.
         */
        this.getValidationSettings = function (address) {
            var entry = getEntry(address);
            return entry ? {
                ranges: entry.getRanges(),
                attributes: entry.getMergedAttributes().validation
            } : null;
        };

        /**
         * Returns the data source ranges for a data validation of type
         * 'source' at the specified cell address.
         *
         * @param {Number[]} address
         *  The logical address of a cell.
         *
         * @returns {Array|Null}
         *  A list of logical range addresses, if the specified cell contains a
         *  data validation of type 'source', and the formula expression
         *  contained in the attribute 'value1' could be resolved successfully
         *  to range addresses; otherwise null. Each range address object
         *  contains the additional property 'sheet' with the zero-based sheet
         *  index of the range.
         */
        this.getSourceRanges = function (address) {
            var entry = getEntry(address);
            return entry ? entry.getSourceRanges(address) : null;
        };

        /**
         * Restricts the valid values of a range of cells according to the
         * specified settings.
         *
         * @param {Array} ranges
         *  The logical addresses of all cell ranges that contain the data
         *  validation settings.
         *
         * @param {Object} attrs
         *  The validation attributes.
         */
        this.insertValidationRanges = function (ranges, attrs) {
            // TODO: handle overlapping ranges
            // TODO: trigger and handle change events
            entries.push(new EntryModel(app, sheetModel, ranges, { validation: attrs }));
        };

        // initialization -----------------------------------------------------

        // update ranges after inserting/deleting columns or rows in the own sheet
        sheetModel.registerTransformationHandler(transformRanges);

        // update source formulas after inserting/deleting columns or rows in any sheet
        model.registerTransformationHandler(transformValues);

        // constructor called from BaseObject.clone()
        this.registerCloneConstructor(function (newSheetModel, newSheet) {
            // pass own entries as hidden argument to the constructor
            return new ValidationCollection(app, newSheetModel, { entries: entries, newSheet: newSheet });
        });

        // clone collection entries passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[ValidationCollection.length];
            if (_.isObject(cloneData)) {
                entries = ModelObject.cloneArray(cloneData.entries, sheetModel, cloneData.newSheet);
            }
        }(arguments));

        // destroy all class members
        this.registerDestructor(function () {
            model.unregisterTransformationHandler(transformValues);
            _(entries).invoke('destroy');
            model = entries = null;
        });

    } // class ValidationCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: ValidationCollection });

});
