/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/namecollection',
    ['io.ox/office/tk/utils',
     'io.ox/office/baseframework/model/modelobject',
     'io.ox/office/spreadsheet/model/tokenarray'
    ], function (Utils, ModelObject, TokenArray) {

    'use strict';

    // private class NameModel ================================================

    /**
     * Stores settings for a single defined name in a specific sheet, or for a
     * global defined name in the document.
     *
     * @constructor
     *
     * @extends AttributedModel
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this defined name.
     *
     * @param {SheetModel} [sheetModel]
     *  The model of the sheet that contains this defined name. If omitted,
     *  this instance is a global document name.
     *
     * @param {String} formula
     *  The formula expression bound to the defined name.
     */
    var NameModel = ModelObject.extend({ constructor: function (app, sheetModel, formula) {

        var // the token array representing the definition of this name
            tokenArray = new TokenArray(app, sheetModel);

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // methods ------------------------------------------------------------

        /**
         * Returns the token array representing the formula definition of this
         * defined name.
         *
         * @returns {TokenArray}
         *  The token array containing the name definition.
         */
        this.getTokenArray = function () {
            return tokenArray;
        };

        // initialization -----------------------------------------------------

        // parse formula definition
        tokenArray.parseFormula(formula);

        // constructor called from BaseObject.clone()
        this.registerCloneConstructor(function (newSheetModel, newSheet) {
            var cloneData = sheetModel ? { oldSheet: sheetModel.getIndex(), newSheet: newSheet } : null;
            return new NameModel(app, newSheetModel, tokenArray.getFormula(), cloneData);
        });

        // adjust formula according to sheet indexes passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[NameModel.length];
            if (_.isObject(cloneData)) {
                tokenArray.relocateSheet(cloneData.oldSheet, cloneData.newSheet);
            }
        }(arguments));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            tokenArray.destroy();
            sheetModel = tokenArray = null;
        });

    }}); // class NameModel

    // class NameCollection ===================================================

    /**
     * Collects all global defined names of the spreadsheet document, or the
     * defined names contained in a single sheet of the document.
     *
     * Triggers the following events:
     * - 'insert:name'
     *      After a new defined name has been inserted into the collection.
     *      Event handlers receive the name of the new defined name.
     * - 'delete:name'
     *      After a defined name has been deleted from the collection. Event
     *      handlers receive the name of the deleted defined name.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this collection instance.
     *
     * @param {SheetModel} [sheetModel]
     *  The model of the sheet that contains this name collection. If omitted,
     *  the name collection is intended to store global document names.
     */
    function NameCollection(app, sheetModel) {

        var // the definitions of all defined names, mapped by upper-case name
            entries = {};

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // methods ------------------------------------------------------------

        /**
         * Returns the token array representing the formula definition of the
         * specified defined name.
         *
         * @param {String} name
         *  The name of the new defined name whose token array will be
         *  returned.
         *
         * @returns {TokenArray|Null}
         *  The token array containing the name definition of the specified
         *  name, if existing; otherwise null.
         */
        this.getTokenArray = function (name) {
            var entry = entries[name.toUpperCase()];
            return entry ? entry.getTokenArray() : null;
        };

        /**
         * Creates and stores a new defined name, and triggers an 'insert:name'
         * event.
         *
         * @param {String} name
         *  The name of the new defined name.
         *
         * @param {String} formula
         *  The formula expression bound to the defined name.
         *
         * @returns {Boolean}
         *  Whether the defined name has been created successfully.
         */
        this.insertName = function (name, formula) {

            // validate input parameters (name must not exist)
            name = name.toUpperCase();
            if ((name.length === 0) || (formula.length === 0) || (name in entries)) {
                return false;
            }

            // create a new collection entry and notify listeners
            entries[name] = new NameModel(app, sheetModel, formula);
            this.trigger('insert:name', name);
            return true;
        };

        /**
         * Deletes a defined name from this collection, and triggers a
         * 'delete:name' event.
         *
         * @param {String} name
         *  The name of the new defined name.
         *
         * @returns {Boolean}
         *  Whether the defined name has been deleted successfully.
         */
        this.deleteName = function (name) {

            // validate input parameters (name must exist)
            name = name.toUpperCase();
            if (!(name in entries)) { return false; }

            // delete the collection entry and notify listeners
            entries[name].destroy();
            delete entries[name];
            this.trigger('delete:name', name);
            return true;
        };

        // initialization -----------------------------------------------------

        // constructor called from BaseObject.clone()
        this.registerCloneConstructor(function (newSheetModel, newSheet) {
            // pass own entries as hidden argument to the constructor
            return new NameCollection(app, newSheetModel, { entries: entries, newSheet: newSheet });
        });

        // clone collection entries passed as hidden argument to the c'tor
        (function (args) {
            var cloneData = args[NameCollection.length];
            if (_.isObject(cloneData)) {
                entries = ModelObject.cloneMap(cloneData.entries, sheetModel, cloneData.newSheet);
            }
        }(arguments));

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _(entries).invoke('destroy');
            sheetModel = entries = null;
        });

    } // class NameCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: NameCollection });

});
