/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/manualcolorchart',
        ['io.ox/office/tk/utils',
         'io.ox/office/drawinglayer/view/chartstyleutil'], function (Utils, ChartStyleUtil) {

    'use strict';

    var BACKGROUNDTRANSFORMATION = ChartStyleUtil.getBackgroundTransformation();

    var STYLESET = ChartStyleUtil.getStyleSet();

    var COLORSET = ChartStyleUtil.getColorSet();

    function ManualColorChart(app, chartmodel, data) {
        var specialVaryColor = false;

        data.colorSet = null;

        // private methods ------------------------------------------------------------

        function getColorOfPattern(meth, index, schemeColors, variations) {
            var variationIndex = null;
            var colorIndex = null;
            if (meth === 'cycle') {
                variationIndex = index % variations.length;
                colorIndex = index % schemeColors.length;
            } else {
                Utils.warn('cant handle color meth:' + meth);
                return null;
            }

            var color = schemeColors[colorIndex];
            var variation = variations[variationIndex];

            var tmpColor = {};
            tmpColor.type = color.type;
            tmpColor.value = color.value;
            if (variation) {
                tmpColor.transformations = variation.transformations;
            }
            return tmpColor;
        }

        function getColor(index) {
            var attributes = chartmodel.getMergedAttributes();

            var colors = attributes.chart.chartColors;

            var result = null;

            if (colors) {
                //modern color pattern
                result = getColorOfPattern(colors.meth, index, colors.schemeClr, colors.variation);
            } else if (attributes.chart.chartStyleId) {
                var styleId = attributes.chart.chartStyleId - 1;
                var colorSet = COLORSET[styleId % 8];

                var res = getColorOfPattern('cycle', index, colorSet.colors, colorSet.variations);
                result = {
                    color: res
                };
            }

            return result;
        }

        function getStyleSet(index, color) {
            var attributes = chartmodel.getMergedAttributes();

            var result = color;

            if (attributes.chart.chartStyleId) {
                var styleId = attributes.chart.chartStyleId - 1;
                var styleSet = STYLESET[(styleId / 8)|0];

                var dataSeries = data.data[index];
                dataSeries.bevelEnabled = styleSet.bevelEnabled;
            }

            if (index === 0) {
                data.backgroundColor = null;
                if (attributes.chart.chartStyleId && result) {

                    var styleId = attributes.chart.chartStyleId - 1;
                    var styleSet = STYLESET[(styleId / 8)|0];

                    var cells = app.getModel().getStyleSheets('cell');

                    var givenStyle = styleSet.bg;
                    if (!givenStyle) {
                        givenStyle = {type: result.color.type, value: result.color.value, transformations: BACKGROUNDTRANSFORMATION};

                    }

                    var bg = cells.getCssColor(givenStyle, 'fill');
                    data.backgroundColor = bg;

                }
            }
        }

        function getVaryColor() {
            var attributes = chartmodel.getMergedAttributes();
            var vary = attributes.chart.varyColors;
            if (!specialVaryColor && data.data.length > 1) {
                vary = false;
            }
            return vary;
        }

        // public functions ------------------------------------------------------------

        this.isVaryColorEnabled = function () {
            if (specialVaryColor) {
                return true;
            } else {
                return data.data.length === 1;
            }
        };

        this.setSpecialVaryColor = function (vary) {
            specialVaryColor = vary;
        };

        this.handleColor = function (attrs, index) {
            var vary = getVaryColor();
            var fill = attrs.fill;
            var dataSeries = data.data[index];
            var attributes = chartmodel.getMergedAttributes();
            var cells = app.getModel().getStyleSheets('cell');

            _(dataSeries.dataPoints).each(function (dataPoint) {
                dataPoint.color = null;
            });

            if (!fill || fill.color.type === 'auto') {
                if (vary) {
                    dataSeries.color = null;
                    _(dataSeries.dataPoints).each(function (dataPoint, pi) {
                        var colorHolder = getColor(pi);
                        if (pi === 0) {
                            getStyleSet(0, colorHolder);
                        }
                        dataPoint.color = cells.getCssColor(colorHolder.color, 'fill');
                    });
                } else {
                    fill = getColor(index);
                }
            }

            if (fill && fill.color) {
                dataSeries.color = cells.getCssColor(fill.color, 'fill');
            }

            if (!vary) {
                getStyleSet(index, fill);
            }
            if (index === 0) {
                var bg = attributes.fill.color;
                if (!data.backgroundColor && bg && bg.type != 'auto') {
                    var cells = app.getModel().getStyleSheets('cell');
                    data.backgroundColor = cells.getCssColor(bg, 'fill');
                }
            }
        };

    } // class ManualColorChart

    // exports ================================================================

    return _.makeExtendable(ManualColorChart);

});
