/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/dataseriesmodel',
        ['io.ox/office/editframework/model/format/attributedmodel',
         'io.ox/office/spreadsheet/model/tokenarray'], function (AttributedModel, TokenArray) {

    'use strict';

    var NAMES_ORDER = ['names', 'title', 'values', 'bubbles'];

    /**
     * A class that implements a dataseries model to be inserted into a chart model.
     *
     * Series can have formulas 'Title', 'Names', 'Values' and 'Bubbles'
     * on 'change:attributes' it refreshs all formulas to TokenArrays
     *
     * Additionally it can have 'fill' and 'line' for styling
     *
     * @constructor
     *
     * @extends DrawingModel
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this chart model.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model that contains the chart object with this data series.
     *
     * @param {Object} [initAttributes]
     *  Initial formatting attribute set for this data series model.
     */
    function DataSeriesModel(app, sheetModel, initAttributes) {
        var self = this;
        var tokens = {};

        // base constructor ---------------------------------------------------

        AttributedModel.call(this, app, {}, { additionalFamilies: ['series', 'fill'] });

        function changeAttributes() {
            var attrs = self.getMergedAttributes();
            _.each(NAMES_ORDER, function (name) {
                var formula = attrs.series[name];
                if (formula && _.isArray(formula)) {
                    formula = null;
                }
                if (!formula) {
                    if (token) {
                        token.destroy();
                        delete tokens[name];
                    }
                } else {
                    var token = tokens[name];
                    if (!token) {
                        token = new TokenArray(app, sheetModel);
                        tokens[name] = token;
                    }
                    token.parseFormula(formula);
                }
            });
        }

        this.on('change:attributes', changeAttributes);

        this.getFormula = function () {
            var sep = app.getListSeparator();
            var formula = '';

            this.iterateTokens(function (token) {
                if (formula.length) {
                    formula += sep;
                }
                formula += token.getFormula();
            });

            return formula;
        };

        this.checkRangeInsideInterval = function (sheetModelIndex, interval, insert, columns) {
            var na = null;

            this.iterateTokens(function (token, name) {
                var change = token.transformFormula(sheetModelIndex, interval, insert, columns);
                if (change) {
                    if (!na) {
                        na = {};
                    }
                    na[name] = token.getFormula();
                }
            });

            if (na) {
                this.setAttributes({series: na});
                return true;
            } else {
                return false;
            }
        };

        this.iterateTokens = function (fn) {
            _.each(tokens, function (token, name) {
                if (token) {
                    fn(token, name);
                }
            });
        };

        this.setA = function (att) {
            att.fill = att.series.fill;
            att.line = att.series.line;
            delete att.series.fill;
            delete att.series.line;
            this.setAttributes(att);
        };

        // destroy all class members on destruction
        this.registerDestructor(function () {
            _.each(NAMES_ORDER, function (name) {
                var token = tokens[name];
                if (token) {
                    token.destroy();
                }
            });
            tokens = null;
            self = null;

        });


        this.setA(initAttributes);
        changeAttributes();
    }

    return AttributedModel.extend({ constructor: DataSeriesModel });

});
