/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2011
 * Mail: info@open-xchange.com
 *
 * @author Kai Ahrens <kai.ahrens@open-xchange.com>
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/app/rtconnection',
    ['io.ox/core/event',
     'io.ox/realtime/groups',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/object/triggerobject'
    ], function (Events, RTGroups, Utils, TriggerObject) {

    'use strict';

    var // names of internal events for the real-time connection state
        INTERNAL_EVENTS = 'online offline reset',

        // names of runtime requests expecting an answer
        REQUEST_EVENTS = 'addresource getdocument',

        // names of runtime push messages
        PUSH_EVENTS = 'update hangup preparelosingeditrights declinedacquireeditrights acceptedacquireeditrights',

        // max number of action send calls until an update must be received
        MAX_NUM_OF_APPLYACTIONS_UNTIL_UPDATE = 50,

        // repeated time to check for pending ACKs
        CHECK_PENDING_INTERVAL = 30000;

    // class RTConnection =====================================================

    /**
     * Represents the connection to the real-time framework, used to send
     * message to and receive message from the server.
     *
     * Triggers the following events:
     * - 'update': Update notifications containing actions created by remote
     *  clients, and state information for the application (read-only, name and
     *  identifier of the current editor, etc.).
     *
     * @extends TriggerObject
     *
     * @param {EditApplication} app
     *  The application containing this operations queue.
     *
     * @param {Object} [initOptions]
     *  A map of options to be called for certain states of the realtime
     *  connection.
     *  @param {Number} [initOptions.realTimeDelay=700]
     *      The duration in milliseconds the realtime framework will collect
     *      messages before sending them to the server.
     */
    function RTConnection(app, initOptions) {

        var // self reference
            self = this,

            // file descriptor of the edited document
            file = app.getFileDescriptor(),

            // create the core real-time group instance for the document
            rtGroup = RTGroups.getGroup('synthetic.office://operations/' + file.folder_id + '.' + file.id),

            // private event hub for internal event handling
            eventHub = new Events(),

            // duration for the realtime framework collecting messages before sending
            realTimeDelay = Utils.getIntegerOption(initOptions, 'realTimeDelay', 700, 0),

            traceActions = {},

            // pending deferreds
            pendingSendAcks = [],

            // count sending actions
            applyActionsSent = 0,

            // pending connects deferred
            connectDeferreds = [],

            // in prepare losing edit rights, for debug purposes only!
            inPrepareLosingEditRights = false;

        // base constructor ---------------------------------------------------

        TriggerObject.call(this);

        // private methods ----------------------------------------------------

        /**
         * Log the data
         */
        function debugLogUpdateNotification(data) {
            var // the operations array (may be missing completely)
                actions = Utils.getArrayOption(data, 'actions', []),
                // editing user id
                editUserId = Utils.getStringOption(data, 'editUserId', ''),
                // server osn
                serverOSN = Utils.getIntegerOption(data, 'serverOSN', -1),
                // has errors
                hasErrors = Utils.getBooleanOption(data, 'hasErrors', false),
                // server osn
                activeClients = Utils.getIntegerOption(data, 'activeClients', 0),
                // write protected
                writeProtected = Utils.getBooleanOption(data, 'writeProtected', false),
                // editing user id
                lockedByUser = Utils.getStringOption(data, 'lockedByUser', ''),
                // output string
                output = '';

            // output content of update notification
            output = 'editUserId: ' + editUserId + '; ';
            output += 'server osn: ' + serverOSN + '; ';
            output += 'active clients: ' + activeClients + '; ';
            output += 'hasErrors: ' + hasErrors + '; ';
            output += 'writeProtected: ' + writeProtected + '; ';
            output += 'locked by user: ' + lockedByUser;
            Utils.log(output);

            // check that all array elements are objects
            if (_(actions).all(_.isObject)) {
                try {
                    Utils.log('Last remote action received by RT-Update: ' + JSON.stringify(actions[actions.length - 1]));
                } catch (ex) {
                    // do nothing
                }
            }
        }

        /**
         * Register events for specific RT actions.
         * The event handler 'eventName' is triggered, when
         * an RT event with the action name 'eventName' is received.
         * The triggered eventName will be equal to the 'actionName'.
         * The eventHandler will be called with the received payload data
         */
        function registerRTEvents(eventNames) {
            _(eventNames.split(/\s+/)).each(function (eventName) {
                rtGroup.on('receive', function (event, stanza) {
                    _(stanza.getAll('office', eventName)).each(function (payload) {
                        RTConnection.log('RTConnection.eventHub.trigger: office:' + eventName);
                        eventHub.trigger(eventName, payload.data);
                    });
                });
            });
        }

        /**
         * Determines whether a certain action should be traced on its way
         * through the realtime backend.
         *
         * @returns {Boolean}
         *  Whether to trace the stanzas for the specified action.
         */
        function shouldTrace(action) {
            return traceActions[action.toLowerCase()];
        }

        /**
         * Checks the pending ACKs for their pending time. Outputs a log
         * entry if the pending time >= CHECK_PENDING_INTERVAL.
         * Debug only function.
         */
        function checkPendingAcks() {
            var // current time
                now = new Date();

            _(pendingSendAcks).each(function (entry) {
                var timeSpan;

                if (entry.deferred && entry.deferred.state() === 'pending') {
                    timeSpan = now.getTime() - entry.timeStamp.getTime();
                    if (timeSpan > CHECK_PENDING_INTERVAL) {
                        Utils.log('RTConnection.send: ACK pending for ' + Math.round(timeSpan / 1000) + ' sec,' + ' action=' + entry.action);
                    }
                }
            });
        }

       /**
         * Send a message to the connected RT object,
         * waiting for the acknowledge to this call.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  acknowledge for this request will have arrived, or
         *  rejected after a time out otherwise.
         */
        function send(action, element, data) {

            var payloads = [{ element: 'action', data: action }],
                defer = null;

            if (_.isString(element) && !_.isUndefined(data)) {
                payloads.push({
                    namespace: 'office',
                    element: element,
                    data: data,
                    verbatim: true
                });
            }

            defer = rtGroup.send({
                element: 'message',
                payloads: payloads,
                trace: shouldTrace(action),
                bufferinterval: realTimeDelay
            });

            if (RTConnection.debug) {
                // add logging code for pending ACKs bound to sends
                pendingSendAcks.push({ deferred: defer, timeStamp: new Date(), action: action });
                defer.always(function () {
                    var // current time
                        now = new Date(),
                        // time span
                        timeSpan = 0,
                        // i
                        i = 0;

                    for (i = 0; i < pendingSendAcks.length; i++) {
                        if (pendingSendAcks[i].deferred === defer) {
                            timeSpan = now.getTime() - pendingSendAcks[i].timeStamp.getTime();
                            // log message if deferred needed at least CHECK_PENDING_INTERVAL time
                            // to be resolved/rejected
                            if (timeSpan > CHECK_PENDING_INTERVAL) {
                                RTConnection.log('RTConnection.send: pending ACK resolved/rejected after ' + Math.round(timeSpan / 1000) + ' seconds.');
                            }
                            // remove entry from pending send acks array
                            pendingSendAcks.splice(i, 1);
                            break;
                        }
                    }

                    if (pendingSendAcks.length > 0) {
                        RTConnection.log('RTConnection.send: pending ACKs for send found: ' + pendingSendAcks.length);
                    }
                });
            }

            // Time out detection now uses the deferred to trigger an optional timeout handler
            defer.fail(function () {
                RTConnection.error('Send message with RT failed, payloads: ' + payloads.toString());
                // RT calls us even after connection, rtgroup are destroyed and references are cleared.
                if (!self.destroyed) { self.trigger('timeout'); }
            });

            return defer.promise();
        }

        /**
         * Send a message request to the connected RT object,
         * returning a promise object that will be resolved,
         * if the result data to this request has arrived.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  result for this request has arrived, or rejected otherwise.
         */
        function sendQuery(action, element, data) {
            var payloads = [{ element: 'action', data: action }],
                defer;

            if (_.isString(element) && !_.isUndefined(data)) {
                payloads.push({
                    namespace: 'office',
                    element: element,
                    data: data,
                    verbatim: true
                });
            }

            defer = rtGroup.query({
                element: 'message',
                payloads: payloads,
                trace: shouldTrace(action),
                bufferinterval: realTimeDelay
            });

            // Time out detection now uses the deferred to trigger an optional timeout handler
            defer.fail(function () {
                RTConnection.error('Send query message with RT failed, payloads: ' + payloads.toString());
                // RT calls us even after connection, rtgroup are destroyed and references are cleared.
                if (!self.destroyed) { self.trigger('timeout'); }
            });

            return defer.promise();
        }

        /**
         * Send a message request to the connected RT object,
         * returning a deferred object that will be resolved,
         * if the result data to this request will have arrived.
         * @deprecated use sendQuery instead
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  result for this request has arrived, or rejected otherwise.
         */
        function sendRequest(action, element, data) {

            var // the result Deferred object
                def = $.Deferred();

            data = Utils.extendOptions(data, { uniqueId: _.uniqueId() });

            // resolves the Deferred object with the 'action' result
            function requestHandler(event, resultData) {
                // stop listening to 'requestHandler' events if we received exactly
                // one event with the message id we sent with the initial request
                if (_.isObject(resultData)) {
                    RTConnection.log('RTConnection.sendRequest: Answer for sendRequest arrived: ' + action + ' , expected uniqueId: ' + data.uniqueId + ', received uniqueId:' + resultData.uniqueId);
                    if (_.isString(resultData.uniqueId) && (resultData.uniqueId === data.uniqueId)) {
                        stopListening();
                        RTConnection.log('RTConnection.sendRequest: resolve deferred with resultData');
                        def.resolve(resultData);
                    }
                } else {
                    RTConnection.error('RTConnection.sendRequest: Unexpected answer - deferred rejected');
                    def.reject();
                }
            }

            // stops listening to action events
            function stopListening() {
                eventHub.off(action, requestHandler);
            }

            // resolves the Deferred object with the 'action' result
            eventHub.on(action, requestHandler);

            // connect and listen to the 'action' result
            RTConnection.log('RTConnection.sendRequest: Sending request: ' + action + ', uniqueId: ' + data.uniqueId);
            send(action, element, data).fail(function () {
                if (def.state() === 'pending') {
                    def.reject({cause: 'timeout'});
                }
            });
            RTConnection.log('RTConnection.sendRequest: Waiting for answer to request ' + action + ' ,uniqueId: ' + data.uniqueId);

            // return a Promise with an abort() extension that disconnects the
            // event listener, leaving the Deferred in pending state
            return _.extend(def.promise(), { abort: stopListening });
        }

        /**
         * Handles the response from the RT low-level code and extract the
         * data part from, providing it as result via the deferred/promise.
         *
         * @param {jQuery.Deferred|jQuery.Promise} orgDeferred
         *  The original promise or deferred provided by the RT low-level
         *  code.
         *
         * @param {Number} [timeout]
         *  If specified, the delay time in milliseconds, after the Promise
         *  returned by this method will be rejected, if the passed original
         *  Deferred object is still pending.
         *
         * @returns {jQuery.Deferred}
         *  The Deferred object that will be resolved if the result for the
         *  original request has arrived, or rejected otherwise.
         */
        function handleResponse(orgDeferred, timeout) {

            var // the Deferred object to be returned
                def = $.Deferred();

            orgDeferred.then(function (response) {
                var payloads = Utils.getArrayOption(response, 'payloads', []);
                if (self.destroyed) {
                    // reject deferred after destroy
                    def.reject({cause: 'closing'});
                }
                if (payloads.length > 0) {
                    // Extract the data and provide it
                    def.resolve(payloads[0].data);
                } else {
                    // Response from server is not in the expected format, therefore
                    // we reject the deferred here.
                    def.reject(response);
                }
            }, function (response) {
                var payloads = Utils.getArrayOption(response, 'payloads', []);
                if (payloads.length > 0) {
                    // Extract data even for the rejected case
                    def.reject(payloads[0].data);
                } else {
                    // If we don't find a payload just provide the original response
                    def.reject(response);
                }
            });

            // prevent endless waiting
            if (_.isNumber(timeout) && (timeout > 0)) {
                app.executeDelayed(function () {
                    if (def.state() === 'pending') {
                        RTConnection.log('Timeout reached for RT request, timeout=' + Math.round(timeout / 1000) + ' seconds.');
                        def.reject({ cause: 'timeout' });
                    }
                }, { delay: timeout });
            }

            return def;
        }

        /**
         * Handles the response from the RT low-level code and adds timeout
         *
         * @param {jQuery.Deferred|jQuery.Promise} orgDeferred
         *  The original promise or deferred provided by the RT low-level
         *  code.
         *
         * @param {Number} [timeout]
         *  If specified, the delay time in milliseconds, after the Promise
         *  returned by this method will be rejected, if the passed original
         *  Deferred object is still pending.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved if the
         *  result for the original request has arrived, or rejected otherwise.
         */
        function handleGenericResponse(orgDeferred, timeout) {
            var // the Deferred object to be returned
                def = $.Deferred();

            orgDeferred.then(function (data) { def.resolve(data); }, function (data) { def.reject(data); });

            // prevent endless waiting
            if (_.isNumber(timeout) && (timeout > 0)) {
                app.executeDelayed(function () { def.reject({ cause: 'timeout' }); }, { delay: timeout });
            }

            return def.promise();
        }

        // methods ------------------------------------------------------------

        /**
         * Connects the real-time group and sends the initial request for the
         * document import operations.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  getActions results.
         */
        this.connect = function () {
            var connectDeferred = null;

            RTConnection.log('RTConnection.connect called');
            connectDeferred = handleResponse(rtGroup.join({expectWelcomeMessage: true}));
            // Remember *connect* deferred to be able to reject them if user
            // closes document while we are still connecting to the realtime group!
            connectDeferreds.push(connectDeferred);

            return connectDeferred;
        };

        /**
         * Send the given operations to the connected RT object
         * for further distribution to other connected clients
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  ACK of the internal send arrives.
         */
        this.sendActions = function (actions) {
            var tmp;

            RTConnection.log('RTConnection.applyactions called');
            if (RTConnection.debug) {
                if (inPrepareLosingEditRights && actions && (actions.length > 0)) {
                    tmp = actions[actions.length - 1].operations;
                    if (tmp && tmp.length > 0) {
                        RTConnection.log('preparelosingeditrights: applyactions, last operation: ' + JSON.stringify(_.last(tmp)));
                    }
                }
            }
            applyActionsSent++;
            if (applyActionsSent > MAX_NUM_OF_APPLYACTIONS_UNTIL_UPDATE) {
                self.trigger('noUpdateFromServer');
                // reset counter after we have triggered the missing update
                applyActionsSent = 0;
            }
            return send('applyactions', 'actions', actions);
        };

        /**
         * Sends a real-time request to the server and waits for a response.
         *
         * @param {String} action
         *  The identifier of the action sent to the server.
         *
         * @param {Number} [timeout]
         *  If specified, the delay time in milliseconds, after the Promise
         *  returned by this method will be rejected, if the query is still
         *  pending.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  answer of the server request, or rejected on error or timeout.
         */
        this.sendQuery = function (action, timeout) {
            RTConnection.log('RTConnection.sendQuery(' + action + ')');
            return handleResponse(sendQuery(action), timeout);
        };

        /**
         * Initiates flushing of the document using a
         * synchronous request.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  answer for flushDocument from the server arrives.
         */
        this.flushDocument = function () {
            RTConnection.log('RTConnection.flushDocument called');
            return handleGenericResponse(sendQuery('flushdocument'), 20000);
        };

        /**
         * Add the given resource id to the connected RT object
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  result data of the request.
         */
        this.addResourceId = function (resourceId) {
            RTConnection.log('RTConnection.addResourceId called');
            return sendRequest('addresource', 'resource', { resourceid: resourceId });
        };

        /**
         * Get the resource id of the connected RT object for the specified document
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  result data of the request.
         */
        this.getDocumentId = function (data) {
            RTConnection.log('RTConnection.getDocumentId called');
            return sendRequest('getdocument', 'document', Utils.extendOptions(data, { returntype: 'resourceid' }));
        };

        /**
         * Send our wish to acquire the edit rights to the connected RT object
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  ACK of the internal send arrives.
         */
        this.acquireEditRights = function () {
            RTConnection.log('RTConnection.acquireedit called');
            return send('acquireedit', 'state', { osn: app.getModel().getOperationStateNumber() });
        };

        /**
         * Acknowledge to the server that our preparation to lose the
         * edit rights are completed.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  ACK of the internal send arrives.
         */
        this.canLoseEditRights = function () {
            RTConnection.log('RTConnection.canloseeditrights called');
            if (RTConnection.debug) {
                inPrepareLosingEditRights = false;
            }
            return send('canloseeditrights', 'state', { osn: app.getModel().getOperationStateNumber() });
        };

        /**
         * Close the document at the connected RT object
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved with the
         *  closing document status of the request.
         */
        this.closeDocument = function () {
            RTConnection.log('RTConnection.closeDocument called');
            return handleResponse(rtGroup.leave({ expectSignOffMessage: true }), 20000);
        };

        // initialization -----------------------------------------------------

        RTConnection.log('RTConnection initialization');
        if (RTConnection.debug) {
            app.repeatDelayed(checkPendingAcks, {delay: CHECK_PENDING_INTERVAL});
        }

        // read traceable actions from page URL
        if (_.url.hash('office:trace')) {
            _(_.url.hash('office:trace').split(/\s*,\s*/)).each(function (action) {
                traceActions[action.toLowerCase()] = true;
            });
        }

        // register the internal event hub for request events and custom runtime events
        registerRTEvents(REQUEST_EVENTS + ' ' + PUSH_EVENTS);

        // forward runtime realtime events to own listeners
        rtGroup.on(INTERNAL_EVENTS, function (event) {
            self.trigger(event.type);
        });

        // forward OX Documents runtime events to own listeners
        eventHub.on(PUSH_EVENTS, function (event, data) {
            if (event.type === 'update') {
                applyActionsSent = 0;
                if (RTConnection.debug) {
                    debugLogUpdateNotification(data);
                }
            } else if (event.type === 'preparelosingeditrights' && RTConnection.debug) {
                inPrepareLosingEditRights = true;
            }
            self.trigger(event.type, data);
        });

        // disconnect from the RT object, no further calls should be made at this object from now on
        this.registerDestructor(function () {

            // check all connect() deferred objects and reject them if they are still pending
            if (connectDeferreds) {
                _(connectDeferreds).each(function (deferred) {
                    if (deferred && deferred.state() === 'pending') {
                        deferred.reject({cause: 'closing'});
                    }
                });
                connectDeferreds = [];
            }

            // destroy realtime group
            try {
                rtGroup.destroy();
                rtGroup = null;
                RTConnection.log('RTConnection.destroy(): RT group destroyed');
            } catch (ex) {
                Utils.exception(ex);
            }

            // destroy private event hub
            eventHub.destroy();

            if (RTConnection.debug) {
                pendingSendAcks = [];
            }
        });

    } // class RTConnection

    // static methods ---------------------------------------------------------

    RTConnection.log = function (msg) { if (RTConnection.debug) { Utils.log(msg); } };
    RTConnection.info = function (msg) { if (RTConnection.debug) { Utils.info(msg); } };
    RTConnection.warn = function (msg) { if (RTConnection.debug) { Utils.warn(msg); } };
    RTConnection.error = function (msg) { if (RTConnection.debug) { Utils.error(msg); } };

    // global initialization ==================================================

    // enable realtime debugging
    require(['io.ox/realtime/rt', 'io.ox/office/tk/config']).done(function (Realtime, Config) {
        // whether to show console output of real-time core
        Realtime.debug = Config.isRealtimeDebug();
        // whether to show console output of this RTConnection class
        RTConnection.debug = Config.isRealtimeDebug();
        // log time stamps in all console messages
        Utils.LOG_TIME_STAMP = Utils.LOG_TIME_STAMP || RTConnection.debug;
    });

    // exports ================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: RTConnection });

});
