/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2009-2011 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at packager/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 *
 * Portions Copyright 2012 OPEN-XCHANGE, licensed under GPL Version 2.
 */

package com.openexchange.http.grizzly.service.http;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Dictionary;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import javax.servlet.Servlet;
import javax.servlet.ServletException;
import org.glassfish.grizzly.http.server.HttpHandler;
import org.glassfish.grizzly.http.server.Request;
import org.glassfish.grizzly.http.server.Response;
import org.glassfish.grizzly.http.server.util.MappingData;
import org.glassfish.grizzly.http.util.HttpStatus;
import org.osgi.framework.Bundle;
import org.osgi.service.http.HttpContext;
import org.osgi.service.http.HttpService;
import org.osgi.service.http.NamespaceException;
import com.openexchange.exception.OXException;
import com.openexchange.http.grizzly.servletfilter.RequestReportingFilter;
import com.openexchange.http.grizzly.servletfilter.WrappingFilter;
import com.openexchange.log.LogProperties;
import com.openexchange.tools.exceptions.ExceptionUtils;

/**
 * OSGi Main HttpHandler.
 * <p/>
 * Dispatching HttpHandler. Grizzly integration.
 * <p/>
 * Responsibilities:
 * <ul>
 * <li>Manages registration data.</li>
 * <li>Dispatching {@link HttpHandler#service(Request, Response)} method call to registered {@link HttpHandler}s.</li>
 * </ul>
 *
 * @author Hubert Iwaniuk
 * @author <a href="mailto:marc.arens@open-xchange.com">Marc Arens</a>
 */
public class OSGiMainHandler extends HttpHandler implements OSGiHandler {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(OSGiMainHandler.class);

    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();

    private final Bundle bundle;

    private final OSGiCleanMapper mapper;

    /**
     * Constructor.
     *
     * @param logger Logger utility.
     * @param bundle Bundle that we create if for, for local data reference.
     */
    public OSGiMainHandler(Bundle bundle) {
        this.bundle = bundle;
        this.mapper = new OSGiCleanMapper();
    }

    /**
     * Service method dispatching to registered handlers.
     * <p/>
     * {@inheritDoc}
     */
    @Override
    public void service(Request request, Response response) throws Exception {
        boolean invoked = false;
        String alias = request.getDecodedRequestURI();
        String originalAlias = alias;
        LOG.debug("Serviceing URI: {}", alias);
        // first lookup needs to be done for full match.
        boolean cutOff = false;
        while (true) {
            LOG.debug("CutOff: {}, alias: {}", cutOff, alias);
            alias = OSGiCleanMapper.map(alias, cutOff);
            if (alias == null) {
                if (cutOff) {
                    // not found
                    break;
                } else {
                    // switching to reducing mapping mode (removing after last '/' and searching)
                    LOG.debug("Switching to reducing mapping mode.");
                    cutOff = true;
                    alias = originalAlias;
                }
            } else {
                HttpHandler httpHandler = OSGiCleanMapper.getHttpHandler(alias);

                ((OSGiHandler) httpHandler).getProcessingLock().lock();
                try {
                    updateMappingInfo(request, alias, originalAlias);

                    httpHandler.service(request, response);
                } catch (Throwable t) {
                    ExceptionUtils.handleThrowable(t);
                    StringBuilder logBuilder = new StringBuilder(128).append("Error processing request:\n");
                    logBuilder.append(LogProperties.getAndPrettyPrint(LogProperties.Name.SESSION_SESSION));
                    appendRequestInfo(logBuilder, request);
                    LOG.error(logBuilder.toString(), t);
                    // 500 - Internal Server Error
                    response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR_500);
                } finally {
                    ((OSGiHandler) httpHandler).getProcessingLock().unlock();
                }
                invoked = true;
                if (response.getStatus() != 404) {
                    break;
                } else if ("/".equals(alias)) {
                    // 404 in "/", cutoff algo will not escape this one.
                    break;
                } else if (!cutOff) {
                    // not found and haven't run in cutoff mode
                    cutOff = true;
                }
            }
        }
        if (!invoked) {
            response.setStatus(HttpStatus.NOT_FOUND_404);
            try {
                customizedErrorPage(request, response);
            } catch (Exception e) {
                LOG.warn("Failed to commit 404 status.", e);
            }
        }
    }

    /**
     * Appends request information.
     *
     * @param builder The builder to append to
     */
    private void appendRequestInfo(final StringBuilder builder, Request request) {
        builder.append("request-URI=''");
        builder.append(request.getRequestURI());
        builder.append("'', query-string=''");
        builder.append(request.getQueryString());
        builder.append("''");
    }

    /**
     * Registers {@link services.http.OSGiServletHandler} in OSGi Http Service.
     * <p/>
     * Keeps track of all registrations, takes care of thread safety.
     *
     * @param alias Alias to register, if wrong value than throws {@link org.osgi.service.http.NamespaceException}.
     * @param servlet Servlet to register under alias, if fails to {@link javax.servlet.Servlet#init(javax.servlet.ServletConfig)} throws
     *            {@link javax.servlet.ServletException}.
     * @param initparams Initial parameters to populate {@link javax.servlet.ServletContext} with.
     * @param context OSGi {@link org.osgi.service.http.HttpContext}, provides mime handling, security and bundle specific resource access.
     * @param httpService Used to {@link HttpService#createDefaultHttpContext()} if needed.
     * @throws org.osgi.service.http.NamespaceException If alias was invalid or already registered.
     * @throws javax.servlet.ServletException If {@link javax.servlet.Servlet#init(javax.servlet.ServletConfig)} fails.
     * @throws OXException
     */
    public void registerServletHandler(final String alias, Servlet servlet, Dictionary initparams, HttpContext context, HttpService httpService) throws NamespaceException, ServletException {

        ReentrantLock lock = OSGiCleanMapper.getLock();
        lock.lock();
        try {
            // TODO: clean up OX servlet structure so we can apply alias and servlet validation

            validateAlias4RegOk(alias);

            /*
             * Currently only checks if servlet is already registered. This prevents the same servlet with different aliases. Disabled until
             * we don't have to register the DispatcherServlet multiple times under different aliases.
             */
            // validateServlet4RegOk(servlet, alias);

            /*
             * A context provides methods to getResources, Mimetypes and handle security. It's implemented by users of the httpservice.
             * Servlets with the same HttpContext share the same ServletContext. If the reference is null we use a default context.
             */
            if (context == null) {
                LOG.debug("No HttpContext provided, creating default");
                context = httpService.createDefaultHttpContext();
            }
            OSGiServletHandler servletHandler = findOrCreateOSGiServletHandler(servlet, context, initparams);
            servletHandler.setServletPath(alias);
            addServletFilters(servletHandler);

            /*
             * Servlet would be started several times if registered with multiple aliases. Starting means: 1. Set ContextPath 2. Instantiate
             * Servlet if null 3. Call init(config) on the Servlet.
             */
            servletHandler.startServlet(); // this might throw ServletException, throw it to offending bundle.

            // Add the servletPath and the OSGiServletHandler to the backing map.
            mapper.addHttpHandler(alias, servletHandler);
        } finally {
            lock.unlock();
        }
    }

    /**
     * Add our default set of Filters to the ServletHandler.
     *
     * @param servletHandler The ServletHandler with the FilterChain
     * @throws ServletException
     */
    private void addServletFilters(OSGiServletHandler servletHandler) throws ServletException {
        // wrap it
        servletHandler.addFilter(new WrappingFilter(), WrappingFilter.class.getName(), null);

        // watch it
            try {
                servletHandler.addFilter(
                    new RequestReportingFilter(),
                    RequestReportingFilter.class.getName(),
                    null);
            } catch (OXException e) {
                throw new ServletException(e);
            }

    }

    /**
     * Registers {@link OSGiResourceHandler} in OSGi Http Service.
     * <p/>
     * Keeps truck of all registrations, takes care of thread safety.
     *
     * @param alias Alias to register, if wrong value than throws {@link NamespaceException}.
     * @param context OSGi {@link HttpContext}, provides mime handling, security and bundle specific resource access.
     * @param internalPrefix Prefix to map request for this alias to.
     * @param httpService Used to {@link HttpService#createDefaultHttpContext()} if needed.
     * @throws NamespaceException If alias was invalid or already registered.
     */
    public void registerResourceHandler(String alias, HttpContext context, String internalPrefix, HttpService httpService) throws NamespaceException {

        ReentrantLock lock = OSGiCleanMapper.getLock();
        lock.lock();
        try {
            validateAlias4RegOk(alias);

            if (context == null) {
                LOG.debug("No HttpContext provided, creating default");
                context = httpService.createDefaultHttpContext();
            }
            if (internalPrefix == null) {
                internalPrefix = "";
            }

            mapper.addHttpHandler(alias, new OSGiResourceHandler(alias, internalPrefix, context));
        } finally {
            lock.unlock();
        }
    }

    /**
     * Unregisters previously registered alias.
     * <p/>
     * Keeps truck of all registrations, takes care of thread safety.
     *
     * @param alias Alias to unregister, if not owning alias {@link IllegalArgumentException} is thrown.
     * @throws IllegalArgumentException If alias was not registered by calling bundle.
     */
    public void unregisterAlias(String alias) {

        ReentrantLock lock = OSGiCleanMapper.getLock();
        lock.lock();
        try {
            if (mapper.isLocalyRegisteredAlias(alias)) {
                mapper.doUnregister(alias, true);
            } else {
                LOG.warn("Bundle: {} tried to unregister not owned alias '{}{}", bundle, alias, '\'');
                throw new IllegalArgumentException(new StringBuilder(64).append("Alias '").append(alias).append(
                    "' was not registered by you.").toString());
            }
        } finally {
            lock.unlock();
        }
    }

    /**
     * Unregisters all <code>alias</code>es registered by owning bundle.
     */
    public void uregisterAllLocal() {
        LOG.info("Unregistering all aliases registered by owning bundle");

        ReentrantLock lock = OSGiCleanMapper.getLock();
        lock.lock();
        try {
            for (String alias : mapper.getLocalAliases()) {
                LOG.debug("Unregistering '{}'", alias);
                // remember not to call Servlet.destroy() owning bundle might be stopped already.
                mapper.doUnregister(alias, false);
            }
        } finally {
            lock.unlock();
        }
    }

    /**
     * Part of Shutdown sequence. Unregister and clean up.
     */
    public void unregisterAll() {
        LOG.info("Unregistering all registered aliases");

        ReentrantLock lock = OSGiCleanMapper.getLock();
        lock.lock();
        try {
            Set<String> aliases = OSGiCleanMapper.getAllAliases();
            while (!aliases.isEmpty()) {
                String alias = ((TreeSet<String>) aliases).first();
                LOG.debug("Unregistering '{}'", alias);
                // remember not to call Servlet.destroy() owning bundle might be stopped already.
                mapper.doUnregister(alias, false);
            }
        } finally {
            lock.unlock();
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ReentrantReadWriteLock.ReadLock getProcessingLock() {
        return lock.readLock();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public ReentrantReadWriteLock.WriteLock getRemovalLock() {
        return lock.writeLock();
    }

    /**
     * Chek if <code>alias</code> has been already registered.
     *
     * @param alias Alias to check.
     * @throws NamespaceException If <code>alias</code> has been registered.
     */
    private void validateAlias4RegOk(String alias) throws NamespaceException {
        if (!alias.startsWith("/")) {
            // have to start with "/"
            String msg = new StringBuilder(64).append("Invalid alias '").append(alias).append("', have to start with '/'.").toString();
            LOG.warn(msg);
            throw new NamespaceException(msg);
        }
        if (alias.length() > 1 && alias.endsWith("/")) {
            // if longer than "/", should not end with "/"
            String msg = new StringBuilder(64).append("Alias '").append(alias).append("' can't end with '/' with exception to alias '/'.").toString();
            LOG.warn(msg);
            throw new NamespaceException(msg);
        }
        if (alias.length() > 1 && alias.endsWith("*")) {
            // if longer than "/", wildcards/mappings aren't supported
            String msg = new StringBuilder(64).append("Alias '").append(alias).append(
                "' can't end with '*'. Wildcards/mappings aren't supported.").toString();
            LOG.warn(msg);
            throw new NamespaceException(msg);
        }
        if (OSGiCleanMapper.containsAlias(alias)) {
            String msg = "Alias: '" + alias + "', already registered";
            LOG.warn(msg);
            throw new NamespaceException(msg);
        }
    }

    /**
     * Check if <code>servlet</code> has been already registered.
     * <p/>
     * An instance of {@link Servlet} can be registed only once, so in case of servlet been registered before will throw
     * {@link ServletException} as specified in OSGI HttpService Spec.
     *
     * @param servlet {@link Servlet} to check if can be registered.
     * @param servletPath the path under which the servlet should be registered
     * @throws ServletException Iff <code>servlet</code> has been registered before.
     */
    private void validateServlet4RegOk(Servlet servlet, String servletPath) throws ServletException {
        if (OSGiCleanMapper.containsServlet(servlet)) {

            String msg = new StringBuilder(64).append("Servlet: '").append(servlet).append("', already registered.").append(
                "\n Tried to register under path:").append(servletPath).toString();
            LOG.warn(msg);
            throw new ServletException(msg);
        }
        throw new UnsupportedOperationException("go implement validation");
    }

    /**
     * Looks up {@link OSGiServletHandler}.
     * <p/>
     * If is already registered for <code>httpContext</code> then create new instance based on already registered. Else Create new one.
     * <p/>
     *
     * @param servlet {@link Servlet} been registered.
     * @param httpContext {@link HttpContext} used for registration.
     * @param initparams Init parameters that will be visible in {@link javax.servlet.ServletContext}.
     * @return Found or created {@link OSGiServletHandler}.
     */
    private OSGiServletHandler findOrCreateOSGiServletHandler(Servlet servlet, HttpContext httpContext, Dictionary initparams) {
        OSGiServletHandler osgiServletHandler;

        if (mapper.containsContext(httpContext)) {
            LOG.debug("Reusing ServletHandler");
            // new servlet handler for same configuration, different servlet and alias
            List<OSGiServletHandler> servletHandlers = mapper.getContext(httpContext);
            osgiServletHandler = servletHandlers.get(0).newServletHandler(servlet);
            servletHandlers.add(osgiServletHandler);
        } else {
            LOG.debug("Creating new ServletHandler");
            HashMap<String, String> params;
            if (initparams != null) {
                params = new HashMap<String, String>(initparams.size());
                Enumeration names = initparams.keys();
                while (names.hasMoreElements()) {
                    String name = (String) names.nextElement();
                    params.put(name, (String) initparams.get(name));
                }
            } else {
                params = new HashMap<String, String>(0);
            }
            osgiServletHandler = new OSGiServletHandler(servlet, httpContext, params);
            ArrayList<OSGiServletHandler> servletHandlers = new ArrayList<OSGiServletHandler>(1);
            servletHandlers.add(osgiServletHandler);
            mapper.addContext(httpContext, servletHandlers);
        }
        osgiServletHandler.addFilter(new OSGiAuthFilter(httpContext), "AuthorisationFilter", Collections.<String, String> emptyMap());
        return osgiServletHandler;
    }

    /*
     * alias = /pathTo/theServlet
     * originalAlias = /pathTo/theServlet/additional/path/info
     */
    private void updateMappingInfo(final Request request, final String alias, final String originalAlias) {
        final MappingData mappingData = request.obtainMappingData();
        //Change contextPath from "/" to the empty Sring for the default context in the httpservice
        mappingData.contextPath.setString("");

        mappingData.wrapperPath.setString(alias);

        if (alias.length() != originalAlias.length()) {
            String pathInfo = originalAlias.substring(alias.length());
            if (pathInfo.charAt(0) != '/') {
                pathInfo = "/" + pathInfo;
            }

            mappingData.pathInfo.setString(pathInfo);
        }

        updatePaths(request, mappingData);
    }
}
