/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2011 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Matthias Biggeleben <matthias.biggeleben@open-xchange.com>
 * @author Christoph Kopp <christoph.kopp@open-xchange.com>
 */

define.async('io.ox/mail/accounts/view-form',
    ['io.ox/core/tk/view',
     'io.ox/core/notifications',
     'io.ox/core/api/account',
     'text!io.ox/mail/accounts/account_detail.html',
     'settings!io.ox/mail',
     'gettext!io.ox/settings/settings'
    ], function (View, notifications, accountAPI, tmpl, settings, gt) {

    'use strict';

    var staticStrings =  {
            TITLE_ACCOUNT_SETTINGS: gt('Account settings'),
            ACCOUNT_NAME: gt('Account name'),
            PERSONAL: gt('Your name'),
            EMAIL_ADDRESS: gt('Email address'),
            UNIFIED_MAIL: gt('Use unified mail for this account'),
            SERVER_SETTINGS: gt('Server settings'),
            SERVER_TYPE: gt('Server type'),
            SERVER_SSL: gt('Use SSL connection'),
            SERVER_NAME: gt('Server name'),
            SERVER_PORT: gt('Server port'),
            LOGIN: gt('Username'),
            PASSWORD: gt('Password'),
            POP_3_REFRESH: gt('Refresh rate in minutes:'),
            EXPUNGE_MESSAGES: gt('Remove copy from server after retrieving a message'),
            DELETING_ON_LOCAL: gt('Deleting messages on local storage also deletes them on server'),
            TITLE_SERVER_OUT: gt('Outgoing server settings (SMTP)'),
            SERVER_OUT_SSL: gt('Use SSL connection'),
            SERVER_OUT_NAME: gt('Server name'),
            SERVER_OUT_PORT: gt('Server port'),
            LOGIN_AND_PASS: gt('Use username and password'),
            LOGIN_OUT: gt('Username'),
            PASSWORD_OUT: gt('Password'),
            TITLE_FOLDER_SETTINGS: gt('Folder settings'),
            FOLDER_SENT: gt('Sent folder'),
            FOLDER_TRASH: gt('Trash folder'),
            FOLDER_DRAFTS: gt('Drafts folder'),
            FOLDER_SPAM: gt('Spam folder'),
            SPAN_SELECT_FOLDER: gt('Select')
        },

        optionsServerType = _(['imap', 'pop3']).map(gt.noI18n),

        optionsRefreshRatePop = _([3, 5, 10, 15, 30, 60, 360]).map(gt.noI18n),

        portDefaults = {
            mail_port: '143',
            transport_port: '587'
        },

        //conditional defaults
        defaults = {
            pop3: {
                pop3_refresh_rate: optionsRefreshRatePop[0],
                pop3_delete_write_through: false,
                pop3_expunge_on_quit: false
            }
        },

        oldModel,

        validationCheck = function (data) {

            data = _.extend({ unified_inbox_enabled: false /*, transport_auth: true */ }, data);
            data.name = data.personal = data.primary_address;
            return accountAPI.validate(data);
        },

        returnPortMail = function (model) {
            var secure = model.get('mail_secure'),
                protocol = model.get('mail_protocol') || 'imap';

            if (protocol === 'imap') {
                return secure ? '993' : '143';
            } else {
                return secure ? '995' : '110';
            }
        },

        defaultDisplayName = '',

        AccountDetailView = Backbone.View.extend({
            tagName: 'div',
            _modelBinder: undefined,
            initialize: function () {
                // create template
                this.template = doT.template(tmpl);
                this._modelBinder = new Backbone.ModelBinder();

                //check if login and mailaddress are synced
                this.inSync = false;

                // use mail display name?
                var personal = this.model.get('personal');
                if (!personal) {
                    this.model.set('personal', defaultDisplayName);
                } else if (personal === ' ') {
                    this.model.set('personal', '');
                }

                oldModel = _.copy(this.model.attributes, true);
                // forceUpdate needed otherwise model is always valid even if inputfields contain wrong values
                Backbone.Validation.bind(this, { selector: 'data-property', forceUpdate: true });
            },
            render: function () {
                var self = this,
                    hideAccountDetails = self.model.isHidden();
                self.$el.empty().append(self.template({
                    strings: staticStrings,
                    optionsServer: optionsServerType,
                    optionsRefreshRate: optionsRefreshRatePop,
                    settings: {
                        hideAccountDetails: hideAccountDetails
                    }
                }));
                var pop3nodes = self.$el.find('.control-group.pop3');

                var defaultBindings = Backbone.ModelBinder.createDefaultBindings(self.el, 'data-property');
                self._modelBinder.bind(self.model, self.el, defaultBindings);
                //check if pop3 refresh rate needs to be displayed
                if (self.model.get('mail_protocol') !== 'pop3') {
                    pop3nodes.hide();
                }

                function syncLogin(model, value) {
                    model.set('login', value, {validate: true});
                }

                //just change port settings if this is a new account
                if (self.model.get('id') === undefined) {

                    //setting port defefaults
                    _.each(portDefaults, function (value, key) {
                        self.model.set(key, value);
                    });

                    self.model.on('change:mail_protocol', function (model) {
                        self.model.set('mail_port', returnPortMail(model));
                    });

                    self.model.on('change:mail_secure', function (model) {
                        self.model.set('mail_port', returnPortMail(model));
                    });

                    self.model.on('change:transport_secure', function (model, value) {
                        var portValue = value ? '465' : '587';
                        self.model.set('transport_port', portValue);
                    });
                }

                if (self.model.get('id') !== 0) {//check for primary account

                    //refreshrate field needs to be toggled
                    self.model.on('change:mail_protocol', function (model, value) {
                        if (value !== 'pop3') {
                            pop3nodes.hide();
                        } else {
                            //conditional defaults
                            _.each(defaults.pop3, function (value, key) {
                                if (!model.has(key))
                                    model.set(key, value);
                            });
                            pop3nodes.show();
                        }
                    });

                    //login for server should be email-address by default;
                    if (self.model.get('login') === undefined && self.model.get('primary_address') !== '') {
                        self.model.set('login', self.model.get('primary_address'), {validate: true});
                    }

                    //if login and mailadress are the same change login if mailadress changes
                    if (self.model.get('primary_address') === self.model.get('login') && !self.inSync) {
                        self.model.on('change:primary_address', syncLogin);
                        self.inSync = true;
                    }

                    //react to loginchange
                    self.model.on('change:login', function (model, value) {
                        if (value === model.get('primary_address')) {
                            if (!self.inSync) {//no need to sync if its allready synced...would cause multiple events to be triggerd
                                self.model.on('change:primary_address', syncLogin);
                                self.inSync = true;
                            }
                        } else {
                            self.model.off('change:primary_address', syncLogin);
                            self.inSync = false;
                        }
                    });
                } else {//primary account does not allow editing besides display name and unified mail
                    self.$el.find('input, select').not('#personal, [data-property="unified_inbox_enabled"]').prop('disabled', true);
                    self.$el.find('button.btn.folderselect').hide();
                }
                //disable folderselect if no account is defined
                if (self.model.get('id') === undefined) {
                    self.$el.find('button.btn.folderselect').hide();
                }

                return self;
            },
            events: {
                'save': 'onSave',
                'click .folderselect': 'onFolderSelect'
            },

            onSave: function () {
                var self = this,
                    list = ['name', 'personal', 'unified_inbox_enabled'],
                    differences = returnDifferences(this.model.attributes, oldModel);

                function returnDifferences(a, b) {
                    var array = [];
                    _.each(a, function (single, key) {
                        if (b[key] !== single) {
                            array.push(key);
                        }
                    });
                    return array;
                }

                function needToValidate(list, differences) {
                    var result = false;
                    _.each(differences, function (value) {
                        if (_.indexOf(list, value) === -1) {
                            result = true;
                        }
                    });
                    return result;
                }

                function saveAccount() {
                    if (!self.model.isNew()) {
                        // updating account, since we save on close of the dialog,
                        // dialog is already gone, tell the user that something is happening
                        notifications.yell('busy', gt('Updating account data. This might take a few seconds.'));
                    }

                    // revert default display name
                    var personal = self.model.get('personal');
                    if (personal === defaultDisplayName) {
                        self.model.set('personal', null, { silent: true }); // empty!
                    } else if ($.trim(personal) === '') {
                        self.model.set('personal', ' ', { silent: true }); // yep, one space!
                    }

                    self.model.save().then(
                        function success(data) {
                            self.dialog.close();
                            if (self.collection) {
                                self.collection.add([data]);
                            }
                            if (self.model.isNew()) {
                                self.succes();
                            } else {
                                notifications.yell('success', gt('Account updated'));
                            }
                        },
                        function fail(data) {
                            if (data.code === 'ACC-0004' && data.error_params[0].substring(8, 13) === 'login') {//string comparison is ugly, maybe backend has a translated version of this
                                notifications.yell('error', gt('Username must not be empty.'));
                            } else if (data.code === 'SVL-0002') {
                                notifications.yell('error',
                                   //#. %1$s the missing request parameter
                                   //#, c-format
                                   gt('Please enter the following data: %1$s', _.noI18n(data.error_params[0])));
                            } else {
                                notifications.yell('error', _.noI18n(data.error));
                            }
                            self.dialog.idle();
                        }
                    );
                }

                if (needToValidate(list, differences)) {
                    validationCheck(this.model.attributes).done(function (response, error) {
                        //an undefined response variable implies an error (f.e. category 'USER_INPUT')
                        var hasError = _.isUndefined(response) || (error ? [].concat(error.categories || []).indexOf('ERROR') > -1 : false),
                            hasWarning = error && error.warnings;

                        if (hasError) {
                            //on error yell
                            notifications.yell(error);
                            self.dialog.idle();
                        } else if (hasWarning) {
                            //on warning ask user
                            require(['io.ox/core/tk/dialogs'], function (dialogs) {
                                var messages = _.map([].concat(error.warnings), function (warn) {
                                    return $('<div>')
                                            .addClass('alert alert-warning')
                                            .css('margin', '10px')
                                            .text(warn.error);
                                });

                                new dialogs.ModalDialog()
                                    .header(
                                        $('<h4>').text(gt('Warnings'))
                                    )
                                    .build(function () {
                                        this.getContentNode().append(messages);
                                    })
                                    .addPrimaryButton('proceed', gt('Ignore Warnings'))
                                    .addButton('cancel', gt('Cancel'))
                                    .show()
                                    .done(function (action) {
                                        if (action === 'proceed') {
                                            saveAccount();
                                        } else {
                                            self.dialog.idle();
                                        }
                                    });
                            });
                        } else {
                            //on success save
                            saveAccount();
                        }
                    });
                } else {
                    saveAccount();
                }

            },

            onFolderSelect: function (e) {

                var self = this;
                self.dialog.getPopup().hide();

                if (self.model.get('id') !== 0) {
                    var property = $(e.currentTarget).prev().attr('data-property'),
                        id = self.model.get(property);
                    require(['io.ox/core/tk/dialogs', 'io.ox/core/tk/folderviews'], function (dialogs, views) {

                        var label = gt('Select folder'),
                            dialog = new dialogs.ModalDialog()
                            .header($('<h4>').text(label))
                            .addPrimaryButton('select', label, 'select', {'tabIndex': '1'})
                            .addButton('cancel', gt('Cancel'), 'cancel', {'tabIndex': '1'});
                        dialog.getBody().css({ height: '250px' });
                        var tree = new views.FolderTree(dialog.getBody(), {
                                type: 'mail',
                                tabindex: 1,
                                rootFolderId: 'default' + self.model.get('id')
                            });
                        dialog.show(function () {
                            tree.paint().done(function () {
                                tree.select(id).done(function () {
                                    tree.selection.updateIndex();
                                    dialog.getBody().focus();
                                });
                            });
                        })
                        .done(function (action) {
                            if (action === 'select') {
                                var target = _(tree.selection.get()).first();
                                self.model.set(property, target, {validate: true});
                            }
                            tree.destroy().done(function () {
                                tree = dialog = null;
                            });
                            self.dialog.getPopup().show();
                        });
                    });
                }
            }
        });

    return accountAPI.getDefaultDisplayName().then(function (name) {
        defaultDisplayName = name;
        return AccountDetailView;
    });
});
