/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.uuid.impl;

import gnu.trove.ConcurrentTIntObjectHashMap;
import gnu.trove.map.TIntObjectMap;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Properties;
import java.util.UUID;
import org.apache.commons.logging.Log;
import com.openexchange.usm.api.database.DatabaseAccess;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.cache.Cache;
import com.openexchange.usm.cache.CacheManager;
import com.openexchange.usm.util.Toolkit;
import com.openexchange.usm.uuid.OXObjectID;
import com.openexchange.usm.uuid.UUIDMappingBundleErrorCodes;
import com.openexchange.usm.uuid.UUIDMappingService;
import com.openexchange.usm.uuid.exceptions.OXObjectAlreadyMappedException;
import com.openexchange.usm.uuid.exceptions.UUIDAlreadyMappedException;

public class UUIDMappingImpl implements UUIDMappingService {

	private static final String FULL_INSERT_PART1 = "INSERT INTO UUIDMapping (cid, contentType, objectId, uuid) VALUES (?, ?, ?, ";
	private static final String FULL_INSERT_PART2 = "(REPLACE(?,'-','')))";
	private static final String SELECT_BY_UUID_PART1 = "SELECT cid, contentType, objectId FROM UUIDMapping WHERE uuid = ";
	private static final String SELECT_BY_UUID_PART2 = "(REPLACE(?,'-','')) AND cid = ?";
	private static final String OX_OBJECT_IDENTIFIER = " cid = ? AND contentType = ? AND objectId = ? ";
	private static final String SELECT_UUID = "(uuid) FROM UUIDMapping WHERE " + OX_OBJECT_IDENTIFIER;
	private static final String DELETE = "DELETE FROM UUIDMapping WHERE ";

	private static final long GENERATED_UUID_FIXED_LSB_BITS = 0xFFFFFF0000000000L;

	private final TIntObjectMap<UUID> _contextUUIDs = new ConcurrentTIntObjectHashMap<UUID>();

	private Log _journal;
	private DatabaseAccess _dbAccess;

	private Cache<UUID, UUID> _cacheUUIDtoOXObjectID;
	private Cache<UUID, UUID> _cacheOXObjectIDtoUUID;

	public void activate(Log journal, DatabaseAccess dbAccess, CacheManager cacheManager) {
		_journal = journal;
		_dbAccess = dbAccess;
		Properties oxIdToUUIDCacheConfigProperties = new Properties();
		Properties uuidToOxIdCacheConfigProperties = new Properties();
		try {
		    oxIdToUUIDCacheConfigProperties.load(new ByteArrayInputStream(UUIDCacheConfiguration.oxObjectToUIDCache));
		    uuidToOxIdCacheConfigProperties.load(new ByteArrayInputStream(UUIDCacheConfiguration.uuidToOxObjectCache));
		} catch (IOException e) {
			_journal.info("Can not load default configuration for UUID Mapping cache", e);
			oxIdToUUIDCacheConfigProperties = null;
		}
		_cacheOXObjectIDtoUUID = createCache(UUIDCacheConfiguration.USMOX_OBJECT_ID_TO_UUID_CACHE, cacheManager, oxIdToUUIDCacheConfigProperties);
		_cacheUUIDtoOXObjectID = createCache(UUIDCacheConfiguration.USMUUID_TO_OX_OBJECT_ID_CACHE, cacheManager, uuidToOxIdCacheConfigProperties);
		_journal.info("USM UUID mapping service activated");
	}

	/**
	 * Create a cache using the given configuration. If that fails, create one using the default configuration.
	 *
	 * @param name
	 * @param cacheManager
	 * @param cacheConfigProperties
	 * @return
	 */
	private Cache<UUID, UUID> createCache(String name, CacheManager cacheManager, Properties cacheConfigProperties) {
		try {
			return cacheManager.createCache(name, UUID.class, UUID.class, cacheConfigProperties);
		} catch (Exception ignored) {
			return cacheManager.createCache(name, UUID.class, UUID.class, null);
		}
	}

	public void deactivate() {
		_contextUUIDs.clear();
		_dbAccess = null;
		if (_cacheUUIDtoOXObjectID != null) {
			_cacheUUIDtoOXObjectID.dispose(); //freeCache
			_cacheUUIDtoOXObjectID = null;
		}
		if (_cacheOXObjectIDtoUUID != null) {
			_cacheOXObjectIDtoUUID.dispose(); //freeCache
			_cacheOXObjectIDtoUUID = null;
		}
		_journal.info("USM UUID mapping service deactivated");
		_journal = null;
	}

	public void storeMapping(UUID uuid, OXObjectID oxObjectId) throws DatabaseAccessException, USMSQLException,
			OXObjectAlreadyMappedException, UUIDAlreadyMappedException {
		if (isComputedUUID(oxObjectId, uuid)) {
			String errorMessage = oxObjectId + " with new UUID " + uuid + " already mapped by server";
			_journal.error(errorMessage);
			throw new OXObjectAlreadyMappedException(UUIDMappingBundleErrorCodes.UUID_RESERVED_FOR_SERVER, errorMessage);
		}
		OXObjectID oldOXObjId = getMappedObject(oxObjectId.getCid(), uuid);
		if (oldOXObjId == null) {
			//uuid does not exist, check if this object is already mapped to another uuid
			UUID oldUUID = getMappedUUID(oxObjectId);
			if (oldUUID != null && !oldUUID.equals(uuid) && !isComputedUUID(oxObjectId, oldUUID)) {
				String errorMessage = oxObjectId + " with new UUID " + uuid + " already mapped to " + oldUUID;
				_journal.error(errorMessage);
				throw new OXObjectAlreadyMappedException(UUIDMappingBundleErrorCodes.OXOBJECT_ALREADY_MAPPED,
						errorMessage);
			}
			saveIdentifierInDB(uuid, oxObjectId);
		} else if (!oldOXObjId.equals(oxObjectId)) {
			//uuid exists already and is mapped to another object
			String errorMessage = "UUID " + uuid + " for " + oxObjectId + " already mapped to " + oldOXObjId;
			_journal.error(errorMessage);
			throw new UUIDAlreadyMappedException(UUIDMappingBundleErrorCodes.UUID_ALREADY_EXIST, errorMessage);
		}
	}

	private boolean isComputedUUID(OXObjectID oxObjectId, UUID uuid) throws DatabaseAccessException, USMSQLException {
		UUID contextUUID = getContextUUID(oxObjectId.getCid());
		return (contextUUID.getMostSignificantBits() == uuid.getMostSignificantBits() && (((contextUUID
				.getLeastSignificantBits() ^ uuid.getLeastSignificantBits()) & GENERATED_UUID_FIXED_LSB_BITS) == 0L));
	}

	public UUID getUUID(int cid, int contentType, int objectId) throws DatabaseAccessException, USMSQLException {
		OXObjectID oxObjId = new OXObjectID(cid, contentType, objectId);
		UUID uuid = getMappedUUID(oxObjId);
		if (uuid == null) {
			uuid = getContextUUID(cid);
			long lsb = uuid.getLeastSignificantBits();
			lsb ^= (objectId | ((long) contentType) << 32);
			uuid = new UUID(uuid.getMostSignificantBits(), lsb);
			storeUUIDMappingInCache(uuid, oxObjId, false); // loaded from db, no need to invalidate
		}
		return uuid;
	}

	public OXObjectID getMappedObject(int cid, UUID uuid) throws DatabaseAccessException, USMSQLException {
		OXObjectID objectID = OXObjectID.fromSimplifiedUUID(_cacheUUIDtoOXObjectID.get(uuid));
		if (objectID == null)
			objectID = getOXIdentifierFromDB(cid, uuid);
		return objectID;
	}

	private UUID getMappedUUID(OXObjectID oxObjectId) throws DatabaseAccessException, USMSQLException {
		UUID uuid = _cacheOXObjectIDtoUUID.get(oxObjectId.convertToSimplifiedUUID());
		if (uuid == null)
			uuid = getUUIDFromDB(oxObjectId);
		return uuid;
	}

	private OXObjectID getOXIdentifierFromDB(int contextId, UUID uuid) throws DatabaseAccessException, USMSQLException {
		OXObjectID oxObjectID = null;
		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		if (_journal.isDebugEnabled())
			_journal.debug("Retrieving OX object for UUID " + uuid);
		try {
			con = _dbAccess.getReadOnly(contextId);
			statement = con.prepareStatement(SELECT_BY_UUID_PART1 + _dbAccess.retrieveUnHexFunctionName()
					+ SELECT_BY_UUID_PART2);
			statement.setString(1, uuid.toString());
			statement.setInt(2, contextId);
			ResultSet result = statement.executeQuery();
			if (result.next()) {
				int cid = result.getInt(1);
				int contentType = result.getInt(2);
				int objectId = result.getInt(3);
				oxObjectID = new OXObjectID(cid, contentType, objectId);
				storeUUIDMappingInCache(uuid, oxObjectID, false); // loaded from db, no need to invalidate
			}
			Toolkit.close(result);
		} catch (SQLException e) {
			String errorMessage = " SQL error reading object with uuid: " + uuid + " and contextId: " + contextId;
			_journal.error(errorMessage, e);
			throw new USMSQLException(UUIDMappingBundleErrorCodes.SELECT_FAILED, errorMessage, e);
		} finally {
			Toolkit.close(statement);
			Toolkit.close(con);
		}
		return oxObjectID;
	}

	private UUID getUUIDFromDB(OXObjectID oxObjectID) throws DatabaseAccessException, USMSQLException {
		UUID uuid = null;
		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		if (_journal.isDebugEnabled())
			_journal.debug("Retrieving UUID for OX object " + oxObjectID);
		try {
			con = _dbAccess.getReadOnly(oxObjectID.getCid());
			statement = con.prepareStatement("SELECT " + _dbAccess.retrieveHexFunctionName() + SELECT_UUID);
			setOxObjectParameters(oxObjectID, statement);
			ResultSet result = statement.executeQuery();
			if (result.next()) {
				uuid = generateUUIDfromHexString(oxObjectID, result.getString(1));
				storeUUIDMappingInCache(uuid, oxObjectID, false); // loaded from db, no need to invalidate
			}
			Toolkit.close(result);
		} catch (SQLException e) {
			String errorMessage = " SQL error reading object " + oxObjectID.toString();
			_journal.error(errorMessage, e);
			throw new USMSQLException(UUIDMappingBundleErrorCodes.SELECT_FAILED_2, errorMessage, e);
		} finally {
			Toolkit.close(statement);
			Toolkit.close(con);
		}
		return uuid;
	}

	private UUID generateUUIDfromHexString(OXObjectID oxObjectID, String str) throws USMSQLException {
		if (str != null && str.length() > 16) {
			int len = str.length();
			int split1 = len - 24; // count from end just in case the UUID starts with 0 and the database doesn't report leading 0s.
			int split2 = len - 16;
			int split3 = len - 8;
			try {
				long v1 = (split1 > 0) ? Long.parseLong(str.substring(0, split1), 16) : 0L;
				long v2 = Long.parseLong(str.substring(split1, split2), 16);
				long v3 = Long.parseLong(str.substring(split2, split3), 16);
				long v4 = Long.parseLong(str.substring(split3), 16);
				return new UUID(v1 << 32 | v2, v3 << 32 | v4);
			} catch (NumberFormatException ignored) {
				// fall through to error reporting
			}
		}
		throw new USMSQLException(UUIDMappingBundleErrorCodes.ILLEGAL_UUID_IN_DB, "Illegal UUID in database for "
				+ oxObjectID + ": " + str);
	}

	private void saveIdentifierInDB(UUID uuid, OXObjectID oxObjectID) throws DatabaseAccessException, USMSQLException {
		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		if (_journal.isDebugEnabled())
			_journal.debug("Storing UUID " + uuid + " for OX object " + oxObjectID);
		try {
			con = _dbAccess.getWritable(oxObjectID.getCid());
			statement = con.prepareStatement(FULL_INSERT_PART1 + _dbAccess.retrieveUnHexFunctionName()
					+ FULL_INSERT_PART2);
			setOxObjectParameters(oxObjectID, statement);
			statement.setString(4, uuid.toString());
			if (statement.executeUpdate() != 1)
				_journal.error(" Insert of object with UUID was not performed");
			storeUUIDMappingInCache(uuid, oxObjectID, true); // invalidate remote cache entries
		} catch (SQLException e) {
			String errorMessage = " SQL error saving uuid for " + oxObjectID.toString();
			_journal.error(errorMessage, e);
			throw new USMSQLException(UUIDMappingBundleErrorCodes.SAVE_FAILED, errorMessage, e);
		} finally {
			Toolkit.close(statement);
			Toolkit.close(con);
		}
	}

	private void storeUUIDMappingInCache(UUID uuid, OXObjectID oxObjectID, boolean invalidate) {
		UUID simplifiedUUID = oxObjectID.convertToSimplifiedUUID();
		_cacheUUIDtoOXObjectID.put(uuid, simplifiedUUID, invalidate);
		_cacheOXObjectIDtoUUID.put(simplifiedUUID, uuid, invalidate);
	}

	private void setOxObjectParameters(OXObjectID oxObjectID, PreparedStatement statement) throws SQLException {
		statement.setInt(1, oxObjectID.getCid());
		statement.setInt(2, oxObjectID.getContentType());
		statement.setInt(3, oxObjectID.getObjectId());
	}

	public void removeMapping(int contextId, OXObjectID oxObjectID) throws DatabaseAccessException, USMSQLException {
		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		try {
			con = _dbAccess.getWritable(contextId);
			statement = con.prepareStatement(DELETE + OX_OBJECT_IDENTIFIER);
			setOxObjectParameters(oxObjectID, statement);
			statement.executeUpdate();

			// AFE: Logging removed since it is normal that a UUID mapping does not exist for some random OX object
			//				if (statement.executeUpdate() == 0)
			//					_journal.warn("DELETE of mapping for object " + oxObjectId.toString() + " was not performed");
			UUID simplifiedUUID = oxObjectID.convertToSimplifiedUUID();
			UUID uuid = _cacheOXObjectIDtoUUID.get(simplifiedUUID);
			_cacheOXObjectIDtoUUID.remove(simplifiedUUID);
			if (uuid != null)
				_cacheUUIDtoOXObjectID.remove(uuid);
		} catch (SQLException e) {
			String errorMessage = " SQL error deleting mapping ";
			_journal.error(errorMessage, e);
			throw new USMSQLException(UUIDMappingBundleErrorCodes.DELETE_FAILED_1, errorMessage, e);
		} finally {
			Toolkit.close(statement);
			Toolkit.close(con);
		}
	}

	public UUID getContextUUID(int cid) throws DatabaseAccessException, USMSQLException {
		UUID uuid = _contextUUIDs.get(cid);
		if (uuid != null)
			return uuid;

		EncapsulatedConnection con = null;
		PreparedStatement statement = null;
		ResultSet result = null;

		try {
			con = _dbAccess.getReadOnly(cid);
			statement = con.prepareStatement("SELECT header FROM UUIDHeader where cid = ?");
			statement.setInt(1, cid);

			result = statement.executeQuery();
			if (result.next()) {
				uuid = UUID.fromString(result.getString(1));
			} else {
				Toolkit.close(result);
				Toolkit.close(statement);
				Toolkit.close(con);
				//generate the header
				uuid = UUID.randomUUID();
				con = _dbAccess.getWritable(cid);
				statement = con.prepareStatement("INSERT INTO UUIDHeader (cid, header) VALUES(?, ?)");
				statement.setInt(1, cid);
				statement.setString(2, uuid.toString());
				if (statement.executeUpdate() != 1)
					_journal.error("Insert of UUID header was not performed for cid " + cid);
			}
		} catch (SQLException e) {
			String errorMessage = " SQL error getting UUID header ";
			_journal.error(errorMessage, e);
			throw new USMSQLException(UUIDMappingBundleErrorCodes.UUID_HEADER_SQL_ERROR, errorMessage, e);
		} finally {
			Toolkit.close(result);
			Toolkit.close(statement);
			Toolkit.close(con);
		}
		_contextUUIDs.put(cid, uuid);
		return uuid;
	}
}
