/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.session.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.sql.Clob;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import org.apache.commons.logging.Log;
import com.openexchange.usm.api.cache.SyncStateCache;
import com.openexchange.usm.api.contenttypes.ContentType;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.exceptions.DeserializationFailedException;
import com.openexchange.usm.api.exceptions.USMIllegalStateException;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.api.session.CompleteSessionID;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.session.dataobject.DataObjectSet;
import com.openexchange.usm.util.Toolkit;

/**
 * DB DataObject storage.
 * 
 * @author ldo
 */
public class SyncStateStorage {

    private static final int MAX_RETRIES_SYNC_STATE_INSERT_IN_DB = 10;

    private static final int NUMBER_OF_SESSION_ID_FIELDS = 2;

    private static final int NUMBER_OF_OBJECT_ID_FIELDS = 3;

    private final SessionImpl _session;

    private final Map<String, long[]> _timestamps = new HashMap<String, long[]>();

    private final SyncStateCache _syncStateCache;

    public SyncStateStorage(SessionImpl session) {
        _session = session;
        _syncStateCache = session.getSessionManager().getSyncStateCacheProvider().getCache(session.getCompleteSessionID());
    }

    public void remapStates(String oldObjectID, String newObjectID) throws DatabaseAccessException, USMSQLException {
        Log journal = getJournal();
        if (journal.isDebugEnabled())
            journal.debug(_session + " Remapping states for object :  " + oldObjectID + " to new object id: " + newObjectID);
        remapLocalTimestamps(oldObjectID, newObjectID);
        _syncStateCache.remapStates(getSessionID(), oldObjectID, newObjectID);
        // restore the data in the DB
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        try {
            String updateStatement = "UPDATE USMDataStorage SET objectId = ? WHERE objectId = ? AND cid = ? AND usmSessionId = ?";
            con = _session.getWritableDBConnection();
            statement = con.prepareStatement(updateStatement);
            statement.setString(1, newObjectID);
            statement.setString(2, oldObjectID);
            statement.setInt(3, _session.getContextId());
            statement.setInt(4, _session.getSessionId());
            if (statement.executeUpdate() == 0) {
                if (journal.isDebugEnabled())
                    journal.debug(_session + " No states were remapped for object:  " + oldObjectID);
            }
        } catch (SQLException e) {
            throw new USMSQLException(
                USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER5,
                _session + " Error re-storing cache data for " + oldObjectID,
                e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
    }

    private long[] getLocalTimestamps(String objectID) {
        synchronized (_timestamps) {
            return _timestamps.get(objectID);
        }
    }

    private void storeLocalTimestamps(String objectID, long[] newTimestamps) {
        synchronized (_timestamps) {
            _timestamps.put(objectID, newTimestamps);
        }
    }

    private void removeLocalTimestamps(String... objectIDs) {
        synchronized (_timestamps) {
            for (String id : objectIDs)
                _timestamps.remove(id);
        }
    }

    private void remapLocalTimestamps(String oldObjectID, String newObjectID) {
        synchronized (_timestamps) {
            long[] timestamps = _timestamps.remove(oldObjectID);
            if (timestamps != null)
                _timestamps.put(newObjectID, timestamps);
        }
    }

    private void retainLocalTimestamps(Set<String> objectIDSet) {
        synchronized (_timestamps) {
            _timestamps.keySet().retainAll(objectIDSet);
        }
    }

    private long[] getTimestamps(String objectID) throws DatabaseAccessException, USMSQLException {
        long[] timestamps = getLocalTimestamps(objectID);
        if (timestamps != null && timestamps.length > 0)
            return timestamps;
        readSyncStatesFromDB(objectID);
        return getLocalTimestamps(objectID);
    }

    private static long getNewestTimestamp(long... timestamps) {
        long newest = 0L;
        for (long timestamp : timestamps)
            newest = Math.max(newest, timestamp);
        return newest;
    }

    private static Long findOldestTimestamp(long timestampToIgnore, Set<Long> timestamps) {
        Long oldestTimestamp = null;
        for (Long timestamp : timestamps) {
            if (timestamp != timestampToIgnore && (oldestTimestamp == null || oldestTimestamp > timestamp))
                oldestTimestamp = timestamp;
        }
        return oldestTimestamp;
    }

    private void removeOldestTimestamp(String objectID, long timestampToKeep, Set<Long> timestamps) {
        Long oldestTimestamp = findOldestTimestamp(timestampToKeep, timestamps);
        if (oldestTimestamp != null) {
            _syncStateCache.remove(getSessionID(), objectID, oldestTimestamp);
            timestamps.remove(oldestTimestamp);
        }
    }

    private long updateLocalTimestamps(String objectID, int limit, long timestampToKeep, long timestampToAdd, long[] timestamps) {
        Set<Long> ts = new HashSet<Long>();
        for (long timestamp : timestamps) {
            if (timestamp < timestampToKeep)
                _syncStateCache.remove(getSessionID(), objectID, timestamp);
            else
                ts.add(timestamp);
        }
        for (int i = 0; ts.size() >= limit && i < ts.size(); i++)
            removeOldestTimestamp(objectID, timestampToKeep, ts);
        long[] newTimestamps = new long[ts.size() + 1];
        int i = 0;
        for (Long timestamp : ts)
            newTimestamps[i++] = timestamp;
        newTimestamps[i] = timestampToAdd;
        storeLocalTimestamps(objectID, newTimestamps);
        Long oldestTimestamp = findOldestTimestamp(timestampToKeep, ts);
        return (oldestTimestamp == null) ? timestampToKeep : oldestTimestamp;
    }

    public long put(String objectID, long timestamp, DataObjectSet objectSet, long timestampToKeep, SessionImpl session) throws DatabaseAccessException, USMSQLException {
        Log journal = getJournal();

        Serializable[][] serializedObjects = serializeObjects(objectSet);
        byte[] data = compress(serializedObjects);
        long[] timestamps = getTimestamps(objectID);
        long newTS = timestamp;
        long oldTS = timestampToKeep;
        if (timestamps != null && timestamps.length > 0) {
            long newestTimestamp = getNewestTimestamp(timestamps);
            byte[] newestData = getCompressedData(objectID, newestTimestamp);
            if (Arrays.equals(data, newestData) || !hasStateChanged(newestTimestamp, newestData, objectSet, session)) {
                if (journal.isDebugEnabled())
                    journal.debug(_session + " Storing in DB cache not necessary, no changes for " + objectID + ", timestamp " + timestamp + "->" + newestTimestamp + ", keeping " + timestampToKeep);
                return newestTimestamp;
            }
            newTS = Math.max(newestTimestamp + 1L, newTS);
        }

        // SyncStateStorageStatistics.collectCompressionStatistics(serializedObjects, data);
        boolean tooManyTimestamps = oldTS != timestampToKeep;
        String dbData = Toolkit.encodeBase64(data);
        if (journal.isDebugEnabled()) {
            if (tooManyTimestamps)
                journal.debug(_session + " Storing in DB cache '" + objectID + "' for " + newTS + ", keeping " + timestampToKeep + ", removing older than " + oldTS);
            else
                journal.debug(_session + " Storing in DB cache '" + objectID + "' for " + newTS + ", removing older than " + oldTS);
        }
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        try {
            con = _session.getWritableDBConnection();
            // make delete/insert
            String deleteStatement = tooManyTimestamps ? DataObjectStorageSQLStatements.COMPLEX_SYNCSTATE_DELETE : DataObjectStorageSQLStatements.SIMPLE_SYNCSTATE_DELETE;
            statement = con.prepareStatement(deleteStatement);
            addObjectIdentifierToStatement(statement, objectID);
            statement.setLong(NUMBER_OF_OBJECT_ID_FIELDS + 1, oldTS);
            if (tooManyTimestamps)
                statement.setLong(NUMBER_OF_OBJECT_ID_FIELDS + 2, timestampToKeep);
            statement.executeUpdate();
            statement.close();

            int retries = 0;
            int executeUpdateAffectedRows = 0;
            do {
                statement = con.prepareStatement(_session.getSessionManager().getDatabaseAccess().supportsInsertIgnoreStatement() ? DataObjectStorageSQLStatements.FULL_INSERT_IGNORE : DataObjectStorageSQLStatements.FULL_INSERT);
                addObjectIdentifierToStatement(statement, objectID);
                statement.setLong(NUMBER_OF_OBJECT_ID_FIELDS + 1, newTS);
                statement.setString(NUMBER_OF_OBJECT_ID_FIELDS + 2, dbData);
                executeUpdateAffectedRows = statement.executeUpdate();
                if (executeUpdateAffectedRows == 0) {
                    if (!_session.getSessionManager().getDatabaseAccess().supportsInsertIgnoreStatement())
                        throw new SQLException("Insert did not produce any results");
                    else {
                        retries++;
                        newTS = newTS + 1L;
                        if(getJournal().isDebugEnabled()) 
                            getJournal().debug("Retry sync state insert for timestamp: " + newTS + " sessionId: " + _session.getSessionId() + " in context: " + _session.getContextId());
                    }
                }
            } while (retries < MAX_RETRIES_SYNC_STATE_INSERT_IN_DB && executeUpdateAffectedRows == 0);
            
            if(executeUpdateAffectedRows == 0) // throw exception if after 10 retries still no insert was possible
                throw new SQLException("Insert did not produce any results");

            //store the new timestamp in the local storage
            if (timestamps != null && timestamps.length > 0) {
                int timestampLimit = _session.getSessionManager().getMaxSyncStatesInDB();
                oldTS = updateLocalTimestamps(objectID, timestampLimit, timestampToKeep, newTS, timestamps);
            } else {
                storeLocalTimestamps(objectID, new long[] { newTS });
            }

        } catch (SQLException e) {
            throw new USMSQLException(
                USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER1,
                _session + " Error storing cache data for " + objectID,
                e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
        _session.getSessionManager().syncStateSavedToDatabase();
        return newTS;
    }

    private static boolean hasStateChanged(long newestTimestamp, byte[] compressedData, DataObjectSet objects, SessionImpl session) {
        try {
            Serializable[][] serializedData = decompress(compressedData);
            int len1 = (serializedData == null) ? 0 : serializedData.length;
            int len2 = (objects == null) ? 0 : objects.size();
            if (len1 != len2)
                return true;
            if (len1 == 0)
                return false;

            ContentType contentType = objects.iterator().next().getContentType();
            DataObject[] cachedObjects = deserializeObjects(serializedData, newestTimestamp, contentType, session);
            return !objects.isEqualTo(cachedObjects);
        } catch (DeserializationFailedException e) {
            return true;
        }
    }

    private static DataObject[] deserializeObjects(Serializable[][] values, long timestamp, ContentType contentType, SessionImpl session) {
        DataObject[] result = new DataObject[values.length];
        for (int i = 0; i < result.length; i++) {
            DataObject o = contentType.newDataObject(session);
            try {
                o.deserialize(timestamp, values[i]);
            } catch (DeserializationFailedException e) {
                return null;
            }
            result[i] = o;
        }
        return result;
    }

    private static Serializable[][] serializeObjects(DataObjectSet objects) {
        List<String> ids = new ArrayList<String>(objects.getIDs());
        Collections.sort(ids);
        Serializable[][] data = new Serializable[ids.size()][];
        for (int i = 0; i < ids.size(); i++)
            data[i] = objects.get(ids.get(i)).serialize();
        return data;
    }

    public Serializable[][] get(String objectID, long timestamp) throws DatabaseAccessException, USMSQLException {
        byte[] data = getCompressedData(objectID, timestamp);
        try {
            return (data == null) ? null : decompress(data);
        } catch (DeserializationFailedException e) {
            _session.logDeserializationError(objectID, timestamp, e);
            return null;
        }
    }

    private byte[] getCompressedData(String objectID, long timestamp) throws DatabaseAccessException, USMSQLException {
        byte[] data = _syncStateCache.get(getSessionID(), objectID, timestamp);
        if (data != null)
            return data;
        readSyncStatesFromDB(objectID);
        return _syncStateCache.get(getSessionID(), objectID, timestamp);
    }

    /**
     * Special method that returns the newest timestamp for internal USM methods that do not know of the current sync key / timestamp.
     * 
     * @param objectID
     * @return
     * @throws DatabaseAccessException
     * @throws USMSQLException
     */
    public long getNewestTimestamp(String objectID) throws DatabaseAccessException, USMSQLException {
        long[] timestamps = getLocalTimestamps(objectID);
        if (timestamps == null || timestamps.length == 0) {
            readSyncStatesFromDB(objectID);
            timestamps = getLocalTimestamps(objectID);
        }
        long newestTimestamp = 0L;
        if (timestamps != null) {
            for (long timestamp : timestamps)
                newestTimestamp = Math.max(newestTimestamp, timestamp);
        }
        return newestTimestamp;
    }

    /**
     * Removes all DB entries for the given user and device that have object IDs as the ones provided as parameters
     * 
     * @param user
     * @param device
     * @param objectIDs
     * @throws USMSQLException
     */
    public void remove(String... objectIDs) throws USMSQLException {
        Log journal = getJournal();
        if (journal.isDebugEnabled())
            journal.debug(_session + " Removing from DB cache " + Arrays.toString(objectIDs));
        removeLocalTimestamps(objectIDs);
        _syncStateCache.removeSyncStates(getSessionID(), objectIDs);
        executeUpdate(
            "",
            "Error while deleting cached information for folders",
            completeComplexStatement(
                DataObjectStorageSQLStatements.DELETE + DataObjectStorageSQLStatements.UNIQUE_SESSION_IDENTIFIER + " and ObjectID IN (",
                objectIDs.length),
            objectIDs);
    }

    /**
     * Removes all DB entries for the given user and device that have object IDs other than the ones provided as parameters
     * 
     * @param objectIDs
     * @throws USMSQLException
     */
    public void retain(String... objectIDs) throws USMSQLException {
        Log journal = getJournal();
        if (journal.isDebugEnabled())
            journal.debug(_session + " Removing from DB cache all except " + Arrays.toString(objectIDs));
        // preparation
        Set<String> objectIDSet = new HashSet<String>();
        for (String objectId : objectIDs) {
            objectIDSet.add(objectId);
        }
        retainLocalTimestamps(objectIDSet);
        _syncStateCache.retainSyncStates(getSessionID(), objectIDSet); // TODO verify that Folder Hierarchy is not removed by accident
        // search which DB entries need to be removed
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        List<String> toDelete = new ArrayList<String>();
        try {
            con = _session.getReadOnlyDBConnection();
            statement = con.prepareStatement(DataObjectStorageSQLStatements.SELECT_OBJECT_IDS);
            addSessionIdentifierToStatement(statement);
            ResultSet result = statement.executeQuery();
            while (result.next()) {
                String id = result.getString(1);
                if (!objectIDSet.contains(id))
                    toDelete.add(id);
            }
        } catch (DatabaseAccessException e) {
            String errorMessage = _session + " Error while deleting cached information for unused folders";
            getJournal().error(errorMessage, e);
            throw new USMSQLException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER2, errorMessage, e);
        } catch (SQLException e) {
            String errorMessage = _session + " Error retrieving list of folders stored in DB cache";
            getJournal().error(errorMessage, e);
            throw new USMSQLException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER6, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
        // If any have been found, remove those DB entries
        if (!toDelete.isEmpty())
            remove(toDelete.toArray(new String[toDelete.size()]));
    }

    /**
     * Removes all saved states for this session
     * 
     * @throws USMSQLException
     */
    public void removeAllObjectsForSession() throws USMSQLException {
        Log journal = getJournal();
        if (journal.isDebugEnabled())
            journal.debug(_session + " Removing from DB cache all elements for session: " + _session.getSessionId());
        _session.getSessionManager().getSyncStateCacheProvider().removeCacheCompletely(_session.getCompleteSessionID());
        executeUpdate(
            "",
            "Error while deleting cached information for one session",
            DataObjectStorageSQLStatements.DELETE + DataObjectStorageSQLStatements.UNIQUE_SESSION_IDENTIFIER);
    }

    private void addSessionIdentifierToStatement(PreparedStatement statement) throws SQLException {
        statement.setInt(1, _session.getContextId());
        statement.setInt(2, _session.getSessionId());
    }

    private void addObjectIdentifierToStatement(PreparedStatement statement, String objectID) throws SQLException {
        addSessionIdentifierToStatement(statement);
        statement.setString(NUMBER_OF_SESSION_ID_FIELDS + 1, objectID);
    }

    private static String completeComplexStatement(String prefix, int length) {
        StringBuilder sb = new StringBuilder(prefix);
        for (int i = 0; i < length; i++) {
            if (i > 0)
                sb.append(',');
            sb.append('?');
        }
        return sb.append(')').toString();
    }

    /**
     * Execute a SQL statement, the first 2 parameters must be user and session and need not be specified as parameters to this method. The
     * remaining parameters are filled from the provided values.
     * 
     * @param warnMessage
     * @param errorMessage
     * @param command
     * @param parameters
     * @throws USMSQLException
     */
    private void executeUpdate(String warnMessage, String errorMessage, String command, String... parameters) throws USMSQLException {
        EncapsulatedConnection con = null;
        try {
            con = _session.getWritableDBConnection();
            executeUpdate(con, warnMessage, errorMessage, command, parameters);
        } catch (DatabaseAccessException e) {
            getJournal().error(String.valueOf(_session) + ' ' + errorMessage, e);
        } finally {
            Toolkit.close(con);
        }
    }

    private void executeUpdate(EncapsulatedConnection con, String warnMessage, String errorMessage, String command, String... parameters) throws USMSQLException {
        PreparedStatement statement = null;
        try {
            statement = con.prepareStatement(command);
            addSessionIdentifierToStatement(statement);
            int i = NUMBER_OF_SESSION_ID_FIELDS + 1;
            for (String param : parameters)
                statement.setString(i++, param);
            if (statement.executeUpdate() == 0 && warnMessage.length() > 0)
                getJournal().warn(String.valueOf(_session) + ' ' + warnMessage);
        } catch (SQLException e) {
            getJournal().error(String.valueOf(_session) + ' ' + errorMessage, e);
            throw new USMSQLException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER3, _session + ": " + errorMessage, e);
        } finally {
            Toolkit.close(statement);
        }
    }

    private void readSyncStatesFromDB(String objectID) throws DatabaseAccessException, USMSQLException {
        EncapsulatedConnection con = null;
        PreparedStatement statement = null;
        List<CachedDBEntry> dbList = new ArrayList<CachedDBEntry>();
        try {
            con = _session.getReadOnlyDBConnection();
            statement = con.prepareStatement(DataObjectStorageSQLStatements.SELECT_CACHED_DATA + DataObjectStorageSQLStatements.UNIQUE_OBJECT_IDENTIFIER + DataObjectStorageSQLStatements.SELECT_CACHED_DATA_ORDER_BY);
            addObjectIdentifierToStatement(statement, objectID);
            ResultSet result = statement.executeQuery();
            while (result.next()) {
                long timestamp = result.getLong(1);
                Clob element = result.getClob(2);
                dbList.add(new CachedDBEntry(timestamp, element.getSubString(1L, (int) element.length())));
            }
        } catch (SQLException e) {
            String errorMessage = _session + " Error retrieving cached data for " + objectID;
            getJournal().error(errorMessage, e);
            throw new USMSQLException(USMSessionManagementErrorCodes.DATA_STORAGE_ERROR_NUMBER4, errorMessage, e);
        } finally {
            Toolkit.close(statement);
            Toolkit.close(con);
        }
        long[] timestamps = new long[dbList.size()];
        _syncStateCache.removeSyncStates(getSessionID(), objectID);
        int i = 0;
        for (CachedDBEntry entry : dbList) {
            timestamps[i] = entry.getTimestamp();
            _syncStateCache.put(getSessionID(), objectID, entry.getTimestamp(), Toolkit.decodeBase64(entry.getData()));
        }
        storeLocalTimestamps(objectID, timestamps);
        _session.getSessionManager().syncStatesLoadedFromDatabase(dbList.size());
    }

    private static Serializable[][] decompress(byte[] compressedData) throws DeserializationFailedException {
        try {
            ObjectInputStream in = new ObjectInputStream(new GZIPInputStream(new ByteArrayInputStream(compressedData)));
            return (Serializable[][]) in.readObject();
        } catch (ClassCastException e) {
            throw new DeserializationFailedException(
                USMSessionManagementErrorCodes.DATA_STORAGE_CACHE_CLASS_CAST_EXCEPTION,
                "Stored object is not of correct type",
                e);
        } catch (IOException e) {
            throw new DeserializationFailedException(
                USMSessionManagementErrorCodes.DATA_STORAGE_CACHE_IO_EXCEPTION,
                "Can not retrieve object from bytes",
                e);
        } catch (ClassNotFoundException e) {
            throw new DeserializationFailedException(
                USMSessionManagementErrorCodes.DATA_STORAGE_CACHE_CLASS_NOT_FOUND_EXCEPTION,
                "Can not find the class of the serializable object",
                e);
        }
    }

    private static byte[] compress(Serializable[][] objects) {
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            ObjectOutputStream oos = new ObjectOutputStream(new GZIPOutputStream(baos));
            oos.writeObject(objects);
            oos.close();
            return baos.toByteArray();
        } catch (IOException e) {
            throw new USMIllegalStateException(
                USMSessionManagementErrorCodes.SERIALIZATION_ERROR,
                "Can not serialize data to byte array",
                e);
        }
    }

    private Log getJournal() {
        return _session.getSessionManager().getJournal();
    }

    public SyncStateCache getSyncStateCache() {
        return _syncStateCache;
    }

    private CompleteSessionID getSessionID() {
        return _session.getCompleteSessionID();
    }
}
