/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.database.ox;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.List;
import org.apache.commons.logging.Log;
import org.osgi.service.component.ComponentContext;
import com.openexchange.context.ContextService;
import com.openexchange.database.DatabaseService;
import com.openexchange.exception.OXException;
import com.openexchange.usm.api.database.DatabaseAccess;
import com.openexchange.usm.api.database.DatabaseAccessException;
import com.openexchange.usm.api.database.EncapsulatedConnection;
import com.openexchange.usm.api.database.SQLConnectionReleaser;
import com.openexchange.usm.api.exceptions.USMSQLException;
import com.openexchange.usm.journal.Journal;
import com.openexchange.usm.util.SQLToolkit;
import com.openexchange.usm.util.Toolkit;

public class OXDBService implements DatabaseAccess, SQLConnectionReleaser {

    private static final String UNHEX = "UNHEX";

    private static final String HEX = "HEX";

    private DatabaseService _delegate;

    private Log _journal;

    private ContextService _contextService;

    /*
     * (non-Javadoc)
     * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
     */
    public void activate(ComponentContext context) {
        Journal journal = (Journal) context.locateService(Journal.class.getName());
        _journal = journal.getLog(DatabaseAccess.class);
        _delegate = (DatabaseService) context.locateService(DatabaseService.class.getName());
        _contextService = (ContextService) context.locateService(ContextService.class.getName());
        _journal.info("USM Connection to OX DB service activated");
    }

    /*
     * (non-Javadoc)
     * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
     */
    public void deactivate(ComponentContext context) {
        _delegate = null;
        _contextService = null;
        _journal.info("USM Connection to OX DB service deactivated");
        _journal = null;
    }

    public void backReadOnly(int contextID, Connection connection) throws DatabaseAccessException {
        if (_journal.isDebugEnabled())
            _journal.debug("Releasing read only JDBC Connection for " + contextID);
        _delegate.backReadOnly(contextID, connection);
    }

    public void backWritable(int contextID, Connection connection) throws DatabaseAccessException {
        if (_journal.isDebugEnabled())
            _journal.debug("Releasing read/write JDBC Connection for " + contextID);
        _delegate.backWritable(contextID, connection);
    }

    public EncapsulatedConnection getReadOnly(int contextID) throws DatabaseAccessException {
        if (_journal.isDebugEnabled())
            _journal.debug("Retrieving read only JDBC Connection for " + contextID);
        try {
            return new EncapsulatedConnection(this, _delegate.getReadOnly(contextID), contextID, true);
        } catch (OXException e) {
            String errorMessage = "Couldn't get read only connection for " + contextID;
            _journal.error(errorMessage, e);
            throw new DatabaseAccessException(USMDatabaseOXErrorCodes.GET_READ_ONLY_ERROR, errorMessage, e);
        }
    }

    public EncapsulatedConnection getWritable(int contextID) throws DatabaseAccessException {
        if (_journal.isDebugEnabled())
            _journal.debug("Retrieving read/write JDBC Connection for " + contextID);
        try {
            return new EncapsulatedConnection(this, _delegate.getWritable(contextID), contextID, false);
        } catch (OXException e) {
            String errorMessage = "Couldn't get read/write connection for " + contextID;
            _journal.error(errorMessage, e);
            throw new DatabaseAccessException(USMDatabaseOXErrorCodes.GET_READ_WRITE_ERROR, errorMessage, e);
        }
    }

    public void generateSequence(int contextID, String sequenceName) throws USMSQLException, DatabaseAccessException {
        EncapsulatedConnection con = getWritable(contextID);
        Statement stmt = null;
        try {
            try {
                stmt = con.createStatement();
                con.setAutoCommit(false);
                stmt.execute(SQLToolkit.createSequenceStatement(sequenceName));
                stmt.execute(SQLToolkit.createSequenceProcedure(sequenceName));
                con.commit();
                _journal.info("Creation of sequence " + sequenceName + " successfully performed");
            } catch (SQLException e) {
                con.rollback();
                throw new USMSQLException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR, e.getMessage(), e);
            } finally {
                con.setAutoCommit(true);
                Toolkit.close(stmt);
                Toolkit.close(con);
            }
        } catch (SQLException e) {
            throw new USMSQLException(USMDatabaseOXErrorCodes.CREATE_SEQUENCE_ERROR_AUTOCOMMIT_SETTINGS, e.getMessage(), e);
        }

    }

    public int getNextIdFromSequence(int contextID, String sequenceName) throws USMSQLException, DatabaseAccessException {
        EncapsulatedConnection con = getWritable(contextID);
        try {
            return SQLToolkit.getNextIdFromSequence(con, sequenceName, contextID);
        } catch (SQLException e) {
            throw new USMSQLException(USMDatabaseOXErrorCodes.GET_NEXT_FROM_SEQUENCE_ERROR, e.getMessage(), e);
        } finally {
            Toolkit.close(con);
        }
    }

    public String retrieveHexFunctionName() {
        return HEX;
    }

    public String retrieveUnHexFunctionName() {
        return UNHEX;
    }

    public List<Integer> getAllContextIds() throws DatabaseAccessException {
        try {
            return _contextService.getAllContextIds();
        } catch (OXException e) {
            throw new DatabaseAccessException(
                USMDatabaseOXErrorCodes.GET_ALL_CONTEXT_IDS_ERROR,
                "Error while retrieving all context ids",
                e);
        }
    }

    public int[] getContextsInSameSchema(int contextId) throws DatabaseAccessException {
        try {
            return _delegate.getContextsInSameSchema(contextId);
        } catch (OXException e) {
            throw new DatabaseAccessException(
                USMDatabaseOXErrorCodes.GET_CONTEXTS_IN_SAME_SCHEMA_ERROR,
                "Error while retrieving all context ids in same schema as " + contextId,
                e);
        }
    }
    
    public boolean supportsOnDuplicateKeyUpdateStatement() {
        return true;
    }

    public boolean supportsInsertIgnoreStatement() {
        return true;
    }
}
