/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * {@link ConnectionStatus}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class ConnectionStatus {

    static final String KEY_EDIT_USER = "editUser";

    static final String KEY_EDIT_USERID = "editUserId";

    static final String KEY_HAS_ERRORS = "hasErrors";

    static final String KEY_WRITE_PROTECTED = "writeProtected";

    static final String KEY_ACTIVE_CLIENTS = "activeClients";

    static final String KEY_LOCKED = "locked";

    static final String KEY_LOCKED_BY_USER = "lockedByUser";

    static final String KEY_FAILSAFESAVEDONE = "failSafeSaveDone";

    static final String KEY_WANTSEDIT_USER = "wantsToEditUser";

    static final String KEY_WANTSEDIT_USERID = "wantsToEditUserId";

    static final String KEY_SERVER_OSN = "serverOSN";

    // - Implementation --------------------------------------------------------

    /**
     * Initializes a new {@link ConnectionStatus}.
     */
    public ConnectionStatus() {
        super();
    }

    /**
     * Gets the m_hasErrors
     *
     * @return The m_hasErrors
     */
    boolean isHasErrors() {
        return m_hasErrors;
    }

    /**
     * Sets the m_hasErrors
     *
     * @param hasErrors The m_hasErrors to set
     */
    void setHasErrors(boolean hasErrors) {
        this.m_hasErrors = hasErrors;
    }

    /**
     * Get the lock state
     *
     * @return The current lock state
     */
    boolean isLocked() {
        return m_locked;
    }

    /**
     * Sets the lock state
     *
     * @param locked The new lock state
     */
    void setLockState(boolean locked) {
        this.m_locked = locked;
    }

    /**
     * Get user name who locked the file
     *
     * @return The name of the user who locked the file
     * or empty string if user is unknown or nobody set
     * a lock.
     */
    String getLockedByUser() {
        return m_lockedByUser;
    }

    /**
     * Sets the user who locked the file
     *
     * @param lockedByUser The name of the user
     * or empty string if unknown or nobody set
     * a lock.
     */
    void setLockedByUser(String lockedByUser) {
        this.m_lockedByUser = lockedByUser;
    }

    /**
     * Gets the m_isWriteProtected
     *
     * @return The m_isWriteProtected
     */
    boolean isWriteProtected() {
        return m_isWriteProtected;
    }

    /**
     * Sets the m_isWriteProtected
     *
     * @param isWriteProtected The isWriteProtected to set
     */
    void setWriteProtected(boolean isWriteProtected) {
        this.m_isWriteProtected = isWriteProtected;
    }

    /**
     * Gets the m_currentEditingUserName
     *
     * @return The m_currentEditingUserName
     */
    void setCurrentEditingUser(String currentEditingUserId, String currentEditingUserName) {
        m_currentEditingUserId = currentEditingUserId;
        m_currentEditingUserName = currentEditingUserName;
    }

    /**
     * Gets the m_currentEditingUserName
     *
     * @return The m_currentEditingUserName
     */
    String getCurrentEditingUserId() {
        return m_currentEditingUserId;
    }

    /**
     * Sets the m_currentEditingUserName
     *
     * @param currentEditingUserName The currentEditingUserName to set
     */
    String getCurrentEditingUserName() {
        return m_currentEditingUserName;
    }

    /**
     * Gets the user id who wants to receive the edit rights
     */
    String getWantsEditRightsUserId() {
    	return m_wantsEditRightsUserId;
    }

    /**
     * Gets the user name who wants to receive the edit rights
     * @return
     */
    String getWantsEditRightsUserName() {
    	return m_wantsEditRightsUserName;
    }

    /**
     * Sets the user id who wants to receive the edit rights
     *
     * @param userId
     */
    void setWantsEditRightsUser(String userId, String userName) {
    	m_wantsEditRightsUserId = userId;
    	m_wantsEditRightsUserName = userName;
    }

    /**
     * Gets the m_activeClients
     *
     * @return The m_activeClients
     */
    int getActiveClients() {
        return m_activeClients;
    }

    /**
     * Sets the m_activeClients
     *
     * @param activeClients The activeClients to set
     */
    void setActiveClients(int activeClients) {
        this.m_activeClients = activeClients;
    }

    /**
     * Gets the m_operationStateNumber
     *
     * @return The m_operationStateNumber
     */
    int getOperationStateNumber() {
        return this.m_operationStateNumber;
    }

    /**
     * Sets the m_operationStateNumber
     *
     * @param operationStateNumber The operation state number to set
     */
    void setOperationStateNumber(int operationStateNumber) {
        this.m_operationStateNumber = operationStateNumber;
    }

    /**
     * Provides information that a fail safe
     * save operation was done successfully.
     *
     * @return
     *  TRUE if there recently was a successful
     *  fail safe save operation, otherwise FALSE.
     */
    boolean isFailSafeSaveDone() {
        return m_failSafeSaveDone;
    }

    /**
     * Sets the state of a fail safe save
     * operation.
     *
     * @param saveDone The state of the fail safe save
     */
    void setFailSafeSaveDone(boolean saveDone) {
        this.m_failSafeSaveDone = saveDone;
    }

    /**
     * Creates a shallow clone of this connection status
     * object.
     *
     * @return
     *  A shallow clone.
     */
    @Override
    public Object clone() {
        ConnectionStatus clone = new ConnectionStatus();
        clone.setActiveClients(m_activeClients);
        clone.setCurrentEditingUser(m_currentEditingUserId, m_currentEditingUserName);
        clone.setFailSafeSaveDone(m_failSafeSaveDone);
        clone.setHasErrors(m_hasErrors);
        clone.setLockedByUser(m_lockedByUser);
        clone.setLockState(m_locked);
        clone.setOperationStateNumber(m_operationStateNumber);
        clone.setWantsEditRightsUser(m_wantsEditRightsUserId, m_wantsEditRightsUserName);
        clone.setWriteProtected(m_isWriteProtected);
        return clone;
    }

    /**
     * @return
     */
    public JSONObject toJSON() {
        final JSONObject ret = new JSONObject();

        try {
            ret.put(KEY_HAS_ERRORS, m_hasErrors);
            ret.put(KEY_WRITE_PROTECTED, m_isWriteProtected);
            ret.put(KEY_EDIT_USERID, m_currentEditingUserId);
            ret.put(KEY_EDIT_USER, m_currentEditingUserName);
            ret.put(KEY_ACTIVE_CLIENTS, m_activeClients);
            ret.put(KEY_LOCKED, m_locked);
            ret.put(KEY_LOCKED_BY_USER, m_lockedByUser);
            ret.put(KEY_FAILSAFESAVEDONE, m_failSafeSaveDone);
            ret.put(KEY_WANTSEDIT_USER, m_wantsEditRightsUserName);
            ret.put(KEY_WANTSEDIT_USERID, m_wantsEditRightsUserId);
            ret.put(KEY_SERVER_OSN, m_operationStateNumber);
        } catch (JSONException e) {
            ;
        }

        return ret;
    }

    /**
     * @param jsonData
     * @return
     */
    static public ConnectionStatus createFrom(JSONObject jsonData) {
        ConnectionStatus queueStatus = new ConnectionStatus();
        boolean found = false;

        try {
            if (jsonData.has(KEY_HAS_ERRORS)) {
                queueStatus.m_hasErrors = jsonData.getBoolean(KEY_HAS_ERRORS);
                found = true;
            }

            if (jsonData.has(KEY_WRITE_PROTECTED)) {
                queueStatus.m_isWriteProtected = jsonData.getBoolean(KEY_WRITE_PROTECTED);
                found = true;
            }

            if (jsonData.has(KEY_EDIT_USERID)) {
                queueStatus.m_currentEditingUserId = jsonData.getString(KEY_EDIT_USERID);
                found = true;
            }

            if (jsonData.has(KEY_EDIT_USER)) {
                queueStatus.m_currentEditingUserName = jsonData.getString(KEY_EDIT_USER);
                found = true;
            }

            if (jsonData.has(KEY_ACTIVE_CLIENTS)) {
                queueStatus.m_activeClients = jsonData.getInt(KEY_ACTIVE_CLIENTS);
                found = true;
            }

            if (jsonData.has(KEY_LOCKED)) {
                queueStatus.m_locked = jsonData.getBoolean(KEY_LOCKED);
                found = true;
            }

            if (jsonData.has(KEY_LOCKED_BY_USER)) {
                queueStatus.m_lockedByUser = jsonData.getString(KEY_LOCKED_BY_USER);
                found = true;
            }

            if (!jsonData.has(KEY_FAILSAFESAVEDONE)) {
                queueStatus.m_failSafeSaveDone = jsonData.getBoolean(KEY_FAILSAFESAVEDONE);
                found = true;
            }

            if (!jsonData.has(KEY_WANTSEDIT_USER)) {
                queueStatus.m_wantsEditRightsUserName = jsonData.getString(KEY_WANTSEDIT_USER);
                found = true;
            }

            if (!jsonData.has(KEY_WANTSEDIT_USERID)) {
                queueStatus.m_wantsEditRightsUserId = jsonData.getString(KEY_WANTSEDIT_USERID);
                found = true;
            }
        } catch (JSONException e) {
            ;
        }

        return (found ? queueStatus : null);
    }

    // - Members ---------------------------------------------------------------

    private boolean m_hasErrors = false;
    private boolean m_isWriteProtected = false;
    private boolean m_locked = false;
    private String m_currentEditingUserName = "";
    private String m_currentEditingUserId = "";
    private String m_wantsEditRightsUserId = "";
    private String m_wantsEditRightsUserName = "";
    private int m_activeClients = 0;
    private int m_operationStateNumber = 0;
    private String m_lockedByUser = "";
    private boolean m_failSafeSaveDone = false;
}
