/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.tools;

import java.math.BigInteger;
import java.net.URI;
import java.util.List;
import org.docx4j.dml.BaseStyles;
import org.docx4j.dml.CTColorScheme;
import org.docx4j.dml.Theme;
import org.docx4j.openpackaging.URIHelper;
import org.docx4j.openpackaging.parts.Part;
import org.docx4j.openpackaging.parts.WordprocessingML.MainDocumentPart;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.relationships.Relationship;
import org.docx4j.wml.Body;
import org.docx4j.wml.CTBorder;
import org.docx4j.wml.CTShd;
import org.docx4j.wml.Color;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.Document;
import org.docx4j.wml.STShd;
import org.docx4j.wml.STThemeColor;
import org.docx4j.wml.SectPr;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.tools.Commons;

public class Utils {

    public static String themeColorToString(Theme theme, STThemeColor themeColor) {
        String rgbValue = null;
        if(theme!=null) {
            BaseStyles themeElements = theme.getThemeElements();
            if(themeElements!=null) {
                CTColorScheme clrScheme = themeElements.getClrScheme();
                if(clrScheme!=null) {
                    org.docx4j.dml.CTColor ctColor = null;
                    switch(themeColor) {
                        case ACCENT_1 : ctColor = clrScheme.getAccent1(); break;
                        case ACCENT_2 : ctColor = clrScheme.getAccent2(); break;
                        case ACCENT_3 : ctColor = clrScheme.getAccent3(); break;
                        case ACCENT_4 : ctColor = clrScheme.getAccent4(); break;
                        case ACCENT_5 : ctColor = clrScheme.getAccent5(); break;
                        case ACCENT_6 : ctColor = clrScheme.getAccent6(); break;
                        case BACKGROUND_1 : ctColor = clrScheme.getLt1(); break;
                        case BACKGROUND_2 : ctColor = clrScheme.getLt2(); break;
                        case DARK_1 : ctColor = clrScheme.getDk1(); break;
                        case DARK_2 : ctColor = clrScheme.getDk2(); break;
                        case FOLLOWED_HYPERLINK : ctColor = clrScheme.getFolHlink(); break;
                        case HYPERLINK : ctColor = clrScheme.getHlink(); break;
                        case LIGHT_1 : ctColor = clrScheme.getLt1(); break;
                        case LIGHT_2 : ctColor = clrScheme.getLt2(); break;
                        case TEXT_1 : ctColor = clrScheme.getDk1(); break;
                        case TEXT_2 : ctColor = clrScheme.getDk2(); break;
                        case NONE : break;
                    }
                    if(ctColor!=null) {
                        rgbValue = Commons.ctColorToString(ctColor);
                    }
                }
            }
        }
        return rgbValue;
    }


    public static void initBorderColorFromJSONColor(OperationDocument operationDocument, JSONObject jsonColor, CTBorder border)
        throws JSONException {

        if(border!=null&&jsonColor.has("type")) {
            String type = jsonColor.getString("type");
            border.setThemeShade(null);
            border.setThemeTint(null);
            if(type.equals("rgb")&&jsonColor.has("value")) {
                border.setColor(jsonColor.getString("value"));
                border.setThemeColor(null);
            } else if (type.equals("scheme")&&jsonColor.has("value")) {
                STThemeColor themeColor = STThemeColor.fromValue(jsonColor.getString("value"));
                border.setThemeColor(themeColor);
                if (jsonColor.has("transformations")) {
                    JSONArray transformations = jsonColor.getJSONArray("transformations");
                    addTransformationsToBorder(border, transformations);
                }
                border.setColor(themeColorToString(operationDocument.getTheme(), themeColor));
            } else if (type.equals("auto")) {
                border.setColor("auto");
                border.setThemeColor(null);
            }
        }
    }

    public static void initShdFromJSONColor(OperationDocument operationDocument, JSONObject jsonColor, CTShd shd)
        throws JSONException {

        if(shd!=null&&jsonColor.has("type")) {
            String type = jsonColor.getString("type");
            shd.setThemeFillShade(null);
            shd.setThemeFillTint(null);
            if(type.equals("rgb")&&jsonColor.has("value")) {
                shd.setFill(jsonColor.getString("value"));
                shd.setThemeFill(null);
            } else if (type.equals("scheme")&&jsonColor.has("value")) {
                STThemeColor themeColor = STThemeColor.fromValue(jsonColor.getString("value"));
                shd.setThemeFill(themeColor);
                if (jsonColor.has("transformations")) {
                    JSONArray transformations = jsonColor.getJSONArray("transformations");
                    addTransformationsToShd(shd, transformations);
                }
                shd.setFill(themeColorToString(operationDocument.getTheme(), themeColor));
            } else if (type.equals("auto")) {
                shd.setFill("auto");
                shd.setThemeFill(null);
            }
            shd.setVal(STShd.CLEAR);
        }
    }

    public static void initColorFromJSONColor(OperationDocument operationDocument, JSONObject jsonColor, Color color)
        throws JSONException {
        if(color!=null&&jsonColor.has("type")) {
            String type = jsonColor.getString("type");
            color.setThemeTint(null);
            color.setThemeShade(null);
            if(type.equals("rgb")&&jsonColor.has("value")) {
                color.setVal(jsonColor.getString("value"));
                color.setThemeColor(null);
            } else if (type.equals("scheme")&&jsonColor.has("value")) {
                STThemeColor themeColor = STThemeColor.fromValue(jsonColor.getString("value"));
                color.setThemeColor(themeColor);
                if (jsonColor.has("transformations")) {
                    JSONArray transformations = jsonColor.getJSONArray("transformations");
                    addTransformationsToColor(color, transformations);
                }
                color.setVal(themeColorToString(operationDocument.getTheme(), themeColor));
            } else if (type.equals("auto")) {
                color.setVal("auto");
                color.setThemeColor(null);
            }
        }
    }

    private static void addTransformationsToColor(Color color, JSONArray transformations) {
        if ((transformations != null) && (transformations.length() > 0)) {
            Object temp = transformations.opt(0);
            if (temp instanceof JSONObject) {
                JSONObject transformation = (JSONObject)temp;
                String type = transformation.optString("type");
                Double value = transformation.optDouble("value");

                if ((type != null) && (value != null)) {
                    if (type.equals("tint"))
                        color.setThemeTint(Integer.toHexString((int)Math.round(value * Commons.themeTransformDiv)));
                    else if (type.equals("shade"))
                        color.setThemeShade(Integer.toHexString((int)Math.round(value * Commons.themeTransformDiv)));
                }
            }
        }
    }

    private static void addTransformationsToShd(CTShd shd, JSONArray transformations) {
        if ((transformations != null) && (transformations.length() > 0)) {
            Object temp = transformations.opt(0);
            if (temp instanceof JSONObject) {
                JSONObject transformation = (JSONObject)temp;
                String type = transformation.optString("type");
                Double value = transformation.optDouble("value");

                if ((type != null) && (value != null)) {
                    if (type.equals("tint"))
                        shd.setThemeFillTint(Integer.toHexString((int)Math.round(value * Commons.themeTransformDiv)));
                    else if (type.equals("shade"))
                        shd.setThemeFillShade(Integer.toHexString((int)Math.round(value * Commons.themeTransformDiv)));
                }
            }
        }
    }

    private static void addTransformationsToBorder(CTBorder border, JSONArray transformations) {
        if ((transformations != null) && (transformations.length() > 0)) {
            Object temp = transformations.opt(0);
            if (temp instanceof JSONObject) {
                JSONObject transformation = (JSONObject)temp;
                String type = transformation.optString("type");
                Double value = transformation.optDouble("value");

                if ((type != null) && (value != null)) {
                    if (type.equals("tint"))
                        border.setThemeTint(Integer.toHexString((int)Math.round(value * Commons.themeTransformDiv)));
                    else if (type.equals("shade"))
                        border.setThemeShade(Integer.toHexString((int)Math.round(value * Commons.themeTransformDiv)));
                }
            }
        }
    }

    public static boolean fillJsonColor(JSONObject jsonColor, STThemeColor themeColor, String STHexColor)
        throws JSONException {

        if(themeColor!=null&&themeColor.value()!=null&&themeColor.value().length()>0) {
            jsonColor.put("type", "scheme");
            jsonColor.put("value", themeColor.value());
        } else if (STHexColor!=null&&STHexColor.length()>0) {
            if(STHexColor.equals("auto"))
                jsonColor.put("type", "auto");
            else if(Commons.highlightColorToRgb.containsKey(STHexColor)){
                jsonColor.put("type", "rgb");
                jsonColor.put("value", Commons.highlightColorToRgb.get(STHexColor));
            }
            else {
                jsonColor.put("type", "rgb");
                jsonColor.put("value", STHexColor);
            }
        }
        return jsonColor.length()>0;
    }

    public static JSONObject createColor(STThemeColor themeColor, String STHexColor)
        throws JSONException {

        JSONObject jsonColor = new JSONObject();
        return fillJsonColor(jsonColor, themeColor, STHexColor) ? jsonColor : null;
    }

    public static JSONObject createFillColor(CTShd shd)
        throws JSONException {

        if (shd!=null) {
            JSONObject jsonColor = new JSONObject();
            JSONArray transformations = Utils.createFillTransformations(shd);
            if (transformations!=null) {
                jsonColor.put("transformations", transformations);
            }
            return fillJsonColor(jsonColor, shd.getThemeFill(), shd.getFill()) ? jsonColor : null;
        }
        return null;
    }

    public static JSONObject createColor(Color color)
        throws JSONException {

        if (color!=null) {
            JSONObject jsonColor = new JSONObject();
            JSONArray transformations = Utils.createTransformations(color);
            if (transformations!=null) {
                jsonColor.put("transformations", transformations);
            }
            return fillJsonColor(jsonColor, color.getThemeColor(), color.getVal()) ? jsonColor : null;
        }
        return null;
    }

    private static JSONArray createFillTransformations(CTShd shd)
        throws JSONException {

        JSONArray jsonColorTransformations = null;
        if (shd!=null) {
            String shade = shd.getThemeFillShade();
            if (shade!=null) {
                jsonColorTransformations = new JSONArray();
                jsonColorTransformations.put(Utils.createShadeTransformation(shade));
            }
            String tint = shd.getThemeFillTint();
            if (tint!=null) {
                jsonColorTransformations = (jsonColorTransformations == null) ? new JSONArray() : jsonColorTransformations;
                jsonColorTransformations.put(Utils.createTintTransformation(tint));
            }
        }
        return jsonColorTransformations;
    }

    private static JSONArray createTransformations(Color color)
        throws JSONException {

        JSONArray jsonColorTransformations = null;
        if (color!=null) {
            String shade = color.getThemeShade();
            if (shade!=null) {
                jsonColorTransformations = new JSONArray();
                jsonColorTransformations.put(Utils.createShadeTransformation(shade));
            }
            String tint = color.getThemeTint();
            if (tint!=null) {
                jsonColorTransformations = (jsonColorTransformations == null) ? new JSONArray() : jsonColorTransformations;
                jsonColorTransformations.put(Utils.createTintTransformation(tint));
            }
        }
        return jsonColorTransformations;
    }

    private static JSONObject createShadeTransformation(String shade)
        throws JSONException {

        JSONObject shadeTransform = null;
        if (shade!=null) {
            shadeTransform = new JSONObject();
            shadeTransform.put("type", "shade");
            shadeTransform.put("value", Math.round(Integer.parseInt(shade, 16) * Commons.themeTransformFactor));
        }
        return shadeTransform;
    }

    private static JSONObject createTintTransformation(String tint)
        throws JSONException {

        JSONObject tintTransform = null;
        if (tint!=null) {
            tintTransform = new JSONObject();
            tintTransform.put("type", "tint");
            tintTransform.put("value", Math.round(Integer.parseInt(tint, 16) * Commons.themeTransformFactor));
        }
        return tintTransform;
    }

    public static SectPr getDocumentProperties(MainDocumentPart mainDocPart) {

        SectPr sectPr = null;
        if(mainDocPart!=null) {
            Document doc = mainDocPart.getJaxbElement();
            if(doc!=null) {
                Body body = doc.getBody();
                if(body!=null)
                    sectPr = body.getSectPr();
            }
        }
        return sectPr;
    }

    public static long mapTwipTo100THMM(BigInteger bigInt) {
        return bigInt!=null?((bigInt.longValue() * 2540) / 72) / 20:0;
    }

    public static long mapTwipTo100THMM(long longValue) {
        return ((longValue * 2540) / 72) / 20;
    }

    public static BigInteger map100THMMToTwip(long longValue) {
        return BigInteger.valueOf( longValue * 1440 /2540 );
    }

    public static BigInteger map100THMMToTwip(double doubleValue) {
        return BigInteger.valueOf( (long)(doubleValue * 1440. /2540.) );
    }

    /**
     *
     * @param part (for docx the uri of the part is something like: /word/document.xml)
     * @param url (should be like /word/media/xxx.png)
     * @return the relationship of an url, the relationship is null if there is no relationship available
     */
    public static Relationship getRelationship(Part part, String url) {

        Relationship relationship = null;
        if(part!=null&&url!=null) {

            final RelationshipsPart relationshipsPart = part.getRelationshipsPart();
            if(relationshipsPart!=null) {

                final List<Relationship> relationships = relationshipsPart.getRelationships().getRelationship();
                if(relationships!=null) {

                    for(Relationship rel:relationships) {

                        try {

                            final String targetPath = URIHelper.resolvePartUri(part.getPartName().getURI(), new URI(rel.getTarget())).getPath();
                            if(targetPath!=null&&targetPath.equals(url)) {
                                relationship = rel;
                                break;
                            }
                        } catch (Exception e) {
                            // TODO Auto-generated catch block
                        }
                    }
                }
            }
        }
        return relationship;
    }

    /**
     * @param part
     * @param rId
     * @return the targetUrl of the corresponding relationship
     */
    public static String getUrl(Part part, String rId) {

        String url = "";

        if(part!=null&&rId!=null&&rId.length()>0) {

            final RelationshipsPart relationshipsPart = part.getRelationshipsPart();
            if(relationshipsPart!=null) {

                final Relationship relationShip = relationshipsPart.getRelationshipByID(rId);
                if(relationShip!=null) {

                    try {

                        final String targetMode = relationShip.getTargetMode();
                        if(targetMode!=null&&targetMode.equals("External"))
                            url = relationShip.getTarget();
                        else
                            url = URIHelper. resolvePartUri(part.getPartName().getURI(), new URI(relationShip.getTarget())).getPath();

                        if(url.charAt(0)=='/') {
                            url = url.substring(1);
                        }

                    } catch(Exception e) {
                        //
                    }
                }
            }
        }
        return url;
    }

    /**
     * sets an external target url to the relationship. if rId is null or empty, a new hyperlink relationship
     * will be created, otherwise the existing relationship will be reused.
     *
     * @param part
     * @param rId
     * @param Url
     * @return rId
     */
    public static String setUrl(Part part, String rId, String Url) {

        if(part!=null) {

            final RelationshipsPart relationshipsPart = part.getRelationshipsPart();
            if(relationshipsPart!=null) {
                Relationship relationShip = null;
                if(rId==null||rId.length()==0) {
                    final org.docx4j.relationships.ObjectFactory factory =
                        new org.docx4j.relationships.ObjectFactory();
                    relationShip = factory.createRelationship();
                    relationShip.setType(Namespaces.HYPERLINK);
                    relationshipsPart.addRelationship(relationShip);
                    rId = relationShip.getId();
                }
                else
                    relationShip = relationshipsPart.getRelationshipByID(rId);

                if(relationShip!=null) {
                    relationShip.setTarget(Url);
                    relationShip.setTargetMode("External");
                }
            }
        }
        return rId;
    }

    public static boolean equals(final String s1, final String s2) {
        return (s1==null&&s2==null)||((s1!=null)&&s1.equals(s2));
    }

    public static List<Object> getContent(Object o) {
        List<Object> r = null;
        if(o instanceof ContentAccessor)
            r = ((ContentAccessor)o).getContent();
        return r;
    }
}
