/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.component;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.json.JSONArray;
import org.json.JSONException;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.text.TextListHeaderElement;
import org.odftoolkit.odfdom.dom.element.text.TextListItemElement;
import org.odftoolkit.odfdom.dom.element.text.TextParagraphElementBase;
import org.odftoolkit.odfdom.dom.style.OdfStyleFamily;
import org.odftoolkit.odfdom.dom.style.props.OdfStylePropertiesSet;
import org.odftoolkit.odfdom.pkg.OdfElement;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * The component is a logical modular entity, to abstract from the
 * implementation details of the XML.
 *
 *
 * @author svante.schubertATgmail.com
 */
public class Component {

	private static final Logger LOG = Logger.getLogger(Component.class.getName());
//    
//public enum OPERATION {
//    INSERT_PARAGRAPH ("insertParagraph", 2),
//    INSERT_TEXT   ("insertText", 3);
//
//    private final String name;   // function name
//    private final int parameterCount; // number of arguments
//    Operation(String name, int parameterCount) {
//        this.name = name;
//        this.parameterCount = parameterCount;
//    }
//    public String name()   { return parameterCount; }
//    public name parameterCount() { return parameterCount; }
//
//    public double surfaceGravity() {
//        return G * mass / (radius * radius);
//    }
//    public double surfaceWeight(double otherMass) {
//        return otherMass * surfaceGravity();
//    }
//}

	/**
	 * Only being used to create the root of all components, representing the
	 * document without a parent element
	 */
	Component(OdfElement componentElement) {
		mRootElement = componentElement;
	}

	protected Component(OdfElement componentElement, Component parent) {
		mRootElement = componentElement;
		mParent = parent;
	}
	// All component children
	List<Component> mChildren;
	// the root XML element of the component
	protected OdfElement mRootElement;
	// the parent component
	Component mParent;
	private Component mRootComponent;
	/**
	 * if a repeated attribute was set at the component. In this case the
	 * positioning will change.
	 */
	boolean mHasRepeated = false;

	/**
	 * Returns the parent component
	 */
	public Component getParent() {
		return mParent;
	}

	/**
	 * Sometimes (e.g. if the child is a paragraph within list elements). The
	 * parent root element of the child component root element will not be
	 * directly children. It will be checked if there is a child element or list
	 * level 10 has reached.
	 */
	public Element getChildElementOf(Component existingComponent) {
		// element usually used for positioning of insertion
		Element existingElement = existingComponent.getRootElement();
		Element parentElement = (Element) this.getRootElement();
		// lists elements are boilerplate between paragraph parent and child component (paragraph), 
		Element existingParentElement = (Element) existingElement.getParentNode();
		// if the existing component is a paragraph with list properities
		if ((existingParentElement instanceof TextListItemElement || existingParentElement instanceof TextListHeaderElement) && existingElement instanceof TextParagraphElementBase) {
			TextParagraphElementBase existingParagraph = (TextParagraphElementBase) existingElement;							
			JsonOperationConsumer.isolateListParagraph(existingParagraph);
			Element potentialParent = (Element) existingParagraph.getParentNode();			
			while (potentialParent != null && !parentElement.equals(potentialParent)) {
				potentialParent = (Element) potentialParent.getParentNode();
				if (parentElement.equals(potentialParent)) {
					break;
				} else {
					existingElement = potentialParent;
				}
			}
		}
		return existingElement;
	}

	public Component getLastChild() {
		Component lastChild = null;
		if (mChildren != null) {
			lastChild = mChildren.get(mChildren.size());
		}
		return lastChild;
	}

	public Document getOwnerDocument() {
		return mRootElement.getOwnerDocument();
	}

	/**
	 * @return the child at the given position
	 */
	public Component getChild(int position) {
		Component c = null;
		if (mChildren != null && mChildren.size() > position) {
			c = mChildren.get(position);
		}
		return c;
	}

	public Component get(JSONArray position) {
		return get(position, false, false, 0);
	}

	/**
	 * Get descendant component by its relative position to this component.
	 * Counting starts with 0.
	 *
	 * @param postion relative position of the desired component relative to the
	 * current component
	 * @param needParent if true the parent of the given position is returned
	 * @param needFollowingSibling if true the next sibling of the given
	 * position is returned (exclusive to getPositionsFollowingSibling)
	 */
	protected Component get(JSONArray position, boolean needParent, boolean needFollowingSibling, int depth) {
		// check recursion end conditions
		Component c = null;
		final int maxDepth = position.length() - 1;
		// if not the correct depth is reached, go deeper
		if (!needParent && maxDepth > depth || (needParent && (maxDepth - 1 != depth))) {
			try {
				// get from this level the currect component
				c = getChild(position.getInt(depth));
				// call recursive this method with an additional depth for getting its child
				if (c != null) {
					c = c.get(position, needParent, needFollowingSibling, depth + 1);
				}
			} catch (JSONException ex) {
				Logger.getLogger(Component.class.getName()).log(Level.SEVERE, null, ex);
			}
		} else {
			try {
				// get the desired child
				if (needFollowingSibling) {
					c = getChild(position.getInt(depth) + 1);
				} else {
					c = getChild(position.getInt(depth));
				}
			} catch (JSONException ex) {
				Logger.getLogger(Component.class.getName()).log(Level.SEVERE, null, ex);
			}
		}
		return c;
	}

	/**
	 * Get next sibling component of the given position. Counting start with 0.
	 */
	public Component getNextSiblingOf(JSONArray position) {
		return get(position, false, true, 0);
	}

	/**
	 * Get parent component of the given position
	 */
	public Component getParentOf(JSONArray position) {
		Component c = null;
		if (position.length() == 1) {
			c = getRootComponent();
		} else {
			c = get(position, true, false, 0);
		}
		return c;
	}

	public Component getRootComponent() {
		if (mRootComponent == null) {
			Component parent = this;
			while (parent != null) {
				mRootComponent = parent;
				parent = parent.getParent();
			}
		}
		return mRootComponent;
	}

	public List getChildren() {
		return mChildren;
	}

//    /**
//     * Get child Component by position counting start with 0
//     */
//    public Component getChild(int position) {
//        Component c = null;
//        if (mChildren != null) {
//            if (position > mChildren.size() - 1) {
//                c = null;
//            } else {
//                c = mChildren.get(position);
//            }
//        }
//        return c;
//    }
//
//    /**
//     * Get the XML root element of the child component (counting start with 0)
//     */
//    public Element getChildRoot(int position) {
//        return getChild(position).getRoot();
//    }
	/**
	 * @return the root element of the component
	 */
	public OdfElement getRootElement() {
		return mRootElement;
	}

	/**
	 * @param odfElement the new root element of the component Used after
	 * splitting a list containing paragraphs and assigning the new cloned
	 * paragraph elements to the existing components.
	 */
	void setRootElement(OdfElement odfElement) {
		mRootElement = odfElement;
	}

	/**
	 * Appending a child element to the component
	 */
	public Component createChildComponent(OdfElement componentRoot) {
		componentRoot.markAsComponentRoot(true);
		return createChildComponent(-1, componentRoot);
	}

	/**
	 * Inserts a component at the given position as child
	 */
	public Component createChildComponent(int position, OdfElement newComponentRootElement) {
		Component c = null;

		// if the component is a text container (have to deal with text nodes and elements)
		if (OdfFileSaxHandler.isTextComponentRoot(newComponentRootElement)) {
			c = new TextComponent<Component>(newComponentRootElement, this);
			newComponentRootElement.setComponent(c);

		} else {
			c = new Component(newComponentRootElement, this);
			newComponentRootElement.setComponent(c);
		}
		if (!(this instanceof TextComponent)) {
			// IMPORTANT: NO ADDING OF COMPONENTS - ONLY IMPLIZIT ADDING
			add(position, c);
		}

		LOG.log(Level.FINEST, "***\n***  New Component: {0}\n*** {1}\n***", new Object[]{getPosition(c), newComponentRootElement.toString()});
		return c;
	}

	/**
	 * Adds the given component as new child component.
	 *
	 * @param index the position of the child, if -1 it will be appended
	 */
	public void add(int index, Component c) {
		if (mChildren == null) {
			if (mChildren == null) {
				mChildren = new LinkedList<Component>();
			}
		}
		if (index >= 0) {
			mChildren.add(index, c);
		} else {
			mChildren.add(c);
		}

	}

//	/** Creates a component with the given element as root XML element */
//	public Component add(JSONArray position, OdfElement componentRoot) {
//		Component c = getParentOf(position);
//		if (c == null) {
//			LOG.log(Level.WARNING, "Could not find parent of position: {0}", position);
//		}
//		Component newComponent = new Component(componentRoot, c);
//		try {
//			mChildren.add(position.getInt(position.length() - 1), newComponent);
//		} catch (JSONException ex) {
//			Logger.getLogger(Component.class.getName()).log(Level.SEVERE, null, ex);
//		}
//		return newComponent;
//	}
	/**
	 * Only removes from the component list, not from the DOM
	 */
	public Component remove(int position) {
		Component c = null;
		if (mChildren != null) {
			c = mChildren.remove(position);
		}
		return c;
	}

	/**
	 * Returns the number of child components
	 */
	public int componentLength() {
		int size = 0;
		if (mChildren != null) {
			size = mChildren.size();
		}
		return size;
	}

	public void hasRepeated(boolean hasRepeated) {
		mHasRepeated = hasRepeated;
	}

	public boolean hasRepeated() {
		return mHasRepeated;
	}

	/**
	 * @return the position as a slash separated string
	 */
	protected String getPosition(Component c) {
		String s;
		List<Integer> position;
		int childPos;
		if (c.mParent != null) {
			position = new LinkedList();
			Component parent;
			while ((parent = c.getParent()) != null) {
				childPos = parent.indexOf(c);
				if (childPos < 0) {
					childPos = indexOf(c);
				}
				position.add(childPos);
				c = parent;
			}
			StringBuilder sb = new StringBuilder();
			for (int i = position.size() - 1; i >= 0; i--) {
				sb.append("/").append(position.get(i));
			}
			s = sb.toString();
		} else {
			s = "/";
		}
		return s;
	}

	/**
	 * @returns the position of the child component C within the parents
	 * children list
	 */
	public int indexOf(Object c) {
		int index = -1;
		if (mChildren != null) {
			index = mChildren.indexOf(c);
		}
		return index;
	}

	public String toString() {
		String s = "POS:" + getPosition(this);
		if (mRootElement != null) {
			s += mRootElement.getPrefix() + ":" + mRootElement.getLocalName();
//			if (mRootElement instanceof OdfElement) {
//				s = ((OdfElement) mRootElement).toString();
//			} else {
//				s = mRootElement.toString();
//			}
		} else {
			s += "NO ROOT ELEMENT!!!";
//			if (mChildren != null) {
//				StringBuffer sb = new StringBuffer();
//				for (Component c : mChildren) {
//					sb.append(c.toString()).append('\n');
//				}
//				s += sb.toString();
//			}
		}
		return s;
	}

	// These sets should be static initialized and a central place, best generated at the family superclass
	public static Map<String, OdfStylePropertiesSet> getAllOxStyleGroupingIdProperties(OdfStylableElement styleElement) {
		return getAllOxStyleGroupingIdProperties(styleElement.getStyleFamily());
	}

	public static Map<String, OdfStylePropertiesSet> getAllOxStyleGroupingIdProperties(OdfStyleFamily styleFamily) {
		Map<String, OdfStylePropertiesSet> familyProperties = new HashMap<String, OdfStylePropertiesSet>();
		if (styleFamily.equals(OdfStyleFamily.Paragraph)) {
			familyProperties.put("paragraph", OdfStylePropertiesSet.ParagraphProperties);
			familyProperties.put("character", OdfStylePropertiesSet.TextProperties);
		} else if (styleFamily.equals(OdfStyleFamily.Text)) {
			familyProperties.put("character", OdfStylePropertiesSet.TextProperties);
		} else if (styleFamily.equals(OdfStyleFamily.Table)) {
			familyProperties.put("table", OdfStylePropertiesSet.TableProperties);
		} else if (styleFamily.equals(OdfStyleFamily.TableRow)) {
			familyProperties.put("row", OdfStylePropertiesSet.TableRowProperties);
		} else if (styleFamily.equals(OdfStyleFamily.TableCell)) {
			familyProperties.put("cell", OdfStylePropertiesSet.TableCellProperties);
			familyProperties.put("paragraph", OdfStylePropertiesSet.ParagraphProperties);
			familyProperties.put("character", OdfStylePropertiesSet.TextProperties); //changed due to OX naming from text to character
		} else if (styleFamily.equals(OdfStyleFamily.TableColumn)) {
			familyProperties.put("column", OdfStylePropertiesSet.TableColumnProperties);
		} else if (styleFamily.equals(OdfStyleFamily.Section)) {
			familyProperties.put("section", OdfStylePropertiesSet.SectionProperties);
		} else if (styleFamily.equals(OdfStyleFamily.List)) {
			familyProperties.put("list", OdfStylePropertiesSet.ListLevelProperties);
		} else if (styleFamily.equals(OdfStyleFamily.Chart)) {
			familyProperties.put("chart", OdfStylePropertiesSet.ChartProperties);
			familyProperties.put("drawing", OdfStylePropertiesSet.GraphicProperties); //changed due to OX naming from graphic to drawing
			familyProperties.put("paragraph", OdfStylePropertiesSet.ParagraphProperties);
			familyProperties.put("character", OdfStylePropertiesSet.TextProperties); //changed due to OX naming from text to character
		} else if (styleFamily.equals(OdfStyleFamily.Graphic) || styleFamily.equals(OdfStyleFamily.Presentation)) {
			familyProperties.put("drawing", OdfStylePropertiesSet.GraphicProperties); //changed due to OX naming from graphic to drawing
			familyProperties.put("paragraph", OdfStylePropertiesSet.ParagraphProperties);
			familyProperties.put("character", OdfStylePropertiesSet.TextProperties); //changed due to OX naming from text to character
		} else if (styleFamily.equals(OdfStyleFamily.DrawingPage)) {
			familyProperties.put("drawing", OdfStylePropertiesSet.DrawingPageProperties);
		} else if (styleFamily.equals(OdfStyleFamily.Ruby)) {
			familyProperties.put("ruby", OdfStylePropertiesSet.RubyProperties);
		}
		return familyProperties;
	}

	// In the end, this meths should be better moved to the component class
	public static String getFamilyID(OdfStylableElement styleElement) {
		return getFamilyID(styleElement.getStyleFamily());
	}

	// In the end, this meths should be better moved to the component class
	public static String getMainOxStyleGroupingId(OdfStylableElement styleElement) {
		return getMainOxStyleGroupingId(styleElement.getStyleFamily());
	}

	// In the end, this meths should be better moved to the component class
	public static String getStyleNamePrefix(OdfStylableElement styleElement) {
		return getStyleNamePrefix(styleElement.getStyleFamily());
	}

	// In the end, this meths should be better moved to the component class
	public static String getMainOxStyleGroupingId(OdfStyleFamily styleFamily) {
		String familyID = null;
		if (styleFamily.equals(OdfStyleFamily.Paragraph)) {
			familyID = "paragraph";
		} else if (styleFamily.equals(OdfStyleFamily.Text)) {
			familyID = "character";
		} else if (styleFamily.equals(OdfStyleFamily.Table)) {
			familyID = "table";
		} else if (styleFamily.equals(OdfStyleFamily.TableRow)) {
			familyID = "row";
		} else if (styleFamily.equals(OdfStyleFamily.TableCell)) {
			familyID = "cell";
		} else if (styleFamily.equals(OdfStyleFamily.TableColumn)) {
			familyID = "column";
		} else if (styleFamily.equals(OdfStyleFamily.Section)) {
			familyID = "section";
		} else if (styleFamily.equals(OdfStyleFamily.List)) {
			familyID = "list";
		} else if (styleFamily.equals(OdfStyleFamily.Chart)) {
			familyID = "chart";
		} else if (styleFamily.equals(OdfStyleFamily.Graphic) || styleFamily.equals(OdfStyleFamily.Presentation)) {
			familyID = "drawing";
		} else if (styleFamily.equals(OdfStyleFamily.DrawingPage)) {
			familyID = "drawing";
		} else if (styleFamily.equals(OdfStyleFamily.Ruby)) {
			familyID = "ruby";
		}
		return familyID;
	}

	// In the end, this meths should be better moved to the component class
	public static String getStyleNamePrefix(OdfStyleFamily styleFamily) {
		String familyID = null;
		if (styleFamily.equals(OdfStyleFamily.Paragraph) || styleFamily.equals(OdfStyleFamily.Text) || styleFamily.equals(OdfStyleFamily.Section) || styleFamily.equals(OdfStyleFamily.List) || styleFamily.equals(OdfStyleFamily.Ruby)) {
			familyID = "text";
		} else if (styleFamily.equals(OdfStyleFamily.Table) || styleFamily.equals(OdfStyleFamily.TableRow) || styleFamily.equals(OdfStyleFamily.TableCell) || styleFamily.equals(OdfStyleFamily.TableColumn)) {
			familyID = "table";
		} else if (styleFamily.equals(OdfStyleFamily.Chart)) {
			familyID = "chart";
		} else if (styleFamily.equals(OdfStyleFamily.Graphic) || styleFamily.equals(OdfStyleFamily.Presentation) || styleFamily.equals(OdfStyleFamily.DrawingPage)) {
			familyID = "draw";
		} else if (styleFamily.equals(OdfStyleFamily.Presentation)) {
			familyID = "presentation";
		}
		return familyID;
	}

	// In the end, this meths should be better moved to the component class
	public static String getFamilyID(OdfStyleFamily styleFamily) {
		String familyID = null;
		if (styleFamily.equals(OdfStyleFamily.Paragraph)) {
			familyID = "paragraph";
		} else if (styleFamily.equals(OdfStyleFamily.Text)) {
			familyID = "character";
		} else if (styleFamily.equals(OdfStyleFamily.Table)) {
			familyID = "table";
		} else if (styleFamily.equals(OdfStyleFamily.TableRow)) {
			familyID = "row";
		} else if (styleFamily.equals(OdfStyleFamily.TableCell)) {
			familyID = "cell";
		} else if (styleFamily.equals(OdfStyleFamily.TableColumn)) {
			familyID = "column";
		} else if (styleFamily.equals(OdfStyleFamily.Section)) {
			familyID = "section";
		} else if (styleFamily.equals(OdfStyleFamily.List)) {
			familyID = "list";
		} else if (styleFamily.equals(OdfStyleFamily.Chart)) {
			familyID = "chart";
		} else if (styleFamily.equals(OdfStyleFamily.Graphic) || styleFamily.equals(OdfStyleFamily.Presentation)) {
			familyID = "drawing";
		} else if (styleFamily.equals(OdfStyleFamily.DrawingPage)) {
			familyID = "drawing";
		} else if (styleFamily.equals(OdfStyleFamily.Ruby)) {
			familyID = "ruby";
		}
		return familyID;
	}

// In the end, this meths should be better moved to the component class
	public static String getFamilyDisplayName(OdfStyleFamily styleFamily) {
		String familyID = null;
		if (styleFamily.equals(OdfStyleFamily.Paragraph)) {
			familyID = "Paragraph";
		} else if (styleFamily.equals(OdfStyleFamily.Text)) {
			familyID = "Character";
		} else if (styleFamily.equals(OdfStyleFamily.Table)) {
			familyID = "Table";
		} else if (styleFamily.equals(OdfStyleFamily.TableRow)) {
			familyID = "Row";
		} else if (styleFamily.equals(OdfStyleFamily.TableCell)) {
			familyID = "Cell";
		} else if (styleFamily.equals(OdfStyleFamily.TableColumn)) {
			familyID = "Column";
		} else if (styleFamily.equals(OdfStyleFamily.Section)) {
			familyID = "Section";
		} else if (styleFamily.equals(OdfStyleFamily.List)) {
			familyID = "List";
		} else if (styleFamily.equals(OdfStyleFamily.Presentation)) {
			familyID = "Presentation";
		} else if (styleFamily.equals(OdfStyleFamily.Chart)) {
			familyID = "Chart";
		} else if (styleFamily.equals(OdfStyleFamily.Graphic)) {
			familyID = "Graphic";
		} else if (styleFamily.equals(OdfStyleFamily.DrawingPage)) {
			familyID = "Drawing";
		} else if (styleFamily.equals(OdfStyleFamily.Ruby)) {
			familyID = "Ruby";
		}
		return familyID;
	}

	// In the end, this meths should be better moved to the component class
	/**
	 * @return styleFamilyValue the <code>String</code> value * * *
	 * of <code>StyleFamilyAttribute</code>,
	 */
	public static String getFamilyName(String styleId) {
		String familyID = null;
		if (styleId.equals("paragraph")) {
			familyID = OdfStyleFamily.Paragraph.getName();
		} else if (styleId.equals("character")) {
			familyID = OdfStyleFamily.Text.getName();
		} else if (styleId.equals("table")) {
			familyID = OdfStyleFamily.Table.getName();
		} else if (styleId.equals("row")) {
			familyID = OdfStyleFamily.TableRow.getName();
		} else if (styleId.equals("cell")) {
			familyID = OdfStyleFamily.TableCell.getName();
		} else if (styleId.equals("column")) {
			familyID = OdfStyleFamily.TableColumn.getName();
		} else if (styleId.equals("graphic")) {
			familyID = OdfStyleFamily.Graphic.getName();
		}
		return familyID;
	}

	// In the end, this meths should be better moved to the component class
	/**
	 * @return styleFamily the <code>OdfStyleFamily</code> representation * * *
	 * of <code>StyleFamilyAttribute</code>,
	 */
	public static OdfStyleFamily getFamily(String styleId) {
		OdfStyleFamily family = null;
		if (styleId.equals("paragraph")) {
			family = OdfStyleFamily.Paragraph;
		} else if (styleId.equals("character")) {
			family = OdfStyleFamily.Text;
		} else if (styleId.equals("table")) {
			family = OdfStyleFamily.Table;
		} else if (styleId.equals("row")) {
			family = OdfStyleFamily.TableRow;
		} else if (styleId.equals("cell")) {
			family = OdfStyleFamily.TableCell;
		} else if (styleId.equals("column")) {
			family = OdfStyleFamily.TableColumn;
		} else if (styleId.equals("drawing")) {
			family = OdfStyleFamily.Graphic;
		}
		return family;
	}
}
