/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/headerpane',
    ['io.ox/core/event',
     'io.ox/office/tk/utils',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Events, Utils, SheetUtils) {

    'use strict';

    var // the ratio of the visible pane size to be added before and after
        ADDITIONAL_SIZE_RATIO = 0.5,

        // size of the resizer nodes, in pixels
        RESIZER_SIZE = 6,

        // margin at trailing border of the entire sheet area, in pixels
        SCROLL_AREA_MARGIN = 30,

        // the names of all tracking events but 'tracking:start'
        TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel';

    // class HeaderPane =======================================================

    /**
     * Represents a single row or column header pane in the spreadsheet view.
     * If the view has been split or frozen at a specific cell position, the
     * view will consist of up to four header panes (left and right column
     * header panes, and top and bottom row header panes).
     *
     * Triggers the following events:
     * - 'change:interval': After the column/row interval shown by this header
     *      pane has been changed. Event handlers receive the new column/row
     *      interval.
     * - 'select:start': After selection tracking has been started.
     * - 'select:update': After the selection has changed while selection
     *      tracking is active.
     * - 'select:end': After selection tracking has been finished (either
     *      successfully or not).
     * - 'resize:start': After column/row resize tracking has been started.
     *      Receives the current offset and initial size of the column/row.
     * - 'resize:update': After the new column/row size has changed while
     *      resize tracking is active. Receives the current offset and size of
     *      the column/row.
     * - 'resize:end': After column/row resize tracking has been finished
     *      (either successfully or not).
     *
     * @constructor
     *
     * @extends Events
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this header pane.
     *
     * @param {String} paneSide
     *  The position of this header pane (one of 'left', 'right', 'top',
     *  'bottom', or 'corner').
     *
     * @param {Boolean} columns
     *  Whether this header pane contains column headers (true) or row headers
     *  (false).
     */
    function HeaderPane(app, paneSide, columns) {

        var // self reference
            self = this,

            // the spreadsheet view
            view = app.getView(),

            // type identifier of this header pane, either 'columns' or 'rows'
            paneType = columns ? 'columns' : 'rows',

            // the container node of this header pane (must be focusable for tracking support)
            rootNode = $('<div>').addClass('header-pane unselectable ' + paneType).attr('tabindex', 0),

            // the content node with the header cell and resizer nodes
            contentNode = $('<div>').addClass('header-content noI18n').appendTo(rootNode),

            // whether this header pane is currently visible
            visible = false,

            // the size of this header pane usable to show column/row header cells
            paneSize = 0,

            // the column/row collection of the active sheet
            collection = null,

            // the column/row interval to be displayed in the DOM
            interval = null,

            // the indexes of all visible columns/rows, as array
            indexes = null,

            // the absolute position and size of the column/row interval in the sheet
            position = { offset: 0, size: 0 },

            // the absolute scroll position in the associated grid panes
            scrollPos = 0,

            // distance between cell A1 and the top-left visible cell of this pane
            hiddenSize = 0,

            // the type of the current tracking mode ('select', 'resize', null)
            trackingType = null;

        // base constructor ---------------------------------------------------

        Events.extend(this);

        // private methods ----------------------------------------------------

        /**
         * Returns an array of indexes of all visible columns/rows in the
         * passed interval.
         */
        function getVisibleIndexes(interval) {

            var // the array index of the interval start position
                index1 = _(indexes).sortedIndex(interval.first),
                // the array index after the interval end position
                index2 = _(indexes).sortedIndex(interval.last + 1);

            return indexes.slice(index1, index2 - index1);
        }

        /**
         * Renders all column header cells in the current column interval.
         */
        function renderColumnCells() {

            var // the height of the entire header pane (used as line height)
                lineHeight = rootNode.height(),
                // the HTML mark-up for all visible column header cells
                markup = '';

            // generate HTML mark-up text for all visible header cells
            collection.iterateVisibleEntries(interval, function (colEntry) {

                var cellStyleAttr = ' style="left:' + self.getRelativeOffset(colEntry.offset) + 'px;width:' + colEntry.size + 'px;line-height:' + lineHeight + 'px;"',
                    cellText = SheetUtils.getColName(colEntry.index),
                    resizeStyleAttr = ' style="left:' + self.getRelativeOffset(colEntry.offset + colEntry.size - RESIZER_SIZE / 2) + 'px;width:' + RESIZER_SIZE + 'px;"',
                    dataAttr = ' data-index="' + colEntry.index + '"';

                markup += '<div class="cell"' + cellStyleAttr + dataAttr + '>' + cellText + '</div><div class="resizer"' + resizeStyleAttr + dataAttr + '></div>';
            });

            // insert entire HTML mark-up into the cell container node
            contentNode[0].innerHTML = markup;
        }

        /**
         * Renders all row header cells in the current row interval.
         */
        function renderRowCells() {

            var // the HTML mark-up for all visible row header cells
                markup = '';

            // generate HTML mark-up text for all visible header cells
            collection.iterateVisibleEntries(interval, function (rowEntry) {

                var cellStyleAttr = ' style="top:' + self.getRelativeOffset(rowEntry.offset) + 'px;height:' + rowEntry.size + 'px;line-height:' + rowEntry.size + 'px;"',
                    cellText = SheetUtils.getRowName(rowEntry.index),
                    resizeStyleAttr = ' style="top:' + self.getRelativeOffset(rowEntry.offset + rowEntry.size - RESIZER_SIZE / 2) + 'px;height:' + RESIZER_SIZE + 'px;"',
                    dataAttr = ' data-index="' + rowEntry.index + '"';

                markup += '<div class="cell"' + cellStyleAttr + dataAttr + '>' + cellText + '</div><div class="resizer"' + resizeStyleAttr + dataAttr + '></div>';
            });

            // insert entire HTML mark-up into the cell container node
            contentNode[0].innerHTML = markup;
        }

        /**
         * Redraws the selection highlighting in the header cells of this
         * header pane.
         */
        function renderSelection(selection) {

            var // the merged ranges of selected columns/rows
                selectionIntervals = (columns ? SheetUtils.getColIntervals : SheetUtils.getRowIntervals)(selection.ranges),
                // current index into the intervals array
                arrayIndex = 0;

            // find the first selection interval that ends at or before the passed index
            function findSelectionInterval(index) {
                while ((arrayIndex < selectionIntervals.length) && (selectionIntervals[arrayIndex].last < index)) { arrayIndex += 1; }
                return selectionIntervals[arrayIndex];
            }

            // find the first interval, and iterate all cell nodes
            findSelectionInterval(interval.first);
            Utils.iterateSelectedDescendantNodes(contentNode, '.cell', function (cellNode) {
                var index = Utils.getElementAttributeAsInteger(cellNode, 'data-index'),
                    selectionInterval = findSelectionInterval(index);
                $(cellNode).toggleClass(Utils.SELECTED_CLASS, _.isObject(selectionInterval) && (selectionInterval.first <= index));
            }, undefined, { children: true });
        }

        /**
         * Updates the relative scroll position and size of the content node.
         */
        function updateContentNodePosition() {
            if (columns) {
                contentNode.css({ left: position.offset - self.getVisibleOffset(), width: position.size });
            } else {
                contentNode.css({ top: position.offset - self.getVisibleOffset(), height: position.size });
            }
        }

        /**
         * Recalculates the current row/column interval and its position and
         * size in the sheet, and renders all visible header cells and the
         * selection on demand.
         */
        function updateAndRenderInterval(operation, changedInterval) {

            var // the absolute offset of the first visible pixel
                visibleOffset = self.getVisibleOffset(),
                // the absolute offset behind the last visible pixel
                visibleEndOffset = visibleOffset + paneSize,
                // the end offset of the available header cells
                intervalEndOffset = position.offset + position.size,
                // the data of the new first or last column/row
                entryData = null,
                // the old visible indexes
                oldIndexes = indexes,
                // all information passed to the change event
                eventOptions = null;

            // recalculate the interval on demand (missing, or visible area near its borders)
            if (_.isString(operation) || !interval ||
                    ((position.offset > 0) && (visibleOffset - position.offset < paneSize * ADDITIONAL_SIZE_RATIO / 3)) ||
                    ((intervalEndOffset < collection.getTotalSize()) && (intervalEndOffset - visibleEndOffset < paneSize * ADDITIONAL_SIZE_RATIO / 3))
            ) {

                Utils.takeTime('HeaderPane.updateAndRenderInterval(): paneSide=' + paneSide + ', recalculating interval', function () {
                    interval = {};

                    // start position of the interval
                    entryData = collection.getEntryByOffset(visibleOffset - paneSize * ADDITIONAL_SIZE_RATIO);
                    interval.first = entryData.index;
                    position.offset = entryData.offset;

                    // end position of the interval
                    entryData = collection.getEntryByOffset(visibleOffset + paneSize * (1 + ADDITIONAL_SIZE_RATIO));
                    interval.last = entryData.index;
                    position.size = entryData.offset + entryData.size - position.offset;

                    // set the new array of visible column/row indexes
                    indexes = collection.getVisibleIndexes(interval);

                    // calculate data for the change event
                    if (oldIndexes) {
                        eventOptions = {
                            removed: _(oldIndexes).difference(indexes),
                            added: _(indexes).difference(oldIndexes)
                        };
                    } else {
                        eventOptions = { added: _.clone(indexes) };
                    }
                });

                // render all header cells and the selection
                Utils.takeTime('HeaderPane.updateAndRenderInterval(): paneSide=' + paneSide + ', rendering interval ' + (columns ? SheetUtils.getColIntervalName : SheetUtils.getRowIntervalName)(interval), function () {
                    updateContentNodePosition();
                    (columns ? renderColumnCells : renderRowCells)();
                    renderSelection(view.getSelection());
                });

                // notify listeners
                Utils.info('HeaderPane.updateAndRenderInterval(): paneSide=' + paneSide + ', triggering change for interval ' + (columns ? SheetUtils.getColIntervalName : SheetUtils.getRowIntervalName)(interval));
                if (eventOptions.removed) { Utils.log('  removed ' + paneType + ': ' + _(eventOptions.removed).map(columns ? SheetUtils.getColName : SheetUtils.getRowName).join(',')); }
                if (eventOptions.added) { Utils.log('  added ' + paneType + ': ' + _(eventOptions.added).map(columns ? SheetUtils.getColName : SheetUtils.getRowName).join(',')); }
                self.trigger('change:interval', _.clone(interval), eventOptions);

            } else {
                // update the relative scroll position and size of the content node
                updateContentNodePosition();
            }
        }

        /**
         * Registers an event handler at the view instance that will only be
         * called when this header pane is visible.
         */
        function registerViewEventHandler(type, handler) {
            view.on(type, function () {
                if (visible) {
                    handler.apply(self, _.toArray(arguments));
                }
            });
        }

        /**
         * Initializes this header pane after the active sheet has been
         * changed.
         */
        function changeActiveSheetHandler() {

            var // the model of the active sheet
                sheetModel = view.getActiveSheetModel();

            // get column/row collection from the current active sheet
            collection = columns ? sheetModel.getColCollection() : sheetModel.getRowCollection();
            // always recalculate the column/row interval of this pane
            interval = indexes = null;
        }

        /**
         * Redraws the selected header cells of this header pane, after the
         * selection in the active sheet has been changed.
         */
        function changeSelectionHandler(event, selection) {
            if (trackingType === 'resize') {
                $.cancelTracking();
            }
            renderSelection(selection);
        }

        /**
         * Handles inserted entries in the column/row collection.
         */
        function insertEntriesHandler(event, changedInterval) {
            if (changedInterval.first <= interval.last) {
                updateAndRenderInterval('insert', changedInterval);
            }
        }

        /**
         * Handles deleted entries in the column/row collection.
         */
        function deleteEntriesHandler(event, changedInterval) {
            if (changedInterval.first <= interval.last) {
                updateAndRenderInterval('delete', changedInterval);
            }
        }

        /**
         * Handles changed entries in the column/row collection.
         */
        function changeEntriesHandler(event, changedInterval) {
            if (changedInterval.first <= interval.last) {
                updateAndRenderInterval('change', changedInterval);
            }
        }

        /**
         * Returns the layout data of the column/row matching the 'data-index'
         * attribute of the target DOM node in the passed 'tracking:start'
         * event.
         *
         * @param {jQuery.Event} event
         *  The 'tracking:start' event.
         *
         * @returns {Object}
         *  The layout data of the matching column/row, in the properties
         *  'index', 'offset', and 'size'.
         */
        function getTrackingStartEntryData(event) {
            return collection.getEntry(Utils.getElementAttributeAsInteger(event.target, 'data-index', -1));
        }

        /**
         * Returns the current absolute sheet position in the header pane for
         * the passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event.
         *
         * @returns {Number}
         *  The absolute sheet position, in pixels.
         */
        function getTrackingOffset(event) {
            return (columns ? (event.pageX - contentNode.offset().left) : (event.pageY - contentNode.offset().top)) + position.offset;
        }

        /**
         * Handles all tracking events while cell selection is active.
         */
        var selectionTrackingHandler = (function () {

            var // the column/row index where tracking has started
                startIndex = null,
                // current index (prevent updates while tracking over the same column/row)
                currentIndex = -1;

            function makeCell(index1, index2) {
                return columns ? [index1, index2] : [index2, index1];
            }

            // makes an adjusted range between start index and current index
            function makeRange() {
                var selectionInterval = { first: Math.min(startIndex, currentIndex), last: Math.max(startIndex, currentIndex) };
                return app.getModel()[columns ? 'makeColRange' : 'makeRowRange'](selectionInterval);
            }

            // initializes the selection according to the passed tracking event
            function initializeTracking(event) {

                var // the layout data of the initial column/row
                    entryData = getTrackingStartEntryData(event),
                    // whether to append a new range to the selection
                    append = !event.shiftKey && (event.ctrlKey || event.metaKey),
                    // whether to extend the active range
                    extend = event.shiftKey && !event.ctrlKey && !event.metaKey;

                currentIndex = entryData.index;
                if (extend) {
                    // pressed SHIFT key: modify the active range (from active cell to tracking start position)
                    startIndex = view.getActiveCell()[columns ? 0 : 1];
                    view.changeActiveRange(makeRange());
                } else {
                    // no SHIFT key: create/append a new selection range
                    startIndex = currentIndex;
                    // set active cell to current first visible cell in the column/row
                    view.selectRange(makeRange(), {
                        append: append,
                        active: makeCell(startIndex, Math.max(0, view.getVisibleHeaderPane(!columns).getFirstVisibleIndex()))
                    });
                }
                view.scrollPaneSideToEntry(paneSide, currentIndex);
                self.trigger('select:start');
            }

            // updates the selection according to the passed tracking event
            function updateSelection(event) {

                var // the column/row entry currently hovered
                    entryData = collection.getEntryByOffset(getTrackingOffset(event));

                if ((currentIndex >= 0) && (entryData.index !== currentIndex)) {
                    currentIndex = entryData.index;
                    view.changeActiveRange(makeRange());
                    self.trigger('select:update');
                }
            }

            // updates the scroll position while tracking
            function updateScroll(event) {
                view.scrollPaneSide(paneSide, scrollPos + (columns ? event.scrollX : event.scrollY));
            }

            // finalizes the selection tracking
            function finalizeTracking(event) {
                if (currentIndex >= 0) {
                    view.scrollPaneSideToEntry(paneSide, currentIndex);
                    currentIndex = -1;
                }
                self.trigger('select:end');
                trackingType = null;
                $(event.delegateTarget).off(TRACKING_EVENT_NAMES);
                view.grabFocus();
            }

            // return the actual selectionTrackingHandler() method
            return function (event) {

                switch (event.type) {
                case 'tracking:start':
                    initializeTracking(event);
                    break;
                case 'tracking:move':
                    updateSelection(event);
                    break;
                case 'tracking:scroll':
                    updateScroll(event);
                    updateSelection(event);
                    break;
                case 'tracking:end':
                case 'tracking:cancel':
                    // cancel selection tracking: keep current selection
                    updateSelection(event);
                    finalizeTracking(event);
                    break;
                }
            };

        }()); // end of selectionTrackingHandler() local scope

        /**
         * Handles all tracking events while resizing a column/row is active.
         */
        var resizeTrackingHandler = (function () {

            var // the layout data of the resized column/row
                entryData = null,
                // difference between exact start position and trailing position of column/row
                correction = 0,
                // the current size of the column/row while tracking
                currentSize = 0,
                // minimum/maximum scroll position allowed while resizing
                minScrollPos = 0, maxScrollPos = 0,
                // maximum size of columns/rows
                MAX_SIZE = Utils.convertHmmToLength(columns ? SheetUtils.MAX_COLUMN_WIDTH : SheetUtils.MAX_ROW_HEIGHT, 'px', 1);

            // initializes column/row resizing according to the passed tracking event
            function initializeTracking(event) {
                entryData = getTrackingStartEntryData(event);
                // calculate difference between exact tracking start position and end of column/row
                // (to prevent unwanted resize actions without moving the tracking point)
                correction = getTrackingOffset(event) - (entryData.offset + entryData.size);
                currentSize = entryData.size;
                minScrollPos = Math.max(0, entryData.offset - 20 - hiddenSize);
                maxScrollPos = Math.max(minScrollPos, entryData.offset + MAX_SIZE + 20 - paneSize - hiddenSize);
                self.trigger('resize:start', entryData.offset, currentSize);
            }

            // updates column/row resizing according to the passed tracking event
            function updateSize(event) {
                currentSize = Utils.minMax(getTrackingOffset(event) - correction - entryData.offset, 0, MAX_SIZE);
                self.trigger('resize:update', entryData.offset, currentSize);
            }

            // updates scroll position according to the passed tracking event
            function updateScroll(event) {
                var newScrollPos = scrollPos + (columns ? event.scrollX : event.scrollY);
                if ((newScrollPos < scrollPos) && (scrollPos > minScrollPos)) {
                    view.scrollPaneSide(paneSide, Math.max(newScrollPos, minScrollPos));
                } else if ((newScrollPos > scrollPos) && (scrollPos < maxScrollPos)) {
                    view.scrollPaneSide(paneSide, Math.min(newScrollPos, maxScrollPos));
                }
            }

            // finalizes the resize tracking
            function finalizeTracking(event, applySize) {
                if (applySize && (currentSize !== entryData.size)) {
                    // pass tracked column/row as target index, will cause to modify all columns/rows selected completely
                    view[columns ? 'setColumnWidth' : 'setRowHeight'](Utils.convertLengthToHmm(currentSize, 'px'), entryData.index);
                }
                self.trigger('resize:end');
                trackingType = null;
                $(event.delegateTarget).off(TRACKING_EVENT_NAMES);
                view.grabFocus();
            }

            // return the actual resizeTrackingHandler() method
            return function (event) {

                switch (event.type) {
                case 'tracking:start':
                    initializeTracking(event);
                    break;
                case 'tracking:move':
                    updateSize(event);
                    break;
                case 'tracking:scroll':
                    updateScroll(event);
                    updateSize(event);
                    break;
                case 'tracking:end':
                    updateSize(event);
                    finalizeTracking(event, true);
                    break;
                case 'tracking:cancel':
                    finalizeTracking(event, false);
                    break;
                }
            };

        }()); // end of resizeTrackingHandler() local scope

        /**
         * Handles 'tracking:start' events end decides whether to select cells,
         * or to resize columns or rows.
         */
        function trackingStartHandler(event) {
            if ($(event.target).is('.cell')) {
                trackingType = 'select';
                $(event.delegateTarget).on(TRACKING_EVENT_NAMES, selectionTrackingHandler);
                selectionTrackingHandler.call(this, event);
            } else if ($(event.target).is('.resizer')) {
                trackingType = 'resize';
                $(event.delegateTarget).on(TRACKING_EVENT_NAMES, resizeTrackingHandler);
                resizeTrackingHandler.call(this, event);
            }
        }

        /**
         * Handles a changed document edit mode. If switched to read-only mode,
         * cancels a running resize columns/rows tracking action.
         */
        function editModeHandler(event, editMode) {
            if (!editMode && (trackingType === 'resize')) {
                $.cancelTracking();
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the root DOM node of this header pane.
         *
         * @returns {jQuery}
         *  The root node of this header pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Initializes the settings and layout of this header pane.
         *
         * @param {Object} settings
         *  The view settings of the pane side represented by this header pane.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (settings) {

            // copy passed settings
            visible = settings.visible;
            paneSize = settings.size;
            hiddenSize = settings.hiddenSize;

            // visibility
            rootNode.toggle(visible);
            if (visible) {

                // position and size
                rootNode.css(columns ?
                    { left: settings.offset, width: paneSize, height: view.getCornerPane().getNode().height() } :
                    { top: settings.offset, width: view.getCornerPane().getNode().width(), height: paneSize }
                );

                // initialize auto-scrolling
                rootNode.enableTracking({
                    selector: '.cell, .resizer',
                    autoScroll: settings.scrollable ? (columns ? 'horizontal' : 'vertical') : false,
                    borderNode: rootNode,
                    borderMargin: -30,
                    borderSize: 60,
                    minSpeed: 2,
                    maxSpeed: 250
                });

                // recalculate the interval and header cells
                updateAndRenderInterval();
            }

            return this;
        };

        /**
         * Returns whether this header pane is currently visible.
         *
         * @returns {Boolean}
         *  Whether this header pane is currently visible.
         */
        this.isVisible = function () {
            return visible;
        };

        /**
         * Returns the column/row interval currently shown in this header pane.
         *
         * @returns {Object}
         *  The column/row interval of this header pane, in the zero-based
         *  index properties 'first' and 'last'.
         */
        this.getInterval = function () {
            return _.clone(interval);
        };

        /**
         * Returns the absolute position and size of the column/row interval
         * currently shown in this header pane.
         *
         * @returns {Object}
         *  The absolute position of the column/row interval of this header
         *  pane in the sheet, in the properties 'offset' and 'size', in
         *  pixels.
         */
        this.getIntervalPosition = function () {
            return _.clone(position);
        };

        /**
         * Converts the passed offset to a position relative to the start
         * offset of the current column/row interval.
         *
         * @returns {Number}
         *  The relative offset, in pixels.
         */
        this.getRelativeOffset = function (offset) {
            return offset - position.offset;
        };

        /**
         * Returns the current scroll position of this header pane.
         *
         * @returns {Number}
         *  The current scroll position, in pixels.
         */
        this.getScrollPos = function () {
            return scrollPos;
        };

        /**
         * Calculates and returns the size of the maximum possible scrollable
         * area for the entire sheet, regardless of its used area.
         *
         * @returns {Number}
         *  The size of the maximum scrollable area, in pixels.
         */
        this.getTotalScrollSize = function () {
            return collection.getTotalSize() - hiddenSize + SCROLL_AREA_MARGIN;
        };

        /**
         * Calculates and returns the size of the scrollable area according to
         * the used area in the sheet, and the current scroll position. Tries
         * to reserve additional space for the full size of this header pane at
         * the trailing border, to be able to scroll page-by-page into the
         * unused area of the sheet.
         *
         * @returns {Number}
         *  The size of the scrollable area, in pixels.
         */
        this.getCurrentScrollSize = function () {

            var // the number of columns/rows in the used area
                usedCount = columns ? view.getUsedCols() : view.getUsedRows(),
                // the total size of the used area in the sheet, in pixels
                usedSize = (usedCount === 0) ? 0 : collection.getIntervalPosition({ first: 0, last: usedCount - 1 }).size,
                // the absolute size of the scrollable area
                scrollSize = 0;

            // scrollable area includes the used area, and the current (but extended) scroll position
            scrollSize = Math.max(usedSize, this.getVisibleOffset() + 2 * paneSize);

            // restrict to the total size of the sheet, plus a little margin
            scrollSize = Math.min(scrollSize, collection.getTotalSize() + SCROLL_AREA_MARGIN);

            // remove the hidden area (the header pane may not start at cell A1 in frozen mode)
            return scrollSize - hiddenSize;
        };

        /**
         * Returns the absolute position of the leading border of the visible
         * area currently shown in this header pane.
         *
         * @returns {Number}
         *  The absolute position of the visible area of this header pane in
         *  the sheet, in pixels.
         */
        this.getVisibleOffset = function () {
            return hiddenSize + scrollPos;
        };

        /**
         * Returns the zero-based index of the first visible column/row in this
         * header pane. The column/row is considered visible, it at least half
         * of its total size is really visible.
         *
         * @returns {Number}
         *  The zero-based index of the first visible column/row.
         */
        this.getFirstVisibleIndex = function () {

            var // the offset of the first visible pixel
                offset = this.getVisibleOffset(),
                // the layout data of the column/row containing the first visible offset
                entryData = collection.getEntryByOffset(offset),
                // the column/row index
                index = entryData.index;

            // check if the cell is visible less than half, go to next cell in that case
            if ((entryData.offset + entryData.size / 2 < offset) && (entryData = collection.getNextVisibleEntry(index + 1))) {
                index = entryData.index;
            }

            return index;
        };

        /**
         * Changes the scroll position of this header pane.
         *
         * @param {Number} pos
         *  The new scroll position.
         *
         * @returns {HeaderPane}
         *  A reference to this instance.
         */
        this.scroll = function (pos) {
            scrollPos = Math.max(0, pos);
            updateAndRenderInterval();
            return this;
        };

        this.destroy = function () {
            this.events.destroy();
        };

        // initialization -----------------------------------------------------

        // listen to view events
        view.on('change:activesheet', changeActiveSheetHandler);
        registerViewEventHandler('change:selection', changeSelectionHandler);
        registerViewEventHandler(columns ? 'insert:columns' : 'insert:rows', insertEntriesHandler);
        registerViewEventHandler(columns ? 'delete:columns' : 'delete:rows', deleteEntriesHandler);
        registerViewEventHandler(columns ? 'change:columns' : 'change:rows', changeEntriesHandler);

        // tracking for column/row selection with mouse/touch
        rootNode.on('tracking:start', trackingStartHandler);

        // enable/disable operations that modify the document according to edit mode
        app.getModel().on('change:editmode', editModeHandler);

    } // class HeaderPane

    // exports ================================================================

    return HeaderPane;

});
