/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/gridpane',
    ['io.ox/core/event',
     'io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/model/format/lineheight',
     'io.ox/office/drawinglayer/view/drawingframe',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Events, Utils, KeyCodes, Color, LineHeight, DrawingFrame, SheetUtils) {

    'use strict';

    var // margin at right or bottom border of the entire sheet area, in pixels
        SCROLL_AREA_MARGIN = 30,

        // the names of all tracking events but 'tracking:start'
        TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel';

    // class GridPane =========================================================

    /**
     * Represents a single scrollable grid pane in the spreadsheet view. If the
     * view has been split or frozen at a specific cell position, the view will
     * consist of up to four panes (top-left, top-right, bottom-left, and
     * bottom-right pane).
     *
     * Instances of this class trigger the following events:
     * - 'celledit:enter': After entering cell edit mode.
     * - 'celledit:leave': After leaving cell edit mode.
     *
     * @constructor
     *
     * @extends Events
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this grid pane.
     *
     * @param {String} panePos
     *  The position of this grid pane.
     *
     * @param {HeaderPane} colHeaderPane
     *  The header pane instance showing the column headers associated to this
     *  grid pane.
     *
     * @param {HeaderPane} rowHeaderPane
     *  The header pane instance showing the row headers associated to this
     *  grid pane.
     *
     * @param {CellCollection} cellCollection
     *  The cell collection containing layout information about the cells shown
     *  in this grid pane.
     */
    function GridPane(app, panePos, colHeaderPane, rowHeaderPane, cellCollection) {

        var // self reference
            self = this,

            // the spreadsheet view
            view = app.getView(),

            // the container node of this grid pane (the 'tabindex' attribute makes the node focusable)
            rootNode = $('<div>').addClass('abs grid-pane unselectable f6-target').attr({ tabindex: 0, 'data-focus-role': 'pane' }),

            // the scrollable node of this pane (embed into rootNode to allow to hide scroll bars if needed)
            scrollNode = $('<div>').addClass('grid-scroll').appendTo(rootNode),

            // the node controlling the size and position of the scroll bars
            scrollSizeNode = $('<div>').addClass('grid-scroll-size').appendTo(scrollNode),

            // the container node for all layer nodes, positioned dynamically in the scrollable area
            layerRootNode = $('<div>').addClass('grid-layer-root').appendTo(scrollNode),

            // the drawing layer (container for the drawing frames)
            drawingLayerNode = $('<div>').addClass('abs grid-drawing-layer noI18n').appendTo(layerRootNode),

            // the selection layer (container for the selected ranges)
            selectionLayerNode = $('<div>').addClass('abs grid-selection-layer').appendTo(layerRootNode),

            // the cell layer (container for the cell nodes), insert above other layers for better DOM debugging
            cellLayerNode = $('<div>').addClass('abs grid-cell-layer noI18n').appendTo(layerRootNode),

            // the text area control for in-place cell editing (will be inserted into the root node)
            textArea = Utils.createTextArea(),

            // the busy node shown if the grid pane is not initialized yet
            busyNode = $('<div>').addClass('abs').busy(),

            // the cell style sheets of the document
            cellStyles = app.getModel().getCellStyles(),

            // the collections of the active sheet
            colCollection = null,
            rowCollection = null,
            mergeCollection = null,

            // whether this grid pane is currently visible
            visible = false,

            // the absolute scroll position represented by this grid pane (may differ from position of scroll bars)
            scrollLeft = 0, scrollTop = 0,

            // the maximum scroll position currently possible in this grid pane
            maxScrollLeft = 0, maxScrollTop = 0,

            // the ratio between (public) absolute scroll position, and internal scroll bars
            scrollLeftRatio = 1, scrollTopRatio = 1,

            // the maximum scroll position possible in the entire sheet
            maxTotalScrollLeft = 0, maxTotalScrollTop = 0,

            // distance between cell A1 and the top-left visible cell of this pane
            hiddenWidth = 0, hiddenHeight = 0,

            // the type of the current tracking mode ('select', 'move', 'fill', null)
            trackingType = null,

            // all settings for in-place edit mode
            editSettings = null,

            // the original focus() method of the DOM root node
            rootFocusMethod = rootNode[0].focus;

        // base constructor ---------------------------------------------------

        Events.extend(this);

        // private methods ----------------------------------------------------

        /**
         * Returns the absolute position of the passed cell range in the active
         * sheet.
         *
         * @param {Object} range
         *  The logical range address.
         *
         * @returns {Object}
         *  The absolute position and size of the range in the active sheet, in
         *  the properties 'left', 'top', 'width', and 'height'.
         */
        function getRangePosition(range) {

            var // the position and size of the column interval
                colPosition = colCollection.getIntervalPosition(SheetUtils.getColInterval(range)),
                // the position and size of the row interval
                rowPosition = rowCollection.getIntervalPosition(SheetUtils.getRowInterval(range));

            return { left: colPosition.offset, top: rowPosition.offset, width: colPosition.size, height: rowPosition.size };
        }

        /**
         * Returns the absolute position of the specified cell in the active
         * sheet. If the cell is covered by a merged cell range, returns the
         * position of the entire merged range.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @returns {Object}
         *  The absolute position and size of the cell in the active sheet, in
         *  the properties 'left', 'top', 'width', and 'height'. If this address
         *  refers to the upper left corner of a merged cell range, a further
         *  property 'isMergedRange' is returned with the value 'true'.
         */
        function getCellPosition(address) {

            var // a merged range starting at the cell position
                mergedRange = mergeCollection.getMergedRange(address),
                // the column and row entry of the passed cell
                colEntry = null, rowEntry = null;

            // try to extend to a merged range
            if (mergedRange) {
                return _.extend(getRangePosition(mergedRange), { isMergedRange : true });
            }

            // calculate position of the single cell
            colEntry = colCollection.getEntry(address[0]);
            rowEntry = rowCollection.getEntry(address[1]);
            return { left: colEntry.offset, top: rowEntry.offset, width: colEntry.size, height: rowEntry.size };
        }

        /**
         * Returns the effective offset and size of the visible area in the
         * scrollable node.
         *
         * @returns {Object}
         *  The offset and size of the visible area in the sheet, in pixels.
         */
        function getVisibleAreaPosition() {
            return {
                left: colHeaderPane.getVisibleOffset(),
                top: rowHeaderPane.getVisibleOffset(),
                width: scrollNode[0].clientWidth,
                height: scrollNode[0].clientHeight
            };
        }

        /**
         * Returns the absolute offset and size of the sheet area for which
         * layout information is available in this grid pane and the associated
         * header panes.
         *
         * @returns {Object}
         *  The offset and size of the available area in the sheet, in pixels.
         */
        function getAvailableAreaPosition() {

            var colPosition = colHeaderPane.getIntervalPosition(),
                rowPosition = rowHeaderPane.getIntervalPosition();

            return { left: colPosition.offset, top: rowPosition.offset, width: colPosition.size, height: rowPosition.size };
        }

        /**
         * Empties all layer nodes (cells, selection, drawing frames).
         */
        function clearLayerNodes() {
            cellLayerNode.empty();
            selectionLayerNode.empty();
            drawingLayerNode.empty();
        }

        /**
         * Scrolls this grid pane to make the specified cell visible.
         *
         * @param {Object} [colPosition]
         *  The horizontal position of the rectangle in pixels, in the integer
         *  properties 'offset' and 'size'.
         *
         * @param {Object} [rowPosition]
         *  The vertical position of the rectangle in pixels, in the integer
         *  properties 'offset' and 'size'.
         */
        function scrollToRectangle(colPosition, rowPosition) {

            var // the absolute offset and size of the visible area in the sheet
                visiblePosition = getVisibleAreaPosition(),
                // whether the leading/trailing border is outside the visible area
                leadingOutside = false, trailingOutside = false,
                // new scrolling position
                newScrollLeft = scrollLeft, newScrollTop = scrollTop;

            // calculate new horizontal scroll position
            if (colPosition) {
                leadingOutside = colPosition.offset < visiblePosition.left;
                trailingOutside = colPosition.offset + colPosition.size + 2 > visiblePosition.left + visiblePosition.width;
                if (leadingOutside && !trailingOutside) {
                    newScrollLeft = colPosition.offset - hiddenWidth;
                } else if (!leadingOutside && trailingOutside) {
                    newScrollLeft = colPosition.offset + colPosition.size + 2 - visiblePosition.width - hiddenWidth;
                }
            }

            // calculate new vertical scroll position
            if (rowPosition) {
                leadingOutside = rowPosition.offset < visiblePosition.top;
                trailingOutside = rowPosition.offset + rowPosition.size + 2 > visiblePosition.top + visiblePosition.height;
                if (leadingOutside && !trailingOutside) {
                    newScrollTop = rowPosition.offset - hiddenHeight;
                } else if (!leadingOutside && trailingOutside) {
                    newScrollTop = rowPosition.offset + rowPosition.size + 2 - visiblePosition.height - hiddenHeight;
                }
            }

            // update scroll position
            if ((newScrollLeft !== scrollLeft) || (newScrollTop !== scrollTop)) {
                self.scroll(newScrollLeft, newScrollTop);
            }
        }

        /**
         * Scrolls this grid pane to make the specified cell visible.
         *
         * @param {Number} [col]
         *  The zero-based index of the column to be made visible. If omitted,
         *  the horizontal scroll position will not be changed.
         *
         * @param {Number} [row]
         *  The zero-based index of the row to be made visible. If omitted, the
         *  vertical scroll position will not be changed.
         */
        function scrollToCell(col, row) {

            var // horizontal position and size of the passed cell
                colEntry = _.isNumber(col) ? colCollection.getEntry(col) : null,
                // vertical position and size of the passed cell
                rowEntry = _.isNumber(row) ? rowCollection.getEntry(row) : null;

            scrollToRectangle(colEntry, rowEntry);
        }

        /**
         * Scrolls this grid pane to make the specified drawing frame visible.
         *
         * @param {HTMLElement|jQuery} drawingFrame
         *  The drawing frame, as jQuery object.
         */
        function scrollToDrawing(drawingFrame) {

            var // the position of the drawing frame in the drawing layer
                position = $(drawingFrame).position(),
                // horizontal position of the drawing rectangle
                colPosition = { offset: position.left, size: $(drawingFrame).width() },
                // vertical position of the drawing rectangle
                rowPosition = { offset: position.top, size: $(drawingFrame).height() };

            scrollToRectangle(colPosition, rowPosition);
        }

        /**
         * Updates the size of the scrollable area according to the used area
         * in the sheet, and the current scroll position. Tries to reserve
         * additional space for the full width of this grid pane at the right
         * border, and the full height of this grid pane at the bottom border,
         * to be able to scroll page-by-page into the unused area of the sheet.
         */
        function updateScrollAreaSize() {

            var // the size of the visible area in this grid pane
                visibleWidth = scrollNode[0].clientWidth,
                visibleHeight = scrollNode[0].clientHeight,
                // the new absolute width and height of the scrollable area
                scrollableWidth = colHeaderPane.getCurrentScrollSize(),
                scrollableHeight = rowHeaderPane.getCurrentScrollSize(),
                // the real size of the scroll size node (may be smaller due to browser limits)
                effectiveSize = null;

            // recalculate the maximum possible scroll position
            maxScrollLeft = Math.max(0, scrollableWidth - visibleWidth);
            maxScrollTop = Math.max(0, scrollableHeight - visibleHeight);

            // initialize the size of the scroll sizer node, the resulting size may be smaller
            effectiveSize = Utils.setContainerNodeSize(scrollSizeNode, scrollableWidth, scrollableHeight);

            // recalculate ratio between absolute and effective scroll area size
            scrollLeftRatio = (effectiveSize.width > visibleWidth) ? (maxScrollLeft / (effectiveSize.width - visibleWidth)) : 0;
            scrollTopRatio = (effectiveSize.height > visibleHeight) ? (maxScrollTop / (effectiveSize.height - visibleHeight)) : 0;

            // recalculate the maximum scroll positions in the entire sheet
            maxTotalScrollLeft = Math.max(0, colHeaderPane.getTotalScrollSize() - visibleWidth);
            maxTotalScrollTop = Math.max(0, rowHeaderPane.getTotalScrollSize() - visibleHeight);
        }

        /**
         * Updates the position and size of the layer nodes according to the
         * current scroll position and pane layout data.
         */
        function updateLayerPosition() {

            var // the position and size of the current sheet contents
                availablePosition = getAvailableAreaPosition(),
                // the effective left offset according to current scroll position
                left = availablePosition.left - scrollLeft + scrollNode.scrollLeft() - hiddenWidth,
                // the effective top offset according to current scroll position
                top = availablePosition.top - scrollTop + scrollNode.scrollTop() - hiddenHeight;

            // workaround IE positioning limitations (not possible to set an
            // absolute left/top position of more than Utils.MAX_NODE_SIZE pixels)
            Utils.setPositionInContainerNode(scrollNode, layerRootNode, left, top);

            // set size of the layer root node (should always be inside the node size limits)
            layerRootNode.css({ width: availablePosition.width, height: availablePosition.height });
        }

        /**
         * Updates the position and size of the text area control for the
         * in-place edit mode, according to the current length of the text in
         * the text area control, and the scroll position of the grid pane.
         */
        function updateTextAreaPosition() {

            var // the address of the edited cell
                address = null,
                // the column/row entries of the edit cell
                colEntry = null, rowEntry =  null,
                // the position of the edited cell
                position = {},
                // the visible area of this grid pane
                visiblePosition = null,
                // effective position of the text area
                leftOffset = 0, topOffset = 0,
                // the minimum and maximum width for the edit area
                minWidth = 0, maxWidth = 0;

            // calculates maximum width according to alignment (distance to visible border, but at least one third of the visible area)
            function getMaxWidth() {

                var // the distance from left visible border to right cell border (for right alignment)
                    leftWidth = Utils.minMax(position.left + position.width - visiblePosition.left, 0, visiblePosition.width),
                    // the distance from left cell border to right visible border (for left alignment)
                    rightWidth = Utils.minMax(visiblePosition.left + visiblePosition.width - position.left, 0, visiblePosition.width),
                    // the effective maximum width
                    maxWidth = 0;

                switch (editSettings.align) {
                case 'right':
                    maxWidth = leftWidth;
                    break;
                case 'center':
                    maxWidth = Math.min(leftWidth, rightWidth) * 2 - position.width;
                    break;
                default:
                    maxWidth = rightWidth;
                }

                return Math.max(maxWidth, Math.round(visiblePosition.width / 3));
            }

            // binary search algorithm to find the optimal width of the text field
            function setOptimalTextAreaWidth() {

                var // the width at the beginning of an iteration
                    prevWidth = 0,
                    // last fitting width (in case iteration ends while width is too small)
                    fittingWidth = maxWidth;

                do {
                    prevWidth = textArea.outerWidth();
                    if (textArea[0].clientHeight < textArea[0].scrollHeight) {
                        // text does not fit: enlarge width (round to blocks of 16 pixels)
                        minWidth = prevWidth;
                        textArea.css({ width: Math.ceil((prevWidth + maxWidth) / 32) * 16 });
                    } else {
                        // text fits: reduce width (round to blocks of 16 pixels)
                        maxWidth = fittingWidth = prevWidth;
                        textArea.css({ width: Math.ceil((minWidth + prevWidth) / 32) * 16 });
                    }
                } while (prevWidth !== textArea.outerWidth());

                // set last known fitting width
                textArea.css({ width: fittingWidth });
            }

            // do nothing if in-place edit mode is not active
            if (!editSettings) { return; }

            // get cell position of the edited cell, and visible area of the grid pane
            address = view.getActiveCell();
            visiblePosition = getVisibleAreaPosition();

            // get horizontal cell position and size
            colEntry = colCollection.getEntry(address[0]);
            position.left = colEntry.offset;
            // use at least 20 pixels width, enlarge width by one pixel to fit exactly into the selection border
            position.width = Math.max(20, colEntry.size + 1);

            // get vertical cell position and size
            rowEntry = rowCollection.getEntry(address[1]);
            position.top = rowEntry.offset;
            // enlarge height by one pixel to fit exactly into the selection border
            position.height = rowEntry.size + 1;

            // stick to cell width in auto-wrapping mode
            if (editSettings.wrap) {
                textArea.css({ width: position.width });
                leftOffset = position.left;
            } else {

                // get minimum width (cell width, restricted to visible area)
                minWidth = Math.min(position.width, visiblePosition.width);
                maxWidth = getMaxWidth();

                // set initial size of the text area to maximum allowed width and as many text lines as needed
                textArea.css({
                    width: maxWidth,
                    height: Utils.convertLength(textArea.val().split('\n').length * LineHeight.calculateNormalLineHeight(textArea), 'pt', 'px', 1)
                });

                // if text fits into a single line, try to collapse width to the actual cell width
                if ((minWidth < maxWidth) && (textArea[0].clientHeight === textArea[0].scrollHeight)) {
                    textArea.css({ width: minWidth });
                    // if text does not fit into the cell width, try to find optimal width via binary search
                    if (textArea[0].clientHeight < textArea[0].scrollHeight) {
                        setOptimalTextAreaWidth();
                    }
                }

                // calculate horizontal position according to alignment
                switch (editSettings.align) {
                case 'right':
                    leftOffset = position.left + position.width - textArea.outerWidth();
                    break;
                case 'center':
                    leftOffset = position.left + Math.round((position.width - textArea.outerWidth()) / 2);
                    break;
                default:
                    leftOffset = position.left;
                }
            }

            // width is optimal, enlarge height in case text still does not fit into one line
            textArea.css({ height: Utils.minMax(textArea[0].scrollHeight, position.height, visiblePosition.height) });

            // move cell into visible area, set effective position
            leftOffset = Utils.minMax(leftOffset, visiblePosition.left, visiblePosition.left + visiblePosition.width - textArea.outerWidth());
            topOffset = Utils.minMax(position.top, visiblePosition.top, visiblePosition.top + visiblePosition.height - textArea.outerHeight());
            textArea.css({ left: leftOffset - visiblePosition.left, top: topOffset - visiblePosition.top });

            // focus back to text area control (lost e.g. while scrolling with scroll bars)
            textArea.focus();
        }

        /**
         * Renders all visible selection ranges according to the current pane
         * layout data received from a view layout update notification.
         */
        function renderCellSelection(selection) {

            var // the address of the active cell
                activeCell = view.getActiveCell(),
                // the position of the active cell, extended to the merged range
                activeCellPos = getCellPosition(activeCell),
                // the HTML mark-up for all selection ranges
                markup = '';

            // restrict to ranges in the current cell area to prevent oversized DOM nodes
            // that would collapse to zero size, e.g. when entire columns or rows are selected
            // (for details, see the comments for the Utils.MAX_NODE_SIZE constant)
            SheetUtils.iterateIntersectionRanges(selection.ranges, cellCollection.getRange(), function (intersectRange, originalRange, index) {

                var // whether the range is the active range
                    active = index === selection.activeRange,
                    // position and size of the selection range
                    position = getRangePosition(intersectRange),
                    // position of active cell, relative to current range
                    activePos = null,
                    // the style attribute
                    styleAttr = '',
                    // mark-up for the fill elements
                    fillMarkup = '';

                // insert the semi-transparent fill elements (never cover the active cell with the fill elements)
                if (SheetUtils.rangeContainsCell(intersectRange, activeCell)) {

                    // initialize position of active cell, relative to selection range
                    activePos = _.clone(activeCellPos);
                    activePos.left -= position.left;
                    activePos.top -= position.top;

                    // insert fill element above active cell
                    if (activePos.top > 0) {
                        fillMarkup += '<div class="fill" style="left:0;right:0;top:0;height:' + activePos.top + 'px;"></div>';
                    }

                    // insert fill element left of active cell
                    if (activePos.left > 0) {
                        fillMarkup += '<div class="fill" style="left:0;width:' + activePos.left + 'px;top:' + activePos.top + 'px;height:' + activePos.height + 'px;"></div>';
                    }

                    // insert fill element right of active cell
                    if (activePos.left + activePos.width < position.width) {
                        fillMarkup += '<div class="fill" style="left:' + (activePos.left + activePos.width) + 'px;right:0;top:' + activePos.top + 'px;height:' + activePos.height + 'px;"></div>';
                    }

                    // insert fill element below active cell
                    if (activePos.top + activePos.height < position.height) {
                        fillMarkup += '<div class="fill" style="left:0;right:0;top:' + (activePos.top + activePos.height) + 'px;bottom:0;"></div>';
                    }

                } else {
                    fillMarkup = '<div class="abs fill"></div>';
                }

                // build the style attribute for the root node of the selection range
                styleAttr += 'left:' + colHeaderPane.getRelativeOffset(position.left) + 'px;top:' + rowHeaderPane.getRelativeOffset(position.top) + 'px;width:' + position.width + 'px;height:' + position.height + 'px;';

                // generate the HTML mark-up for the selection range
                markup += '<div class="range' + (active ? ' active' : '') + '" style="' + styleAttr + '"><div class="abs border"></div>' + fillMarkup + '</div>';
            });

            // insert entire HTML mark-up into the selection container node
            selectionLayerNode[0].innerHTML = markup;
        }

        /**
         * Updates the position of the passed drawing frame according to the
         * anchor attributes in the passed attribute set.
         */
        function updateDrawingFramePosition(drawingFrame, attributes) {

            var // the drawing attributes
                drawingAttrs = attributes.drawing,
                // the absolute position of the drawing frame in the sheet area, in pixels
                left = getBorderPosition(colCollection, drawingAttrs.startCol, drawingAttrs.startColOffset),
                top = getBorderPosition(rowCollection, drawingAttrs.startRow, drawingAttrs.startRowOffset),
                right = getBorderPosition(colCollection, drawingAttrs.endCol, drawingAttrs.endColOffset),
                bottom = getBorderPosition(rowCollection, drawingAttrs.endRow, drawingAttrs.endRowOffset),
                // the resulting size, in pixels
                width = (_.isNumber(left) && _.isNumber(right)) ? (right - left) : null,
                height = (_.isNumber(top) && _.isNumber(bottom)) ? (bottom - top) : null;

            // returns the absolute position of a single drawing frame border in the sheet area
            function getBorderPosition(collection, index, offset) {
                var entry = collection.getEntry(index);
                return entry.offset + Math.min(entry.size, Utils.convertHmmToLength(offset, 'px', 1));
            }

            // keep minimum size of 5 pixels
            if (width < 5) { left -= Math.round((5 - width) / 2); width = 5; }
            if (height < 5) { top -= Math.round((5 - height) / 2); height = 5; }
            drawingFrame.show().css({ left: colHeaderPane.getRelativeOffset(left), top: rowHeaderPane.getRelativeOffset(top), width: width, height: height });
        }

        /**
         * Returns the DOM drawing frame node that represents the passed
         * drawing model.
         */
        function getDrawingFrame(drawingModel) {
            return drawingLayerNode.find('>[data-uid="' + drawingModel.getUid() + '"]');
        }

        /**
         * Creates and inserts a new DOM drawing frame node that represents the
         * passed drawing model.
         */
        function createDrawingFrame(drawingModel, position) {

            var // the new drawing frame, as jQuery object
                drawingFrame = DrawingFrame.createDrawingFrame(drawingModel.getType()),
                // the merged drawing attributes
                attributes = drawingModel.getMergedAttributes(),
                // all existing drawing frames, as plain DOM collection
                drawingNodes = drawingLayerNode[0].childNodes,
                // the DOM child index (Z order) specified by the logical position
                index = position[0];

            // add the unique identifier of the drawing model for DOM look-up
            drawingFrame.attr('data-uid', drawingModel.getUid());

            // insert the drawing frame with the correct Z order
            if (drawingNodes.length === index) {
                drawingLayerNode.append(drawingFrame);
            } else {
                drawingFrame.insertBefore(drawingNodes[index]);
            }

            // update position and formatting
            updateDrawingFramePosition(drawingFrame, attributes);
            DrawingFrame.updateFormatting(app, drawingFrame, attributes);
        }

        /**
         * Creates DOM drawing frames for all existing drawing objects in the
         * current active sheet.
         */
        function createDrawingFrames() {
            view.getActiveSheetModel().iterateDrawingModels(createDrawingFrame);
            renderDrawingSelection(view.getSelection());
        }

        /**
         * Updates the position and formatting of the drawing frame that
         * represents the passed drawing model.
         */
        function updateDrawingFrame(drawingModel) {

            var // the existing DOM drawing frame
                drawingFrame = getDrawingFrame(drawingModel),
                // the merged drawing attributes
                attributes = drawingModel.getMergedAttributes();

            // update position and formatting
            updateDrawingFramePosition(drawingFrame, attributes);
            DrawingFrame.updateFormatting(app, drawingFrame, attributes);
        }

        /**
         * Updates position and formatting of all drawing frames in the current
         * visible area or the passed bounding range.
         *
         * @param {Object} [boundRange]
         *  if specified, the logical address of the cell range whose drawing
         *  frames will be updated. If omitted, all visible drawing frames will
         *  be updated.
         */
        function updateDrawingFrames(boundRange) {

            if (_.isObject(boundRange)) {

                // update all drawing frames inside the bounding range
                view.getActiveSheetModel().iterateDrawingModels(function (drawingModel) {
                    if (SheetUtils.rangesOverlap(boundRange, drawingModel.getRange())) {
                        updateDrawingFrame(drawingModel);
                    }
                });

            } else {
                // update all existing drawing frames
                view.getActiveSheetModel().iterateDrawingModels(updateDrawingFrame);
            }
        }

        /**
         * Updates the selection of all drawing frames.
         */
        function renderDrawingSelection(selection) {

            var // all drawing frames currently selected in the DOM (also embedded drawings)
                oldSelectedFrames = drawingLayerNode.find(DrawingFrame.NODE_SELECTOR + Utils.SELECTED_SELECTOR),
                // all existing drawing frames, as plain DOM collection
                drawingNodes = drawingLayerNode[0].childNodes,
                // the current document edit mode
                editMode = app.getModel().getEditMode();

            // process all drawings to be selected
            _(selection.drawings).each(function (position) {

                var // the root drawing frame addressed by the current logical position
                    drawingFrame = drawingNodes[position[0]];

                if (drawingFrame) {
                    // TODO: find embedded drawing frames
                    if (position.length === 1) {
                        DrawingFrame.drawSelection(drawingFrame, { movable: editMode, resizable: editMode });
                        oldSelectedFrames = oldSelectedFrames.not(drawingFrame);
                    }
                } else {
                    Utils.warn('GridPane.renderDrawingSelection(): invalid drawing position: ' + JSON.stringify(position));
                }
            });

            // remove selection border from all drawing frames not selected anymore
            oldSelectedFrames.each(function () { DrawingFrame.clearSelection(this); });
        }

        /**
         * Registers an event handler at the view instance that will only be
         * called when this grid pane is visible.
         */
        function registerViewEventHandler(type, handler) {
            view.on(type, function () {
                if (visible) {
                    handler.apply(self, _.toArray(arguments));
                }
            });
        }

        /**
         * Processes a new drawing object inserted into the drawing collection.
         */
        function insertDrawingHandler(event, drawingModel, position) {
            createDrawingFrame(drawingModel, position);
        }

        /**
         * Processes a drawing object removed from the drawing collection.
         */
        function deleteDrawingHandler(event, drawingModel) {
            getDrawingFrame(drawingModel).remove();
        }

        /**
         * Processes a drawing object whose formatting attributes have changed.
         */
        function changeDrawingHandler(event, drawingModel) {
            updateDrawingFrame(drawingModel);
        }

        /**
         * Returns whether the cell described by the passed cell entry is
         * empty (no contents, but ignoring formatting attributes).
         *
         * @param {Object} [cellEntry]
         *  The entry of a cell collection. If missing, the cell is considered
         *  to be empty.
         *
         * @returns {Boolean}
         *  Whether the cell described by the passed entry is empty.
         */
        function isEmptyCell(cellEntry) {
            return _.isNull(Utils.getOption(cellEntry, 'result', null));
        }

        /**
         * Returns the effective CSS text alignment of the passed cell data,
         * according to the cell result value and the alignment formatting
         * attributes.
         *
         * @param {String|Number|Boolean|Null} result
         *  The result value of the cell.
         *
         * @param {Object} mergedAttributes
         *  The merged (effective) attribute set of a cell. See method
         *  'CellCollection.getMergedAttributes()'.
         *
         * @returns {String}
         *  The effective horizontal CSS text alignment ('left', 'center',
         *  'right', or 'justify').
         */
        function getTextAlignment(result, mergedAttributes) {

            switch (mergedAttributes.cell.alignHor) {
            case 'left':
            case 'center':
            case 'right':
            case 'justify':
                return mergedAttributes.cell.alignHor;
            case 'auto':
                // Error or String
                if (_.isString(result)) { return (result[0] === '#' ? 'center' : 'left'); }
                // Number
                if (_.isNumber(result)) { return 'right'; }
                // Boolean
                if (_.isBoolean(result)) { return 'center'; }
                // empty cells: left aligned
                return 'left';
            }
            Utils.warn('GridPane.getTextAlignment(): unknown alignment attribute value "' + mergedAttributes.cell.alignHor + '"');
            return 'left';
        }

        /**
         * Returns whether a cell formatted with the passed attributes wraps
         * its text contents at the left and right cell borders. This happens
         * either by having the cell attribute 'wrapText' set to true, or by
         * containing a justifying horizontal alignment.
         *
         * @param {Object} mergedAttributes
         *  The merged (effective) attribute set of a cell. See method
         *  'CellCollection.getMergedAttributes()'.
         *
         * @returns {Boolean}
         *  Whether the cell wraps its text contents automatically.
         */
        function hasTextWrap(mergedAttributes) {
            return mergedAttributes.cell.wrapText || (mergedAttributes.cell.alignHor === 'justify');
        }

        /**
         * Returns the DOM cell node at the specified logical address.
         */
        function getCellNode(address) {
            return cellLayerNode.find('>[data-address="' + address[0] + ',' + address[1] + '"]');
        }

        /**
         * Returns all DOM cell nodes in the specified column.
         */
        function getCellNodesInColumn(col) {
            return cellLayerNode.find('>[data-col="' + col + '"]');
        }

        /**
         * Returns all DOM cell nodes in the specified row.
         */
        function getCellNodesInRow(row) {
            return cellLayerNode.find('>[data-row="' + row + '"]');
        }

        /**
         * Returns the clipping and overflow settings for the specified cell
         * (the additional space at the left and right side of the specified
         * cell that can be used for visible overlapping text contents).
         *
         * @param {Number[]} address
         *  The logical cell address.
         *
         * @param {String} textAlign
         *  The effective horizontal CSS text alignment of the cell.
         *
         * @returns {Object}
         *  The available space at the left and right side of the cell, in the
         *  properties 'leftClip' and 'rightClip' (visible space left/right of
         *  the cell), and 'leftOverflow' and 'rightOverflow' (additional space
         *  beyond the clip region needed to have enough space for very long
         *  text without reaching the content node borders).
         */
        function calculateCellClipping(address, textAlign) {

            var // the total clip width left and right of the cell
                clippingData = { leftClip: 0, rightClip: 0, leftOverflow: 0, rightOverflow: 0 };

            // no clipping left of cell needed, if cell is right-aligned or justified
            if ((textAlign === 'left') || (textAlign === 'center')) {
                cellCollection.iterateNextCells(address, 'right', function (address, cellEntry, colEntry) {
                    if (isEmptyCell(cellEntry)) { clippingData.rightClip += colEntry.size; } else { return Utils.BREAK; }
                }, { skipStartCell: true });
                clippingData.rightOverflow = 500000 - clippingData.rightClip;
            }

            // no clipping right of cell needed, if cell is left-aligned or justified
            if ((textAlign === 'right') || (textAlign === 'center')) {
                cellCollection.iterateNextCells(address, 'left', function (address, cellEntry, colEntry) {
                    if (isEmptyCell(cellEntry)) { clippingData.leftClip += colEntry.size; } else { return Utils.BREAK; }
                }, { skipStartCell: true });
                clippingData.leftOverflow = 500000 - clippingData.leftClip;
            }

            return clippingData;
        }

        /**
         * Renders all cells in the passed cell ranges according to the current
         * column, row, and cell collections.
         *
         * @param {Object|Array} [ranges]
         *  The logical address of a single cell range, or an array with cell
         *  range addresses. If omitted, all existing cells will be rendered.
         */
        function renderCellRanges(ranges) {

            var // the bounding range containing all changed ranges
                boundRange = null,
                // the intersection interval for updating trailing columns/rows
                interval = null,
                // additional cells whose clip regions need to be updated
                updateClipCells = [],
                // additional cells with vertical alignment 'middle' (positioned dynamically)
                vertMiddleCells = [],
                // additional cells with vertical alignment 'justify' (positioned dynamically)
                vertJustifyCells = [],
                // merged cells that have been rendered regularly
                mergedCells = [],
                // additional merged cells located outside the rendered ranges
                mergeRefCells = [],
                // the mark-up of all new cells
                markup = '';

            // returns the cell position and performs additional preparations for merged cells
            function getCellPosition2(address, cellEntry) {

                var // the absolute position of the cell in the sheet, in pixels
                    position = getCellPosition(address),
                    // the absolute position of the cell in the sheet, in pixels
                    refAddress = null;

                // additional processing for merged cells and cells covered by merged cells
                if (position.isMergedRange) { // This is the upper left cell of a merged range
                    if ((position.width > 0) && (position.height > 0)) {
                        // register rendered merged cells
                        mergedCells.push(address);
                    }
                } else {
                    refAddress = mergeCollection.getReferenceAddress(address); // This is a hidden cell inside a merged range
                    if ((refAddress) && (!_.isEqual(refAddress, _.last(mergeRefCells)))) {
                        // register references addresses of merged cells
                        mergeRefCells.push(refAddress);
                    }
                }

                return position;
            }

            // create the mark-up of a single cell, and performs additional registration for dynamic updates
            function renderCell(address, cellEntry, colEntry, rowEntry) {

                var // the absolute position of the cell in the sheet, in pixels
                    position = getCellPosition2(address, cellEntry);

                // prevent initialization of further variables (performance!), if cell is not visible
                if ((position.width === 0) || (position.height === 0) || (mergeCollection.isHiddenCell(address))) { return; }

                var // style attribute value for the outer cell node
                    cellStyle = '',
                    // style attribute value for the border node
                    borderStyle = '',
                    // style attribute value for the cell clip node
                    clipStyle = '',
                    // style attribute value for the cell contents node (parent of the text spans)
                    contentsStyle = '',
                    // style attribute value for the text span (for now, only one span for entire text)
                    spanStyle = '',

                    // the effective formatting attributes of the cell
                    mergedAttributes = cellCollection.getMergedAttributes(cellEntry, colEntry, rowEntry),
                    // attribute map for the 'cell' attribute family
                    cellAttributes = mergedAttributes.cell,
                    // attribute map for the 'character' attribute family
                    charAttributes = mergedAttributes.character,

                    // whether any border line of the cell is visible
                    hasBorder = false,
                    // CSS fill color
                    fillColor = cellStyles.getCssColor(cellAttributes.fillColor, 'fill'),
                    // CSS text color, calculated from text color and fill color attributes
                    textColor = cellStyles.getCssTextColor(charAttributes.color, [cellAttributes.fillColor]),
                    // the effective horizontal alignment
                    textAlign = '',
                    // CSS text decoration, calculated from underline and strike attributes
                    textDecoration = 'none',
                    // the text to be shown in the cell
                    displayText = Utils.getStringOption(cellEntry, 'display', ''),
                    // additional width for the left/right border of the clip node
                    clippingData = null;

                // creates and adds CSS attributes for the specified border
                function processBorder(attrName, position, roundUp) {

                    var // the border attribute value
                        border = cellAttributes[attrName],
                        // the effective CSS attributes
                        cssAttrs = cellStyles.getCssBorderAttributes(border),
                        // the number of pixels the border will be moved out of the cell
                        offset = roundUp ? Math.ceil(cssAttrs.width / 2) : Math.floor(cssAttrs.width / 2);

                    borderStyle += position + ':-' + offset + 'px;';
                    if (cssAttrs.width > 0) {
                        borderStyle += 'border-' + position + ':' + cssAttrs.style + ' ' + cssAttrs.width + 'px ' + cssAttrs.color + ';';
                        hasBorder = true;
                    }
                }

                // add effective position and size of the cell to the style attribute
                cellStyle += 'left:' + colHeaderPane.getRelativeOffset(position.left) + 'px;top:' + rowHeaderPane.getRelativeOffset(position.top) + 'px;width:' + position.width + 'px;height:' + position.height + 'px;';

                // offset to left/top cell border: round down (keep middle line for odd line widths on left/top pixel of the OWN cell)
                processBorder('borderLeft', 'left', false);
                processBorder('borderTop', 'top', false);
                // offset to right/bottom cell border: round up (keep middle line for odd line widths on left/top pixel of the NEXT cell)
                processBorder('borderRight', 'right', true);
                processBorder('borderBottom', 'bottom', true);

                // post-processing of some CSS attributes
                if (fillColor !== 'transparent') { cellStyle += 'background-color:' + fillColor + ';'; }

                // generate the HTML mark-up for the cell node
                markup += '<div class="cell" data-col="' + address[0] + '" data-row="' + address[1] + '" data-address="' + address[0] + ',' + address[1] + '" style="' + cellStyle + '">';

                // add border node, if any border is visible
                if (hasBorder) {
                    markup += '<div class="border" style="' + borderStyle + '"></div>';
                }

                // add clip node and content node with the cell text
                if (displayText.length > 0) {

                    // add horizontal alignment
                    textAlign = getTextAlignment(cellEntry.result, mergedAttributes);
                    contentsStyle += 'text-align:' + textAlign + ';';

                    // add vertical alignment
                    switch (cellAttributes.alignVert) {
                    case 'top':
                        contentsStyle += 'top:0;';
                        break;
                    case 'middle':
                        vertMiddleCells.push({ address: address, position: position });
                        break;
                    case 'justify':
                        contentsStyle += 'top:0;line-height:100%;';
                        vertJustifyCells.push({ address: address, position: position, fontSize: charAttributes.fontSize });
                        break;
                    default:
                        // 'bottom' alignment, and fall-back for unknown alignments
                        contentsStyle += 'bottom:0;';
                    }

                    // handle automatic text wrapping
                    if (hasTextWrap(mergedAttributes)) {
                        // CSS's 'white-space:pre-wrap' does not work together with
                        // 'text-align:justify', need to process white-space dynamically
                        contentsStyle += 'white-space:normal;';
                        displayText = _(displayText.split(/\n/)).map(function (textLine) {
                            return Utils.cleanString(textLine).replace(/\s/g, ' ').replace(/^ /, '\xa0').replace(/ {2}/g, ' \xa0').replace(/ $/, '\xa0');
                        });
                    } else {
                        // no text wrapping: calculate available space for the cell text over preceding/following empty cells
                        clippingData = calculateCellClipping(address, textAlign);
                        clipStyle += 'left:' + (-clippingData.leftClip) + 'px;right:' + (-clippingData.rightClip) + 'px;';
                        contentsStyle += 'left:' + (-clippingData.leftOverflow) + 'px;right:' + (-clippingData.rightOverflow) + 'px;';
                        displayText = [displayText.replace(/\n/g, '')];
                    }

                    // create CSS formatting for character attributes
                    spanStyle += 'font-family:' + cellStyles.getCssFontFamily(charAttributes.fontName).replace(/"/g, '\'') + ';';
                    if (charAttributes.fontSize !== 11) { spanStyle += 'font-size:' + charAttributes.fontSize + 'pt;'; }
                    if (charAttributes.bold) { spanStyle += 'font-weight:bold;'; }
                    if (charAttributes.italic) { spanStyle += 'font-style:italic;'; }
                    if (charAttributes.underline) { textDecoration = Utils.addToken(textDecoration, 'underline', 'none'); }
                    if (charAttributes.strike !== 'none') { textDecoration = Utils.addToken(textDecoration, 'line-through', 'none'); }

                    // post-processing of some CSS attributes
                    if (textColor !== '#000000') { spanStyle += 'color:' + textColor + ';'; }
                    if (textDecoration !== 'none') { spanStyle += 'text-decoration:' + textDecoration + ';'; }

                    // Create the clip node and the content node with the cell text. For now, character styles
                    // from 'spanStyle' goes to the content node too (no support for rich-text cells yet).
                    markup += '<div class="clip" style="' + clipStyle + '"><div class="contents" style="' + contentsStyle + spanStyle + '">';
                    markup += _(displayText).map(function (textLine) { return '<span>' + Utils.escapeHTML(textLine) + '</span>'; }).join('<br>');
                    markup += '</div></div>';
                }

                markup += '</div>';
            }

            // finds an additional cell outside the passed ranges whose clip region needs to be updated
            function findUpdateClipCell(address, direction) {
                cellCollection.iterateNextCells(address, direction, function (address, cellEntry, colEntry, rowEntry) {

                    var // the merged attributes of the cell
                        mergedAttributes = null;

                    // skip cells without value (but treat merged cells as filled cells)
                    if (isEmptyCell(cellEntry) && (! mergeCollection.isMergedCell(address))) { return; }

                    // filled cell found: do not add it twice
                    if ((updateClipCells.length === 0) || !_.isEqual(address, _.last(updateClipCells).address)) {
                        mergedAttributes = cellCollection.getMergedAttributes(cellEntry, colEntry, rowEntry);
                        // do not add it, if it wraps its text automatically (no overflow anyway)
                        if (!hasTextWrap(mergedAttributes)) {
                            updateClipCells.push({ address: address, textAlign: getTextAlignment(cellEntry.result, mergedAttributes) });
                        }
                    }

                    // exit the loop on any non-empty cell
                    return Utils.BREAK;

                }, { existing: true, skipStartCell: true });
            }

            // renders additional merged cells registered in the 'mergeRefCells' array
            function renderMergedCells() {

                // remove duplicates and all merged cells already rendered from the 'mergeRefCells' array
                mergeRefCells = _.chain(mergeRefCells).unique(SheetUtils.getCellName).reject(function (refAddress) {
                    // if 'refAddress' is contained in 'mergedCells', the array entry will be rejected
                    return _(mergedCells).any(function (address) { return _.isEqual(address, refAddress); });
                }).value();

                // render the remaining merged cells (starting outside the rendered ranges or visible area)
                if (mergeRefCells.length > 0) {
                    Utils.takeTime('generating HTML mark-up for ' + mergeRefCells.length + ' additional merged cells', function () {
                        _(mergeRefCells).each(function (address) {
                            getCellNode(address).remove();
                            renderCell(address, cellCollection.getCellEntry(address), colCollection.getEntry(address[0]), rowCollection.getEntry(address[1]));
                        });
                    });
                }
            }

            if (_.isObject(ranges)) {

                Utils.takeTime('GridPane.renderCellRanges(): panePos=' + panePos + ', rendering cells ' + _.chain(ranges).getArray().map(SheetUtils.getRangeName).value().join(' '), function () {

                    // process all cells in all passed ranges once (row by row)
                    Utils.takeTime('generating HTML mark-up', function () {
                        cellCollection.iterateCellsInRows(ranges, function (address, cellEntry, colEntry, rowEntry, rowBandRange) {

                            // If cell is located at the left or right border of a row band range,
                            // find preceding/following non-empty cell which needs to be rendered
                            // again if it is not wrapped (its clip region may change due to the
                            // new contents of the current cell).
                            if (address[0] === rowBandRange.start[0]) {
                                findUpdateClipCell(address, 'left');
                            }
                            if (address[0] === rowBandRange.end[0]) {
                                findUpdateClipCell(address, 'right');
                            }

                            // remove the old cell node from the DOM, generate the HTML mark-up of the new cell node
                            getCellNode(address).remove();
                            renderCell(address, cellEntry, colEntry, rowEntry);
                        });
                    });

                    // TODO: remove cell nodes of rows/columns that are now hidden (but were visible before)

                    // render merged cells starting outside the changed ranges/visible area
                    renderMergedCells();

                    // calculate the bounding range of all changed ranges, extend to sheet end address
                    boundRange = SheetUtils.getBoundingRange(ranges);
                    boundRange.end = app.getModel().getSheetRange().end;

                    // update the horizontal positions of remaining cell nodes (changed column width)
//                    if (columnGeometry) {
//                        Utils.takeTime('refreshing horizontal cell positions', function () {
//                            if ((interval = SheetUtils.getIntersectionInterval(colHeaderPane.getInterval(), SheetUtils.getColInterval(boundRange)))) {
//                                colCollection.iterateVisibleEntries(interval, function (colEntry) {
//                                    getCellNodesInColumn(colEntry.index).css({ left: colHeaderPane.getRelativeOffset(colEntry.offset) });
//                                });
//                            }
//                        });
//                    }

                    // update the vertical positions of remaining cell nodes (changed row height)
//                    if (rowGeometry) {
//                        Utils.takeTime('refreshing vertical cell positions', function () {
//                            if ((interval = SheetUtils.getIntersectionInterval(rowHeaderPane.getInterval(), SheetUtils.getRowInterval(boundRange)))) {
//                                rowCollection.iterateVisibleEntries(interval, function (rowEntry) {
//                                    getCellNodesInRow(rowEntry.index).css({ top: rowHeaderPane.getRelativeOffset(rowEntry.offset) });
//                                });
//                            }
//                        });
//                    }

                    // update selection
                    Utils.takeTime('refreshing cell selection', function () {
                        renderCellSelection(view.getSelection());
                    });

                    // update the clip region of the additional cells found while rendering the ranges
                    if (updateClipCells.length > 0) {
                        Utils.takeTime('refreshing clipping regions for ' + updateClipCells.length + ' cells', function () {
                            _(updateClipCells).each(function (cellData) {
                                var cellNode = getCellNode(cellData.address), clippingData = null;
                                // cell node may be missing (if covered by passed cell ranges)
                                if (cellNode.length > 0) {
                                    clippingData = calculateCellClipping(cellData.address, cellData.textAlign);
                                    cellNode.find('>.clip').css({ left: -clippingData.leftClip, right: -clippingData.rightClip })
                                        .find('>.contents').css({ left: -clippingData.leftOverflow, right: -clippingData.rightOverflow });
                                }
                            });
                        });
                    }

                    // append new cells
                    Utils.takeTime('appending HTML mark-up', function () {
                        cellLayerNode.append(markup);
                    });

                    // repaint all visible drawing frames
//                    if (columnGeometry || rowGeometry || cellGeometry) {
                    Utils.takeTime('refreshing drawing frames', function () {
                        updateDrawingFrames(boundRange);
                    });
//                    }
                });

            } else {

                Utils.takeTime('GridPane.renderCellRanges(): panePos=' + panePos + ', rendering all cells', function () {

                    // generate HTML mark-up text for all visible cells (also empty cells not contained in the cell collection)
                    Utils.takeTime('generating HTML mark-up', function () {
                        cellCollection.iterateCells(renderCell);
                    });

                    // render merged cells starting outside the visible area
                    renderMergedCells();

                    // replace all cells with the new HTML mark-up
                    Utils.takeTime('inserting HTML mark-up', function () {
                        cellLayerNode[0].innerHTML = markup;
                    });

                    // repaint the selection
                    Utils.takeTime('refreshing cell selection', function () {
                        renderCellSelection(view.getSelection());
                    });

                    // repaint all visible drawing frames
                    Utils.takeTime('refreshing drawing frames', function () {
                        updateDrawingFrames();
                    });
                });
            }

            // update vertical alignment (middle and justified) dynamically
            if (vertMiddleCells.length + vertJustifyCells.length > 0) {
                Utils.takeTime('GridPane.renderCellRanges(): panePos=' + panePos + ', updating vertical alignment of ' + (vertMiddleCells.length + vertJustifyCells.length) + ' cells', function () {

                    // alignment 'middle': position contents node centered in clip node
                    _(vertMiddleCells).each(function (cellData) {
                        var contentsNode = getCellNode(cellData.address).find('>.clip>.contents');
                        contentsNode.css({ top: (cellData.position.height - contentsNode.height()) / 2 });
                    });

                    // alignment 'justify': manipulate line height of the contents node to make the text fit
                    _(vertJustifyCells).each(function (cellData) {

                        var // the contents node in the cell
                            contentsNode = getCellNode(cellData.address).find('>.clip>.contents'),
                            // the total height of the contents node
                            contentsHeight = contentsNode.height(),
                            // the font size of the cell, in pixels
                            fontSize = Utils.convertLength(cellData.fontSize, 'pt', 'px'),
                            // the 'normal' line height, in pixels
                            normalLineHeight = Utils.convertLength(LineHeight.calculateNormalLineHeight(contentsNode), 'pt', 'px'),
                            // the free space between text line border and characters with 'normal' line height, in pixels
                            normalLinePadding = (normalLineHeight - fontSize) / 2,
                            // the number of text lines
                            lineCount = Math.round(contentsHeight / fontSize),
                            // the free space between text line border and characters with target line height, in pixels
                            targetLinePadding = 0,
                            // the effective line height, in pixels
                            targetLineHeight = 0;

                        if (lineCount > 1) {
                            targetLinePadding = (cellData.position.height - 2 * normalLinePadding - lineCount * fontSize) / (2 * lineCount - 2);
                            targetLineHeight = 2 * targetLinePadding + fontSize;
                        }
                        targetLineHeight = Math.max(targetLineHeight, normalLineHeight);
                        contentsNode.css({ 'line-height': Math.round(targetLineHeight) + 'px', top: Math.round(normalLinePadding - targetLinePadding) });
                    });
                });
            }
        }

        /**
         * Handles change events from the column, row, or cell collection.
         * Initiates to render the affected cells debounced after collecting
         * all changed ranges from multiple change events.
         */
        var registerChangedRanges = (function () {

            var // the cell ranges to be rendered after collection updates (if null, the entire grid pane will be rendered)
                changedRanges = [];

            // direct callback: stores the changed ranges in an internal array
            function registerRanges(ranges) {
                if (visible && _.isArray(changedRanges)) {
                    changedRanges = _.isObject(ranges) ? changedRanges.concat(_.getArray(ranges)) : null;
                }
            }

            // deferred callback: renders all changed cells, updates other layout settings of the grid pane
            function renderChangedRanges() {

                if (_.isArray(changedRanges) && (changedRanges.length === 0)) {
                    changedRanges = null;
                }

                // recalculate global layout settings if column/row geometry has been changed
                if (visible && !_.isObject(changedRanges)) {
                    busyNode.detach();
                    updateScrollAreaSize();
                    updateLayerPosition();
                    updateTextAreaPosition();
                }

                // render all affected cells and the selection
                if (visible) {
                    renderCellRanges(changedRanges);
                }

                // reset status variables
                changedRanges = [];
            }

            // create and return the actual debounced 'registerChangedRanges()' method
            return app.createDebouncedMethod(registerRanges, renderChangedRanges);
        }());

        /**
         * Handles change events from the cell collection. Initiates debounced
         * rendering of all affected cells.
         */
        function changeCellsHandler(event, ranges) {
            Utils.log('GridPane.changeCellsHandler(): panePos=' + panePos + ', "change:cells" received for ' + (ranges ? ('cells ' + SheetUtils.getRangesName(ranges, ' ')) : 'all cells'));
            registerChangedRanges(ranges);
        }

        /**
         * Leaves the in-place cell edit mode before the selection of the
         * sheet will be changed.
         */
        function beforeSelectionHandler() {
            leaveCellEditMode('cell');
        }

        /**
         * Renders the selected ranges after the selection has been changed.
         */
        function changeSelectionHandler(event, selection) {
            renderCellSelection(selection);
            renderDrawingSelection(selection);
        }

        /**
         * Initializes this grid pane after the active sheet has been changed.
         */
        function changeActiveSheetHandler() {

            var // the model of the active sheet
                sheetModel = view.getActiveSheetModel();

            // get the collections from the current active sheet
            colCollection = sheetModel.getColCollection();
            rowCollection = sheetModel.getRowCollection();
            mergeCollection = sheetModel.getMergeCollection();

            // clear the DOM of all layer nodes
            visible = false;
            clearLayerNodes();
            rootNode.prepend(busyNode);
        }

        /**
         * Handles 'scroll' events of the scrollable node and synchronizes the
         * scroll position of the other grid panes.
         */
        var scrollHandler = app.createDebouncedMethod(function () {

            // calculate absolute scroll position from current scroll bar positions
            scrollLeft = Math.round(scrollNode.scrollLeft() * scrollLeftRatio);
            scrollTop = Math.round(scrollNode.scrollTop() * scrollTopRatio);

            // update header panes and other grid panes
            view.syncGridPaneScroll(panePos, scrollLeft, scrollTop);

            // update position of layer nodes and text area immediately
            updateLayerPosition();
            updateTextAreaPosition();

        }, updateScrollAreaSize, { delay: 250 });

        /**
         * Returns the current absolute sheet position in the grid pane for the
         * passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event.
         *
         * @returns {Object}
         *  The absolute sheet position in the properties 'left' and 'top', in
         *  pixels.
         */
        function getTrackingPosition(event) {

            var // the absolute position of the layer root node
                layerOffset = layerRootNode.offset();

            return {
                left: event.pageX - layerOffset.left + colHeaderPane.getIntervalPosition().offset,
                top: event.pageY - layerOffset.top + rowHeaderPane.getIntervalPosition().offset
            };
        }

        /**
         * Returns the column and row entries matching the position in the
         * passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event.
         *
         * @returns {Object}
         *  The column and row entries in the properties 'col' and 'row'.
         */
        function getTrackingEntries(event) {

            var // the absolute sheet position of the event
                position = getTrackingPosition(event);

            return {
                col: colCollection.getEntryByOffset(position.left),
                row: rowCollection.getEntryByOffset(position.top)
            };
        }

        /**
         * Handles all tracking events for cell selection.
         */
        var selectionTrackingHandler = (function () {

            var // the cell address where tracking has started
                startAddress = null,
                // current address (prevent updates while tracking over the same cell)
                currentAddress = null;

            // makes an adjusted range from the start address to the current address
            function makeRange() {
                return SheetUtils.adjustRange({ start: _.clone(startAddress), end: _.clone(currentAddress) });
            }

            // initializes the selection according to the passed tracking event
            function initializeTracking(event) {

                var // initial column/row entries
                    entries = getTrackingEntries(event),
                    // whether to append a new range to the selection
                    append = !event.shiftKey && (event.ctrlKey || event.metaKey),
                    // whether to extend the active range
                    extend = event.shiftKey && !event.ctrlKey && !event.metaKey;

                currentAddress = [entries.col.index, entries.row.index];
                if (extend) {
                    // pressed SHIFT key: modify the active range (from active cell to tracking start position)
                    startAddress = view.getActiveCell();
                    view.changeActiveRange(makeRange());
                } else {
                    // no SHIFT key: create a new selection range
                    startAddress = currentAddress;
                    view.selectCell(startAddress, { append: append });
                }

                self.scrollToCell(currentAddress);

                // Workaround for Firefox, it sends focusout events after the
                // old selection DOM node that has been clicked is removed (the
                // focusout event causes canceling the selection tracking).
                _.defer(function () { self.grabFocus(); });
            }

            // updates the selection according to the passed tracking event
            function updateSelection(event) {

                var // the column/row entries currently hovered
                    entries = getTrackingEntries(event);

                if (currentAddress && ((entries.col.index !== currentAddress[0]) || (entries.row.index !== currentAddress[1]))) {
                    currentAddress = [entries.col.index, entries.row.index];
                    view.changeActiveRange(makeRange());
                }
            }

            // updates the scroll position while tracking
            function updateScroll(event) {
                self.scroll(scrollLeft + event.scrollX, scrollTop + event.scrollY);
            }

            // finalizes the selection tracking
            function finalizeTracking(event) {
                if (currentAddress) {
                    self.scrollToCell(currentAddress);
                    currentAddress = null;
                }
                trackingType = null;
                $(event.delegateTarget).off(TRACKING_EVENT_NAMES);
                view.grabFocus();
            }

            // return the actual selectionTrackingHandler() method
            return function (event) {

                switch (event.type) {
                case 'tracking:start':
                    initializeTracking(event);
                    break;
                case 'tracking:move':
                    updateSelection(event);
                    break;
                case 'tracking:scroll':
                    updateScroll(event);
                    updateSelection(event);
                    break;
                case 'tracking:end':
                case 'tracking:cancel': // cancel tracking: keep current selection
                    updateSelection(event);
                    finalizeTracking(event);
                    break;
                }
            };

        }()); // end of selectionTrackingHandler() local scope

        /**
         * Handles all tracking events for drawing frames.
         */
        var drawingTrackingHandler = (function () {

            var // the current drawing frame being tracked
                drawingFrame = null,
                // the logical position of the tracked drawing frame
                drawingPos = null;

            // initializes the drawing according to the passed tracking event
            function initializeTracking(event) {

                var // whether to toggle the selection of the clicked drawing frame
                    toggle = event.shiftKey || event.ctrlKey || event.metaKey,
                    // whether the clicked drawing frame is already selected
                    selected = false;

                drawingFrame = Utils.findFarthest(drawingLayerNode, event.target, DrawingFrame.NODE_SELECTOR);
                if (!drawingFrame) {
                    $.cancelTracking();
                    return;
                }

                // do not modify the selection, if clicked on a selected drawing without modifier key
                selected = DrawingFrame.isSelected(drawingFrame);
                if (toggle || !selected) {
                    drawingPos = [$(drawingFrame).index()];
                    scrollToDrawing(drawingFrame);
                    view.selectDrawing(drawingPos, { toggle: toggle });
                }
            }

            // updates the position of the drawing frame according to the passed tracking event
            function updateDrawingPosition(event) {
            }

            // updates the scroll position while tracking
            function updateScroll(event) {
                self.scroll(scrollLeft + event.scrollX, scrollTop + event.scrollY);
            }

            // finalizes the drawing tracking
            function finalizeTracking(event, applyPosition) {
                trackingType = null;
                $(event.delegateTarget).off(TRACKING_EVENT_NAMES);
                view.grabFocus();
            }

            // return the actual drawingTrackingHandler() method
            return function (event) {

                switch (event.type) {
                case 'tracking:start':
                    initializeTracking(event);
                    break;
                case 'tracking:move':
                    updateDrawingPosition(event);
                    break;
                case 'tracking:scroll':
                    updateScroll(event);
                    updateDrawingPosition(event);
                    break;
                case 'tracking:end':
                    updateDrawingPosition(event);
                    finalizeTracking(event, true);
                    break;
                case 'tracking:cancel':
                    finalizeTracking(event, false);
                    break;
                }
            };

        }()); // end of drawingTrackingHandler() local scope

        /**
         * Handles all tracking events for cell selection.
         */
        function trackingStartHandler(event) {
            if ($(event.target).closest(DrawingFrame.NODE_SELECTOR).length > 0) {
                trackingType = 'drawing';
                $(event.delegateTarget).on(TRACKING_EVENT_NAMES, drawingTrackingHandler);
                drawingTrackingHandler.call(this, event);
            } else {
                trackingType = 'select';
                $(event.delegateTarget).on(TRACKING_EVENT_NAMES, selectionTrackingHandler);
                selectionTrackingHandler.call(this, event);
            }
        }

        /**
         * Handles double clicks and starts the in-place edit mode.
         */
        function doubleClickHandler() {
            if (!editSettings) {
                enterCellEditMode();
            }
        }

        /**
         * Tries to find the previous/next visible column/row index in the
         * passed collection, relative to the specified index.
         *
         * @param {ColRowCollection} collection
         *  The collection to be searched for a visible entry.
         *
         * @param {Number} index
         *  The zero-based index of the start column/row.
         *
         * @param {Number} diff
         *  The number of columns/rows to be moved. If the target column/row is
         *  available but hidden, looks for the next visible column or row.
         *
         * @returns {Number|Null}
         *  The zero-based index of the found visible column or row; or null,
         *  if the target index is outside the current index interval of this
         *  collection.
         */
        function findCollectionIndex(collection, index, diff) {
            var entry = (diff < 0) ? collection.getPrevVisibleEntry(index + diff) : (diff > 0) ? collection.getNextVisibleEntry(index + diff) : null;
            return entry ? entry.index : (diff === 0) ? index : null;
        }

        /**
         * Clears the current selection and moves the active cell into the
         * specified direction. If the target cell is hidden, moves the cursor
         * to the next available visible cell.
         *
         * @param {Number} cols
         *  The number of columns the active cell will be moved. Negative
         *  values will move the cursor to the left.
         *
         * @param {Number} rows
         *  The number of rows the active cell will be moved. Negative values
         *  will move the cursor up.
         */
        function selectNextCell(cols, rows) {

            var // current selection in the active sheet
                selection = view.getSelection(),
                // the merged cell range of cells, that are upper left cells of merged cell ranges
                mergedCellRange = null,
                // a reference address
                address = null;

            if ((cols > 0) || (rows > 0)) {
                mergedCellRange = mergeCollection.getMergedRange(selection.activeCell);
                // taking care of merged cells
                if (mergedCellRange) {
                    cols += (SheetUtils.getColCount(mergedCellRange) - 1);
                    rows += (SheetUtils.getRowCount(mergedCellRange) - 1);
                }
            }

            address = [
                findCollectionIndex(colCollection, selection.activeCell[0], cols),
                findCollectionIndex(rowCollection, selection.activeCell[1], rows)
            ];

            // collapse selection to current active cell, if no other cell has been found
            if (!_.isNumber(address[0]) || !_.isNumber(address[1])) {
                address = selection.activeCell;
            }

            // set selection to the new active cell
            view.selectCell(address);
            self.scrollToCell(address);
        }

        /**
         * Clears the current selection and moves the active cell up or down by
         * one page.
         *
         * @param {Boolean} down
         *  If set to true, moves the active cell down, otherwise up.
         */
        function selectNextPage(down) {
        }

        /**
         * Extends the active range in the current selection into the specified
         * direction. If the target column/row is hidden, extends the active
         * range to the next available visible column/row.
         *
         * @param {Number} cols
         *  The number of columns to extend the active range. Negative values
         *  will select previous columns.
         *
         * @param {Number} rows
         *  The number of rows to extend the active range. Negative values will
         *  select upper rows.
         */
        function extendToNextCell(cols, rows) {

            var // current selection in the active sheet
                selection = view.getSelection(),
                // the current active range address
                activeRange = selection.ranges[selection.activeRange],
                // the new active range address
                newRange = _.copy(activeRange, true),
                // reference to the start or end address in the new active range
                address = null,
                // the new index inside the active range
                index = null;

            // find previous or next column index
            if (cols < 0) {
                address = (selection.activeCell[0] === activeRange.start[0]) ? newRange.end: newRange.start;
            } else if (cols > 0) {
                address = (selection.activeCell[0] === activeRange.end[0]) ? newRange.start : newRange.end;
            } else {
                address = null;
            }
            if (address && _.isNumber(index = findCollectionIndex(colCollection, address[0], cols))) {
                address[0] = index;
            }

            // find previous or next row index
            if (rows < 0) {
                address = (selection.activeCell[1] === activeRange.start[1]) ? newRange.end: newRange.start;
            } else if (rows > 0) {
                address = (selection.activeCell[1] === activeRange.end[1]) ? newRange.start : newRange.end;
            } else {
                address = null;
            }
            if (address && _.isNumber(index = findCollectionIndex(rowCollection, address[1], rows))) {
                address[1] = index;
            }

            // change the active range
            if (!_.isEqual(newRange, activeRange)) {
                view.changeActiveRange(newRange);
                self.scrollToCell([
                    (newRange.start[0] < activeRange.start[0]) ? newRange.start[0] : newRange.end[0],
                    (newRange.start[1] < activeRange.start[1]) ? newRange.start[1] : newRange.end[1]
                ]);
            }
        }

        /**
         * Moves the active cell in the current selection into the specified
         * direction. Cycles through all ranges contained in the selection. In
         * case the selection is a single cell, moves it in the entire sheet.
         *
         * @param {String} direction
         *  The direction the active cell will be moved to. Supported values
         *  are 'left', 'right', 'up', and 'down'.
         */
        function moveActiveCell(direction) {

            var // current selection in the active sheet
                selection = view.getSelection(),
                // whether the selection consists of a single cell (taking care of merged cells)
                singleCell = (selection.ranges.length === 1) && view.isSingleCellInRange(selection.ranges[0]),
                // the index of the cell address array element in primary direction
                index1 = ((direction === 'left') || (direction === 'right')) ? 0 : 1,
                // the index of the cell address array element in secondary direction
                index2 = 1 - index1,
                // the current active range address
                activeRange = null,
                // whether the active cell is a hidden merged cell
                isHiddenMergedCell = false;


            // single cell selection: use entire sheet as 'selection range'
            if (singleCell) {
                selection.ranges = [app.getModel().getSheetRange()];
                selection.activeRange = 0;
            }

            // get reference to the active range in the selection
            activeRange = selection.ranges[selection.activeRange];

            // move active cell forward
            // TODO: skip protected cells (horizontal only)
            if ((direction === 'right') || (direction === 'down')) {

                do {
                    if (selection.activeCell[index1] < activeRange.end[index1]) {
                        selection.activeCell[index1] += 1;
                    } else if (selection.activeCell[index2] < activeRange.end[index2]) {
                        selection.activeCell[index1] = activeRange.start[index1];
                        selection.activeCell[index2] += 1;
                    } else {
                        selection.activeRange = (selection.activeRange + 1) % selection.ranges.length;
                        selection.activeCell = _.clone(selection.ranges[selection.activeRange].start);
                    }

                    // the active cell must not be a merged cell with 'ref' attribute
                    isHiddenMergedCell = mergeCollection.isHiddenCell(selection.activeCell);
                } while (isHiddenMergedCell);

            // move active cell backward
            // TODO: skip protected cells (horizontal only)
            } else {

                do {
                    if (selection.activeCell[index1] > activeRange.start[index1]) {
                        selection.activeCell[index1] -= 1;
                    } else if (selection.activeCell[index2] > activeRange.start[index2]) {
                        selection.activeCell[index1] = activeRange.end[index1];
                        selection.activeCell[index2] -= 1;
                    } else {
                        selection.activeRange = (selection.activeRange + selection.ranges.length - 1) % selection.ranges.length;
                        selection.activeCell = _.clone(selection.ranges[selection.activeRange].end);
                    }

                    // the active cell must not be a merged cell with 'ref' attribute
                    isHiddenMergedCell = mergeCollection.isHiddenCell(selection.activeCell);
                } while (isHiddenMergedCell);
            }

            // select the cell or set the new selection, scroll to active cell
            if (singleCell) {
                view.selectCell(selection.activeCell);
            } else {
                view.setCellSelection(selection);
            }
            self.scrollToCell(selection.activeCell);
        }

        /**
         * Expands the selected ranges to entire columns or rows.
         *
         * @param {String} direction
         *  If set to 'columns', the current selection will be expanded to full
         *  columns. If set to 'rows', the current selection will be expanded
         *  to full rows.
         */
        function extendSelectionToLimits(direction) {

            var // current selection in the active sheet
                selection = view.getSelection(),
                // the index of the cell address array element to be modified
                index = (direction === 'columns') ? 1 : 0,
                // the last available column/row index in the sheet
                lastIndex = (direction === 'columns') ? app.getModel().getMaxRow() : app.getModel().getMaxCol();

            _(selection.ranges).each(function (range) {
                range.start[index] = 0;
                range.end[index] = lastIndex;
            });
            view.setCellSelection(selection);
        }

        /**
         * Starts the in-place edit mode for the active cell.
         *
         * @param {String} [text]
         *  The initial text to insert into the edit area. If omitted, the
         *  current value of the active cell will be inserted.
         */
        function enterCellEditMode(text) {

            var // the contents (display string, result, formula) and formatting of the active cell
                contents = view.getCellContents();

            // first, check document edit mode
            if (!app.getModel().getEditMode()) {
                app.rejectEditAttempt();
                return;
            }

            // cancel current tracking
            $.cancelTracking();

            // check whether in-place edit mode is already active
            if (editSettings) { return; }

            // first, scroll to the active cell
            self.scrollToCell(view.getActiveCell());

            // initialize edit mode settings
            editSettings = {
                value: _.isString(text) ? text : _.isString(contents.formula) ? contents.formula : contents.display,
                align: getTextAlignment(contents.result, contents.attrs),
                wrap: hasTextWrap(contents.attrs)
            };

            // copy formatting and editing text from active cell to the text area
            textArea.css({
                fontFamily: cellStyles.getCssFontFamily(contents.attrs.character.fontName),
                fontSize: contents.attrs.character.fontSize + 'pt',
                textAlign: editSettings.align
            }).val(editSettings.value);

            // insert and activate the text area
            textArea.appendTo(rootNode).focus();

            // IE9 does not trigger 'input' events when deleting characters or
            // pasting text, use a timer interval as a workaround
            if (Utils.IE9) {
                textArea.data('lastValue', textArea.val());
                textArea.data('updateTimer', app.repeatDelayed(function () {
                    var value = textArea.val();
                    if (value !== textArea.data('lastValue')) {
                        textArea.data('lastValue', value);
                        updateTextAreaPosition();
                    }
                }, { delay: 250 }));
            }

            // set initial position, size, and selection of the text area
            updateTextAreaPosition();
            Utils.setTextFieldSelection(textArea, editSettings.value.length);

            // forward focus requests at the root node to the text area
            rootNode[0].focus = function () { textArea.focus(); };

            // notify listeners
            self.trigger('celledit:enter');
        }

        /**
         * Leaves the in-place edit mode for the active cell.
         *
         * @param {Boolean} [commitMode='discard']
         *  If set to 'cell', the current value will be committed to the active
         *  cell of the selection. If set to 'fill', the current value will be
         *  filled to all cells of the selection. If set to 'array', the
         *  current value will be inserted as array formula. Otherwise, the
         *  cell in the document will not be changed.
         */
        function leaveCellEditMode(commitMode) {

            var // the new cell text
                value = textArea.val(),
                // additional cell attributes
                attributes = (value.indexOf('\n') >= 0) ? { cell: { wrapText: true } } : undefined;

            // nothing to do, if in-place edit mode is not active
            if (!editSettings) { return; }

            // IE9 does not trigger 'input' events when deleting characters or pasting text,
            // remove the timer interval that has been started as a workaround
            if (Utils.IE9) {
                textArea.data('updateTimer').abort();
                textArea.data('updateTimer', null);
            }

            // send current value to document model
            switch (commitMode) {
            case 'cell':
                view.setCellContents(value, attributes);
                break;
            case 'fill':
                view.fillCellContents(value, attributes);
                break;
            case 'array':
                // TODO
                break;
            default:
                commitMode = 'discard';
            }

            // restore original focus() method of the root node
            rootNode[0].focus = rootFocusMethod;

            // back to cell selection mode
            editSettings = null;
            rootNode.focus();
            textArea.detach();

            // notify listeners
            self.trigger('celledit:leave');
        }

        /**
         * Selects the previous or next drawing frame.
         */
        function selectNextDrawing(backwards) {

            var // last drawing selected in the active sheet
                position = _.last(view.getSelectedDrawings()),
                // number of drawings in the sheet
                count = view.getActiveSheetModel().getDrawingCollection().getModelCount();

            // TODO: selection in embedded objects
            if ((count > 1) && _.isArray(position) && (position.length === 1) && (0 <= position[0]) && (position[0] < count)) {
                if (backwards) {
                    position[0] = (position[0] + count - 1) % count;
                } else {
                    position[0] = (position[0] + 1) % count;
                }
                scrollToDrawing(drawingLayerNode[0].childNodes[position[0]]);
                view.selectDrawing(position);
            }
        }

        /**
         * Handles 'keydown' events of the root node. Adjusts the selection, or
         * performs other actions according to the current selection.
         */
        function keyDownHandler(event) {

            // do nothing while in-place edit mode is active
            if (editSettings) { return; }

            // special handling for drawing selection
            if (view.hasDrawingSelection()) {

                // traverse through drawing frames with TAB key
                if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                    selectNextDrawing(event.shiftKey);
                    return false;
                }

                // back to cell selection on ESCAPE key
                if (event.keyCode === KeyCodes.ESCAPE) {
                    view.removeDrawingSelection();
                    return false;
                }

                // delete selected drawings
                if (KeyCodes.matchKeyCode(event, 'DELETE')) {
                    view.deleteDrawings();
                    return false;
                }

                // TODO: handle other keys

                return;
            }

            // no modifier keys: delete selection, move active cell
            // SHIFT key: extend active range
            if (KeyCodes.matchModifierKeys(event, { shift: null })) {

                switch (event.keyCode) {
                case KeyCodes.LEFT_ARROW:
                    (event.shiftKey ? extendToNextCell : selectNextCell)(-1, 0);
                    return false;
                case KeyCodes.RIGHT_ARROW:
                    (event.shiftKey ? extendToNextCell : selectNextCell)(1, 0);
                    return false;
                case KeyCodes.UP_ARROW:
                    (event.shiftKey ? extendToNextCell : selectNextCell)(0, -1);
                    return false;
                case KeyCodes.DOWN_ARROW:
                    (event.shiftKey ? extendToNextCell : selectNextCell)(0, 1);
                    return false;
                case KeyCodes.TAB:
                    moveActiveCell(event.shiftKey ? 'left' : 'right');
                    return false;
                case KeyCodes.ENTER:
                    moveActiveCell(event.shiftKey ? 'up' : 'down');
                    return false;
                }
            }

            // additional actions (without any modifier keys)
            if (!event.shiftKey && !event.altKey && !event.ctrlKey && !event.metaKey) {
                switch (event.keyCode) {
                case KeyCodes.PAGE_UP:
                    selectNextPage(false);
                    return false;
                case KeyCodes.PAGE_DOWN:
                    selectNextPage(true);
                    return false;
                case KeyCodes.F2:
                    enterCellEditMode();
                    return false;
                case KeyCodes.BACKSPACE:
                    enterCellEditMode('');
                    return false;
                case KeyCodes.DELETE:
                    view.fillCellContents('');
                    return false;
                }
            }

            // convert selection to entire columns
            if (KeyCodes.matchKeyCode(event, 'SPACE', { ctrlOrMeta: true })) {
                extendSelectionToLimits('columns');
                return false;
            }

            // convert selection to entire rows
            if (KeyCodes.matchKeyCode(event, 'SPACE', { shift: true })) {
                extendSelectionToLimits('rows');
                return false;
            }
        }

        /**
         * Handles 'keypress' events of the root node. Starts the in-place edit
         * mode on-the-fly for valid Unicode characters.
         */
        function keyPressHandler(event) {
            // do not handle 'keypress' events bubbled up from the active in-place text
            // area, ignore key events where either CTRL/META or ALT is pressed, ignore
            // SPACE keys with any control keys (used as shortcuts for column/row selection)
            if (!editSettings && !view.hasDrawingSelection() &&
                    ((event.charCode > 32) && ((event.ctrlKey || event.metaKey) === event.altKey) || ((event.charCode === 32) && KeyCodes.matchModifierKeys(event)))) {
                enterCellEditMode(String.fromCharCode(event.charCode));
                return false;
            }
        }

        /**
         * Handles 'keydown' events in the text input control, while in-place
         * editing is active.
         */
        function editKeyDownHandler(event) {

            // quit in-place edit mode on ESCAPE key regardless of any modifier keys
            if (event.keyCode === KeyCodes.ESCAPE) {
                leaveCellEditMode();
                return;
            }

            // ENTER without modifier keys (but ignoring SHIFT): set value, move cell down/up
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {
                leaveCellEditMode('cell');
                // let key event bubble up to move the cell cursor
                return;
            }

            // ENTER with CTRL or META key: fill value to all cells (no SHIFT), or insert array formula (SHIFT)
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null, ctrlOrMeta: true })) {
                leaveCellEditMode(event.shiftKey ? 'array' : 'fill');
                // do not let key event bubble up (no change of selection)
                return false;
            }

            // ENTER with ALT key without any other modifier keys: insert line break
            if (KeyCodes.matchKeyCode(event, 'ENTER', { alt: true })) {
                Utils.replaceTextInTextFieldSelection(textArea, '\n');
                updateTextAreaPosition();
                return false;
            }

            // ENTER with any other modifier keys: discard the entire event
            if (event.keyCode === KeyCodes.ENTER) {
                return false;
            }

            // TAB key without modifier keys (but ignoring SHIFT): set value, move cell to next/previous cell
            if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                leaveCellEditMode('cell');
                // let key event bubble up to move the cell cursor
                return;
            }

            // all other events will be processed by the text field
        }

        /**
         * Handles a changed document edit mode. If switched to read-only mode,
         * cancels in-place cell edit mode, or a running tracking action to
         * move or resize drawing objects.
         */
        function editModeHandler(event, editMode) {

            var // the current selection
                selection = null;

            // nothing to do, if this grid pane is currently not visible
            if (!visible) { return; }

            // stop cell in-place edit mode and drawing tracking, if edit rights have been lost
            if (!editMode) {
                leaveCellEditMode();
                if (trackingType === 'drawing') {
                    $.cancelTracking();
                }
            }

            // redraw selection according to edit mode
            selection = view.getSelection();
            renderCellSelection(selection);
            renderDrawingSelection(selection);
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the root DOM node of this grid pane.
         *
         * @returns {jQuery}
         *  The root node of this grid pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns whether the root node of this grid pane is currently focused
         * or contains a DOM node that is currently focused (e.g. the text area
         * node while edit mode is active).
         *
         * @returns {Boolean}
         *  Whether this grid pane is currently focused.
         */
        this.hasFocus = function () {
            return visible && ((rootNode[0] === window.document.activeElement) || Utils.containsFocus(rootNode));
        };

        /**
         * Sets the browser focus into this grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            (editSettings ? textArea : rootNode).focus();
            return this;
        };

        /**
         * Initializes the settings and layout of this grid pane.
         *
         * @param {Object} horSettings
         *  The view settings of the horizontal pane side (left or right).
         *
         * @param {Object} vertSettings
         *  The view settings of the vertical pane side (top or bottom).
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (horSettings, vertSettings) {

            // visibility
            visible = horSettings.visible && vertSettings.visible;
            rootNode.toggle(visible);
            if (visible) {

                // position and size
                rootNode.css({ left: horSettings.offset, top: vertSettings.offset, width: horSettings.size, height: vertSettings.size });

                // initialize the scroll node containing the scroll bars
                scrollNode.css({
                    overflowX: horSettings.scrollable ? '' : 'hidden',
                    overflowY: vertSettings.scrollable ? '' : 'hidden',
                    bottom: (horSettings.scrollable && !vertSettings.showOppositeScroll) ? -Math.max(6, Utils.SCROLLBAR_HEIGHT) : 0,
                    right: (vertSettings.scrollable && !horSettings.showOppositeScroll) ? -Math.max(6, Utils.SCROLLBAR_WIDTH) : 0
                });

                // initialize auto-scrolling
                layerRootNode.enableTracking({
                    autoScroll: horSettings.scrollable ? (vertSettings.scrollable ? true : 'horizontal') : (vertSettings.scrollable ? 'vertical' : false),
                    borderNode: rootNode,
                    borderMargin: -30,
                    borderSize: 60,
                    minSpeed: 2,
                    maxSpeed: 250
                });

                // other layout options
                hiddenWidth = horSettings.hiddenSize;
                hiddenHeight = vertSettings.hiddenSize;

                // update the DOM
                updateScrollAreaSize();
                updateLayerPosition();

                // create all drawing frames if not done yet (e.g. enabled split in current sheet)
                if (!drawingLayerNode[0].hasChildNodes()) {
                    createDrawingFrames();
                }
            } else {
                clearLayerNodes();
            }

            return this;
        };

        /**
         * Returns whether this grid pane is currently visible.
         *
         * @returns {Boolean}
         *  Whether this grid pane is currently visible.
         */
        this.isVisible = function () {
            return visible;
        };

        /**
         * Returns the position of the first pixel in the entire sheet visible
         * in this grid pane.
         *
         * @returns {Object}
         *  The position of the first visible pixel, in the properties 'left'
         *  and 'top', in pixels.
         */
        this.getTopLeftPosition = function () {
            return { left: colHeaderPane.getVisibleOffset(), top: rowHeaderPane.getVisibleOffset() };
        };

        /**
         * Returns the logical address of the top-left cell visible in this
         * grid pane. The cell is considered visible, it it's column and row
         * are at least half visible.
         *
         * @returns {Number[]}
         *  The logical address of the top-left cell in this grid pane.
         */
        this.getTopLeftAddress = function () {
            return [colHeaderPane.getFirstVisibleIndex(), rowHeaderPane.getFirstVisibleIndex()];
        };

        /**
         * Changes the scroll position of this grid pane.
         *
         * @param {Number} left
         *  The new horizontal scroll position, in pixels.
         *
         * @param {Number} top
         *  The new vertical scroll position, in pixels.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scroll = function (left, top) {

            var // the new scroll positions
                newScrollLeft = 0, newScrollTop = 0;

            // set the new absolute scroll position, prevent one-pixel rounding errors when
            // synchronizing scroll position, by converting to DOM scroll position and back once
            newScrollLeft = Math.round(Math.round(left / scrollLeftRatio) * scrollLeftRatio);
            newScrollTop = Math.round(Math.round(top / scrollLeftRatio) * scrollLeftRatio);

            // restrict to maximum scroll position in the sheet
            newScrollLeft = Utils.minMax(left, 0, maxTotalScrollLeft);
            newScrollTop = Utils.minMax(top, 0, maxTotalScrollTop);

            // Only set scroll position if it differs from current position.
            // This test MUST be done, otherwise Firefox scrolls extremely slow
            // when using the mouse wheel!
            if ((newScrollLeft !== scrollLeft) || (newScrollTop !== scrollTop)) {

                // set the new absolute scroll position
                scrollLeft = newScrollLeft;
                scrollTop = newScrollTop;

                // if the scroll position is outside the current scroll size (but
                // inside the entire sheet area), enlarge scroll area in the header
                // panes and this grid pane, before setting DOM scroll position
                if ((scrollLeft > maxScrollLeft) || (scrollTop > maxScrollTop)) {
                    colHeaderPane.scroll(scrollLeft);
                    rowHeaderPane.scroll(scrollTop);
                    updateScrollAreaSize();
                }

                // set the scroll position (the scroll event handler will synchronize the other panes)
                scrollNode
                    .scrollLeft((scrollLeftRatio > 0) ? Math.round(scrollLeft / scrollLeftRatio) : 0)
                    .scrollTop((scrollTopRatio > 0) ? Math.round(scrollTop / scrollTopRatio) : 0);
            }

            return this;
        };

        /**
         * Changes the horizontal scroll position of this grid pane.
         *
         * @param {Number} left
         *  The new horizontal scroll position, in pixels.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollLeft = function (left) {
            return this.scroll(left, scrollTop);
        };

        /**
         * Changes the vertical scroll position of this grid pane.
         *
         * @param {Number} top
         *  The new vertical scroll position, in pixels.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollTop = function (top) {
            return this.scroll(scrollLeft, top);
        };

        /**
         * Scrolls this grid pane to make the specified cell visible.
         *
         * @param {Number[]} address
         *  The logical cell position to be made visible.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToCell = function (address) {
            scrollToCell(address[0], address[1]);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified column visible.
         *
         * @param {Number} col
         *  The zero-based index of the column to be made visible.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToColumn = function (col) {
            scrollToCell(col, null);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified row visible.
         *
         * @param {Number} row
         *  The zero-based index of the row to be made visible.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToRow = function (row) {
            scrollToCell(null, row);
            return this;
        };

        this.destroy = function () {
            this.events.destroy();
            colCollection = rowCollection = mergeCollection = null;
        };

        // initialization -----------------------------------------------------

        // listen to layout update events from the view
        view.on({
            'change:activesheet': changeActiveSheetHandler,
            // leave in-place cell edit mode when resize tracking starts in any header pane
            'resize:start': function () { leaveCellEditMode('cell'); },
            // disable global F6 focus traveling into this grid pane while in-place cell edit mode is active in another grid pane
            'celledit:enter': function (event, editPos) { rootNode.toggleClass('f6-target', editPos === panePos).addClass('fade-out'); },
            'celledit:leave': function () { rootNode.addClass('f6-target').removeClass('fade-out'); }
        });

        // update cell layer node
        cellCollection.on({
            'insert:cells': changeCellsHandler,
            'delete:cells': changeCellsHandler,
            'change:cells': changeCellsHandler
        });

        registerViewEventHandler('change:mergedCells', changeCellsHandler);

        // update drawing layer node (only, if this grid pane is visible)
        registerViewEventHandler('insert:drawing', insertDrawingHandler);
        registerViewEventHandler('delete:drawing', deleteDrawingHandler);
        registerViewEventHandler('change:drawing', changeDrawingHandler);

        // update selection layer node (only, if this grid pane is visible)
        registerViewEventHandler('before:selection', beforeSelectionHandler);
        registerViewEventHandler('change:selection', changeSelectionHandler);

        // listen to scroll events and synchronize scroll position of other grid panes
        scrollNode.on('scroll', scrollHandler);

        // tracking for cell selection with mouse/touch
        layerRootNode.on({
            'tracking:start': trackingStartHandler,
            dblclick: doubleClickHandler
        });

        // keyboard events in grid pane (scrolling etc.)
        rootNode.on({
            keydown: keyDownHandler,
            keypress: keyPressHandler
        });

        // events for in-place edit mode
        textArea.on({
            keydown: editKeyDownHandler,
            input: updateTextAreaPosition
        });

        // leave in-place edit mode or cancel tracking, if document goes into read-only mode
        app.getModel().on('change:editmode', editModeHandler);

    } // class GridPane

    // exports ================================================================

    return GridPane;

});
