/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/sheetutils', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    var // maximum number of columns that will ever be allowed in a sheet (A to ZZZZ)
        MAX_COL_COUNT = 475254,

        // maximum number of rows that will ever be allowed in a sheet (1 to 99,999,999)
        MAX_ROW_COUNT = 99999999,

        // regular expression pattern matching a valid column name (A to ZZZZ)
        COL_PATTERN = '[A-Z]{1,4}',

        // regular expression pattern matching a row name (1 to 99,999,999)
        ROW_PATTERN = '0*[1-9][0-9]{0,7}',

        // regular expression object matching a column name only
        RE_COL_NAME = new RegExp('^' + COL_PATTERN + '$'),

        // regular expression object matching a row name only
        RE_ROW_NAME = new RegExp('^' + ROW_PATTERN + '$'),

        // regular expression object matching a cell address, row and column part in parentheses
        RE_ADDRESS = new RegExp('^(' + COL_PATTERN + ')(' + ROW_PATTERN + ')$');

    // static class SheetUtils ================================================

    var SheetUtils = {};

    // constants --------------------------------------------------------------

    // TODO: get maximum size of columns/rows from somewhere
    /**
     * Maximum width of columns, in 1/100 millimeters.
     *
     * @constant
     */
    SheetUtils.MAX_COLUMN_WIDTH = 20000;

    // TODO: get maximum size of columns/rows from somewhere
    /**
     * Maximum height of rows, in 1/100 millimeters.
     *
     * @constant
     */
    SheetUtils.MAX_ROW_HEIGHT = 20000;

    // methods ----------------------------------------------------------------

    /**
     * Returns whether the passed value is a valid logical cell address.
     *
     * @param {Any} address
     *  Any value that will be checked whether it is a valid logical cell
     *  address.
     *
     * @returns {Boolean}
     *  Whether the passed value is an array with two numerical non-negative
     *  elements. The first element (column index) must be less than the global
     *  maximum number of supported columns (currently 475,254 - columns A to
     *  ZZZZ); the second element (row index) must be less than the global
     *  maximum number of supported rows (currently 99,999,999). This static
     *  method does not take the real grid size of a spreadsheet document into
     *  account.
     */
    SheetUtils.isValidAddress = function (address) {
        return _.isArray(address) && (address.length === 2) &&
            _.isFinite(address[0]) && (0 <= address[0]) && (address[0] < MAX_COL_COUNT) &&
            _.isFinite(address[1]) && (0 <= address[1]) && (address[1] < MAX_ROW_COUNT);
    };

    /**
     * Returns whether the passed value is a valid logical cell range address.
     *
     * @param {Any} range
     *  Any value that will be checked whether it is a valid logical cell range
     *  address.
     *
     * @returns {Boolean}
     *  Whether the passed value is an object with the two properties 'start'
     *  and 'end', both containing valid cell addresses (see method
     *  SheetUtils.isValidAddress()). Does not check whether the start and end
     *  positions are ordered.
     */
    SheetUtils.isValidRange = function (range) {
        return _.isObject(range) && SheetUtils.isValidAddress(range.start) && SheetUtils.isValidAddress(range.end);
    };

    /**
     * Returns whether the passed ranges are equal. This is the case, if the
     * start and the end position are equal.
     *
     * @param {Object} range1
     *  The range address of the first range, that shall be compared.
     *
     * @param {Object} range2
     *  The range address of the second range, that shall be compared.
     *
     * @returns {Boolean}
     *  Whether the passed ranges are equal. Does not check whether the start and end
     *  positions are ordered.
     */
    SheetUtils.equalRanges = function (range1, range2) {
        return (_.isEqual(range1.start, range2.start) && _.isEqual(range1.end, range2.end));
    };

    /**
     * Checks the column and row indexes contained in the passed range, and
     * swaps them (in-place) if they are not ordered.
     *
     * @param {Object} range
     *  (in/out) The logical address of a range, that will be adjusted in-place
     *  if required.
     *
     * @returns {Object}
     *  A reference to the passed range.
     */
    SheetUtils.adjustRange = function (range) {
        var tmp = null;
        // sort columns and rows independently
        if (range.start[0] > range.end[0]) { tmp = range.start[0]; range.start[0] = range.end[0]; range.end[0] = tmp; }
        if (range.start[1] > range.end[1]) { tmp = range.start[1]; range.start[1] = range.end[1]; range.end[1] = tmp; }
        return range;
    };

    /**
     * Returns a new range with column and row indexes in the correct order.
     *
     * @param {Object} range
     *  The logical address of a range.
     *
     * @returns {Object}
     *  A new adjusted cell range address.
     */
    SheetUtils.getAdjustedRange = function (range) {
        return {
            start: [Math.min(range.start[0], range.end[0]), Math.min(range.start[1], range.end[1])],
            end: [Math.max(range.start[0], range.end[0]), Math.max(range.start[1], range.end[1])]
        };
    };

    /**
     * Returns the number of columns covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @returns {Number}
     *  The number of columns in the passed range.
     */
    SheetUtils.getColCount = function (range) {
        return range.end[0] - range.start[0] + 1;
    };

    /**
     * Returns the number of rows covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @returns {Number}
     *  The number of rows in the passed range.
     */
    SheetUtils.getRowCount = function (range) {
        return range.end[1] - range.start[1] + 1;
    };

    /**
     * Returns the number of columns/rows covered by the passed cell range,
     * corresponding to the specified index for column (0) or row (1).
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @param {Number} index
     *  The index for column or row. Must be 0 for columns and 1 for rows.
     *
     * @returns {Number}
     *  The number of columns or rows in the passed range.
     */
    SheetUtils.getColRowCount = function (range, index) {
        return range.end[index] - range.start[index] + 1;
    };

    /**
     * Returns the number of cells covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @returns {Number}
     *  The number of cells in the passed range.
     */
    SheetUtils.getCellCount = function (range) {
        return SheetUtils.getColCount(range) * SheetUtils.getRowCount(range);
    };

    /**
     * Returns whether the passed range contains the specified column index.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @param {Number} col
     *  The zero-based column index to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed range contains the column index.
     */
    SheetUtils.rangeContainsCol = function (range, col) {
        return (range.start[0] <= col) && (col <= range.end[0]);
    };

    /**
     * Returns whether the passed range contains the specified row index.
     *
     * @param {Object} range
     *  The logical address of the range.
     *
     * @param {Number} row
     *  The zero-based row index to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed range contains the row index.
     */
    SheetUtils.rangeContainsRow = function (range, row) {
        return (range.start[1] <= row) && (row <= range.end[1]);
    };

    /**
     * Returns whether the passed bounding cell range contains the specified
     * cell.
     *
     * @param {Object} boundRange
     *  The logical addresses of the bounding cell range.
     *
     * @param {Number[]} address
     *  The logical cell address to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed bounding cell range contains the cell address.
     */
    SheetUtils.rangeContainsCell = function (boundRange, address) {
        return SheetUtils.rangeContainsCol(boundRange, address[0]) && SheetUtils.rangeContainsRow(boundRange, address[1]);
    };

    /**
     * Returns whether the cell address is one of the border cells in the
     * passed bounding cell range (a cell located at the left, right, top, or
     * bottom border of the cell range).
     *
     * @param {Object} boundRange
     *  The logical addresses of the bounding cell range.
     *
     * @param {Number[]} address
     *  The logical cell address to be checked whether it is one of the border
     *  cells in the passed bounding range.
     *
     * @returns {Boolean}
     *  Whether the passed cell is one of the border cells.
     */
    SheetUtils.isRangeBorderCell = function (boundRange, address) {
        return (address[0] === boundRange.start[0]) || (address[0] === boundRange.end[0]) || (address[1] === boundRange.start[1]) || (address[1] === boundRange.end[1]);
    };

    /**
     * Returns whether the passed bounding cell range contains another cell
     * range.
     *
     * @param {Object} boundRange
     *  The logical addresses of the bounding cell range.
     *
     * @param {Object} range
     *  The logical cell range address to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed bounding cell range contains the other cell range.
     */
    SheetUtils.rangeContainsRange = function (boundRange, range) {
        return SheetUtils.rangeContainsCell(boundRange, range.start) && SheetUtils.rangeContainsCell(boundRange, range.end);
    };

    // range intersections and bounding range ---------------------------------

    /**
     * Returns the unique ranges of the passed range addresses. Does NOT join
     * overlapping or adjacent ranges.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single cell range, or an array with cell range
     *  addresses.
     *
     * @returns {Array}
     *  The address of the resulting unique cell ranges.
     */
    SheetUtils.getUniqueRanges = function (ranges) {
        return _.chain(ranges).getArray().unique(SheetUtils.getRangeName).value();
    };

    /**
     * Returns whether the passed ranges overlap with at least one cell.
     *
     * @param {Object} range1
     *  The logical address of the first range.
     *
     * @param {Object} range2
     *  The logical address of the second range.
     *
     * @returns {Boolean}
     *  Whether the passed ranges are overlapping.
     */
    SheetUtils.rangesOverlap = function (range1, range2) {
        return (range1.start[0] <= range2.end[0]) && (range2.start[0] <= range1.end[0]) && (range1.start[1] <= range2.end[1]) && (range2.start[1] <= range1.end[1]);
    };

    /**
     * Returns whether any two ranges in the passed array of cell range
     * addresses overlap each other.
     *
     * @param {Array} ranges
     *  The array of logical range addresses.
     *
     * @returns {Boolean}
     *  Whether any two of the passed ranges are overlapping each other.
     */
    SheetUtils.anyRangesOverlap = function (ranges) {
        // TODO: this has O(n^2) complexity, better algorithm?
        return _(ranges).any(function (range1, index) {
            return _(ranges.slice(index + 1)).any(function (range2) {
                return SheetUtils.rangesOverlap(range1, range2);
            });
        });
    };

    /**
     * Returns the intersecting range of the passed ranges.
     *
     * @param {Object} range1
     *  The logical address of the first range.
     *
     * @param {Object} range2
     *  The logical address of the second range.
     *
     * @returns {Object|Null}
     *  The address of the cell range covered by both passed ranges, if
     *  existing; otherwise null.
     */
    SheetUtils.getIntersectionRange = function (range1, range2) {
        return SheetUtils.rangesOverlap(range1, range2) ? {
            start: [Math.max(range1.start[0], range2.start[0]), Math.max(range1.start[1], range2.start[1])],
            end: [Math.min(range1.end[0], range2.end[0]), Math.min(range1.end[1], range2.end[1])]
        } : null;
    };

    /**
     * Returns the range addresses resulting from intersecting all ranges in
     * the passed array and the bounding range.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single cell range, or an array with cell range
     *  addresses.
     *
     * @param {Object} boundRange
     *  The logical address of the bounding range.
     *
     * @returns {Array}
     *  The address of the resulting cell ranges covered by the bounding range.
     *  May be an empty array, if the bounding range does not contain any of
     *  the cells in the passed ranges.
     */
    SheetUtils.getIntersectionRanges = function (ranges, boundRange) {
        var result = [];
        _.chain(ranges).getArray().each(function (range) {
            var intersectRange = SheetUtils.getIntersectionRange(range, boundRange);
            if (intersectRange) { result.push(intersectRange); }
        });
        return result;
    };

    /**
     * Iterates the range addresses resulting from intersecting all ranges in
     * the passed array and the bounding range. Ranges not contained in the
     * bounding range at all will be skipped in the iteration process.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single cell range, or an array with cell range
     *  addresses.
     *
     * @param {Object} boundRange
     *  The logical address of the bounding range.
     *
     * @param {Function} iterator
     *  The iterator function called for all intersection ranges. Receives the
     *  following parameters:
     *  (1) {Object} intersectionRange
     *      The intersection of the original range and the bounding range.
     *  (2) {Object} originalRange
     *      The original range.
     *  (3) {Number} index
     *      The index of the original range in the passed array of ranges.
     *  If the iterator returns the Utils.BREAK object, the iteration
     *  process will be stopped immediately.
     *
     * @param {Object} [options]
     *  A map with options controlling the behavior of this method. The
     *  following options are supported:
     *  @param {Object} [options.context]
     *      If specified, the iterator will be called with this context
     *      (the symbol 'this' will be bound to the context inside the
     *      iterator function).
     *  @param {Boolean} [options.reverse=false]
     *      If set to true, the resulting intersection ranges will be visited
     *      in reversed order.
     *
     * @returns {Utils.BREAK|Undefined}
     *  A reference to the Utils.BREAK object, if the iterator has returned
     *  Utils.BREAK to stop the iteration process, otherwise undefined.
     */
    SheetUtils.iterateIntersectionRanges = function (ranges, boundRange, iterator, options) {

        var // the calling context for the iterator function
            context = Utils.getOption(options, 'context'),
            // whether to iterate in reversed order
            reverse = Utils.getBooleanOption(options, 'reverse', false);

        return Utils.iterateArray(_.getArray(ranges), function (range, index) {

            var // the intersection between current range and bound range
                intersectRange = SheetUtils.getIntersectionRange(range, boundRange);

            // invoke the iterator, if the intersection is not empty
            return intersectRange ? iterator.call(context, intersectRange, range, index) : undefined;
        }, { reverse: reverse });
    };

    /**
     * Returns the bounding range of the passed cell ranges (the smallest range
     * that contains all passed ranges).
     *
     * @param {Object|Array} [...]
     *  The logical address of a single cell range, or an array with cell range
     *  addresses. The number of parameters that can be passed to this method
     *  is not limited.
     *
     * @returns {Object|Null}
     *  The logical address of the bounding range containing all passed ranges,
     *  or null, if no range address has been passed.
     */
    SheetUtils.getBoundingRange = function () {
        var boundRange = null;
        _(arguments).each(function (ranges) {
            _.chain(ranges).getArray().each(function (range) {
                if (!boundRange) {
                    boundRange = _.copy(range, true);
                } else {
                    boundRange.start[0] = Math.min(boundRange.start[0], range.start[0]);
                    boundRange.start[1] = Math.min(boundRange.start[1], range.start[1]);
                    boundRange.end[0] = Math.max(boundRange.end[0], range.end[0]);
                    boundRange.end[1] = Math.max(boundRange.end[1], range.end[1]);
                }
            });
        });
        return boundRange;
    };

    // column/row intervals ---------------------------------------------------

    /**
     * Returns the number of indexes covered by the passed index interval.
     *
     * @param {Object} interval
     *  A single index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @returns {Number}
     *  The number of indexes covered by the passed interval.
     */
    SheetUtils.getIntervalSize = function (interval) {
        return interval.last - interval.first + 1;
    };

    /**
     * Returns whether the passed interval contains the specified index.
     *
     * @param {Object} interval
     *  The index interval, with the zero-based index properties 'first' and
     *  'last'.
     *
     * @param {Number} index
     *  The zero-based index to be checked.
     *
     * @returns {Boolean}
     *  Whether the passed interval contains the index.
     */
    SheetUtils.intervalContainsIndex = function (interval, index) {
        return (interval.first <= index) && (index <= interval.last);
    };

    /**
     * Returns whether the passed bounding interval contains another interval.
     *
     * @param {Object} boundInterval
     *  The bounding index interval, with the zero-based index properties
     *  'first' and 'last'.
     *
     * @param {Object} range
     *  The index interval to be checked, with the zero-based index properties
     *  'first' and 'last'.
     *
     * @returns {Boolean}
     *  Whether the passed bounding interval contains the other interval.
     */
    SheetUtils.intervalContainsInterval = function (boundInterval, interval) {
        return (boundInterval.first <= interval.first) && (interval.last <= boundInterval.last);
    };

    /**
     * Returns the intersection of the passed index intervals.
     *
     * @param {Object} interval1
     *  The first index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @param {Object} interval2
     *  The second index interval, with the zero-based index properties 'first'
     *  and 'last'.
     *
     * @returns {Object|Null}
     *  The intersection interval covered by both passed intervals, if
     *  existing; otherwise null.
     */
    SheetUtils.getIntersectionInterval = function (interval1, interval2) {
        var first = Math.max(interval1.first, interval2.first),
            last = Math.min(interval1.last, interval2.last);
        return (first <= last) ? { first: first, last: last } : null;
    };

    /**
     * Merges the passed column or row intervals.
     */
    function mergeIntervals(sourceIntervals) {

        var // the resulting intervals
            intervals = [];

        // sort the source intervals by start index
        sourceIntervals.sort(function (i1, i2) { return i1.first - i2.first; });

        // merge overlapping intervals
        _(sourceIntervals).each(function (interval) {

            var // last interval in the resulting list
                lastInterval = _.last(intervals);

            if (!lastInterval || (lastInterval.last + 1 < interval.first)) {
                // append a new interval
                intervals.push({ first: interval.first, last: interval.last });
            } else {
                // extend end of last interval
                lastInterval.last = Math.max(lastInterval.last, interval.last);
            }
        });

        return intervals;
    }

    /**
     * Returns the column interval covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the cell range.
     *
     * @returns {Object}
     *  The column interval covering the passed cell range, containing the
     *  zero-based index properties 'first' and 'last'.
     */
    SheetUtils.getColInterval = function (range) {
        return { first: range.start[0], last: range.end[0] };
    };

    /**
     * Returns the sorted and merged column intervals covering the passed range
     * addresses.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single cell range, or an array with cell range
     *  addresses.
     *
     * @returns {Array}
     *  The column intervals covering the passed cell ranges. Each interval
     *  object contains the zero-based index properties 'first' and 'last'. The
     *  intervals are sorted, and intervals covering overlapping or adjacent
     *  column ranges have been merged into a single interval object.
     */
    SheetUtils.getColIntervals = function (ranges) {
        return mergeIntervals(_.chain(ranges).getArray().map(SheetUtils.getColInterval).value());
    };

    /**
     * Returns the row interval covered by the passed cell range.
     *
     * @param {Object} range
     *  The logical address of the cell range.
     *
     * @returns {Object}
     *  The row interval covering the passed cell range, containing the
     *  zero-based index properties 'first' and 'last'.
     */
    SheetUtils.getRowInterval = function (range) {
        return { first: range.start[1], last: range.end[1] };
    };

    /**
     * Returns the sorted and merged row intervals covering the passed range
     * addresses.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single cell range, or an array with cell range
     *  addresses.
     *
     * @returns {Array}
     *  The row intervals covering the passed cell ranges. Each interval object
     *  contains the zero-based index properties 'first' and 'last'. The
     *  intervals are sorted, and intervals covering overlapping or adjacent
     *  row ranges have been merged into a single interval object.
     */
    SheetUtils.getRowIntervals = function (ranges) {
        return mergeIntervals(_.chain(ranges).getArray().map(SheetUtils.getRowInterval).value());
    };

    /**
     * Builds a logical range address from the passed column and row interval.
     *
     * @param {Object} colInterval
     *  The column interval, containing the zero-based index properties 'first'
     *  and 'last'.
     *
     * @param rowInterval
     *  The row interval, containing the zero-based index properties 'first'
     *  and 'last'.
     *
     * @returns {Object}
     *  The logical range address built from the passed intervals.
     */
    SheetUtils.makeRangeFromIntervals = function (colInterval, rowInterval) {
        return { start: [colInterval.first, rowInterval.first], end: [colInterval.last, rowInterval.last] };
    };

    // generate/parse UI names ------------------------------------------------

    /**
     * Returns the string representation of the passed column index.
     *
     * @param {Number} col
     *  The zero-based column index.
     *
     * @returns {String}
     *  The upper-case string representation of the column index.
     */
    SheetUtils.getColName = function (col) {
        var name = '';
        while (col >= 0) {
            name = String.fromCharCode(65 + col % 26) + name;
            col = Math.floor(col / 26) - 1;
        }
        return name;
    };

    /**
     * Returns the string representation of the passed column interval.
     *
     * @param {Object|Number} interval
     *  The column interval, in the zero-based column index properties 'first'
     *  and 'last', or a single zero-based column index.
     *
     * @returns {String}
     *  The upper-case string representation of the column interval.
     */
    SheetUtils.getColIntervalName = function (interval) {
        return SheetUtils.getColName(_.isNumber(interval) ? interval : interval.first) + ':' + SheetUtils.getColName(_.isNumber(interval) ? interval : interval.last);
    };

    /**
     * Returns the string representation of the passed column intervals.
     *
     * @param {Object|Array} intervals
     *  A single column interval, or an array of column intervals. Each
     *  interval object contains the zero-based index properties 'first' and
     *  'last'.
     *
     * @param {String} [separator=',']
     *  The separator character inserted between the column interval names.
     *
     * @returns {String}
     *  The upper-case string representation of the column intervals.
     */
    SheetUtils.getColIntervalsName = function (intervals, separator) {
        return _.chain(intervals).getArray().map(SheetUtils.getColIntervalName).value().join(separator);
    };

    /**
     * Returns the string representation of the passed row index.
     *
     * @param {Number} row
     *  The zero-based row index.
     *
     * @returns {String}
     *  The one-based string representation of the row index.
     */
    SheetUtils.getRowName = function (row) {
        return String(row + 1);
    };

    /**
     * Returns the string representation of the passed row interval.
     *
     * @param {Object|Number} interval
     *  The row interval, in the zero-based row index properties 'first' and
     *  'last', or a single zero-based row index.
     *
     * @returns {String}
     *  The one-based string representation of the row interval.
     */
    SheetUtils.getRowIntervalName = function (interval) {
        return SheetUtils.getRowName(_.isNumber(interval) ? interval : interval.first) + ':' + SheetUtils.getRowName(_.isNumber(interval) ? interval : interval.last);
    };

    /**
     * Returns the string representation of the passed row intervals.
     *
     * @param {Object|Array} intervals
     *  A single row interval, or an array of row intervals. Each interval
     *  object contains the zero-based index properties 'first' and 'last'.
     *
     * @param {String} [separator=',']
     *  The separator character inserted between the row interval names.
     *
     * @returns {String}
     *  The upper-case string representation of the row intervals.
     */
    SheetUtils.getRowIntervalsName = function (intervals, separator) {
        return _.chain(intervals).getArray().map(SheetUtils.getRowIntervalName).value().join(separator);
    };

    /**
     * Returns the string representation of the passed logical cell address.
     *
     * @param {Number[]} address
     *  The logical cell address as two-element array, containing the
     *  zero-based column and row indexes of the cell.
     *
     * @returns {String}
     *  The string representation of the cell position, in A1 notation.
     */
    SheetUtils.getCellName = function (address) {
        return SheetUtils.getColName(address[0]) + SheetUtils.getRowName(address[1]);
    };

    /**
     * Returns the string representation of the passed logical range address.
     *
     * @param {Object} range
     *  The logical cell range address as object with 'start' and 'end'
     *  properties, both containing logical cell addresses.
     *
     * @param {Object} [options]
     *  A map of options controlling the result of this method. The following
     *  options are supported:
     *  @param {Boolean} [options.shorten=false]
     *      If set to true, a cell range referring to a single cell will be
     *      converted to a cell address instead of a range address consisting
     *      of two equal cell addresses.
     *
     * @returns {String}
     *  The string representation of the cell range, in A1:A1 notation.
     */
    SheetUtils.getRangeName = function (range, options) {
        var name = SheetUtils.getCellName(range.start);
        if (!Utils.getBooleanOption(options, 'shorten', false) || !_.isEqual(range.start, range.end)) {
            name += ':' + SheetUtils.getCellName(range.end);
        }
        return name;
    };

    /**
     * Returns the string representation of the passed logical range addresses.
     *
     * @param {Object|Array} ranges
     *  The logical address of a single cell range, or an array with cell range
     *  addresses.
     *
     * @param {String} [separator=',']
     *  The separator character inserted between the range names.
     *
     * @param {Object} [options]
     *  A map of options controlling the result of this method. The following
     *  options are supported:
     *  @param {Boolean} [options.shorten=false]
     *      If set to true, a cell range referring to a single cell will be
     *      converted to a cell address instead of a range address consisting
     *      of two equal cell addresses.
     *
     * @returns {String}
     *  The string representation of the cell ranges, in A1:A1 notation.
     */
    SheetUtils.getRangesName = function (ranges, separator, options) {
        return _.chain(ranges).getArray().map(function (range) {
            return SheetUtils.getRangeName(range, options);
        }).value().join(separator);
    };

    /**
     * Returns the column index of the passed column string representation.
     *
     * @param {String} name
     *  The string representation of a column (case-insensitive).
     *
     * @returns {Number}
     *  The zero-based column index, if the passed column name is valid,
     *  otherwise -1.
     */
    SheetUtils.parseColName = function (name) {
        var col = -1;
        name = name.toUpperCase();
        if (RE_COL_NAME.test(name)) {
            for (var ichar = 0; ichar < name.length; ichar += 1) {
                col = (col + 1) * 26 + name.charCodeAt(ichar) - 65;
            }
        }
        return (col < MAX_COL_COUNT) ? col : -1;
    };

    /**
     * Returns the row index of the passed row string representation.
     *
     * @param {String} name
     *  The string representation of a row (one-based).
     *
     * @returns {Number}
     *  The zero-based row index, if the passed row name is valid, otherwise
     *  the value -1.
     */
    SheetUtils.parseRowName = function (name) {
        var row = RE_ROW_NAME.test(name) ? (parseInt(name, 10) - 1) : -1;
        return (row < MAX_ROW_COUNT) ? row : -1;
    };

    SheetUtils.parseCellName = function (name) {
    };

    SheetUtils.parseRangeName = function (name, options) {
    };


    // exports ================================================================

    return SheetUtils;

});
