/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/modelobject',
    ['io.ox/core/event',
     'io.ox/office/tk/utils'
    ], function (Events, Utils) {

    'use strict';

    // class ModelObject ======================================================

    /**
     * An abstract model object that allows to trigger change events to event
     * listeners. As long as the application still imports the document, no
     * events will be triggered to the event listeners.
     *
     * @constructor
     *
     * @extends Events
     *
     * @param {EditApplication} app
     *  The application instance containing this attributed model object.
     *
     * @param {Object} [options]
     *  A map with additional options for this instance. The following options
     *  are supported:
     *  @param {Boolean} [options.silent=false]
     *      If set to true, this instance will never trigger any change events.
     *      Thus, specific sub classes may chose to be always silent although
     *      their base class derives from this class.
     */
    function ModelObject(app, options) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        if (!Utils.getBooleanOption(options, 'silent', false)) {
            Events.extend(this);
        }

        // private methods ----------------------------------------------------

        /**
         * Enables triggering event after the document has been imported.
         */
        function importFinishedHandler() {
            app.off('docs:import:success', importFinishedHandler);
            self.trigger = self.events.trigger;
        }

        // methods ------------------------------------------------------------

        this.destroy = function () {
            if (this.events) { this.events.destroy(); }
        };

        // initialization -----------------------------------------------------

        // start triggering change events after the document has been imported
        if (this.events && !app.isImportFinished()) {
            app.on('docs:import:success', importFinishedHandler);
            this.trigger = function () { return this; };
        } else if (!this.events) {
            this.trigger = function () { return this; };
        }

    } // class ModelObject

    // exports ================================================================

    return _.makeExtendable(ModelObject);

});
