/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/format/documentstyles',
    ['io.ox/core/event',
     'io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/fonts',
     'io.ox/office/editframework/model/format/themes',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/model/format/border'
    ], function (Events, Utils, Fonts, Themes, Color, Border) {

    'use strict';

    // class DocumentStyles ===================================================

    /**
     * Provides the style sheet containers for all attribute families used in a
     * document, and other containers collecting special formatting information
     * for the document. Creates a font container (instance of class Fonts)
     * assigned to the container key 'fonts', and a themes container (instance
     * of class Themes) assigned to the container key 'themes'.
     *
     * Triggers the following events:
     * - 'change:attributes': After the document attributes have been changed
     *      via the method 'DocumentStyles.setAttributes()'. The event handlers
     *      receive the new attribute map, and the old attribute map.
     *
     * @constructor
     *
     * @extends Events
     *
     * @param {EditApplication} app
     *  The root application instance.
     *
     * @param {Object} attributeDefaultValues
     *  Default values of all document attributes supported by the current
     *  document.
     */
    function DocumentStyles(app, attributeDefaultValues) {

        var // self reference
            self = this,

            // the collection of document fonts
            fonts = new Fonts(app),

            // the collection of document themes
            themes = new Themes(app),

            // style sheet containers mapped by attribute family
            styleSheets = {},

            // other specific containers
            containers = {},

            // global document attributes
            documentAttributes = _.copy(attributeDefaultValues),

            // attribute definitions registry (mapped by attribute family)
            definitionsRegistry = {};

        // base constructor ---------------------------------------------------

        Events.extend(this);

        // private methods ----------------------------------------------------

        /**
         * Returns an existing entry in the attribute definition registry, or
         * creates and initializes a new entry.
         */
        function getOrCreateRegistryEntry(styleFamily) {
            return definitionsRegistry[styleFamily] || (definitionsRegistry[styleFamily] = {
                definitions: {},
                parentFamilies: {},
                supportedFamilies: [styleFamily]
            });
        }

        // methods ------------------------------------------------------------

        this.registerAttributeDefinitions = function (styleFamily, definitions, options) {

            // updates the supported families of the parent registry entries (recursively)
            function updateParentEntries(registryEntry) {
                _(registryEntry.parentFamilies).each(function updateParentEntry(entry, parentFamily) {
                    var parentEntry = getOrCreateRegistryEntry(parentFamily);
                    parentEntry.supportedFamilies = _.unique(parentEntry.supportedFamilies.concat(registryEntry.supportedFamilies));
                    updateParentEntries(parentEntry);
                });
            }

            var // get existing or create a new entry in the global registry
                registryEntry = getOrCreateRegistryEntry(styleFamily);

            // insert information for the new style family
            registryEntry.definitions = definitions;
            registryEntry.parentFamilies = Utils.getObjectOption(options, 'parentFamilies', {});

            // store references back to this style families in specified parent families
            updateParentEntries(registryEntry);
        };

        /**
         * Adds a style sheet container to this document styles collection.
         *
         * @internal
         *  Called from constructor functions of derived classes.
         *
         * @param {StyleSheets} styleSheetsContainer
         *  The new style sheets container.
         *
         * @returns {DocumentStyles}
         *  A reference to this instance.
         */
        this.addStyleSheetsContainer = function (styleSheetsContainer) {
            styleSheets[styleSheetsContainer.getStyleFamily()] = styleSheetsContainer;
            return this;
        };

        /**
         * Returns the style sheet container for the specified attribute
         * family.
         *
         * @param {String} family
         *  The name of the attribute family.
         *
         * @returns {StyleSheets|Null}
         *  The specified style sheets container if existing, otherwise null.
         */
        this.getStyleSheets = function (family) {
            return (family in styleSheets) ? styleSheets[family] : null;
        };

        /**
         * Returns the static definitions map of all attributes registered for
         * the specified style family.
         *
         * @param {String} styleFamily
         *  The style family whose attribute definitions will be returned.
         *
         * @returns {Object|Null}
         *  A reference to the static attribute definitions map, or null if the
         *  passed style family is invalid.
         */
        this.getAttributeDefinitions = function (styleFamily) {
            return Utils.getObjectOption(definitionsRegistry[styleFamily], 'definitions', null);
        };

        /**
         * Returns the supported attribute families of the style sheet container
         * class associated to the specified style family.
         *
         * @param {String} styleFamily
         *  The style family whose supported attribute families will be
         *  returned.
         *
         * @returns {String[]|Null}
         *  A string array containing the supported attribute families, or null
         *  if the passed style family is invalid.
         */
        this.getSupportedFamilies = function (styleFamily) {
            return Utils.getArrayOption(definitionsRegistry[styleFamily], 'supportedFamilies', null);
        };

        /**
         * Returns the parent style families of the style sheet container class
         * associated to the specified style family.
         *
         * @param {String} styleFamily
         *  The style family whose parent style families will be returned.
         *
         * @returns {Object|Null}
         *  A map containing the parent style families as keys, mapping the
         *  element resolver functions used to get the parent DOM element from
         *  a descendant element.
         */
        this.getParentFamilies = function (styleFamily) {
            return Utils.getObjectOption(definitionsRegistry[styleFamily], 'parentFamilies', null);
        };

        /**
         * Builds an attribute map containing all formatting attributes
         * registered for the specified style family, set to the null value.
         *
         * @param {String} styleFamily
         *  The main style family of the attributes to be inserted into the
         *  returned attribute map.
         *
         * @param {Object} [options]
         *  A map of options to control the operation. The following options
         *  are supported:
         *  @param {Boolean} [options.supportedFamilies=false]
         *      If set to true, adds null values for all additional attribute
         *      families supported by the style sheet container associated to
         *      the passed style family. Otherwise, only the attributes of the
         *      passed style family will be added to the returned attribute
         *      set.
         *
         * @returns {Object}
         *  An attribute set with null values for all attributes registered for
         *  the supported attribute families.
         */
        this.buildNullAttributes = function (styleFamily, options) {

            var // the resulting attribute map
                attributes = { styleId: null };

            // adds the null values for all attributes of the specified attribute family
            function addNullAttributes(family) {

                var // the attribute definitions of the current family
                    definitions = self.getAttributeDefinitions(family),
                    // attribute values of the current family
                    attributeValues = attributes[family] = {};

                // add null values for all registered attributes
                _(definitions).each(function (definition, name) {
                    if (!definition.special) {
                        attributeValues[name] = null;
                    }
                });
            }

            // add null values for all attributes
            if (Utils.getBooleanOption(options, 'supportedFamilies', false)) {
                // process all supported attribute families
                _(this.getSupportedFamilies(styleFamily)).each(addNullAttributes);
            } else {
                // only for the passed style family
                addNullAttributes(styleFamily);
            }

            return attributes;
        };

        /**
         * Extends the passed attribute set with the existing values of the
         * second attribute set. If the definitions of a specific attribute
         * contain a merger function, and both attribute sets contain an
         * attribute value, uses that merger function to merge the values from
         * both attribute sets, otherwise the value of the second attribute set
         * will be copied to the first attribute set.
         *
         * @param {Object} attributes1
         *  (in/out) The first attribute set that will be extended in-place, as
         *  map of attribute value maps (name/value pairs), keyed by attribute
         *  family.
         *
         * @param {Object} attributes2
         *  The second attribute set, as map of attribute value maps
         *  (name/value pairs), keyed by attribute family, whose attribute
         *  values will be inserted into the first attribute set.
         *
         * @returns {Object}
         *  A reference to the first passed and extended attribute set.
         */
        this.extendAttributes = function (attributes1, attributes2) {

            // add attributes existing in attributes2 to attributes1
            _(attributes2).each(function (attributeValues, family) {

                var // the attribute definitions of the current family
                    definitions = this.getAttributeDefinitions(family);

                // copy style sheet identifier directly (may be string or null)
                if (family === 'styleId') {
                    attributes1.styleId = attributeValues;
                    return;
                }

                // restrict to valid attribute families
                if (!definitions || !_.isObject(attributeValues)) { return; }

                // insert attribute values of attributes2
                attributes1[family] = attributes1[family] || {};
                _(attributeValues).each(function (value, name) {

                    var // the merger function from the attribute definition
                        merger = null;

                    if (DocumentStyles.isRegisteredAttribute(definitions, name, { special: true })) {
                        // try to find merger function from attribute definition
                        merger = definitions[name].merge;
                        // either set return value from merger, or copy the attribute directly
                        if ((name in attributes1[family]) && !_.isNull(attributes1[family][name]) && _.isFunction(merger)) {
                            attributes1[family][name] = merger.call(this, attributes1[family][name], value);
                        } else {
                            attributes1[family][name] = value;
                        }
                    }
                });
            }, this);

            return attributes1;
        };

        /**
         * Adds a custom formatting container to this document styles
         * collection.
         *
         * @internal
         *  Called from constructor functions of derived classes.
         *
         * @param {String} key
         *  The unique key of the custom container.
         *
         * @param {Container} container
         *  The new custom formatting container.
         *
         * @returns {DocumentStyles}
         *  A reference to this instance.
         */
        this.addCustomContainer = function (key, container) {
            containers[key] = container;
            return this;
        };

        /**
         * Returns the specified custom formatting container.
         *
         * @param {String} key
         *  The unique key of the custom container.
         *
         * @returns {Container|Null}
         *  The specified custom container if existing, otherwise null.
         */
        this.getContainer = function (key) {
            return (key in containers) ? containers[key] : null;
        };

        /**
         * Returns the global attributes of the document.
         *
         * @param {Object}
         *  A deep clone of the global document attributes.
         */
        this.getAttributes = function () {
            return _.copy(documentAttributes, true);
        };

        /**
         * Changes the global attributes of the document.
         *
         * @param {Object} attributes
         *  The new document attributes.
         *
         * @returns {DocumentStyles}
         *  A reference to this instance.
         */
        this.setAttributes = function (attributes) {

            var // copy of the old attributes for change listeners
                oldAttributes = this.getAttributes(),
                // whether any attribute has been changed
                changed = false;

            // update all valid attributes
            _(attributes).each(function (value, name) {
                if ((name in documentAttributes) && !_.isEqual(documentAttributes[name], value)) {
                    documentAttributes[name] = _.copy(value, true);
                    changed = true;
                }
            });

            // notify all change listeners
            if (changed) {
                this.trigger('change:attributes', this.getAttributes(), oldAttributes);
            }

            return this;
        };

        /**
         * Returns the value of the CSS 'font-family' attribute containing the
         * specified font name and all alternative font names.
         *
         * @param {String} fontName
         *  The name of of the font (case-insensitive).
         *
         * @returns {String}
         *  The value of the CSS 'font-family' attribute containing the
         *  specified font name and all alternative font names.
         */
        this.getCssFontFamily = function (fontName) {
            return fonts.getCssFontFamily(fontName);
        };

        /**
         * Converts the passed color attribute object to a CSS color value.
         * Scheme colors will be resolved by using the current theme.
         *
         * @param {Object} color
         *  The color object as used in operations.
         *
         * @param {String} context
         *  The context needed to resolve the color type 'auto'.
         *
         * @returns {String}
         *  The CSS color value converted from the passed color object.
         */
        this.getCssColor = function (color, context) {
            // use the static helper function from module Color, pass current theme
            return Color.getCssColor(color, context, themes.getTheme());
        };

        /**
         * Resolves the passed text color to an explicit color if it is set to
         * 'auto', according to the specified fill colors.
         *
         * @param {Object} textColor
         *  The source text color object, as used in operations.
         *
         * @param {Object[]} fillColors
         *  The source fill color objects, from outermost to innermost level.
         *
         * @returns {String}
         *  The CSS text color value converted from the passed color objects.
         */
        this.getCssTextColor = function (textColor, fillColors) {

            var // effective fill color
                effectiveFillColor = null;

            if (!_.isObject(textColor) || Color.isAutoColor(textColor)) {

                // find last non-transparent fill color in the passed array
                // TODO: merge partly-transparent colors?
                _(fillColors).each(function (fillColor) {
                    if (fillColor && !Color.isAutoColor(fillColor)) {
                        effectiveFillColor = fillColor;
                    }
                });

                // convert fill color to RGB (still transparent: fall-back to white)
                effectiveFillColor = this.getCssColor(effectiveFillColor || Color.WHITE, 'fill');

                // set text color to black or white according to fill color brightness
                textColor = Color.isDark(effectiveFillColor) ? Color.WHITE : Color.BLACK;
            }

            // convert to CSS color string
            return this.getCssColor(textColor, 'text');
        };

        /**
         * Returns the effective CSS attributes for the passed border value.
         * Scheme colors will be resolved by using the current theme.
         *
         * @param {Object} border
         *  The border object as used in operations.
         *
         * @param {Object} [options]
         *  A map of options controlling the operation. Supports the following
         *  options:
         *  @param {Boolean} [options.preview=false]
         *      If set to true, the border will be rendered for a preview
         *      element in the GUI. The border width will be restricted to two
         *      pixels in that case.
         *
         * @returns {Object}
         *  A map with CSS border attributes. Contains the property 'style'
         *  with the effective CSS border style as string; the property 'width'
         *  with the effective border width in pixels (as number, rounded to
         *  entire pixels); and the property 'color' with the effective CSS
         *  color (as string with leading hash sign).
         */
        this.getCssBorderAttributes = function (border, options) {
            // use the static helper function from module Border, pass current theme
            return Border.getCssBorderAttributes(border, themes.getTheme(), options);
        };

        /**
         * Converts the passed border attribute object to a CSS border value.
         * Scheme colors will be resolved by using the current theme.
         *
         * @param {Object} border
         *  The border object as used in operations.
         *
         * @param {Object} [options]
         *  A map of options controlling the operation. Supports the following
         *  options:
         *  @param {Boolean} [options.clearNone=false]
         *      If set to true, the return value for invisible borders will be
         *      the empty string instead of the keyword 'none'.
         *  @param {Boolean} [options.preview=false]
         *      If set to true, the border will be rendered for a preview
         *      element in the GUI. The border width will be restricted to two
         *      pixels in that case.
         *
         * @returns {String}
         *  The CSS border value converted from the passed border object.
         */
        this.getCssBorder = function (border, options) {
            // use the static helper function from module Border, pass current theme
            return Border.getCssBorder(border, themes.getTheme(), options);
        };

        /**
         * Returns whether the current theme contains scheme color definitions.
         */
        this.hasSchemeColors = function () {
            return themes.getTheme().hasSchemeColors();
        };

        this.destroy = function () {
            this.events.destroy();
            _(containers).invoke('destroy');
            _(styleSheets).invoke('destroy');
            styleSheets = containers = null;
        };

        // initialization -----------------------------------------------------

        this.addCustomContainer('fonts', fonts)
            .addCustomContainer('themes', themes);

    } // class DocumentStyles

    // static methods ---------------------------------------------------------

    /**
     * Returns whether the passed string is the name of a attribute that is
     * registered in the passed attribute definition map.
     *
     * @param {Object} definitions
     *  The attribute definitions map.
     *
     * @param {String} name
     *  The attribute name to be checked.
     *
     * @param {Object} [options]
     *  A map of options controlling the operation. Supports the following
     *  options:
     *  @param {Boolean} [options.special=false]
     *      If set to true, returns true for special attributes (attributes
     *      that are marked with the 'special' flag in the attribute
     *      definitions). Otherwise, special attributes will not be recognized
     *      by this function.
     *
     * @returns {Boolean}
     *  Whether the attribute is registered in the attribute definitions map.
     */
    DocumentStyles.isRegisteredAttribute = function (definitions, name, options) {

        var // whether to include special attributes
            special = Utils.getBooleanOption(options, 'special', false);

        return (name in definitions) && (special || (definitions[name].special !== true));
    };

    // exports ================================================================

    return _.makeExtendable(DocumentStyles);

});
