/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/operations',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/operationsgenerator',
     'io.ox/office/editframework/model/format/stylesheets',
     'io.ox/office/drawinglayer/view/drawingframe',
     'io.ox/office/text/dom',
     'io.ox/office/text/position',
     'io.ox/office/text/image'
    ], function (Utils, OperationsGenerator, StyleSheets, DrawingFrame, DOM, Position, Image) {

    'use strict';

    // class TextOperationsGenerator ==========================================

    /**
     * An instance of this class contains an operations array and provides
     * methods to generate operations for various element nodes.
     *
     * @constructor
     *
     * @extends OperationsGenerator
     *
     * @param {TextDocumentStyles} documentStyles
     *  Global collection with the style sheet containers and custom formatting
     *  containers of a document.
     */
    function TextOperationsGenerator(documentStyles) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        OperationsGenerator.call(this, documentStyles);

        // methods ------------------------------------------------------------

        /**
         * Generates all operations needed to recreate the child nodes of the
         * passed paragraph.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph element whose content nodes will be converted to
         *  operations. If this object is a jQuery collection, uses the first
         *  node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed paragraph node. The generated
         *  operations will contain positions starting with this address.
         *
         * @param {Object} [options]
         *  A map with options controlling the operation generation process.
         *  Supports the following options:
         *  @param {Number} [options.start]
         *      The logical index of the first character to be included into
         *      the generated operations. By default, operations will include
         *      all contents from the beginning of the paragraph.
         *  @param {Number} [options.end]
         *      The logical index of the last character to be included into the
         *      generated operations (closed range). By default, operations
         *      will include all contents up to the end of the paragraph.
         *  @param {Boolean} [options.clear=false]
         *      If set to true, a 'setAttributes' operation will be generated
         *      for the first 'insertText' operation that clears all character
         *      attributes of the inserted text. This prevents that applying
         *      the operations at another place in the document clones the
         *      character formatting of the target position.
         *  @param {Number} [options.targetOffset]
         *      If set to a number, the logical positions in the operations
         *      generated for the child nodes will start at this offset. If
         *      omitted, the original node offset will be used in the logical
         *      positions.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateParagraphChildOperations = function (paragraph, position, options) {

            var // start of text range to be included in the operations
                rangeStart = Utils.getIntegerOption(options, 'start'),
                // end of text range to be included in the operations
                rangeEnd = Utils.getIntegerOption(options, 'end'),
                // start position of text nodes in the generated operations
                targetOffset = Utils.getIntegerOption(options, 'targetOffset'),

                // used to merge several text portions into the same operation
                lastTextOperation = null,
                // formatting ranges for text portions, must be applied after the contents
                attributeRanges = [],
                // attributes passed to the first insert operation to clear all formatting
                clearAttributes = null;

            // generates the specified operation, adds that attributes in clearAttributes on first call
            function generateOperationWithClearAttributes(name, operationOptions) {

                // add the character attributes that will be cleared on first insertion operation
                if (_.isObject(clearAttributes)) {
                    operationOptions.attrs = clearAttributes;
                    clearAttributes = null;
                }

                return self.generateOperation(name, operationOptions);
            }

            // generates a usable image operation for an internal & external URL
            function generateImageOperationWithAttributes(node, operationOptions) {

                var // the new operation
                    operation = self.generateOperationWithAttributes(node, TextOperationsGenerator.DRAWING_INSERT, _.extend({ type: 'image' }, operationOptions)),
                    // the image URL from the attributes map
                    imageUrl = operation.attrs && operation.attrs.drawing && operation.attrs.drawing.imageUrl;

                // set image attributes so they can be used in the same instance or between different instances
                if (imageUrl && DOM.isDocumentImageNode(node)) {
                    _(operation.attrs.drawing).extend({
                        imageData: DOM.getBase64FromImageNode(node, Image.getMimeTypeFromImageUri(imageUrl)),
                        sessionId: DOM.getUrlParamFromImageNode(node, 'session'),
                        fileId: DOM.getUrlParamFromImageNode(node, 'id')
                    });
                }

                return operation;
            }

            // clear all attributes of the first inserted text span
            if (Utils.getBooleanOption(options, 'clear', false)) {
                clearAttributes = documentStyles.buildNullAttributes('character', { supportedFamilies: true });
            }

            // process all content nodes in the paragraph and create operations
            Position.iterateParagraphChildNodes(paragraph, function (node, nodeStart, nodeLength, offsetStart, offsetLength) {

                var // logical start index of the covered part of the child node
                    startIndex = _.isNumber(targetOffset) ? targetOffset : (nodeStart + offsetStart),
                    // logical end index of the covered part of the child node (closed range)
                    endIndex = startIndex + offsetLength - 1,
                    // logical start position of the covered part of the child node
                    startPosition = Position.appendNewIndex(position, startIndex),
                    // logical end position of the covered part of the child node
                    endPosition = Position.appendNewIndex(position, endIndex),
                    // text of a portion span
                    text = null,
                    // type of the drawing
                    type = null;

                // operation to create a (non-empty) generic text portion
                if (DOM.isTextSpan(node)) {

                    // extract the text covered by the specified range
                    text = node.firstChild.nodeValue.substr(offsetStart, offsetLength);
                    // append text portions to the last 'insertText' operation
                    if (lastTextOperation) {
                        lastTextOperation.text += text;
                    } else {
                        lastTextOperation = generateOperationWithClearAttributes(TextOperationsGenerator.TEXT_INSERT, { start: startPosition, text: text });
                    }
                    attributeRanges.push({ node: node, position: _.extend({ start: startPosition }, (text.length > 1) ? { end: endPosition } : undefined) });

                } else {

                    // anything else than plain text will be inserted, forget last text operation
                    lastTextOperation = null;

                    // operation to create a text field
                    // TODO: field type
                    if (DOM.isFieldNode(node)) {
                        // extract text of all embedded spans representing the field
                        text = $(node).text();
                        generateOperationWithClearAttributes(TextOperationsGenerator.FIELD_INSERT, { start: startPosition, representation: text, type: '' });
                        // attributes are contained in the embedded span elements
                        attributeRanges.push({ node: node.firstChild, position: { start: startPosition } });
                    }

                    // operation to create a tabulator
                    else if (DOM.isTabNode(node)) {
                        generateOperationWithClearAttributes(TextOperationsGenerator.TAB_INSERT, { start: startPosition });
                        // attributes are contained in the embedded span elements
                        attributeRanges.push({ node: node.firstChild, position: { start: startPosition } });
                    }

                    // operation to create a hard break
                    else if (DOM.isHardBreakNode(node)) {
                        generateOperationWithClearAttributes(TextOperationsGenerator.HARDBREAK_INSERT, { start: startPosition });
                        // attributes are contained in the embedded span elements
                        attributeRanges.push({ node: node.firstChild, position: { start: startPosition } });
                    }

                    // operation to create a drawing (including its attributes)
                    else if (DrawingFrame.isDrawingFrame(node)) {
                        type = DrawingFrame.getDrawingType(node);
                        if (type === 'image') {
                            // Special image handling: We need to take care that images are always readable
                            // for the target instance
                            generateImageOperationWithAttributes(node, { start: startPosition });
                        } else {
                            this.generateOperationWithAttributes(node, TextOperationsGenerator.DRAWING_INSERT, { start: startPosition, type: type });
                        }
                    }

                    else {
                        Utils.error('TextOperationsGenerator.generateParagraphChildOperations(): unknown content node');
                        return Utils.BREAK;
                    }
                }

                // custom target offset: advance offset by covered node length
                if (_.isNumber(targetOffset)) {
                    targetOffset += offsetLength;
                }

            }, this, { start: rangeStart, end: rangeEnd });

            // Generate 'setAttribute' operations after all contents have been
            // created via 'insertText', 'insertField', etc. Otherwise, these
            // operations would clone the attributes of the last text portion
            // instead of creating a clean text node as expected in this case.
            _(attributeRanges).each(function (range) {
                this.generateSetAttributesOperation(range.node, range.position);
            }, this);

            return this;
        };

        /**
         * Generates all operations needed to recreate the passed paragraph.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph element whose contents will be converted to
         *  operations. If this object is a jQuery collection, uses the first
         *  node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed paragraph node. The generated
         *  operations will contain positions starting with this address.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateParagraphOperations = function (paragraph, position) {

            // operations to create the paragraph element and formatting
            this.generateOperationWithAttributes(paragraph, TextOperationsGenerator.PARA_INSERT, { start: position });

            // process all content nodes in the paragraph and create operations
            return this.generateParagraphChildOperations(paragraph, position);
        };

        /**
         * Generates all operations needed to recreate the passed table cell.
         *
         * @param {HTMLTableCellElement|jQuery} cellNode
         *  The table cell element that will be converted to operations. If
         *  this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed table cell. The generated
         *  operations will contain positions starting with this address.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateTableCellOperations = function (cellNode, position) {

            // operation to create the table cell element
            this.generateOperationWithAttributes(cellNode, TextOperationsGenerator.CELLS_INSERT, { start: position, count: 1 });

            // generate operations for the contents of the cell
            return this.generateContentOperations(cellNode, position);
        };

        /**
         * Generates all operations needed to recreate the passed table row.
         *
         * @param {HTMLTableRowElement|jQuery} rowNode
         *  The table row element that will be converted to operations. If this
         *  object is a jQuery collection, uses the first node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed table row. The generated
         *  operations will contain positions starting with this address.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateTableRowOperations = function (rowNode, position) {

            // operation to create the table row element
            this.generateOperationWithAttributes(rowNode, TextOperationsGenerator.ROWS_INSERT, { start: position });

            // generate operations for all cells
            position = Position.appendNewIndex(position);
            Utils.iterateSelectedDescendantNodes(rowNode, 'td', function (cellNode) {
                this.generateTableCellOperations(cellNode, position);
                position = Position.increaseLastIndex(position);
            }, this, { children: true });

            return this;
        };

        /**
         * Generates all operations needed to recreate the passed table.
         *
         * @param {HTMLTableElement|jQuery} tableNode
         *  The table element that will be converted to operations. If this
         *  object is a jQuery collection, uses the first node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed table node. The generated
         *  operations will contain positions starting with this address.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateTableOperations = function (tableNode, position) {

            // operation to create the table element
            this.generateOperationWithAttributes(tableNode, TextOperationsGenerator.TABLE_INSERT, { start: position });

            // generate operations for all rows
            position = Position.appendNewIndex(position);
            DOM.getTableRows(tableNode).each(function () {
                self.generateTableRowOperations(this, position);
                position = Position.increaseLastIndex(position);
            });

            return this;
        };

        /**
         * Generates all operations needed to recreate the contents of the
         * passed root node. Root nodes are container elements for text
         * paragraphs and other first-level content nodes (e.g. tables).
         * Examples for root nodes are the entire document root node, table
         * cells, or text shapes. Note that the operation to create the root
         * node itself will NOT be generated.
         *
         * @param {HTMLElement|jQuery} rootNode
         *  The root node containing the content nodes that will be converted
         *  to operations. The passed root node may contain an embedded node
         *  that serves as parent for all content nodes.  If this object is a
         *  jQuery collection, uses the first node it contains.
         *
         * @param {Number[]} position
         *  The logical position of the passed node. The generated operations
         *  will contain positions starting with this address.
         *
         * @returns {TextOperationsGenerator}
         *  A reference to this instance.
         */
        this.generateContentOperations = function (rootNode, position) {

            var // the container node (direct parent of the target content nodes)
                containerNode = DOM.getChildContainerNode(rootNode);

            // iterate all child elements of the root node and create operations
            position = Position.appendNewIndex(position);
            Utils.iterateDescendantNodes(containerNode, function (node) {

                if (DOM.isParagraphNode(node)) {
                    // skip implicit paragraph nodes without increasing the position
                    if (DOM.isImplicitParagraphNode(node)) { return; }
                    // operations to create a paragraph
                    this.generateParagraphOperations(node, position);
                } else if (DOM.isTableNode(node)) {
                    // skip oversized table nodes without increasing the position
                    if (DOM.isExceededSizeTableNode(node)) { return; }
                    // operations to create a table with its structure and contents
                    this.generateTableOperations(node, position);
                } else {
                    Utils.error('TextOperationsGenerator.generateContentOperations(): unexpected node "' + Utils.getNodeName(node) + '" at position ' + JSON.stringify(position) + '.');
                    // continue with next child node (do not increase position)
                    return;
                }

                // increase last element of the logical position (returns a clone)
                position = Position.increaseLastIndex(position);

            }, this, { children: true });

            return this;
        };

    } // class TextOperationsGenerator

    // constants --------------------------------------------------------------

    _(TextOperationsGenerator).extend({

        DELETE: 'delete',
        MOVE: 'move',

        TEXT_INSERT: 'insertText',
        DRAWING_INSERT: 'insertDrawing',
        FIELD_INSERT: 'insertField',
        TAB_INSERT: 'insertTab',
        HARDBREAK_INSERT: 'insertHardBreak',

        PARA_INSERT: 'insertParagraph',
        PARA_SPLIT: 'splitParagraph',
        PARA_MERGE: 'mergeParagraph',

        TABLE_INSERT: 'insertTable',
        ROWS_INSERT: 'insertRows',
        CELLS_INSERT: 'insertCells',
        CELL_SPLIT: 'splitCell',
        CELL_MERGE: 'mergeCell',
        COLUMN_INSERT: 'insertColumn',
        COLUMNS_DELETE: 'deleteColumns',

        INSERT_LIST: 'insertListStyle',
        DELETE_LIST: 'deleteListStyle'

    });

    // exports ================================================================

    // derive this class from class OperationsGenerator
    return OperationsGenerator.extend({ constructor: TextOperationsGenerator });

});
