/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 */

define('io.ox/office/text/hyperlink',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/dialogs',
     'io.ox/office/editframework/model/format/stylesheets',
     'io.ox/office/text/dom',
     'io.ox/office/text/position',
     'io.ox/office/text/operations',
     'gettext!io.ox/office/text'
    ], function (Utils, KeyCodes, Dialogs, StyleSheets, DOM, Position, Operations, gt) {

    'use strict';

    // static private functions ===============================================

    /**
     * Shows the hyperlink pop-up menu.
     *
     * @param {TextApplication} app
     *  The text application instance.
     */
    function showPopup(app, hyperlinkPopup) {
        hyperlinkPopup.show();
        app.getView().scrollToChildNode(hyperlinkPopup);
    }

    /**
     * Hides the hyperlink pop-up menu.
     *
     * @param {jQuery} hyperlinkPopup
     *  The hyperlink pop-up which should be hidden.
     */
    function hidePopup(hyperlinkPopup) {
        hyperlinkPopup.hide();
    }

    function createResultFromHyperlinkSelection(position, hyperlinkSelection) {

        var result = null,
            start = _.clone(position),
            end = _.clone(position);

        // create result with correct Position objects
        start[start.length - 1] = hyperlinkSelection.start;
        end[end.length - 1] = hyperlinkSelection.end;
        result = { start: start, end: end, text: hyperlinkSelection.text, url: null };

        return result;
    }

    // static class Hyperlink =================================================

    /**
     * Provides static helper methods for manipulation and calculation
     * of a hyperlink.
     */
    var Hyperlink = {
        Separators: [ '!', '?', '.', ' ', '-', ':', ',', '\u00a0'],
        Protocols: [ 'http://', 'https://', 'ftp://', 'mailto:' ],

        // templates for web/ftp identifications
        TextTemplates: [ 'www.', 'ftp.' ],

        /**
         * Predefined character attributes needed to remove a hyperlink style
         * from a piece of text.
         */
        CLEAR_ATTRIBUTES: { styleId: null, character: { url: null } }
    };

    // methods ----------------------------------------------------------------

    /**
     * Shows a hyperlink input dialog.
     *
     * @param {String} text
     *  The optional text which represents the URL
     *
     * @param {String} url
     *  An optional URL which is set for the supplied text
     *
     * @param {BaseModel} model
     *  The calling model
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the primary
     *  button has been activated, or rejected if the dialog has been canceled.
     *  The done handlers registered at the promise object will receive the
     *  entered text.
     */
    Hyperlink.showDialog = function (text, url, model) {

        return Hyperlink.showHyperlinkDialog(
            model, {
            title: gt('Insert/Edit Hyperlink'),
            valueURL: url,
            placeholderURL: gt('Enter URL'),
            valueText: text,
            placeholderText: gt('Enter visible text'),
            okLabel: gt('Insert')
        });
    };

    /**
     * Provides the URL of a selection without range.
     *
     * @param {Selection} selection
     *  A selection object which describes the current selection.
     *
     * @returns {Object}
     *  - {String|Null} url
     *      The URL or null if no URL character attribute is set at the
     *      selection.
     *  - {Boolean} beforeHyperlink
     *      True if we provide the URL for the hyperlink which is located at
     *      the next position.
     *  - {Boolean} clearAttributes
     *      True if the preselected attributes of the text model have to be
     *      extended by character attributes that remove the current URL style.
     */
    Hyperlink.getURLFromPosition = function (editor, selection) {
        var result = { url: null, beforeHyperlink: false, clearAttributes: false };

        if (!selection.hasRange()) {
            // find out a possible URL set for the current position
            var startPosition = selection.getStartPosition(),
                obj = Position.getDOMPosition(editor.getNode(), startPosition);

            if (obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {
                var attributes = StyleSheets.getExplicitAttributes(obj.node.parentNode, 'character');
                if ((attributes) && (attributes.url) && (attributes.url.length > 0)) {
                    // Now we have to check some edge cases to prevent to show
                    // the popup for a paragraph which contains only an empty span
                    // having set the url attribute.
                    var span = $(obj.node.parentNode);
                    if ((span.text().length > 0) ||
                        (span.prev().length) ||
                        (DOM.isTextSpan(span.next())))
                        result.url = attributes.url;
                    else {
                        result.clearAttributes = true;
                    }
                }
                else {
                    // Next special case: Before a hyperlink we always want to
                    // show the popup
                    var nextPosition = selection.getStartPosition();
                    nextPosition[nextPosition.length - 1] += 1;

                    obj = Position.getDOMPosition(editor.getNode(), nextPosition);
                    if (obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {
                        var nextAttributes = StyleSheets.getExplicitAttributes(obj.node.parentNode, 'character');
                        if ((nextAttributes) && (nextAttributes.url) && (nextAttributes.url.length > 0)) {
                            result.url = nextAttributes.url;
                            result.beforeHyperlink = true;
                        }
                    }
                }
            }
        }

        return result;
    };

    /**
     * Tries to find a selection range based on the current text cursor
     * position. The URL character style is used to find a consecutive
     * range.
     *
     * @param {Editor} editor
     *  The editor object.
     *
     * @param {Selection} selection
     *  The current selection.
     *
     * @returns {Object}
     *  The new selection properties which includes a range with the same URL
     *  character style or null if no URL character style could be found.
     */
    Hyperlink.findSelectionRange = function (editor, selection) {
        var newSelection = null;

        if (!selection.hasRange() && selection.getEnclosingParagraph()) {
            var paragraph = selection.getEnclosingParagraph(),
                startPosition = selection.getStartPosition(),
                pos = null, url = null, result = null;

            pos = startPosition[startPosition.length - 1];

            if (!selection.hasRange()) {
                result = Hyperlink.getURLFromPosition(editor, selection);
                url = result.url;
            }

            if (url) {
                if (result.beforeHyperlink)
                    pos += 1;
                startPosition[startPosition.length - 1] = pos;
                newSelection = Hyperlink.findURLSelection(editor, startPosition, url);
            }
            else {
                newSelection = Hyperlink.findTextSelection(paragraph, pos);
            }
        }

        return newSelection;
    };

    /**
     * Tries to find a selection based on the provided position which includes
     * the provided url as character style.
     *
     * @param {Editor} editor
     *  The editor instance.
     *
     * @param {Number[]} startPosition
     *  The startPosition in the paragraph
     *
     * @param url
     *  The hyperlink URL which is set as character style at pos
     *
     * @returns {Object}
     *  Contains start and end position of the selection where both could be
     *  null which means that there is no selection but the hyperlink should be
     *  inserted at the position.
     */
    Hyperlink.findURLSelection = function (editor, startPosition, url) {
        var startPos,
            endPos,
            startNode = null,
            endNode = null,
            attributes = null,
            obj = null,
            characterStyles = editor.getStyleSheets('character'),
            result = { start: null, end: null };

        obj = Position.getDOMPosition(editor.getNode(), startPosition);
        if (obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {

            startNode = obj.node.parentNode;
            endNode = obj.node.parentNode;

            while (endNode && endNode.nextSibling && DOM.isTextSpan(endNode.nextSibling)) {
                attributes = characterStyles.getElementAttributes(endNode.nextSibling).character;
                if (attributes.url !== url)
                    break;
                endNode = endNode.nextSibling;
            }

            while (startNode && startNode.previousSibling && DOM.isTextSpan(startNode.previousSibling)) {
                attributes = characterStyles.getElementAttributes(startNode.previousSibling).character;
                if (attributes.url !== url)
                    break;
                startNode = startNode.previousSibling;
            }

            startPos = Position.getPositionRangeForNode(editor.getNode(), startNode, true);
            if (startNode !== endNode) {
                endPos = Position.getPositionRangeForNode(editor.getNode(), endNode, true);
            }
            else {
                endPos = startPos;
            }

            result = { start: startPos.start[startPos.start.length - 1], end: endPos.end[endPos.end.length - 1] };
        }

        return result;
    };

    /**
     * Find a text selection based on the provided position which is limited
     * by separator characters.
     *
     * @param {HTMLElement|jQuery} paragraph
     *  The paragraph which contains the position provided as the second
     *  argument.
     *
     * @param {Number[]} pos
     *  The position relative inside the paragraph
     *
     * @param optional {Object}
     *  A object with optional arguments.
     *
     * @returns {Object}
     *  An object which contains the start and end position relative to the
     *  provided paragraph. Both can be null if there is no selection and the
     *  hyperlink should be inserted at pos.
     */
    Hyperlink.findTextSelection = function (paragraph, pos, optional) {
        var text = '',
            startFound = false,
            startPos = -1,
            endPos = -1,
            selection = { start: null, end: null, text: null };

        Position.iterateParagraphChildNodes(paragraph, function (node, nodeStart, nodeLength, nodeOffset, offsetLength) {

            if (DOM.isTextSpan(node)) {
                var str = $(node).text(), mustConcat = true;

                if (nodeStart <= pos) {
                    if (startPos === -1)
                        startPos = nodeStart;
                    text = text.concat(str.slice(nodeOffset, nodeOffset + offsetLength));
                    mustConcat = false;
                }
                if ((nodeStart + nodeLength) > pos) {
                    if (!startFound) {
                        var leftPos = startPos;

                        startFound = true;
                        startPos = Hyperlink.findLeftWordPosition(text, leftPos, pos);
                        if (startPos === -1)
                            return Utils.BREAK;
                        else if (leftPos < startPos)
                            text = text.slice(startPos - leftPos, text.length);
                    }
                    if (mustConcat)
                        text = text.concat(str.slice(nodeOffset, nodeOffset + offsetLength));
                    endPos = Hyperlink.findRightWordPosition(text, startPos, pos);
                    if (endPos < (nodeStart + nodeLength)) {
                        text = text.slice(0, endPos - startPos);
                        return Utils.BREAK;
                    }
                }
            }
            else {
                if (startFound)
                    return Utils.BREAK;
                else {
                    text = '';
                    startPos = -1;
                }
            }
        });

        if ((startPos >= 0) && (endPos >= startPos))
            selection = { start: startPos, end: endPos, text: text };

        return selection;
    };

    /**
     * Tries to find the left position of a word using a predefined separator
     * array.
     *
     * @param {String} text
     *  The text which should be parsed to find the left bound of a selection.
     *
     * @param {Number} offset
     *  An offset to be used to provide correct position data.
     *
     * @param {Number} pos
     *  The absolute position to start with (pos - offset) is the relative
     *  position in the provided text.
     *
     * @param optSeparators {optional, Array}
     *  An array filled with separator characters which are a
     *  border for the left work position search.
     *
     * @returns {Number}
     *  The absolute position of the left boundary or -1 if the current
     *  position is the boundary.
     */
    Hyperlink.findLeftWordPosition = function (text, offset, pos, optSeparators) {
        var i = pos - offset,
            separators = (optSeparators === undefined) ? Hyperlink.Separators : optSeparators;

        if (_.contains(separators, text[i]))
            return -1;

        while (i >= 0 && !_.contains(separators, text[i])) {
            i--;
        }
        return offset + i + 1;
    };

    /**
     * Tries to find the right position of a word using a
     * predefined separator array.
     *
     * @param {String} text
     *  The text which should be parsed to find the right bound of a selection.
     *
     * @param {Number} offset
     *  An offset to be used to provide correct position data.
     *
     * @param {Number} pos
     *  The absolute position to start with (pos - offset) is the relative
     *  position in the provided text.
     *
     * @returns {Number}
     *  The absolute position of the right boundary or -1 if the current
     *  position is the boundary.
     */
    Hyperlink.findRightWordPosition = function (text, offset, pos) {
        var i = pos - offset, length = text.length;

        if (_.contains(Hyperlink.Separators, text[i]))
            return -1;

        while (i < length && !_.contains(Hyperlink.Separators, text[i])) {
            i++;
        }
        return offset + i;
    };

    /**
     * Checks an URL based on some basic rules. This is not a fully compliant
     * URL check.
     *
     * @param {String} url
     *
     * @returns {Boolean}
     *  true if the URL seems to be correct otherwise false
     */
    Hyperlink.checkURL = function (url) {
        return (/^([a-z]([a-z]|\d|\+|-|\.)*):(\/\/(((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:)*@)?((\[(|(v[\da-f]{1,}\.(([a-z]|\d|-|\.|_|~)|[!\$&'\(\)\*\+,;=]|:)+))\])|((\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5])\.(\d|[1-9]\d|1\d\d|2[0-4]\d|25[0-5]))|(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=])*)(:\d*)?)(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*|(\/((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)+(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*)?)|((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)+(\/(([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)*)*)|((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)){0})(\?((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)|[\uE000-\uF8FF]|\/|\?)*)?(\#((([a-z]|\d|-|\.|_|~|[\u00A0-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF])|(%[\da-f]{2})|[!\$&'\(\)\*\+,;=]|:|@)|\/|\?)*)?$/i.test(url));
    };

    Hyperlink.checkMailAddress = function (mailAddress) {
        return (/^[A-Z0-9._%+-]+@[A-Z0-9.-]+\.[A-Z]{2,4}$/i.test(mailAddress));
    };

    /**
     * Shows a hyperlink input dialog.
     *
     * @param {BaseModel} model
     *  The calling model.
     *
     * @param {Object} [options]
     *  Additional options that control the appearance and behavior of the
     *  dialog. The following options are supported:
     *  @param {String} [options.title]
     *      If specified, the title of the dialog window that will be shown in
     *      a larger font.
     *  @param {String} [options.valueText='']
     *      The initial value of the text field.
     *  @param {String} [options.placeholderText='']
     *      The place-holder text that will be shown in the empty text field.
     *  @param {String} [options.valueURL='']
     *      The initial value of the URL field.
     *  @param {String} [options.placeholderURL='']
     *      The place-holder text that will be shown in the empty URL field.
     *  @param {String} [options.okLabel=gt('OK')]
     *      The label of the primary button that triggers the intended action
     *      by resolving the promise object returned by this method.
     *  @param {String} [options.cancelLabel=gt('Cancel')]
     *      The label of the Cancel button that rejects the promise object
     *      returned by this method.
     *
     * @returns {jQuery.Promise}
     *  The promise of a deferred object that will be resolved if the primary
     *  button or the remove button have been activated, or rejected if the
     *  dialog has been canceled. The done handlers registered at the promise
     *  object will receive a object containing the text and URL entered by the
     *  user. The object contains null for text and the URL if remove has been
     *  clicked.
     */
    Hyperlink.showHyperlinkDialog = function (model, options) {

        var // the text input fields
            inputurlid = _.uniqueId('url'),
            inputtextid = _.uniqueId('text'),
            inputUrl = $('<input>', {
                    value: Utils.getStringOption(options, 'valueURL', '')
                }).addClass('span12 nice-input').attr({'data-property': 'url', tabindex: 0, id: inputurlid, defaultEnter: true, 'aria-labelledby': 'label-url-02'}),

            inputText = $('<input>', {
                    value: Utils.getStringOption(options, 'valueText', ''),
                    placeholder: Utils.getStringOption(options, 'placeholderURL', '')
                }).addClass('span12 nice-input').attr({'data-property': 'text', tabindex: 0, id: inputtextid, defaultEnter: true, 'aria-labelledby': 'label-text-01'}),
            // the dialog object
            dialog = Dialogs.createDialog(Utils.extendOptions(options, { enter: Dialogs.defaultKeyEnterHandler }))
            .append(
                $('<div>').addClass('row-fluid').css({'margin-top': '10px'}).append(
                    $('<div>').addClass('control-group').css({'margin-bottom': '0px'}).append(
                        $('<label>').addClass('control-label').attr({'for': inputtextid, 'id': 'label-text-01'}).text(gt('Text:')),
                        $('<div>').addClass('controls').css({'margin-right': '10px'}).append(inputText))
                    )
                )
            .append(
                $('<div>').addClass('row-fluid').css({'margin-top': '10px'}).append(
                    $('<div>').addClass('control-group').css({'margin-bottom': '0px'}).append(
                        $('<label>').addClass('control-label').attr({'for': inputurlid, 'id': 'label-url-02'}).text(gt('URL:')),
                        $('<div>').addClass('controls').css({'margin-right': '10px'}).append(inputUrl))
                    )
                ),

            // the result deferred
            def = $.Deferred(),
            insertButton,
            removeButton = null;

        // special handling for iPad to make nice-input and nice-input:focus work
        if (_.browser.iOS && _.browser.WebKit) {
            inputUrl.css('-webkit-appearance', 'none');
            inputText.css('-webkit-appearance', 'none');
        }

        // add OK and Cancel buttons & remove button to remove hyperlink
        dialog.addPrimaryButton('ok', Utils.getStringOption(options, 'okLabel', gt('OK')), null, { tabIndex: 0 });
        dialog.addButton('cancel', Utils.getStringOption(options, 'cancelLabel', gt('Cancel')), null, { tabIndex: 0 });
        // only add remove button if we already have a set URL
        if (Utils.getStringOption(options, 'valueURL', '').length > 0) {
            dialog.addDangerButton('remove', Utils.getStringOption(options, 'removeLabel', gt('Remove')), null, { tabIndex: 0 });
            removeButton = dialog.getFooter().children('.btn-danger');
        }
        insertButton = dialog.getFooter().children('.btn-primary');

        function checkTextFields() {
            var textUrl = inputUrl.val(),
                text = inputText.val();

            if (!model.getEditMode()) {
                // disable ok and delete button if we don't have edit rights
                insertButton.attr({'disabled': 'disabled', 'aria-disabled': true, 'aria-label': gt('Insert')});
                if (removeButton) {
                    removeButton.attr({'disabled': 'disabled', 'aria-disabled': false, 'aria-label': gt('Insert')});
                }
            } else if ((text && text.length > 0) && (textUrl && textUrl.length > 0)) {
                insertButton.removeAttr('disabled').attr('aria-disabled', false);
            } else {
                insertButton.attr({'disabled': 'disabled', 'aria-disabled': true, 'aria-label': gt('Insert')});
            }
        }

        function editModeChangeHandler(event, editMode) { checkTextFields(); }

        // register a change handlers to toggle the insert button
        inputUrl.bind('input', checkTextFields);
        inputText.bind('input', checkTextFields);

        // register handler to initially focus the url or text input
        dialog.getPopup().on('shown', function () {

            if (Utils.getStringOption(options, 'valueText', '').length > 0)
                dialog.getBody().find('[data-property="url"]').focus();
            else
                dialog.getBody().find('[data-property="text"]').focus();
            checkTextFields();
        });

        // disable the ok button if the dialog is already opened but we lost the edit rights.
        model.on('change:editmode', editModeChangeHandler);

        // show the dialog and register listeners for the results
        dialog.show()
        .done(function (action, data, node) {
            if (action === 'ok') {
                var text = $(node).find('[data-property="text"]').val(),
                    url = $.trim($(node).find('[data-property="url"]').val()),
                    protIndex = -1;

                // add http: as default if protocol is missing
                protIndex = url.indexOf(':');
                if (protIndex === -1) {
                    url = 'http://' + url;
                } else {
                    if (!Hyperlink.isLegalUrl(url)) {
                        Utils.warn('you tried to add illegal code instead of an url: ' + url);
                        url = null;
                    }
                }

                def.resolve({ text: text, url: url });
            } else if (action === 'remove') {
                def.resolve({ text: null, url: null });
            } else {
                def.reject();
            }
            model.off('change:editmode', editModeChangeHandler);
        });

        return def.promise();
    };
    
    Hyperlink.isLegalUrl = function (url) {
        var lower = url.toLowerCase();
        for (var i = 0; i < Hyperlink.Protocols.length; i++) {
            if (lower.indexOf(Hyperlink.Protocols[i]) === 0) {
                return true;
            }
        }
        return null;
    };

    /**
     * Find a text selection based on the provided position to the left
     * which is limited by the provided separator characters.
     *
     * @param paragraph {HTMLElement|jQuery}
     *  The paragraph which contains the position provided as the second
     *  argument.
     *
     * @param pos {Number}
     *  The position relative inside the paragraph
     *
     * @param separators {Array}
     *  An array with separator character to look for the left border.
     *
     * @returns {Object}
     *  An object which contains the Object.start and Object.end position
     *  relative to the provided paragraph. Both can be null if there is
     *  no selection and the hyperlink should inserted at pos. Object.text
     *  contains the text within the selection.
     */
    Hyperlink.findLeftText = function (paragraph, pos, separators) {
        var text = '',
            startFound = false,
            startPos = -1,
            endPos = -1,
            selection = { start: null, end: null, text: null };

        Position.iterateParagraphChildNodes(paragraph, function (node, nodeStart, nodeLength, nodeOffset, offsetLength) {

            if (DOM.isTextSpan(node)) {
                var str = $(node).text();

                if (nodeStart <= pos) {
                    if (startPos === -1) {
                        startPos = nodeStart;
                    }
                    text = text.concat(str.slice(nodeOffset, nodeOffset + offsetLength));
                }
                if ((nodeStart + nodeLength) >= pos) {
                    if (!startFound) {
                        var leftPos = startPos;

                        // we just need all text left inclusive pos
                        text = text.slice(0, pos);

                        startFound = true;
                        startPos = Hyperlink.findLeftWordPosition(text, leftPos, pos, separators);
                        endPos = Math.max(startPos, pos - 1);

                        // make the final slice using startPos and endPos
                        text = text.slice(startPos, endPos + 1);
                        return Utils.Break;
                    }
                }
            } else {
                if (startFound)
                    return Utils.BREAK;
                else {
                    if (DOM.isDrawingFrame(node) || DOM.isTabNode(node) || DOM.isFieldNode(node)) {
                        if (startPos === -1) {
                            startPos = nodeStart;
                        }
                        text = text + ' ';  // <- using an empty space for a drawings, fields or tabs (Fix for 28095)
                    } else {
                        text = '';
                        startPos = -1;
                    }
                }
            }
        });

        if ((startPos >= 0) && (endPos >= startPos))
            selection = { start: startPos, end: endPos, text: text };

        return selection;
    };

    /**
     * Checks for a text in a paragraph which defines a hyperlink
     * e.g. http://www.open-xchange.com
     *
     * @param paragraph {HTMLElement/jQuery}
     *  The paragraph node to check
     *
     * @param position {Position}
     *  The rightmost position to check the text to the left for
     *  a hyperlink text.
     *
     * @returns {Object}
     *  Returns an object containing the start and end position
     *  to set the hyperlink style/url or null if no hyperlink
     *  has been found.
     */
    Hyperlink.checkForHyperlinkText = function (paragraph, position) {
        var result = null;

        if (position !== null) {
            var pos = position[position.length - 1],
                hyperlinkSelection = Hyperlink.findLeftText(paragraph, pos, [' ', '\u00a0']);

            if ((hyperlinkSelection.start !== null) && (hyperlinkSelection.end !== null) &&
                (hyperlinkSelection.text !== null)) {
                var found = false, foundProtocol = null;

                _.each(Hyperlink.Protocols, function (protocol) {
                    if (hyperlinkSelection.text.indexOf(protocol) === 0) {
                        found = true;
                        foundProtocol = protocol;
                    }
                });

                if (found) {

                    if (foundProtocol !== 'mailto:') {
                        var index = hyperlinkSelection.text.indexOf('//');

                        // At least one character after the protocol must be there
                        if (((index + 2) < hyperlinkSelection.text.length) &&
                             (Hyperlink.checkURL(hyperlinkSelection.text))) {
                            // create result with correct Position objects
                            result = createResultFromHyperlinkSelection(position, hyperlinkSelection);
                        }
                    }
                    else if (foundProtocol === 'mailto:') {
                        var atIndex = hyperlinkSelection.text.indexOf('@'),
                            dotIndex = hyperlinkSelection.text.lastIndexOf('.');

                        // mailto: needs a '@' and '.' char and at least one char after the dot
                        // we also use the regexp to check for other problems
                        if ((atIndex > ('mailto:'.length + 1)) &&
                            (dotIndex > atIndex) && (dotIndex < (hyperlinkSelection.text.length - 1)) &&
                            Hyperlink.checkURL(hyperlinkSelection.text)) {

                            // create result with correct Position objects
                            result = createResultFromHyperlinkSelection(position, hyperlinkSelection);
                        }
                    }
                }
                else {
                    //some magic auto detection for text without protocol
                    var textTemplate = null, found = false;

                    _.each(Hyperlink.TextTemplates, function (template) {
                        if (hyperlinkSelection.text.indexOf(template) === 0) {
                            found = true;
                            textTemplate = template;
                        }
                    });

                    if (found) {
                        if (textTemplate === 'www.') {
                            // http://
                            if (hyperlinkSelection.text.length > 'www.'.length) {
                                result = createResultFromHyperlinkSelection(position, hyperlinkSelection);
                                result.url = 'http://' + hyperlinkSelection.text;
                            }
                        }
                        else if (textTemplate === 'ftp.') {
                            // ftp://
                            if (hyperlinkSelection.text.length > 'ftp.'.length) {
                                result = createResultFromHyperlinkSelection(position, hyperlinkSelection);
                                result.url = 'ftp://' + hyperlinkSelection.text;
                            }
                        }
                    }
                    else {
                        // some more magic for a mail address
                        if (Hyperlink.checkMailAddress(hyperlinkSelection.text)) {
                            result = createResultFromHyperlinkSelection(position, hyperlinkSelection);
                            result.url = 'mailto:' + hyperlinkSelection.text;
                        }
                    }
                }
            }
        }

        return result;
    };

    /**
     * 'Inserts' a hyperlink at the provided selection using the
     * provided url.
     *
     * @param editor {Editor}
     *  The editor instance to use.
     *
     * @param start {Position}
     *  The start position of the selection
     *
     * @param end {Position}
     *  The end position of the selection.
     *
     * @param url {String}
     *  The url of the hyperlink to set at the selection
     */
    Hyperlink.insertHyperlink = function (editor, start, end, url) {
        var hyperlinkStyleId = editor.getDefaultUIHyperlinkStylesheet(),
            characterStyles = editor.getStyleSheets('character'),
            generator = editor.createOperationsGenerator();

        if (characterStyles.isDirty(hyperlinkStyleId)) {
            // insert hyperlink style to document
            generator.generateOperation(Operations.INSERT_STYLESHEET, {
                attrs: characterStyles.getStyleSheetAttributeMap(hyperlinkStyleId),
                type: 'character',
                styleId: hyperlinkStyleId,
                styleName: characterStyles.getName(hyperlinkStyleId),
                parent: characterStyles.getParentId(hyperlinkStyleId),
                uiPriority: characterStyles.getUIPriority(hyperlinkStyleId)
            });
            characterStyles.setDirty(hyperlinkStyleId, false);
        }

        generator.generateOperation(Operations.SET_ATTRIBUTES, {
            attrs: { styleId: hyperlinkStyleId, character: { url: url } },
            start: _.clone(start),
            end: _.clone(end)
        });

        editor.applyOperations(generator.getOperations());
    };

    /**
     * 'Removes' the hyperlink at the provided selection.
     *
     * @param editor {Editor}
     *  The editor instance to use.
     *
     * @param start {Position}
     *  The start position of the selection
     *
     * @param end {Position}
     *  The end position of the selection.
     */
    Hyperlink.removeHyperlink = function (editor, start, end) {

        var generator = editor.createOperationsGenerator();

        generator.generateOperation(Operations.SET_ATTRIBUTES, {
            attrs: Hyperlink.CLEAR_ATTRIBUTES,
            start: _.clone(start),
            end: _.clone(end)
        });

        editor.applyOperations(generator.getOperations());
    };

    /**
     * Shortens a hyperlink to a maximal number of
     * characters.
     *
     * @param text {String} Hyperlink to shorten
     */
    Hyperlink.limitHyperlinkText = function (text) {

        var maxChars = 255,
            result = text;

        if (_.isString(text) && text.length > maxChars) {
            var length = text.length,
                start = text.slice(0, Math.round(maxChars / 2)),
                end = text.slice(length - Math.round(maxChars / 2));

            result = "".concat(start, "...", end);
        }

        return result;
    };

    /**
     * Checks if element is a clickable part of the hyperlink popup
     * return true if yes otherwise false.
     *
     * @param element {HTMLElement/jQuery}
     *
     * @param {Boolean}
     */
    Hyperlink.isClickableNode = function (element, readOnlyMode) {

        var node = $(element);

        if (Hyperlink.isPopupNode(node.parent())) {
            return node.is('a');
        }

        return false;
    };

    /**
     * Checks if element is the topmost div of a popup
     * returns true if yes otherwise false.
     *
     * @param element {HTMLElement/jQuery}
     *
     * @returns {Boolean}
     */
    Hyperlink.isPopupNode = function (element) {
        return $(element).is('.inline-popup');
    };

    /**
     * Check if element is the hyperlink popup or a child
     * within the popup.
     *
     * @param element {HTMLElement/jQuery}
     *
     * @returns {Boolean}
     */
    Hyperlink.isPopupOrChildNode = function (element) {
        var node = $(element);

        return (Hyperlink.isClickableNode(node, false) ||
                 Hyperlink.isPopupNode(node));
    };

    /**
     * Creates a hyperlink popup and appends it to the page DOM if it doesn't
     * exist, otherwise nothing is done.
     *
     * @param {Editor} editor
     *  The editor object.
     *
     * @param {TextApplication} app
     *  The application object.
     *
     * @param {Selection} selection
     *  The selection object of the editor.
     */
    Hyperlink.createPopup = function (editor, app, selection) {
        var page = editor.getNode();

        if (page.children('.inline-popup.hyperlink').length === 0) {
            var hyperlinkPopup = $('<div>', { contenteditable: false }).addClass('inline-popup hyperlink f6-target').css('display', 'none')
                .append(
                    $('<a>').addClass('textwrap').attr({ href: '', rel: 'noreferrer', target: '_blank', tabindex: 0 }),
                    $('<br/>'),
                    $('<a>').addClass('popupbutton').attr({ href: '#', tabindex: 0 }).text(gt('Edit')).click(function () { app.getController().change('character/hyperlink/dialog'); return false; }),
                    $('<span>').text(' | '),
                    $('<a>').addClass('popupbutton').attr({ href: '#', tabindex: 0 }).text(gt('Remove')).click(function () { app.getController().change('character/hyperlink/remove'); return false; }));

            // Add keydown handler to handle TAB/SHIFT+TAB within popup
            // as one of our parents (editdiv) installs a handler by it's
            // own. We want to have keyboard handling for our functions.
            hyperlinkPopup.on('keydown', { nextFocus: page }, Hyperlink.popupProcessKeydown);

            if (Modernizr.touch) {
                // some CSS changes for touch devices
                hyperlinkPopup.addClass('touch');
                hyperlinkPopup.find('a.popupbutton').addClass('touch');
            }

            page.append(hyperlinkPopup);

            var updateEditMode = function () {
                var readOnly = editor.getEditMode() !== true;
                hyperlinkPopup.find('br').toggle(!readOnly);
                hyperlinkPopup.find('span').toggle(!readOnly);
                hyperlinkPopup.find('a.popupbutton').toggle(!readOnly);
                Hyperlink.updatePopup(editor, app, selection, hyperlinkPopup);
            };

            editor.on('change:editmode', updateEditMode);

            selection.on('change', function () {
                Hyperlink.updatePopup(editor, app, selection, hyperlinkPopup);
            });

            // must be called to set the initial state correctly
            updateEditMode();
        }
    };

    /**
     * Process keyboard events used to travel in a container and sets the
     * focus to the next/previous element.
     *
     * The function expects that the element which gets the focus on ESCAPE to
     * be in the [data.nextFocus] part of the event. Therefore this function
     * must be set using .on('keydown', {nextFocus: focus}, popupProcessKeydown)
     *
     * @param {Object} event
     *  A keydown event sent by the browser.
     *
     * @returns {Boolean}
     *  Returns false if the browser shouldn't process the keyboard
     *  event.
     */
    Hyperlink.popupProcessKeydown = function (event) {
        var items, focus, index, down;

        if (event.keyCode === KeyCodes.TAB || DOM.isCursorKey(event.keyCode)) {
            event.preventDefault();

            items = $(this).find('[tabindex]');
            focus = $(document.activeElement);
            down = (((event.keyCode === KeyCodes.TAB) && !event.shiftKey) ||
                    (event.keyCode === KeyCodes.DOWN_ARROW) ||
                    (event.keyCode === KeyCodes.RIGHT_ARROW));

            index = (items.index(focus) || 0);
            if (!down) {
                index--;
            } else {
                index++;
            }

            if (index >= items.length) {
                index = 0;
            } else if (index < 0) {
                index = items.length - 1;
            }
            items[index].focus();
            event.preventDefault();
            return false;
        } else if (event.keyCode === KeyCodes.ESCAPE) {
            $(this).hide();
            if (event.data && event.data.nextFocus && event.data.nextFocus.length) {
                $(event.data.nextFocus).focus();
                event.preventDefault();
                return false;
            }
        }
    };

    /**
     * Internal method to update the hyperlink popup
     *
     * @param {Editor} editor
     *  The editor object.
     *
     * @param {TextApplication} app
     *  The application object.
     *
     * @param {Selection} selection
     *  The current selection.
     *
     * @param {jQuery} hyperlinkPopup
     *  The hyperlink popup node as jQuery.
     */
    Hyperlink.updatePopup = function (editor, app, selection, hyperlinkPopup) {
        var result = Hyperlink.getURLFromPosition(editor, selection),
            page = editor.getNode();

        if (result.url) {
            var link = $('a.textwrap', hyperlinkPopup[0]),
                urlSelection = selection.getStartPosition(),
                obj = null;

            if (result.beforeHyperlink) {
                urlSelection[urlSelection.length - 1] += 1;
            }
            obj = Position.getDOMPosition(page, urlSelection);
            if (obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {
                var pos = urlSelection[urlSelection.length - 1],
                    startEndPos = Hyperlink.findURLSelection(editor, urlSelection, result.url),
                    left, top, height, outerWidth, width, diffWidth;

                if (pos !== startEndPos.end || result.beforeHyperlink) {
                    // find out position of the first span of our selection
                    urlSelection[urlSelection.length - 1] = startEndPos.start;
                    obj = Position.getDOMPosition(page, urlSelection, true);
                    left = $(obj.node).offset().left;

                    // find out position of the last span of our selection
                    urlSelection[urlSelection.length - 1] = startEndPos.end - 1;
                    obj = Position.getDOMPosition(page, urlSelection, true);
                    top = $(obj.node).offset().top;
                    height = $(obj.node).height();

                    // calculate position relative to the application pane
                    var parent = page,
                        parentLeft = parent.offset().left,
                        parentTop = parent.offset().top,
                        parentWidth = parent.innerWidth();

                    left = left - parentLeft;
                    top = top + height - parentTop;

                    link.text(Hyperlink.limitHyperlinkText(result.url));
                    if (!result.url || result.url.charAt(0) === '#') {
                        link.attr({
                            title: ''
                        });
                        link.css({
                            color: '',
                            cursor: 'default',
                            textDecoration: 'none'
                        });
                        link.removeAttr('href');
                    } else if (Hyperlink.isLegalUrl(result.url)) {
                        link.css({
                            color: '',
                            cursor: '',
                            textDecoration: ''
                        });
                        link.attr({href: result.url, title: result.url});
                    } else {
                        link.attr({
                            title: 'this is no legal url'
                        });
                        link.css({
                            color: '#0088cc',
                            cursor: 'default',
                            textDecoration: 'none'
                        });
                        link.removeAttr('href');
                    }
                    hyperlinkPopup.css({left: left, top: top, width: '', height: ''});
                    outerWidth = hyperlinkPopup.outerWidth();
                    width = hyperlinkPopup.width();
                    diffWidth = outerWidth - width;
                    if ((left + outerWidth) > parentWidth) {
                        left -= ((left + outerWidth) - parentWidth);
                        left = Math.max(Utils.convertCssLength(page.css('padding-left'), 'px'), left);
                        if (!_.browser.IE) {
                            outerWidth = Math.min(outerWidth, parentWidth);
                            width = outerWidth - diffWidth;
                        }
                        hyperlinkPopup.css({left: left, width: width});
                    }
                    if (pos === startEndPos.start || result.beforeHyperlink) {
                        // special case: at the start of a hyperlink we want to
                        // write with normal style
                        editor.addPreselectedAttributes(Hyperlink.CLEAR_ATTRIBUTES);
                    }
                    if (hyperlinkPopup.css('display') === 'none') {
                        showPopup(app, hyperlinkPopup);
                    }
                }
                else {
                    // special case: at the end of a hyperlink we want to
                    // write with normal style and we don't show the popup
                    editor.addPreselectedAttributes(Hyperlink.CLEAR_ATTRIBUTES);
                    hidePopup(hyperlinkPopup);
                }
            }
        }
        else {
            hidePopup(hyperlinkPopup);
            if (result.clearAttributes) {
                editor.addPreselectedAttributes(Hyperlink.CLEAR_ATTRIBUTES);
            }
        }
    };

    /**
     * Provides the text from a selection.
     *
     * @param {Selection} selection
     *  A selection object which is used to extract the text from.
     */
    Hyperlink.getSelectedText = function (selection) {
        var text = '';

        selection.iterateNodes(function (node, pos, start, length) {
            if ((start >= 0) && (length >= 0) && DOM.isTextSpan(node)) {
                var nodeText = $(node).text();
                if (nodeText) {
                    text = text.concat(nodeText.slice(start, start + length));
                }
            }
        });

        return text;
    };

    /**
     * Closes an open hyperlink pop-up menu.
     *
     * @param {Editor} editor
     *  An editor instance.
     *
     * @returns {Boolean}
     *  Whether the pop-up menu was actually open.
     */
    Hyperlink.closePopup = function (editor) {

        var menuNode = editor.getNode().children('.inline-popup.hyperlink'),
            visible = menuNode.css('display') !== 'none';

        hidePopup(menuNode);
        return visible;
    };

    // exports ================================================================

    return Hyperlink;

});
