/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/controls',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/control/radiogroup',
     'io.ox/office/tk/control/radiolist',
     'gettext!io.ox/office/spreadsheet'
    ], function (Utils, RadioGroup, RadioList, gt) {

    'use strict';

    var // number of frames per slide animation in the sheet radio group
        FRAME_COUNT = 5,

        // delay for a slide animation frames in the sheet radio group, in milliseconds
        FRAME_DELAY = 25;

    // static class SpreadsheetControls =======================================

    var SpreadsheetControls = {};

    // class ActiveSheetGroup =================================================

    /**
     * The selector for the current active sheet, as radio group (all sheet
     * tabs are visible at the same time).
     *
     * @constructor
     *
     * @extends RadioGroup
     */
    SpreadsheetControls.ActiveSheetGroup = RadioGroup.extend({ constructor: function (app) {

        var // self reference
            self = this,

            // the left and right shadow nodes for the fade-out effectes
            leftShadowNode = null,
            rightShadowNode = null,

            // index of the first button currently visible
            firstSheet = 0,

            // timer of the current slide animation
            slideTimer = null;

        // base constructor ---------------------------------------------------

        RadioGroup.call(this, { classes: 'active-sheet-group' });

        // private methods ----------------------------------------------------

        /**
         * Returns the horizontal position of the specified button relative to
         * the left border of the scroll area (the scroll position needed to
         * move the specified button to the left border of the own root node).
         *
         * @param {Number} sheet
         *  The zero-based sheet index. Invalid sheet indexes will be
         *  restricted automatically to the range of available sheets.
         *
         * @param {Boolean} [right=false]
         *  If set to true, the position of the right button border will be
         *  returned instead of the offset of the left border.
         */
        function getButtonOffset(sheet, right) {
            var rootNode = self.getNode(),
                optionButtons = self.getOptionButtons(),
                buttonNode = optionButtons.eq(Utils.minMax(sheet, 0, optionButtons.length - 1)),
                buttonLeftMargin = Utils.convertCssLength(buttonNode.css('margin-left'), 'px', 1),
                rootNodePadding = Utils.convertCssLength(rootNode.css('padding-left'), 'px', 1),
                offset = (buttonNode.length > 0) ? (rootNode[0].scrollLeft + buttonNode.position().left + buttonLeftMargin - rootNodePadding) : 0;
            return (right === true) ? (offset + buttonNode.outerWidth()) : offset;
        }

        /**
         * Returns the maximum scroll position possible for the current
         * contents of this radio group.
         */
        function getMaxScrollPos() {
            var rootNode = self.getNode();
            return rootNode[0].scrollWidth - rootNode[0].clientWidth;
        }

        /**
         * Returns the index of the left-most button that will be completely
         * visible after trying to scroll the specified sheet button to the
         * left border. If this radio group would be scrolled to its right
         * border, the index of the first sheet may be less then the passed
         * sheet index.
         */
        function getFirstSheet(sheet) {

            var // all existing buttons
                optionButtons = self.getOptionButtons(),
                // maximum possible scroll position
                maxScrollPos = getMaxScrollPos(),
                // index of the last sheet button that can be scrolled to the left border
                lastSheet = Math.max(0, sheet);

            // sheet button is located outside the maximum scroll position: find the last
            // button that can be scrolled to the left border, get its sheet index
            if (getButtonOffset(sheet) > maxScrollPos) {
                lastSheet = optionButtons.length;
                _(optionButtons.get()).any(function (button, sheet) {
                    if (getButtonOffset(sheet) >= maxScrollPos) {
                        lastSheet = sheet;
                        return true; // break the loop
                    }
                });
            }

            // restrict passed sheet index to the available range of sheet buttons
            return Utils.minMax(sheet, 0, lastSheet);
        }

        /**
         * Scrolls the sheet tab buttons to the specified pixel position.
         */
        function scrollToPos(offset) {

            var rootNode = self.getNode(),
                maxScrollPos = getMaxScrollPos(),
                scrollPos = 0;

            rootNode.scrollLeft(offset);
            scrollPos = rootNode[0].scrollLeft;
            leftShadowNode.toggle(scrollPos > 0).css({ left: scrollPos });
            rightShadowNode.toggle(scrollPos < maxScrollPos).css({ left: scrollPos + rootNode.outerWidth() - 1 });
        }

        /**
         * Scrolls the specified sheet tab button to the left border of the
         * root node of this radio group. If the passed index is less than
         * zero, scrolls to the first button. If the index is greater or equal
         * to the number of existing buttons, scrolls to the right as far as
         * possible.
         */
        function scrollToLeft(sheet) {
            scrollToPos(getButtonOffset(sheet));
            firstSheet = getFirstSheet(sheet);
        }

        /**
         * Scrolls the specified sheet tab button to the left border of the
         * root node of this radio group, with a short animation.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  scrolling animation is finished; or rejected, id the specified
         *  button does not exist or cannot be moved to the left node border.
         */
        function slideToLeft(sheet) {

            var oldScrollPos = self.getNode()[0].scrollLeft,
                newScrollPos = Math.min(getButtonOffset(sheet), getMaxScrollPos());

            // initialize the first visible index
            firstSheet = getFirstSheet(sheet);

            // abort current animation
            if (slideTimer) {
                slideTimer.abort();
                slideTimer = null;
            }

            // return rejected Deferred object, if scrolling is not possible
            if (oldScrollPos === newScrollPos) {
                return $.Deferred().reject();
            }

            // start the new animation
            slideTimer = app.repeatDelayed(function (frame) {
                var factor = Math.pow((frame + 1) / FRAME_COUNT, 2);
                scrollToPos(oldScrollPos + factor * (newScrollPos - oldScrollPos));
            }, { delay: FRAME_DELAY, cycles: FRAME_COUNT });

            // clear reference to timer when animation is finished
            return slideTimer.always(function () { slideTimer = null; });
        }

        /**
         * Scrolls the specified sheet tab button to the right border of the
         * root node of this radio group, with a short animation.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  scrolling animation is finished; or rejected, id the specified
         *  button does not exist or cannot be moved to the right node border.
         */
        function slideToRight(sheet) {

            var // all existing buttons
                optionButtons = self.getOptionButtons(),
                // the minimum scroll position required to make the sheet button visible
                minScrollPos = getButtonOffset(sheet, true) - self.getNode().width(),
                // index of the sheet button that will be scrolled to the left border
                firstSheet = 0;

            // find the corresponding button to be scrolled to the left border
            _(optionButtons.get()).any(function (button, sheet) {
                if (getButtonOffset(sheet) >= minScrollPos) {
                    firstSheet = sheet;
                    return true;
                }
            });

            // start the slide animation
            return slideToLeft(Math.min(sheet, firstSheet));
        }

        /**
         * Called every time the value of the control (the index of the active
         * sheet) has been changed.
         */
        function updateHandler(sheet, options, oldSheet) {
            var rootNode = self.getNode();
            if (sheet !== oldSheet) {
                if (getButtonOffset(sheet) < rootNode[0].scrollLeft) {
                    slideToLeft(sheet);
                } else if (getButtonOffset(sheet, true) > rootNode[0].scrollLeft + rootNode.width()) {
                    slideToRight(sheet);
                }
            }
        }

        /**
         * Recreates all sheet tabs in this radio group.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {
            self.clearOptionButtons();
            _(app.getModel().getSheetNames()).each(function (sheetName, sheet) {
                self.createOptionButton(sheet, { label: _.noI18n(sheetName) });
            });
            self.trigger('refresh:layout');
        });

        // methods ------------------------------------------------------------

        /**
         * Resets this control to automatic width, making all option buttons
         * visible.
         *
         * @returns {ActiveSheetGroup}
         *  A reference to this instance.
         */
        this.reset = function (scrollable) {
            this.getNode().css({ width: '' }).toggleClass('scrolling', scrollable);
            this.getOptionButtons().css({ marginRight: '' });
            return this;
        };

        /**
         * Sets this control to fixed width mode.
         *
         * @param {Number} width
         *  The new outer width of this control.
         *
         * @returns {ActiveSheetGroup}
         *  A reference to this instance.
         */
        this.setWidth = function (width) {
            this.getNode().css({ width: width });
            this.getOptionButtons().last().css({ marginRight: 8 });
            scrollToLeft(firstSheet);
            return this;
        };

        /**
         * Scrolls the sheet tab located before the current first visible sheet
         * tab to the left border of the root node of this radio group.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  scrolling animation is finished; or rejected, if the control cannot
         *  be scrolled to the left anymore.
         */
        this.scrollToPrev = function () {
            return slideToLeft(firstSheet - 1);
        };

        /**
         * Scrolls the sheet tab located after the current first visible sheet
         * tab to the left border of the root node of this radio group.
         *
         * @returns {jQuery.Promise}
         *  The Promise of a Deferred object that will be resolved when the
         *  scrolling animation is finished; or rejected, if the control cannot
         *  be scrolled to the right anymore.
         */
        this.scrollToNext = function () {
            return slideToLeft(firstSheet + 1);
        };

        // initialization -----------------------------------------------------

        // insert special elements used to create a fade-out effect at left and right border
        this.getNode().append(
            leftShadowNode = $('<div>').addClass('scroll-shadow'),
            rightShadowNode = $('<div>').addClass('scroll-shadow')
        );

        // the update handler scrolls to the active sheet, after it has been changed
        this.registerUpdateHandler(updateHandler);

        // rebuild the list of available sheets after it has been changed
        app.on('docs:import:success', function () {
            fillList();
            app.getModel().on('insert:sheet delete:sheet move:sheet rename:sheet', fillList);
        });

    }}); // class ActiveSheetGroup

    // class ActiveSheetList ==================================================

    /**
     * The selector for the current active sheet, as radio drop-down list.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.ActiveSheetList = RadioList.extend({ constructor: function (app, options) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        RadioList.call(this, {
            icon: 'icon-reorder',
            tooltip: gt('Select sheet'),
            caret: 'none',
            updateCaptionMode: Utils.getBooleanOption(options, 'showNames', false) ? 'label' : 'none'
        });

        // private methods ----------------------------------------------------

        /**
         * Recreates all sheet entries in the drop-down menu.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {
            self.clearOptionButtons();
            _(app.getModel().getSheetNames()).each(function (sheetName, sheet) {
                self.createOptionButton(sheet, { label: _.noI18n(sheetName) });
            });
        });

        // methods ------------------------------------------------------------

        /**
         * Sets this control to fixed width mode.
         *
         * @param {Number} width
         *  The new outer width of this control.
         *
         * @returns {ActiveSheetList}
         *  A reference to this instance.
         */
        this.setWidth = function (width) {
            this.getNode().css({ width: width });
            this.getMenuButton().css({ position: 'absolute', left: 0, right: 0, top: 0, bottom: 0 });
            return this;
        };

        // initialization -----------------------------------------------------

        app.on('docs:import:success', function () {
            fillList();
            app.getModel().on('insert:sheet delete:sheet move:sheet rename:sheet', fillList);
        });

    }}); // class ActiveSheetList

    // class MergeChooser =================================================

    /**
     * The selector for merge or unmerge of cells, as radio drop-down list.
     *
     * @constructor
     *
     * @extends RadioList
     */
    SpreadsheetControls.MergeChooser = RadioList.extend({ constructor: function (app, mergeOptions, options) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            highlight: function (type) {
                // type can be null or string (from setter) or boolean (from getter)
                if (type === null) { type = false; }
                else if (_.isString(type)) { type = (type !== 'unmerge'); }
                return type;
            },
            splitValue: 'toggleMerge',  // simulating toggle behavior for merge button
            updateCaptionMode: 'none'
        }, options));

        // initialization -----------------------------------------------------

        _(mergeOptions).each(function (option) {
            this.createOptionButton(option.name, { label: option.label });
        }, this);

    }}); // class MergeChooser

    // exports ================================================================

    return SpreadsheetControls;

});
