/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/operations',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/operationsgenerator'
    ], function (Utils, OperationsGenerator) {

    'use strict';

    // class SpreadsheetOperationsGenerator ===================================

    /**
     * An instance of this class contains an operations array and provides
     * methods to generate operations for various element nodes.
     *
     * @constructor
     *
     * @extends OperationsGenerator
     *
     * @param {SpreadsheetDocumentStyles} documentStyles
     *  Global collection with the style sheet containers and custom formatting
     *  containers of a spreadsheet document.
     */
    function SpreadsheetOperationsGenerator(documentStyles) {

        // base constructor ---------------------------------------------------

        OperationsGenerator.call(this, documentStyles);

        // methods ------------------------------------------------------------

        /**
         * Creates and appends a new operation with the properties 'sheet' and
         * 'start' to the operations array.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Number} sheet
         *  The zero-based sheet index, will be inserted as 'sheet' property.
         *
         * @param {Number|Number[]} start
         *  The logical start position of the operation.
         *
         * @param {Object} [operationOptions]
         *  Additional options that will be stored in the operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateSheetOperation = function (name, sheet, start, operationOptions) {
            operationOptions = _({ sheet: sheet, start: start }).extend(operationOptions);
            return this.generateOperation(name, operationOptions);
        };

        /**
         * Creates and appends a new operation with 'start' and (optional)
         * 'end' properties to the operations array addressing a cell range. If
         * the range addresses a single cell, the 'end' property will be
         * omitted in the operation.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Number} sheet
         *  The zero-based sheet index, will be inserted as 'sheet' property.
         *
         * @param {Object} range
         *  The logical address of the cell range addressed by the operation.
         *
         * @param {Object} [operationOptions]
         *  Additional options that will be stored in the operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateRangeOperation = function (name, sheet, range, operationOptions) {
            if (!_.isEqual(range.start, range.end)) {
                operationOptions = _({ end: range.end }).extend(operationOptions);
            }
            return this.generateSheetOperation(name, sheet, range.start, operationOptions);
        };

        /**
         * Creates and appends a new operation with 'start' and (optional)
         * 'end' properties to the operations array addressing a column or row
         * interval. If first and last index of the interval are equal, the
         * 'end' property will be omitted in the operation.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Number} sheet
         *  The zero-based sheet index, will be inserted as 'sheet' property.
         *
         * @param {Object} interval
         *  The zero-based column/row indexes of the interval, in the
         *  properties 'first' and 'last'.
         *
         * @param {Object} [operationOptions]
         *  Additional options that will be stored in the operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateIntervalOperation = function (name, sheet, interval, operationOptions) {
            if (interval.first < interval.last) {
                operationOptions = _({ end: interval.last }).extend(operationOptions);
            }
            return this.generateSheetOperation(name, sheet, interval.first, operationOptions);
        };

    } // class SpreadsheetOperationsGenerator

    // constants --------------------------------------------------------------

    _(SpreadsheetOperationsGenerator).extend({

        INSERT_SHEET: 'insertSheet',
        DELETE_SHEET: 'deleteSheet',
        MOVE_SHEET: 'moveSheet',

        SET_SHEET_NAME: 'setSheetName',
        SET_SHEET_ATTRIBUTES: 'setSheetAttributes',

        INSERT_COLUMNS: 'insertColumns',
        DELETE_COLUMNS: 'deleteColumns',
        SET_COLUMN_ATTRIBUTES: 'setColumnAttributes',

        INSERT_ROWS: 'insertRows',
        DELETE_ROWS: 'deleteRows',
        SET_ROW_ATTRIBUTES: 'setRowAttributes',

        INSERT_CELLS: 'insertCells',
        DELETE_CELLS: 'deleteCells',
        FILL_CELL_RANGE: 'fillCellRange',
        SET_CELL_CONTENTS: 'setCellContents',
        MERGE_CELLS: 'mergeCells',

        INSERT_DRAWING: 'insertDrawing',
        DELETE_DRAWING: 'deleteDrawing',
        SET_DRAWING_ATTRIBUTES: 'setDrawingAttributes'
    });

    // exports ================================================================

    // derive this class from class OperationsGenerator
    return OperationsGenerator.extend({ constructor: SpreadsheetOperationsGenerator });

});
