/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/colrowcollection',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/modelobject',
     'io.ox/office/editframework/model/format/attributedmodel',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Utils, ModelObject, AttributedModel, SheetUtils) {

    'use strict';

    // private global functions ===============================================

    /**
     * Converts the passed size of a column/row from 1/100 mm to pixels.
     * Visible entries will be set to at least 3 pixels, regardless of their
     * real size.
     */
    function convertSizeToPixels(size) {
        return (size <= 0) ? 0 : Math.max(3, Utils.convertHmmToLength(size, 'px', 1));
    }

    // class ColRowCollection =================================================

    /**
     * Collects information about all columns or all rows of a single sheet in
     * a spreadsheet document.
     *
     * Triggers the following events:
     * - 'change:entries'
     *      After the formatting attributes of a column/row interval have been
     *      changed. The event handler receives the index interval of the
     *      changed entries.
     * - 'insert:entries'
     *      After new columns/rows have been inserted into the sheet. The event
     *      handler receives the index interval of the inserted entries.
     * - 'delete:entries'
     *      After columns/rows have been deleted from the sheet. The event
     *      handler receives the index interval of the deleted entries.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this collection instance.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model instance containing this collection.
     *
     * @param {Boolean} columns
     *  Whether this collection contains column headers (true) or row headers
     *  (false).
     */
    function ColRowCollection(app, sheetModel, columns) {

        var // self reference
            self = this,

            // the property names for column or row properties
            FAMILY_NAME = columns ? 'column' : 'row',
            SIZE_ATTR_NAME = columns ? 'width' : 'height',
            DEFAULT_SIZE_ATTR_NAME = columns ? 'colWidth' : 'rowHeight',

            // sorted sparse array of index ranges (instances of EntryModel)
            entries = [],

            // the largest valid column/row index
            maxIndex = columns ? app.getModel().getMaxCol() : app.getModel().getMaxRow();

        // base constructor ---------------------------------------------------

        ModelObject.call(this, app);

        // private class EntryModel -------------------------------------------

        var EntryModel = AttributedModel.extend({ constructor: function (first, last, attributes) {

            // column/row interval covered by this entry
            this.first = first;
            this.last = last;

            // effective position/size of the columns/rows, in pixels
            this.offset = 0;
            this.size = 0;

            // base constructor - - - - - - - - - - - - - - - - - - - - - - - -

            // do not trigger any events (collection entries are potential mass objects)
            AttributedModel.call(this, app, 'cell', attributes, { silent: true, additionalFamilies: FAMILY_NAME });

            // initialization - - - - - - - - - - - - - - - - - - - - - - - - -

            // update the initial column/row size of this entry
            this.updateSize();

        }}); // class EntryModel

        // methods (prototype, collection entries are mass objects) - - - - - -

        /**
         * Returns a deep clone of this collection entry.
         *
         * @returns {EntryModel}
         *  A clone of this collection entry.
         */
        EntryModel.prototype.cloneEntry = function () {
            var entry = new EntryModel(this.first, this.last, this.getExplicitAttributes());
            entry.offset = this.offset;
            return entry;
        };

        /**
         * Updates the effective pixel size of a single column/row, according
         * to the current formatting attributes of this entry.
         */
        EntryModel.prototype.updateSize = function () {
            var attributes = this.getMergedAttributes()[FAMILY_NAME];
            this.size = attributes.visible ? convertSizeToPixels(attributes[SIZE_ATTR_NAME]) : 0;
        };

        /**
         * Returns the absolute offset of the passed index (interpreted
         * relative to the start index of this entry), in pixels.
         */
        EntryModel.prototype.getOffset = function (index) {
            return this.offset + this.size * index;
        };

        /**
         * Returns the end offset of this entry in pixels.
         */
        EntryModel.prototype.getEndOffset = function () {
            return this.getOffset(this.last - this.first + 1);
        };

        var // default model for entries missing in the collection
            defaultEntry = new EntryModel(0, maxIndex);

        // private methods ----------------------------------------------------

        /**
         * Searches for an entry that contains or follows the passed column/row
         * index.
         *
         * @param {Number} index
         *  The zero-based column/row index. Must be located in the valid range
         *  covered by this collection.
         *
         * @returns {Number}
         *  The array index of the collection entry containing the passed
         *  index. If the index is located in a gap between two entries,
         *  returns the array index of the entry starting after the passed
         *  index. This may also be an array index pointing after the last
         *  existing collection entry.
         */
        function findEntryIndex(index) {
            return _(entries).sortedIndex({ last: index }, 'last');
        }

        /**
         * Updates the dirty entry offsets up to the end of the collection.
         *
         * @param {Number} index
         *  The zero-based column/row index where updating the offsets starts.
         */
        function updateEntryOffsets(index) {

            var // the array index of the first dirty collection entry
                arrayIndex = 0,
                // the last valid collection entry
                prevEntry = null,
                // the current collection entry to be updated
                currEntry = null;

            // update start offsets of all dirty collection entries
            arrayIndex = findEntryIndex(index);
            prevEntry = entries[arrayIndex - 1];
            for (; arrayIndex < entries.length; arrayIndex += 1) {

                // set offset of current entry to the end offset of the previous entry
                currEntry = entries[arrayIndex];
                currEntry.offset = prevEntry ? prevEntry.getEndOffset() : 0;

                // add size of the gap between previous and current entry
                if (!prevEntry || (prevEntry.last + 1 < currEntry.first)) {
                    currEntry.offset += defaultEntry.size * (currEntry.first - (prevEntry ? (prevEntry.last + 1) : 0));
                }

                prevEntry = currEntry;
            }
        }

        /**
         * Helper function to get the descriptor of a specific column/row,
         * without needing to search for the affected collection entry.
         *
         * @param {Number} arrayIndex
         *  The array index of the collection entry that contains the passed
         *  index, or that adjoins the gap containing the passed index.
         *
         * @param {Number} index
         *  The index of the column/row whose descriptor will be returned. Must
         *  be contained in the collection entry with the passed index, or must
         *  be contained in the gap between that collection entry and its
         *  predecessor or successor.
         *
         * @returns {Object}
         *  A descriptor object containing the following properties:
         *  - {Number} entry.index
         *      The passed column/row index.
         *  - {Number} entry.offset
         *      The absolute position of the column/row, in pixels.
         *  - {Number} entry.size
         *      The effective size (zero for hidden columns/rows), in pixels.
         *  - {Object} entry.attributes
         *      The merged attribute set of the column/row.
         */
        function getEntryDescriptor(arrayIndex, index) {

            var // the current collection entry
                entry = null;

            // array is empty, or entry points into gap before first entry: use default size
            if ((entries.length === 0) || (arrayIndex < 0) || ((arrayIndex === 0) && (index < entries[0].first))) {
                return { index: index, offset: defaultEntry.size * index, size: defaultEntry.size, attributes: defaultEntry.getMergedAttributes() };
            }

            // index points into a gap between two entries, or to the gap after the last entry
            if ((arrayIndex === entries.length) || (index < entries[arrayIndex].first)) {
                entry = entries[arrayIndex - 1];
                return { index: index, offset: entry.getEndOffset() + defaultEntry.size * (index - entry.last - 1), size: defaultEntry.size, attributes: defaultEntry.getMergedAttributes() };
            }

            // index points into a collection entry
            entry = entries[arrayIndex];
            return {
                index: index,
                offset: entry.getOffset(index - entry.first),
                size: entry.size,
                attributes: entry.getMergedAttributes()
            };
        }

        /**
         * Splits the collection entry at the passed array index, and inserts
         * the new entry after the existing entry.
         *
         * @param {Number} arrayIndex
         *  The array index of the collection entry to be split.
         *
         * @param {Number} leadingCount
         *  The new interval size of the existing collection entry. Must be
         *  less than the current interval size of the entry.
         *
         * @returns {EntryModel}
         *  The new collection entry inserted after the existing entry.
         */
        function splitEntry(arrayIndex, leadingCount) {

            var // the existing collection entry
                oldEntry = entries[arrayIndex],
                // the clone of the existing entry
                newEntry = oldEntry.cloneEntry();

            // adjust start and end position of the collection entries
            oldEntry.last = oldEntry.first + leadingCount - 1;
            newEntry.first = oldEntry.first + leadingCount;
            newEntry.offset = oldEntry.offset + leadingCount * oldEntry.size;

            // insert the new entry into the collection
            entries.splice(arrayIndex + 1, 0, newEntry);
            return newEntry;
        }

        /**
         * Tries to merge the collection entry at the passed array index with
         * its predecessor and/or successor, if these entries are equal. If
         * successful, the preceding and/or following entries will be removed
         * from this collection.
         *
         * @param {Number} arrayIndex
         *  The array index of the collection entry to be merged with its
         *  predecessor and /or successor. May be an invalid index (no entry
         *  exists in the collection), or may refer to the first or last entry
         *  (no predecessor or successor exists in the collection).
         *
         * @param {Object} [options]
         *  A map with options specifying which entries will be merged with the
         *  specified entry. The following options are supported:
         *  @param {Boolean} [options.prev=false]
         *      If set to true, the entry and its predecessor will be tried to
         *      merge.
         *  @param {Boolean} [options.next=false]
         *      If set to true, the entry and its successor will be tried to
         *      merge.
         *
         * @returns {Object}
         *  An object containing the Boolean flags 'prev' and 'next' specifying
         *  whether the entry has been merged with its predecessor and/or
         *  successor respectively.
         */
        function mergeEntry(arrayIndex, options) {

            var // the result object
                result = {};

            // tries to merge the passed entries
            function tryMerge(arrayIndex) {

                var // the preceding collection entry
                    prevEntry = entries[arrayIndex - 1],
                    // the current collection entry
                    thisEntry = entries[arrayIndex];

                // check that the entries are equal and do not have a gap
                if (prevEntry && thisEntry && (prevEntry.last + 1 === thisEntry.first) && _.isEqual(prevEntry.getExplicitAttributes(), thisEntry.getExplicitAttributes())) {
                    thisEntry.first = prevEntry.first;
                    thisEntry.offset = prevEntry.offset;
                    entries.splice(arrayIndex - 1, 1);
                    return true;
                }
                return false;
            }

            // try to merge the entry with its successor/predecessor
            result.next = Utils.getBooleanOption(options, 'next', false) && tryMerge(arrayIndex + 1);
            result.prev = Utils.getBooleanOption(options, 'prev', false) && tryMerge(arrayIndex);
            return result;
        }

        /**
         * Invokes the passed iterator function for all collection entries that
         * cover the specified column/row interval.
         *
         * @param {Object} interval
         *  The column/row interval to be iterated, with the zero-based index
         *  properties 'first' and 'last'.
         *
         * @param {Function} iterator
         *  The iterator function called for every collection entry covered by
         *  the specified interval. Receives the following parameters:
         *  (1) {EntryModel} entry
         *      The current collection entry.
         *  (2) {Number} first
         *      The zero-based index of the first column/row. May be located
         *      inside the passed collection entry, if it is the first entry
         *      visited by this method.
         *  (3) {Number} last
         *      The zero-based index of the last column/row. May be located
         *      inside the passed collection entry, if it is the last entry
         *      visited by this method.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with additional options for this method. The following
         *  options are supported:
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, visits the covered collection entries from last
         *      to first. Must not be used together with 'options.modify'.
         *  @param {Boolean} [options.modify=false]
         *      If set to true, the iterator function wants to modify the
         *      collection entries. Entries not yet existing (gaps between
         *      existing entries) will be created and inserted into the
         *      collection before. Entries not covered completely by the first
         *      or last index of the passed interval will be split before.
         *      After iteration is finished, adjacent entries with equal
         *      formatting attributes will be merged to one entry. Must not be
         *      used together with 'options.reverse'.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        function iterateEntries(interval, iterator, options) {

            var // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false),
                // whether the iterator modifies the collection entries
                modify = Utils.getBooleanOption(options, 'modify', false),
                // current column/row index for the next iterator invocation
                index = reverse ? interval.last : interval.first,
                // array index of the next collection entry to be visited
                arrayIndex = findEntryIndex(index),
                // the current collection entry in the loop
                entry = null,
                // whether to break the iteration
                breakLoop = false;

            // invokes the iterator function
            function invokeIterator(entry, other) {
                breakLoop = breakLoop || (iterator.call(self, entry, reverse ? other : index, reverse ? index : other) === Utils.BREAK);
                index = reverse ? (other - 1) : (other + 1);
            }

            // invokes the iterator function for the gap before a collection entry
            function invokeIteratorForGap(other) {

                var // the entry passed to the iterator function
                    entry = null;

                if (modify) {
                    // insert new entry into the collection, if iterator wants to modify the collection
                    entry = defaultEntry.cloneEntry();
                } else {
                    // read-only iterator: create a dummy entry that will not be inserted into the collection
                    entry = _.clone(defaultEntry);
                    delete entry.setAttributes;
                }

                // initialize interval and start offset of the new entry
                entry.first = reverse ? other : index;
                entry.last = reverse ? index : other;
                entry.offset = getEntryDescriptor(arrayIndex, entry.first).offset;

                // insert new entry into array in modifying mode
                if (modify) {
                    entries.splice(arrayIndex, 0, entry);
                    arrayIndex += 1;
                }

                // invoke the iterator
                invokeIterator(entry, other);

                // Try to merge the new entry with its predecessor (arrayIndex
                // points after the new entry, after merging, it has to be decreased).
                if (modify && mergeEntry(arrayIndex - 1, { prev: true }).prev) {
                    arrayIndex -= 1;
                }
            }

            // invokes the iterator function for an existing collection entry
            function invokeIteratorForEntry(other) {

                // invoke the iterator
                invokeIterator(entries[arrayIndex], other);

                // Try to merge the entry with its predecessor. On success, do
                // not change the array index (it already points to the next
                // entry after merging the entries).
                if (!(modify && mergeEntry(arrayIndex, { prev: true }).prev)) {
                    arrayIndex += reverse ? -1 : 1;
                }
            }

            // modify reversely not implemented
            if (reverse && modify) {
                Utils.error('ColRowCollection.iterateEntries(): modify collection reversely not implemented');
                return Utils.BREAK;
            }

            // split the first collection entry not covered completely by the interval
            if (modify && (arrayIndex < entries.length) && (entries[arrayIndex].first < index)) {
                splitEntry(arrayIndex, index - entries[arrayIndex].first);
                arrayIndex += 1;
            }

            // reverse mode: back to previous entry, if interval end points to gap before current entry
            if (reverse && ((arrayIndex === entries.length) || (interval.last < entries[arrayIndex].first))) {
                arrayIndex -= 1;
            }

            // process all existing collection entries and the gaps covered by the interval
            while (!breakLoop && (0 <= arrayIndex) && (arrayIndex < entries.length) && (reverse ? (interval.first <= entries[arrayIndex].last) : (entries[arrayIndex].first <= interval.last))) {

                // the collection entry to be visited
                entry = entries[arrayIndex];

                // visit the gap between current index and start of the next collection entry
                // (reverse mode: gap between end of the previous collection entry and current index)
                if (reverse ? (entry.last < index) : (index < entry.first)) {
                    invokeIteratorForGap(reverse ? (entry.last + 1) : (entry.first - 1));
                }

                // split the last collection entry not covered completely by the interval
                if (modify && (interval.last < entry.last)) {
                    splitEntry(arrayIndex, interval.last - index + 1);
                }

                // visit the collection entry
                invokeIteratorForEntry(reverse ? Math.max(interval.first, entry.first) : Math.min(interval.last, entry.last));
            }

            // Try to merge last visited entry with its successor (arrayIndex
            // already points to the successor of the last visited entry).
            if (modify) {
                mergeEntry(arrayIndex, { prev: true });
            }

            // visit the gap after the last existing collection entry
            // (reverse mode: before the first existing collection entry)
            if (!breakLoop && (reverse ? (interval.first <= index) : (index <= interval.last))) {
                invokeIteratorForGap(reverse ? interval.first : interval.last);
            }
        }

        /**
         * Triggers the specified event, but not while the document is still
         * being imported.
         */
        function triggerEvent(type, interval) {

            // debug: dump contents of the collection
            if (app.isImportFinished()) {
                Utils.info('ColRowCollection: length=' + entries.length);
                _(entries).each(function (entry, index) {
                    Utils.log('  ' + index + ': interval=' + (columns ? SheetUtils.getColIntervalName : SheetUtils.getRowIntervalName)(entry) + ', offset=' + entry.offset + 'px, size=' + entry.size + 'px');
                });
            }

            self.trigger(type, interval);
        }

        /**
         * Updates the default column/row size according to the current
         * formatting attributes of the sheet model.
         */
        function updateDefaultSize() {

            var // the old default size
                oldDefSize = defaultEntry.size,
                // the merged sheet attributes
                defSizeValue = sheetModel.getMergedAttributes().sheet[DEFAULT_SIZE_ATTR_NAME];

            // set new default size at the default collection entry, calculate effective size
            defaultEntry.setAttributes(Utils.makeSimpleObject(FAMILY_NAME, Utils.makeSimpleObject(SIZE_ATTR_NAME, defSizeValue)));
            defaultEntry.updateSize();

            // do nothing, if the effective size has not changed
            if (defaultEntry.size !== oldDefSize) {
                updateEntryOffsets(0);
                triggerEvent('change:entries', { first: 0, last: maxIndex });
            }
        }

        /**
         * Enables triggering event after the document has been imported.
         */
        function importFinishedHandler() {
            app.off('docs:import:success', importFinishedHandler);
            updateDefaultSize();
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the total size of all columns/rows represented by this
         * collection.
         *
         * @returns {Number}
         *  The total size of all columns/rows in the sheet, in pixels.
         */
        this.getTotalSize = function () {
            // calculate the offset behind the last valid column/row
            return getEntryDescriptor(entries.length, maxIndex + 1).offset;
        };

        /**
         * Returns a descriptor object for the column/row with the specified
         * index.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @returns {Object}
         *  An object containing the properties 'index', 'offset', 'size', and
         *  'attributes' describing the column/row with the passed index.
         */
        this.getEntry = function (index) {
            return getEntryDescriptor(findEntryIndex(index), index);
        };

        /**
         * Invokes the passed iterator function for all visible columns/rows
         * covered by the passed index interval.
         *
         * @param {Object} interval
         *  The column/row interval to be iterated, with the zero-based index
         *  properties 'first' and 'last'.
         *
         * @param {Function} iterator
         *  The iterator function called for every column/row covered by the
         *  specified interval. Receives an entry descriptor object with the
         *  following properties:
         *  - {Number} entry.index
         *      The zero-based index of the current column/row.
         *  - {Number} entry.offset
         *      The absolute position of the column/row in the sheet in pixels.
         *  - {Number} entry.size
         *      The effective size of the column/row in pixels.
         *  - {Object} entry.attributes
         *      The merged attribute set of the column/row.
         *  If the iterator returns the Utils.BREAK object, the iteration
         *  process will be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, visits the columns/rows from last to first.
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateVisibleEntries = function (interval, iterator, options) {

            var // the calling context for the iterator function
                context = Utils.getOption(options, 'context'),
                // whether to iterate in reversed order
                reverse = Utils.getBooleanOption(options, 'reverse', false);

            // visit all collection entries covered by the passed interval
            return iterateEntries(interval, function (entry, first, last) {

                // do not invoke iterator for hidden columns/rows
                if (entry.size === 0) { return; }

                // invoke iterator for every column/row covered by the current collection entry
                return Utils.iterateRange(reverse ? last : first, reverse ? (first - 1) : (last + 1), function (index) {
                    var offset = entry.offset + entry.size * (index - entry.first);
                    return iterator.call(context, { index: index, offset: offset, size: entry.size, attributes: entry.getMergedAttributes() });
                }, { step: reverse ? -1 : 1 });
            }, { reverse: reverse });
        };

        /**
         * Returns the indexes of all visible columns/rows in the passed
         * interval as plain array.
         *
         * @param {Object} interval
         *  The column/row interval to be processed, with the zero-based index
         *  properties 'first' and 'last'.
         *
         * @returns {Number[]}
         *  The indexes of all visible columns/rows, in ascending order.
         */
        this.getVisibleIndexes = function (interval) {
            var indexes = [];
            this.iterateVisibleEntries(interval, function (entry) { indexes.push(entry.index); });
            return indexes;
        };

        /**
         * Returns whether the column/row with the specified index is visible.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @returns {Boolean}
         *  Whether the column/row is visible.
         */
        this.isEntryVisible = function (index) {
            return this.getEntry(index).size > 0;
        };

        /**
         * Returns a descriptor object for the column/row with the specified
         * index, if it is visible.
         *
         * @param {Number} index
         *  The zero-based index of the column/row.
         *
         * @returns {Object|Null}
         *  An object containing the properties 'index', 'offset', 'size', and
         *  'attributes' describing the column/row with the passed index, if it
         *  is visible; otherwise null.
         */
        this.getVisibleEntry = function (index) {
            var entryData = this.getEntry(index);
            return (entryData.size > 0) ? entryData : null;
        };

        /**
         * Returns information about the column/row with the passed index, if
         * it is visible; otherwise about the first visible column/row that
         * follows the column/row with the passed index.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @returns {Object|Null}
         *  An object containing the properties 'index', 'offset', 'size', and
         *  'attributes' describing the first visible column/row at or after
         *  the specified index; or null, if no more visible columns/rows are
         *  available.
         */
        this.getNextVisibleEntry = function (index) {

            var // the resulting entry data
                entryData = null;

            // use iterator to visit the first visible entry following the passed index
            this.iterateVisibleEntries({ first: index, last: maxIndex }, function (entry) {
                entryData = entry;
                return Utils.BREAK;
            });

            return entryData;
        };

        /**
         * Returns information about the column/row with the passed index, if
         * it is visible; otherwise about the first visible column/row that
         * precedes the column/row with the passed index.
         *
         * @param {Number} index
         *  The zero-based index of a column/row.
         *
         * @returns {Object|Null}
         *  An object containing the properties 'index', 'offset', 'size', and
         *  'attributes' describing the first visible column/row at or before
         *  the specified index; or null, if no more visible columns/rows are
         *  available.
         */
        this.getPrevVisibleEntry = function (index) {

            var // the resulting entry data
                entryData = null;

            // use iterator to visit the first visible entry following the passed index
            this.iterateVisibleEntries({ first: 0, last: index }, function (entry) {
                entryData = entry;
                return Utils.BREAK;
            }, { reverse: true });

            return entryData;
        };

        /**
         * Reduces the passed column/row interval to an interval starting and
         * ending at visible columns/rows.
         *
         * @param {Object} interval
         *  The index interval, with zero-based properties 'first' and 'last'.
         *
         * @returns {Object|Null}
         *  An index interval with 'start' and 'end' properties containing the
         *  indexes of the nearest columns/rows inside the passed interval, if
         *  available. Otherwise, null will be returned.
         */
        this.getVisibleInterval = function (interval) {

            var // find the first and last available entry in the passed interval
                firstEntryData = this.getNextVisibleEntry(interval.first),
                lastEntryData = this.getPrevVisibleEntry(interval.last);

            return firstEntryData && lastEntryData && (firstEntryData.index <= lastEntryData.index) ? { first: firstEntryData.index, last: lastEntryData.index } : null;
        };

        /**
         * Returns information about the column/row covering the passed offset
         * in the sheet.
         *
         * @param {Number} offset
         *  The absolute offset in the sheet, in pixels.
         *
         * @returns {Object}
         *  An object containing the properties 'index', 'offset', and 'size'
         *  describing the column/row containing the passed offset. If the
         *  passed offset is less than 0, returns information about the first
         *  column/row in the sheet; if the passed offset is greater than the
         *  total size of the sheet, returns information about the last
         *  column/row in the sheet.
         */
        this.getEntryByOffset = function (offset) {

            var // the passed offset, restricted to the valid sheet dimension
                currOffset = Utils.minMax(offset, 0, this.getTotalSize() - 1),
                // the index of the first collection entry after the offset
                arrayIndex = _(entries).sortedIndex({ offset: currOffset + 1 }, 'offset'),
                // the current collection entry
                entry = entries[arrayIndex - 1],
                // the end offset of the previous entry
                endOffset = entry ? entry.getEndOffset() : 0,
                // the column/row index, relative to the entry or the gap
                index = 0;

            // offset points into the previous entry
            if (entry && (currOffset < endOffset)) {
                index = Math.floor((currOffset - entry.offset) / entry.size);
                return {
                    index: entry.first + index,
                    offset: entry.getOffset(index),
                    size: entry.size,
                    attributes: entry.getMergedAttributes()
                };
            }

            // offset points into the gap before the entry at arrayIndex
            index = Math.floor((currOffset - endOffset) / defaultEntry.size);
            return {
                index: (entry ? (entry.last + 1) : 0) + index,
                offset: endOffset + defaultEntry.size * index,
                size: defaultEntry.size,
                attributes: defaultEntry.getMergedAttributes()
            };
        };

        /**
         * Returns the position and size of the specified column/row interval.
         *
         * @param {Object} interval
         *  The index interval of the columns/rows, with the zero-based
         *  properties 'first' and 'last'.
         *
         * @returns {Object}
         *  An object with the properties 'offset' containing the absolute
         *  position of the first column/row, and 'size' containing the total
         *  size of the interval, both in pixels.
         */
        this.getIntervalPosition = function (interval) {
            var offset = this.getEntry(interval.first).offset;
            return { offset: offset, size: this.getEntry(interval.last + 1).offset - offset };
        };

        /**
         * Changes the formatting attributes of all entries contained in the
         * passed index interval.
         *
         * @param {Object} interval
         *  The index interval of the collection entries to be changed, in the
         *  zero-based index properties 'first' and 'last'.
         *
         * @param {Object} attributes
         *  The new attributes for the collection entries.
         *
         * @returns {ColRowCollection}
         *  A reference to this instance.
         */
        this.setAttributes = function (interval, attributes) {

            var // whether any collection entry has been changed
                changed = false,
                // index of the first column/row with dirty pixel offset due to changed sizes
                dirtyIndex = null;

            // update all affected collection entries
            iterateEntries(interval, function (entry, first, last) {

                var // old size of the entry, in pixels
                    oldSize = entry.size;

                // set the passed attributes; post-processing, if any attributes have changed
                if (entry.setAttributes(attributes)) {
                    changed = true;

                    // update effective size of the collection entry
                    entry.updateSize();

                    // set index of entries with dirty pixel offsets, if size has been changed
                    if ((oldSize !== entry.size) && _.isNull(dirtyIndex)) {
                        dirtyIndex = entry.last + 1;
                    }
                }

            }, { modify: true });

            // update pixel offsets of following entries, if the size has been changed
            if (_.isNumber(dirtyIndex)) {
                updateEntryOffsets(dirtyIndex);
            }

            // notify change listeners
            if (changed) {
                triggerEvent('change:entries', interval);
            }
            return this;
        };

        /**
         * Inserts new entries (columns or rows) into this collection.
         *
         * @param {Object} interval
         *  The index interval, with the zero-based index properties 'first'
         *  and 'last'. New entries will be inserted before the entry specified
         *  by the property 'first'. The size of the interval specifies the
         *  number of new entries that will be inserted. The new entries will
         *  contain the formatting attributes of the predecessor of the first
         *  existing entry, except that the new entries will always be visible.
         *  If the new entries will be inserted at the beginning of this
         *  collection, they will contain default formatting.
         *
         * @returns {ColRowCollection}
         *  A reference to this instance.
         */
        this.insertEntries = function (interval) {

            var // the array index of the first entry to be moved
                arrayIndex = findEntryIndex(interval.first),
                // the current collection entry
                entry = entries[arrayIndex],
                // the previous entry used to copy formatting attributes
                prevEntry = null,
                // the explicit attributes to be used for the new entries
                prevAttributes = null,
                // the number of columns/rows to be inserted
                delta = SheetUtils.getIntervalSize(interval);

            // split collection entry if it is not covered completely
            if (entry && (entry.first < interval.first)) {
                splitEntry(arrayIndex, interval.first - entry.first);
                prevEntry = entry;
                arrayIndex += 1;
            } else if ((arrayIndex > 0) && (entries[arrayIndex - 1].last + 1 === interval.first)) {
                // use formatting of previous entry, if it ends exactly before the interval
                prevEntry = entries[arrayIndex - 1];
            }

            // move existing collection entries
            for (var index = arrayIndex; index < entries.length; index += 1) {

                // update index interval of the entry
                entry = entries[index];
                entry.first += delta;
                entry.last += delta;

                // delete following entries moved outside the collection limits
                if (entry.first > maxIndex) {
                    entries.splice(arrayIndex);
                } else if (entry.last >= maxIndex) {
                    entries.splice(arrayIndex + 1);
                    entry.last = maxIndex;
                }
            }

            // do not create a new entry, if the preceding column/row has default format
            if (prevEntry) {

                // prepare the explicit attributes to be used for the new entries (always be visible)
                prevAttributes = prevEntry.getExplicitAttributes();
                if ((FAMILY_NAME in prevAttributes) && (prevAttributes[FAMILY_NAME].visible === false)) {
                    delete prevAttributes[FAMILY_NAME].visible;
                }

                // insert a new collection entry, try to merge with adjacent entries
                entry = new EntryModel(interval.first, interval.last, prevAttributes);
                entries.splice(arrayIndex, 0, entry);
                mergeEntry(arrayIndex, { prev: true, next: true });
            }

            // update pixel offsets of the entries
            updateEntryOffsets(interval.first);

            // notify insert listeners
            triggerEvent('insert:entries', interval);
            return this;
        };

        /**
         * Removes existing entries (columns or rows) from this collection.
         *
         * @param {Object} interval
         *  The index interval of the entries to be deleted, in the zero-based
         *  index properties 'first' and 'last'.
         *
         * @returns {ColRowCollection}
         *  A reference to this instance.
         */
        this.deleteEntries = function (interval) {

            var // the array index of the first entry to be deleted
                arrayIndex = findEntryIndex(interval.first),
                // the current collection entry
                entry = entries[arrayIndex],
                // the number of elements to be deleted from the array
                deleteCount = 0,
                // the number of columns/rows to be removed
                delta = SheetUtils.getIntervalSize(interval);

            // split the part of the entry located before the interval
            if (entry && (entry.first < interval.first)) {
                entry = splitEntry(arrayIndex, interval.first - entry.first);
                arrayIndex += 1;
            }

            // remove and move the remaining entries
            for (var index = arrayIndex; index < entries.length; index += 1) {

                // update number of entries to be deleted, update index interval of moved entries
                entry = entries[arrayIndex];
                if (entry.last <= interval.last) {
                    deleteCount += 1;
                } else {
                    entry.first = Math.max(interval.first, entry.first - delta);
                    entry.last -= delta;
                }
            }

            // delete the array elements
            if (deleteCount > 0) {
                entries.splice(arrayIndex, deleteCount);
            }

            // try to merge with previous entry
            mergeEntry(arrayIndex, { prev: true });

            // update pixel offsets of the entries
            updateEntryOffsets(interval.first);

            // notify delete listeners
            triggerEvent('delete:entries', interval);
            return this;
        };

        this.destroy = (function () {
            var baseMethod = self.destroy;
            return function () {
                sheetModel.off('change:attributes', updateDefaultSize);
                app.off('docs:import:success', importFinishedHandler);
                _(entries).invoke('destroy');
                sheetModel = entries = null;
                baseMethod.call(self);
            };
        }());

        // initialization -----------------------------------------------------

        // update entry sizes, after the default column/row size defined by the sheet has been changed
        sheetModel.on('change:attributes', updateDefaultSize);
        updateDefaultSize();

        // update entry sizes after the document has been loaded
        app.on('docs:import:success', importFinishedHandler);

    } // class ColRowCollection

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: ColRowCollection });

});
