/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/view/editcontrols',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/tk/io',
     'io.ox/office/tk/control/group',
     'io.ox/office/tk/control/button',
     'io.ox/office/tk/control/radiolist',
     'io.ox/office/tk/control/combofield',
     'io.ox/office/tk/dropdown/dropdown',
     'io.ox/office/editframework/model/format/color',
     'io.ox/office/editframework/model/format/fonts',
     'io.ox/office/editframework/model/format/border',
     'gettext!io.ox/office/editframework'
    ], function (Utils, KeyCodes, IO, Group, Button, RadioList, ComboField, DropDown, Color, Fonts, Border, gt) {

    'use strict';

    var // predefined color definitions
        BUILTIN_COLOR_DEFINITIONS = [
            { name: 'dark-red',    label: gt('Dark red'),    color: { type: 'rgb', value: 'C00000' } },
            { name: 'red',         label: gt('Red'),         color: { type: 'rgb', value: 'FF0000' } },
            { name: 'orange',      label: gt('Orange'),      color: { type: 'rgb', value: 'FFC000' } },
            { name: 'yellow',      label: gt('Yellow'),      color: { type: 'rgb', value: 'FFFF00' } },
            { name: 'light-green', label: gt('Light green'), color: { type: 'rgb', value: '92D050' } },
            { name: 'green',       label: gt('Green'),       color: { type: 'rgb', value: '00B050' } },
            { name: 'light-blue',  label: gt('Light blue'),  color: { type: 'rgb', value: '00B0F0' } },
            { name: 'blue',        label: gt('Blue'),        color: { type: 'rgb', value: '0070C0' } },
            { name: 'dark-blue',   label: gt('Dark blue'),   color: { type: 'rgb', value: '002060' } },
            { name: 'purple',      label: gt('Purple'),      color: { type: 'rgb', value: '7030A0' } }
        ],

        // definitions for the theme color table (in order as shown in the GUI color picker)
        // transformation table: 0 = pure color; negative values = 'shade'; positive values = 'tint'
        SCHEME_COLOR_DEFINITIONS = [
            { name: 'background1', label: getBackgroundColorName(1), transformations: [    0, -242, -217, -191, -166, -128 ] },
            { name: 'text1',       label: getTextColorName(1),       transformations: [  128,  166,  191,  217,  242,    0 ] },
            { name: 'background2', label: getBackgroundColorName(2), transformations: [    0, -230, -191, -128,  -64,  -26 ] },
            { name: 'text2',       label: getTextColorName(2),       transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent1',     label: getAccentColorName(1),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent2',     label: getAccentColorName(2),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent3',     label: getAccentColorName(3),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent4',     label: getAccentColorName(4),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent5',     label: getAccentColorName(5),     transformations: [   51,  102,  153,    0, -191, -128 ] },
            { name: 'accent6',     label: getAccentColorName(6),     transformations: [   51,  102,  153,    0, -191, -128 ] }
        ],

        LOCALE_DEFINITIONS = [
            { locale: 'en-US', label: gt('English (US)') },
            { locale: 'de-DE', label: gt('German') },
            { locale: 'fr-FR', label: gt('French') },
            { locale: 'es-ES', label: gt('Spanish') },
            { locale: 'cs-CZ', label: gt('Czech') },
            { locale: 'da-DK', label: gt('Danish') },
            { locale: 'nl-NL', label: gt('Dutch (Netherlands)') },
            { locale: 'fi-FI', label: gt('Finnish') },
            { locale: 'el-GR', label: gt('Greek') },
            { locale: 'hu-HU', label: gt('Hungarian') },
            { locale: 'it-IT', label: gt('Italian (Italy)') },
            { locale: 'pl-PL', label: gt('Polish') },
            { locale: 'pt-PT', label: gt('Portuguese (Portugal)') },
            { locale: 'ro-RO', label: gt('Romanian') },
            { locale: 'ru-RU', label: gt('Russian') },
            { locale: 'sv-SE', label: gt('Swedish (Sweden)') }
        ],

        // all available entries for the border chooser control
        BORDER_ENTRIES = [

            { name: 'left',    value: { left:    true }, icon: 'docs-border-l', label: /*#. in paragraphs and tables cells */ gt('Left border') },
            { name: 'right',   value: { right:   true }, icon: 'docs-border-r', label: /*#. in paragraphs and tables cells */ gt('Right border') },
            { name: 'top',     value: { top:     true }, icon: 'docs-border-t', label: /*#. in paragraphs and tables cells */ gt('Top border') },
            { name: 'bottom',  value: { bottom:  true }, icon: 'docs-border-b', label: /*#. in paragraphs and tables cells */ gt('Bottom border') },
            { name: 'insidev', value: { insidev: true }, icon: 'docs-border-v', label: /*#. in paragraphs and tables cells */ gt('Inner vertical borders') },
            { name: 'insideh', value: { insideh: true }, icon: 'docs-border-h', label: /*#. in paragraphs and tables cells */ gt('Inner horizontal borders') },

            { name: 'none',               value: { left: false, right: false, insidev: false, top: false, bottom: false, insideh: false }, icon: 'docs-border',        label: /*#. in paragraphs and tables cells */ gt('No borders') },
            { name: 'outer',              value: { left: true,  right: true,  insidev: false, top: true,  bottom: true,  insideh: false }, icon: 'docs-border-tblr',   label: /*#. in paragraphs and tables cells */ gt('Outer borders') },
            { name: 'inner',              value: { left: false, right: false, insidev: true,  top: false, bottom: false, insideh: true },  icon: 'docs-border-hv',     label: /*#. in paragraphs and tables cells */ gt('Inner borders') },
            { name: 'all',                value: { left: true,  right: true,  insidev: true,  top: true,  bottom: true,  insideh: true },  icon: 'docs-border-tbhlrv', label: /*#. in paragraphs and tables cells */ gt('All borders') },
            { name: 'left-right',         value: { left: true,  right: true,  insidev: false, top: false, bottom: false, insideh: false }, icon: 'docs-border-lr',     label: /*#. in paragraphs and tables cells */ gt('Left and right borders') },
            { name: 'left-right-insidev', value: { left: true,  right: true,  insidev: true,  top: false, bottom: false, insideh: false }, icon: 'docs-border-lrv',    label: /*#. in paragraphs and tables cells */ gt('All vertical borders') },
            { name: 'outer-insidev',      value: { left: true,  right: true,  insidev: true,  top: true,  bottom: true,  insideh: false }, icon: 'docs-border-tblrv',  label: /*#. in paragraphs and tables cells */ gt('Inner vertical and outer borders') },
            { name: 'top-bottom',         value: { left: false, right: false, insidev: false, top: true,  bottom: true,  insideh: false }, icon: 'docs-border-tb',     label: /*#. in paragraphs and tables cells */ gt('Top and bottom borders') },
            { name: 'top-bottom-insideh', value: { left: false, right: false, insidev: false, top: true,  bottom: true,  insideh: true },  icon: 'docs-border-tbh',    label: /*#. in paragraphs and tables cells */ gt('All horizontal borders') },
            { name: 'outer-insideh',      value: { left: true,  right: true,  insidev: false, top: true,  bottom: true,  insideh: true },  icon: 'docs-border-tbhlr',  label: /*#. in paragraphs and tables cells */ gt('Inner horizontal and outer borders') }
        ],

        // all available border line styles
        BORDER_LINE_STYLES = [
            { value: 'single', label: /*#. Border line style: a solid single line */ gt('Single') },
            { value: 'double', label: /*#. Border line style: a solid double line */ gt('Double') },
            { value: 'dotted', label: /*#. Border line style: a dotted single line */ gt('Dotted') },
            { value: 'dashed', label: /*#. Border line style: a dashed single line */ gt('Dashed') }
        ],

        ACCESS_RIGHTS_DEFINITIONS = [
            { mail: 'user1@example.com', name: 'User 1' },
            { mail: 'user2@example.com', name: 'User 2' },
            { mail: 'user3@example.com', name: 'User 3' },
            { mail: 'user4@example.com', name: 'User 4' }
        ];

    // private global functions ===============================================

    /**
     * Returns the localized name of a text color in a color scheme.
     */
    function getTextColorName(index) {
        var label =
            //#. The name of a text color in a color scheme (a color scheme consists
            //#. of two text colors, two background colors, and six accent colors).
            //#. Example result: "Text 1", "Text 2"
            //#. %1$d is the index of the text color
            //#, c-format
            gt('Text %1$d', _.noI18n(index));
        return label;
    }

    /**
     * Returns the localized name of a background color in a color scheme.
     */
    function getBackgroundColorName(index) {
        var label =
            //#. The name of a background color in a color scheme (a color scheme consists
            //#. of two text colors, two background colors, and six accent colors).
            //#. Example result: "Background 1", "Background 2"
            //#. %1$d is the index of the background color
            //#, c-format
            gt('Background %1$d', _.noI18n(index));
        return label;
    }

    /**
     * Returns the localized name of an accented color in a color scheme.
     */
    function getAccentColorName(index) {
        var label =
            //#. The name of an accent color in a color scheme (a color scheme consists
            //#. of two text colors, two background colors, and six accent colors).
            //#. Example result: "Accent 1", "Accent 2"
            //#. %1$d is the index of the accent color
            //#, c-format
            gt('Accent %1$d', _.noI18n(index));
        return label;
    }

    // static class EditControls ==============================================

    var EditControls = {};

    // constants --------------------------------------------------------------

    /**
     * Standard options for the 'Reload' button.
     *
     * @constant
     */
    EditControls.RELOAD_OPTIONS = { icon: 'icon-repeat', css: {color: 'yellow'}, tooltip: gt('Reload document'), visibility: 'error' };
    /**
     * Standard options for the 'Undo' button.
     *
     * @constant
     */
    EditControls.UNDO_OPTIONS = { icon: 'docs-undo', tooltip: gt('Revert last operation'), visibility: 'editmode' };

    /**
     * Standard options for the 'Redo' button.
     *
     * @constant
     */
    EditControls.REDO_OPTIONS = { icon: 'docs-redo', tooltip: gt('Restore last operation'), visibility: 'editmode' };

    /**
     * Standard options for a 'Bold' toggle button.
     *
     * @constant
     */
    EditControls.BOLD_OPTIONS = { icon: 'docs-font-bold', tooltip: gt('Bold'), toggle: true, visibility: 'editmode' };

    /**
     * Standard options for an 'Italic' toggle button.
     *
     * @constant
     */
    EditControls.ITALIC_OPTIONS = { icon: 'docs-font-italic', tooltip: gt('Italic'), toggle: true, visibility: 'editmode' };

    /**
     * Standard options for an 'Underline' toggle button.
     *
     * @constant
     */
    EditControls.UNDERLINE_OPTIONS = { icon: 'docs-font-underline', tooltip: gt('Underline'), toggle: true, visibility: 'editmode' };

    /**
     * Standard options for a 'Strike through' toggle button.
     *
     * @constant
     */
    EditControls.STRIKEOUT_OPTIONS = { icon: 'docs-font-strikeout', tooltip: gt('Strike through'), toggle: true, visibility: 'editmode' };

    /**
     * Standard options for a 'Clear formatting' button.
     *
     * @constant
     */
    EditControls.CLEAR_FORMAT_OPTIONS = { icon: 'docs-reset-formatting', tooltip: /*#. clear manual formatting of text or table cells */ gt('Clear formatting'), visibility: 'editmode' };

    // class ColorChooser =====================================================

    /**
     * Creates a control with a drop-down menu used to choose a color from a
     * set of color items. Shows a selection of standard colors, and a table of
     * shaded scheme colors from the current document theme.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {EditApplication} app
     *  The application containing this control group.
     *
     * @param {String} context
     *  The color context that will be used to resolve the 'auto' color
     *  correctly.
     *
     * @param {Object} [options]
     *  A map of options to control the properties of the drop-down button and
     *  menu. Supports all options of the RadioList base class.
     */
    EditControls.ColorChooser = RadioList.extend({ constructor: function (app, context, options) {

        var // self reference
            self = this,

            // the color box in the drop-down button
            colorBox = $('<div>').addClass('color-box'),

            // the style sheet containers of the document
            documentStyles = app.getModel().getDocumentStyles(),

            // the collection of themes of the edited document
            themes = app.getModel().getThemes();

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            itemGrid: true,
            itemDesign: 'framed',
            itemColumns: 10,
            itemCreateHandler: createItemHandler,
            updateCaptionMode: 'none',
            equality: sameColors
        }, options));

        // private methods ----------------------------------------------------

        /**
         * Converts the passed color value to a CSS color string.
         */
        function resolveCssColor(color) {
            var cssColor = _.isObject(color) ? Color.getCssColor(color, context, themes.getTheme()) : 'transparent';
            // show 'white' for transparent colors in dark GUI design
            return (cssColor === 'transparent') ? 'white' : cssColor;
        }

        /**
         * Initializes a new color item button.
         */
        function createItemHandler(button) {
            var color = Utils.getControlValue(button);
            button.prepend($('<div>').addClass('color-button').css('background-color', resolveCssColor(color)));
        }

        /**
         * Updates the color box in the drop-down menu button.
         */
        function updateHandler(color) {
            colorBox.css('background-color', (_.isUndefined(color) || _.isNull(color)) ? 'transparent' : resolveCssColor(color));
        }

        function sameColors(color1, color2) {
            var color1Value = color1.value ? color1.value.toLowerCase() : undefined,
                color2Value = color2.value ? color2.value.toLowerCase() : undefined;

            if (app.isODF()) {
                return (((color1.type === color2.type) && (color1Value === color2Value) && (_.isEqual(color1.transformations, color2.transformations))) ||
                         ((color1.type === 'scheme') && (color1.fallbackValue) && (color1.fallbackValue.toLowerCase() === color2Value)) ||
                         ((color2.type === 'scheme') && (color2.fallbackValue) && (color2.fallbackValue.toLowerCase() === color1Value)));
            } else {
                return (color1.type === color2.type) && (color1Value === color2Value) && (_.isEqual(color1.transformations, color2.transformations));
            }
        }

        /**
         * Creates the option buttons for one table row of scheme colors.
         */
        function fillSchemeColorRow(rowIndex) {
            _(SCHEME_COLOR_DEFINITIONS).each(function (definition) {

                var // the encoded transformation
                    encoded = definition.transformations[rowIndex],
                    // decoded tint/shade value
                    tint = encoded > 0,
                    value = Math.round(Math.abs(encoded) / 255 * 100000),
                    // the theme color object
                    color = { type: 'scheme', value: definition.name },
                    // the data value added at the button
                    dataValue = definition.name,
                    // description of the color (tool tip)
                    label = definition.label,
                    // shade/tint value as integral percentage
                    percent = (100 - Math.round(value / 1000)),
                    // fallback value for format without theme support
                    fallbackValue = null;

                if (value !== 0) {
                    color.transformations = [{ type: tint ? 'tint' : 'shade', value: value }];
                    dataValue += (tint ? '-lighter' : '-darker') + percent;
                    label = tint ?
                        //#. The full name of a light theme color (a base color lightened by a specific percentage value)
                        //#. Example result: "Green, lighter 20%"
                        //#. %1$s is the name of the base color
                        //#. %2$d is the percentage value, followed by a literal percent sign
                        //#, c-format
                        gt('%1$s, lighter %2$d%', label, _.noI18n(percent)) :
                        //#. The full name of a dark theme color (a base color darkened by a specific percentage value)
                        //#. Example result: "Green, darker 20%"
                        //#. %1$s is the name of the base color
                        //#. %2$d is the percentage value, followed by a literal percent sign
                        //#, c-format
                        gt('%1$s, darker %2$d%', label, _.noI18n(percent));
                }

                // store the resulting RGB color in the color value object
                if (Color.isThemeColor(color)) {
                    fallbackValue = documentStyles.getCssColor(color, 'fill');
                    if (fallbackValue[0] === '#') {
                        color.fallbackValue = fallbackValue.slice(1);
                    }
                }

                self.createOptionButton(color, { sectionId: 'theme', tooltip: label, dataValue: dataValue });
            });
        }

        /**
         * Inserts all available colors into the drop-down menu.
         */
        var initializeColorTable = app.createDebouncedMethod($.noop, function () {

            self.clearOptionButtons();

            // add automatic color
            self.createMenuSection('auto')
                .createOptionButton(Color.AUTO, {
                    sectionId: 'auto',
                    label: Color.isTransparentColor(Color.AUTO, context) ?
                         /*#. no fill color, transparent */ gt('No color') :
                         /*#. automatic text color (usually white on dark background, otherwise black) */ gt('Automatic color'),
                    dataValue: 'auto'
                });

            // add scheme colors
            if (documentStyles.hasSchemeColors()) {
                self.createMenuSection('theme', { label: gt('Theme Colors') });
                _(SCHEME_COLOR_DEFINITIONS[0].transformations.length).times(fillSchemeColorRow);
            }

            // add predefined colors
            self.createMenuSection('standard', { label: gt('Standard Colors') });
            _(BUILTIN_COLOR_DEFINITIONS).each(function (definition) {
                self.createOptionButton(definition.color, { sectionId: 'standard', tooltip: definition.label, dataValue: definition.name });
            });

        });

        // initialization -----------------------------------------------------

        this.getMenuNode().addClass('color-chooser');

        // add the color preview box to the menu button
        this.getMenuButton().append(colorBox);

        // register an update handler that updates the color box
        this.registerUpdateHandler(updateHandler);

        // insert color buttons in the drop-down menu after import and changed themes
        app.on('docs:import:success', initializeColorTable);
        themes.on('change', initializeColorTable);

    }}); // class ColorChooser

    // class StyleSheetChooser ================================================

    /**
     * A drop-down list control used to select a style sheet from a list. The
     * drop-down list entries will visualize the formatting attributes of the
     * style sheet if possible.
     *
     * @constructor
     *
     * @extends RadioList
     *
     * @param {EditApplication} app
     *  The application containing this control group.
     *
     * @param {String} family
     *  The attribute family of the style sheets visualized by this control.
     *
     * @param {Object} [options]
     *  Additional options passed to the RadioList constructor. Supports all
     *  options of the RadioList class, and the following additional options:
     *  @param {String|String[]} [options.previewFamilies]
     *      The attribute families used to get the formatting options of the
     *      list items representing the style sheets. If omitted, only
     *      attributes of the family specified by the 'family' parameter will
     *      be used.
     *  @param {Object} [options.translationDatabase]
     *      The database map containing the configuration of translated style
     *      sheet names. Each property of the map must be an object with the
     *      English style sheet name as key. That key may contain regular
     *      expressions which must be placed in capturing groups (in
     *      parentheses). The object itself is a map with short locale names
     *      (en, de, fr, etc.) or full locale names (de_DE, fr_CA, etc.) as
     *      keys, mapping the translated style sheet name for that locale or
     *      language. Translations for complete locales will be preferred over
     *      translations for entire languages. The placeholders '$1', '$2',
     *      etc. will match the capturing groups defined in the key pattern.
     */
    EditControls.StyleSheetChooser = RadioList.extend({ constructor: function (app, family, options) {

        var // self reference
            self = this,

            // the model instance
            model = app.getModel(),

            // the style sheet container
            styleSheets = model.getStyleSheets(family),

            // the configuration for translated style sheet names
            translationDatabase = Utils.getObjectOption(options, 'translationDatabase', {}),

            // the cache with translated style sheet names
            translationCache = {};

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            itemGrid: true,
            itemDesign: 'framed',
            sorted: true,
            sortFunctor: Utils.getControlUserData,
            updateCaptionMode: 'label'
        }, options));

        // private methods ----------------------------------------------------

        /**
         * Creates a new button in the drop-down menu for the specified style
         * sheet.
         */
        function createOptionButton(styleName, styleId) {

            var // sorting priority
                priority = styleSheets.getUIPriority(styleId),
                // the sort index stored at the button for lexicographical sorting
                sortIndex = String((priority < 0) ? (priority + 0x7FFFFFFF) : priority);

            // translate and adjust style name
            styleName = self.translateStyleName(styleName);

            // build a sorting index usable for lexicographical comparison:
            // 1 digit for priority sign, 10 digits positive priority,
            // followed by lower-case translated style sheet name
            while (sortIndex.length < 10) { sortIndex = '0' + sortIndex; }
            sortIndex = ((priority < 0) ? '0' : '1') + sortIndex + styleName.toLowerCase();

            // create the list item, pass sorting index as user data
            self.createOptionButton(styleId, { label: styleName, tooltip: styleName, userData: sortIndex });
        }

        /**
         * Fills the drop-down list with all known style names, and adds
         * preview CSS formatting to the list items.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {
            self.clearOptionButtons();
            _(styleSheets.getStyleSheetNames()).each(createOptionButton);
        });

        // public methods -----------------------------------------------------

        /**
         * Returns the translation of the passed style sheet name for the
         * current locale. Additionally, the passed style name will be adjusted
         * further for GUI display. All words will be capitalized, automatic
         * line breaks before numbers will be prevented by converting the
         * preceding space characters to NBSP characters.
         *
         * @param {String} styleName
         *  The original (English) style sheet name.
         *
         * @returns {String}
         *  The translated and adjusted style sheet name. If no translation is
         *  available, the passed style sheet name will be adjusted and
         *  returned.
         */
        this.translateStyleName = function (styleName) {

            var // the lower-case style name, used as database key
                key = styleName.toLowerCase(),
                // the translated style name
                translatedName = null;

            // first, check the cache of the translation database
            if (key in translationCache) { return translationCache[key]; }

            // try to find a database entry for the base style name
            _(translationDatabase).any(function (entry, key) {

                var // the matches for the pattern of the current entry
                    matches = new RegExp('^' + key + '$', 'i').exec(styleName);

                if (!_.isArray(matches)) { return; }

                // matching entry found, translate it and replace the placeholders
                translatedName = entry[Utils.LOCALE] || entry[Utils.LANGUAGE] || null;
                if (translatedName) {
                    for (var index = matches.length - 1; index > 0; index -= 1) {
                        translatedName = translatedName.replace(new RegExp('\\$' + index, 'g'), matches[index]);
                    }
                }

                // exit the _.any() loop
                return true;
            });

            // adjust the resulting style name for GUI display
            translatedName = (translatedName || Utils.capitalizeWords(styleName)).replace(/ ([0-9])/g, '\xa0$1');

            // put the translated name into the cache
            return translationCache[key] = _.noI18n(translatedName);
        };

        // initialization -----------------------------------------------------

        // add CSS class to menu button and drop-down list for special formatting
        this.getNode().addClass('style-chooser family-' + styleSheets.getStyleFamily());
        this.getMenuNode().addClass('app-' + app.getDocumentType() + ' style-chooser family-' + styleSheets.getStyleFamily());

        // add all known style sheets after import and changed style sheets
        app.on('docs:import:success', fillList);
        styleSheets.on('change', fillList);
        // also reinitialize the preview if theme settings have been changed
        model.getThemes().on('change', fillList);

    }}); // class StyleSheetChooser

    // class FontFamilyChooser ================================================

    /**
     * A combobox control used to select a font family.
     *
     * @constructor
     *
     * @extends ComboField
     *
     * @param {Object} [options]
     *  A map with options controlling the appearance of the control. Supports
     *  all options of the base class ComboField.
     */
    EditControls.FontFamilyChooser = ComboField.extend({ constructor: function (app, options) {

        var // self reference
            self = this,

            // the collection of fonts of the edited document
            fonts = app.getModel().getFonts();

        // base constructor ---------------------------------------------------

        ComboField.call(this, Utils.extendOptions({
            width: 100,
            tooltip: gt('Font name'),
            sorted: true,
            typeAhead: true
        }, options));

        // methods ------------------------------------------------------------

        /**
         * Fills the drop-down list with all known font names.
         */
        var fillList = app.createDebouncedMethod($.noop, function () {
            self.clearMenuSections();
            _(fonts.getFontNames()).each(function (fontName) {
                self.createListEntry(fontName, { label: _.noI18n(fontName), labelCss: { fontFamily: fonts.getCssFontFamily(fontName), fontSize: '115%' } });
            });
        });

        // initialization -----------------------------------------------------

        // add all known fonts after import and changed fonts
        app.on('docs:import:success', fillList);
        fonts.on('change', fillList);

    }}); // class FontFamilyChooser

    // class FontHeightChooser ================================================

    /**
     * A combobox control used to select a font height.
     *
     * @constructor
     *
     * @extends ComboField
     *
     * @param {Object} [options]
     *  A map with options controlling the appearance of the control. Supports
     *  all options of the base class ComboField.
     */
    EditControls.FontHeightChooser = ComboField.extend({ constructor: function (options) {

        // base constructor ---------------------------------------------------

        ComboField.call(this, Utils.extendOptions({
            width: 30,
            tooltip: gt('Font size'),
            css: { textAlign: 'right' },
            keyboard: 'number',
            validator: new ComboField.NumberValidator({ min: 1, max: 999.9, precision: 0.1 })
        }, options));

        // initialization -----------------------------------------------------

        _([6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 18, 20, 22, 24, 26, 28, 32, 36, 40, 44, 48, 54, 60, 66, 72, 80, 88, 96]).each(function (size) {
            this.createListEntry(size, { label: _.noI18n(String(size)), css: { textAlign: 'right', paddingRight: 20 } });
        }, this);

    }}); // class FontHeightChooser

    // class LanguageChooser ================================================

    (function () {

        var // a Promise that will resolve when the supported locales have been received
            supportedLocales = IO.sendRequest({
                method: 'POST',
                module: 'spellchecker',
                params: {
                    action: 'supportedlocales'
                }
            })
            .done(function (data) {
                var supportedLocales = Utils.getArrayOption(data, 'SupportedLocales', []);
                _(LOCALE_DEFINITIONS).each(function (definition) {
                    var localeName = definition.locale.replace('-', '_');
                    definition.supported = _(supportedLocales).contains(localeName);
                });
            });

        EditControls.LanguageChooser = RadioList.extend({ constructor: function (options) {

            var // self reference
                self = this;

            // base constructor -----------------------------------------------

            RadioList.call(this, Utils.extendOptions(options, {
                sorted: true,
                tooltip: gt('Text language'),
                css: { textAlign: 'left' }
            }));

            // initialization -------------------------------------------------

            this.getMenuNode().addClass('language-chooser');

            supportedLocales.always(function () {
                _(LOCALE_DEFINITIONS).each(function (definition) {
                    self.createOptionButton(definition.locale, {
                        label: definition.label,
                        icon: definition.supported ? 'icon-check' : 'icon-check-empty'
                    });
                });
            });

        }}); // class LanguageChooser

    }());

    // class BorderChooser ====================================================

    /**
     * A drop-down list control with different settings for single or multiple
     * components supporting outer and inner border lines. The list entries are
     * divided into two sections. The first section contains entries for single
     * border lines that can be toggled. The second section contains list
     * entries for complex border settings for multiple component borders.
     *
     * The method 'BorderChooser.setValue()' expects object values with the
     * optional properties 'top', 'bottom', 'left', 'right', insideh', and
     * 'insidev' of type Boolean or Null specifying the visibility state of the
     * respective border line. The property value null represents an ambiguous
     * state for the border line. Additionally, the method accepts an options
     * map as second parameter. The two options 'showInsideHor' and
     * 'showInsideVert' will override the respective options passed to the
     * constructor of this instance.
     *
     * @constructor
     *
     * @extends RadioGroup
     *
     * @param {Object} [options]
     *  A map with options controlling the appearance of the drop-down list.
     *  Supports all options of the base class RadioList. Additionally, the
     *  following options are supported:
     *  @param {Boolean} [options.showInsideHor=false]
     *      If set to true, the drop-down list will provide list entries for
     *      horizontal inner borders between multiple components that are
     *      stacked vertically.
     *  @param {Boolean} [options.showInsideVert=false]
     *      If set to true, the drop-down list will provide list entries for
     *      vertical inner borders between multiple components that are located
     *      side-by-side.
     */
    EditControls.BorderChooser = RadioList.extend({ constructor: function (options) {

        var // self reference
            self = this,

            // whether list entries with inner horizontal borders are currently shown
            showInsideHor = null,

            // whether list entries with inner vertical borders are currently shown
            showInsideVert = null;

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            icon: 'docs-border-tblr',
            tooltip: /*#. in paragraphs and tables cells */ gt('Borders')
        }, options, {
            equality: matchBorder,
            highlight: highlightHandler,
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Insert all supported list entries according to the passed options.
         */
        function fillList(options) {

            var newInsideHor = Utils.getBooleanOption(options, 'showInsideHor', showInsideHor),
                newInsideVert = Utils.getBooleanOption(options, 'showInsideVert', showInsideVert);

            if ((showInsideHor !== newInsideHor) || (showInsideVert !== newInsideVert)) {
                showInsideHor = newInsideHor;
                showInsideVert = newInsideVert;

                // prepare the sections for the toggle buttons and the complex border buttons
                self.clearMenuSections().createMenuSection('single').createMenuSection('compound', { separator: true });

                // insert all supported list items
                _(BORDER_ENTRIES).each(function (entry) {
                    var sectionId = (_.size(entry.value) === 1) ? 'single' : 'compound';
                    if ((showInsideHor || !entry.value.insideh) && (showInsideVert || !entry.value.insidev)) {
                        self.createOptionButton(entry.value, { sectionId: sectionId, icon: entry.icon, label: entry.label, dataValue: entry.name });
                    }
                });
            }
        }

        /**
         * Returns whether the passed borders matches one of the drop-down list
         * entries.
         *
         * @param {Object} value
         *  The border value to be matched against a list entry.
         *
         * @param {Object} entryValue
         *  The value of one of the list entries of this control.
         *
         * @returns {Boolean}
         *  Whether the border value matches the list entry. All properties
         *  contained in the list entry must be equal in the border value.
         *  Other existing properties of the border value are ignored.
         */
        function matchBorder(value, entryValue) {
            return _.isObject(value) && _(entryValue).all(function (visible, propName) {
                // use false for missing properties in value, but do not match existing null values
                return visible === ((propName in value) ? value[propName] : false);
            });
        }

        /**
         * Returns whether to highlight the button for the current value.
         */
        function highlightHandler(value) {
            // no entry in the border value must be null
            return _.isObject(value) && _(value).all(_.isBoolean);
        }

        /**
         * Updates the icon in the drop-down menu button according to the
         * current value of this control.
         */
        function updateHandler(value, options) {
            var icon = '';
            fillList(options);
            if (_.isObject(value)) {
                if (value.top) { icon += 't'; }
                if (value.bottom) { icon += 'b'; }
                if (value.insideh) { icon += 'h'; }
                if (value.left) { icon += 'l'; }
                if (value.right) { icon += 'r'; }
                if (value.insidev) { icon += 'v'; }
                if (icon.length > 0) { icon = '-' + icon; }
                icon = 'docs-border' + icon;
            }
            Utils.setControlCaption(self.getMenuButton(), icon ? { icon: icon } : self.getOptions());
        }

        // initialization -----------------------------------------------------

        // initialize the drop-down list items
        fillList(options);

        // register custom update handler to select the correct border icon
        this.registerUpdateHandler(updateHandler);

    }}); // class BorderChooser

    // class BorderStyleChooser ===============================================

    EditControls.BorderStyleChooser = RadioList.extend({ constructor: function (options) {

        var // the caption element in the drop-down menu button
            captionNode = $('<div>').addClass('caption').css({ width: '100%' }).append(
                $('<p>').css({ borderTop: '3px white none', marginTop: '15px', marginRight: '10px' })
            );

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions({
            tooltip: /*#. line style (solid, dashed, dotted) in paragraphs and tables cells */ gt('Border style'),
            css: { textAlign: 'left' }
        }, options, {
            itemCreateHandler: createItemHandler,
            updateCaptionMode: 'none'
        }));

        // private methods ----------------------------------------------------

        /**
         * Initializes a new list item according to the border styles.
         */
        function createItemHandler(button, options) {
            button.prepend($('<p>').css({ borderTop: '3px black ' + Border.getCssBorderStyle(options.value), marginTop: '12px' }));
        }

        /**
         * Updates the border style in the drop-down menu button.
         */
        function updateHandler(value) {
            if (value === null) { value = 'none'; }
            captionNode.children('p').css('border-top-style', Border.getCssBorderStyle(value));
        }

        // initialization -----------------------------------------------------

        this.getMenuButton().prepend(captionNode);
        this.registerUpdateHandler(updateHandler);

        _(BORDER_LINE_STYLES).each(function (entry) {
            this.createOptionButton(entry.value, { tooltip: entry.label });
        }, this);

    }}); // class BorderStyleChooser

    // class TableBorderWidthChooser ==========================================

    EditControls.TableBorderWidthChooser = ComboField.extend({ constructor: function (options) {

        // base constructor ---------------------------------------------------

        ComboField.call(this, Utils.extendOptions({
            width: 40,
            tooltip: gt('Border width'),
            css: { textAlign: 'right' },
            keyboard: 'number',
            validator: new ComboField.NumberValidator({ min: 0.5, max: 10, precision: 0.1 })
        }, options));

        // initialization -----------------------------------------------------

        _([0.5, 1, 1.5, 2, 2.5, 3, 4, 6]).each(function (size) {
            this.createListEntry(size, {
                label: gt.format(
                    //#. A single list entry of a GUI border width picker.
                    //#. Example result: "5 points"
                    //#. %1$d is the width of the border line.
                    //#, c-format
                    gt.ngettext('%1$d point', '%1$d points', size),
                    _.noI18n(size)
                ),
                css: { textAlign: 'center' }
            });
        }, this);

    }}); // class TableBorderWidthChooser

    // class CellBorderWidthChooser ==========================================

    EditControls.CellBorderWidthChooser = ComboField.extend({ constructor: function (options) {

        // base constructor ---------------------------------------------------

        ComboField.call(this, Utils.extendOptions({
            width: 40,
            tooltip: gt('Border width'),
            css: { textAlign: 'right' },
            keyboard: 'number',
            validator: new ComboField.NumberValidator({ min: 0.5, max: 3, precision: 0.1 })
        }, options));

        // initialization -----------------------------------------------------

        _([0.5, 1, 2, 3]).each(function (size) {
            this.createListEntry(size, {
                label: gt.format(
                    //#. A single list entry of a GUI border width picker.
                    //#. Example result: "5 points"
                    //#. %1$d is the width of the border line.
                    //#, c-format
                    gt.ngettext('%1$d point', '%1$d points', size),
                    _.noI18n(size)
                ),
                css: { textAlign: 'center' }
            });
        }, this);

    }}); // class CellBorderWidthChooser

    // class TableSizeChooser =================================================

    /**
     * A drop-down button and a drop-down menu containing a resizable grid
     * allowing to select a specific size.
     *
     * @constructor
     *
     * @extends Group
     * @extends DropDown
     *
     * @param {Object} [options]
     *  A map with options controlling the appearance of the control.
     *  @param {Integer} [options.maxCols=15]
     *      The maximal number of columns to be displayed in the control.
     *  @param {Integer} [options.maxRows=15]
     *      The maximal number of rows to be displayed in the control.
     */
    EditControls.TableSizeChooser = Group.extend({ constructor: function (app, options) {

        var // self referemce
            self = this,

            // minimum size allowed to choose
            MIN_SIZE = { width: 1, height: 1 },
            // maximum size allowed to choose (defaulting to 15 columns and rows)
            MAX_SIZE = { width: Utils.getIntegerOption(options, 'maxCols', 15, 1), height: Utils.getIntegerOption(options, 'maxRows', 15, 1) },
            // minimum size visible even if selected size is smaller
            MIN_VIEW_SIZE = { width: Math.min(MAX_SIZE.width, 6), height: Math.min(MAX_SIZE.height, 4) },

            // current grid size
            gridSize = { width: 0, height: 0 },

            // the button control to be inserted into the drop-down view component
            gridButton = Utils.createButton().addClass(Utils.FOCUSABLE_CLASS),

            // the badge label showing the current grid size
            sizeLabel = $('<span>').addClass('size-badge'),

            // whether to grow to the left or right or to the top or bottom
            growTop = false, growLeft = false,

            // last applied page position
            lastPagePos = null;

        // base constructors --------------------------------------------------

        Group.call(this, { tooltip: gt('Insert table') });
        DropDown.call(this, { icon: 'docs-table-insert' });

        // private methods ----------------------------------------------------

        /**
         * Sets the position and growing directions of the menu node according
         * to the available space around the group node.
         */
        function initializeMenuNodePosition() {

            var // position and size of the group node
                groupPosition = Utils.getNodePositionInPage(self.getNode()),
                // sizes available for the drop-down menu per group side
                availableSizes = self.getAvailableMenuSizes(),
                // decide where to grow the grid to
                newGrowTop = (availableSizes.bottom.height < 200) && (availableSizes.top.height > availableSizes.bottom.height),
                newGrowLeft = availableSizes.left.width > availableSizes.right.width;

            // rerender the grid (position of selected cells) if orientation has changed
            if ((growTop !== newGrowTop) || (growLeft !== newGrowLeft)) {
                growTop = newGrowTop;
                growLeft = newGrowLeft;
                setGridSize(gridSize);
            }

            // initialize grid position
            self.getMenuNode().toggleClass('grow-top', growTop).toggleClass('grow-left', growLeft);
            self.setMenuNodePosition({
                top: growTop ? null : (groupPosition.top + groupPosition.height + DropDown.GROUP_BORDER_PADDING),
                bottom: growTop ? (groupPosition.bottom + groupPosition.height + DropDown.GROUP_BORDER_PADDING) : null,
                left: growLeft ? null : groupPosition.left,
                right: growLeft ? groupPosition.right : null
            });
        }

        /**
         * Returns the current size of the grid, as object with 'width' and
         * 'height' attributes.
         */
        function getViewGridSize() {
            var rows = gridButton.children('div');
            return { width: rows.first().children('div').length, height: rows.length };
        }

        /**
         * Changes the current size of the grid, and updates the badge labels.
         */
        function setGridSize(newSize) {

            var // new total size to be shown in the view grid
                newViewSize = {},
                // HTML mark-up of the grid cells
                rowCellsMarkup = '', gridMarkup = '';

            // validate passed size
            gridSize.width = Utils.minMax(newSize.width, MIN_SIZE.width, MAX_SIZE.width);
            gridSize.height = Utils.minMax(newSize.height, MIN_SIZE.height, MAX_SIZE.height);
            newViewSize.width = Math.max(gridSize.width, MIN_VIEW_SIZE.width);
            newViewSize.height = Math.max(gridSize.height, MIN_VIEW_SIZE.height);

            // generate mark-up for all cells in one row
            _(newViewSize.width).times(function (col) {
                var selected = growLeft ? (col >= newViewSize.width - gridSize.width) : (col < gridSize.width);
                rowCellsMarkup += '<div data-width="' + (growLeft ? (newViewSize.width - col) : (col + 1)) + '"';
                if (selected) { rowCellsMarkup += ' class="selected"'; }
                rowCellsMarkup += '></div>';
            });

            // generate markup for all rows
            _(newViewSize.height).times(function (row) {
                var selected = growTop ? (row >= newViewSize.height - gridSize.height) : (row < gridSize.height);
                gridMarkup += '<div data-height="' + (growTop ? (newViewSize.height - row) : (row + 1)) + '"';
                if (selected) { gridMarkup += ' class="selected"'; }
                gridMarkup += '>' + rowCellsMarkup + '</div>';
            });
            gridButton[0].innerHTML = gridMarkup;

            // update badge label
            sizeLabel.text(_.noI18n(gt.format(
                //#. %1$d is the number of columns in the drop-down grid of a table size picker control.
                //#, c-format
                gt.ngettext('%1$d column', '%1$d columns', gridSize.width),
                _.noI18n(gridSize.width)
            ) + ' \xd7 ' + gt.format(
                //#. %1$d is the number of rows in the drop-down grid of a table size picker control.
                //#, c-format
                gt.ngettext('%1$d row', '%1$d rows', gridSize.height),
                _.noI18n(gridSize.height)
            )));
        }

        /**
         * Updates the grid size according to the passed screen coordinates.
         */
        function updateGridSizeForPosition(pageX, pageY) {

            var // current size of the view grid
                viewSize = getViewGridSize(),
                // width and height of one cell
                cellWidth = gridButton.outerWidth() / viewSize.width,
                cellHeight = gridButton.outerHeight() / viewSize.height,
                // position/size of the grid button node
                buttonDim = Utils.extendOptions(gridButton.offset(), { width: gridButton.outerWidth(), height : gridButton.outerHeight() }),
                // mouse position relative to grid origin
                mouseX = growLeft ? (buttonDim.left + buttonDim.width - pageX) : (pageX - buttonDim.left),
                mouseY = growTop ? (buttonDim.top + buttonDim.height - pageY) : (pageY - buttonDim.top);

            // Calculate new grid size. Enlarge width/height of the grid area, if
            // the last column/row is covered more than 80% of its width/height.
            setGridSize({
                width: (cellWidth > 0) ? Math.floor(mouseX / cellWidth + 1.2) : 1,
                height: (cellHeight > 0) ? Math.floor(mouseY / cellHeight + 1.2) : 1
            });

            // store position, needed to refresh grid if its position changes from outside
            lastPagePos = { x: pageX, y: pageY };

            return false;
        }

        /**
         * Handles 'menu:open' events and initializes the drop-down grid.
         * Registers a 'mouseenter' handler at the drop-down menu that starts
         * a 'mousemove' listener when the mouse first hovers the grid element.
         */
        function menuOpenHandler(event) {

            // unbind all running global event handlers
            unbindGlobalEventHandlers();

            // initialize grid
            lastPagePos = null;
            setGridSize(MIN_SIZE);
            initializeMenuNodePosition();

            // wait for mouse to enter the grid before listening globally to mousemove events
            gridButton.off('mouseenter').one('mouseenter', function (event) {
                $(document).on('mousemove', gridMouseMoveHandler);
            });

            // wait for touchstart event before listening globally to other touch events
            gridButton.off('touchstart').one('touchstart', function (event) {
                $(document).on('touchstart touchmove touchend touchcancel', gridTouchHandler);
                return false;
            });
        }

        /**
         * Handles 'menu:close' events.
         */
        function menuCloseHandler() {
            unbindGlobalEventHandlers();
            lastPagePos = null;
        }

        /**
         * Handles 'menu:refresh' events.
         */
        function menuRefreshHandler() {
            if (lastPagePos) {
                updateGridSizeForPosition(lastPagePos.x, lastPagePos.y);
            } else {
                setGridSize(gridSize);
            }
            initializeMenuNodePosition();
        }

        /**
         * Unbind the global event listener registered at the document.
         */
        function unbindGlobalEventHandlers() {
            $(document)
                .off('mousemove', gridMouseMoveHandler)
                .off('touchstart touchmove touchend touchcancel', gridTouchHandler);
        }

        /**
         * Handles 'mousemove' events in the open drop-down menu element.
         */
        function gridMouseMoveHandler(event) {
            updateGridSizeForPosition(event.pageX, event.pageY);
            return false;
        }

        /**
         * Handles touch events while the drop-down menu element is open.
         */
        function gridTouchHandler(event) {

            var // the touch list from the original event object
                touches = event.originalEvent.touches;

            // update grid size as long as a single touch point is moved around
            if ((touches.length === 1) && (event.type === 'touchmove')) {
                updateGridSizeForPosition(touches[0].pageX, touches[0].pageY);
            } else if ((touches.length === 0) && (event.type === 'touchend')) {
                // simulate a mouse click for releasing a single touch point
                gridButton.click();
            } else {
                // hiding the drop-down menu unbinds all touch event handlers
                self.triggerCancel();
            }

            return false;
        }

        /**
         * Handles keyboard events in the open drop-down menu element.
         */
        function gridKeyHandler(event) {

            var // distinguish between event types (ignore keypress events)
                keydown = event.type === 'keydown',
                keyup = event.type === 'keyup',
                // new size of the grid
                newGridSize = _.clone(gridSize);

            switch (event.keyCode) {
            case KeyCodes.LEFT_ARROW:
                if (keydown) {
                    newGridSize.width += (growLeft ? 1 : -1);
                    setGridSize(newGridSize);
                }
                return false;
            case KeyCodes.UP_ARROW:
                if (keydown) {
                    newGridSize.height += (growTop ? 1 : -1);
                    setGridSize(newGridSize);
                }
                return false;
            case KeyCodes.RIGHT_ARROW:
                if (keydown) {
                    newGridSize.width += (growLeft ? -1 : 1);
                    setGridSize(newGridSize);
                }
                return false;
            case KeyCodes.DOWN_ARROW:
                if (keydown) {
                    newGridSize.height += (growTop ? -1 : 1);
                    setGridSize(newGridSize);
                }
                return false;
            case KeyCodes.TAB:
                if (keydown && !event.altKey && !event.ctrlKey && !event.metaKey) {
                    self.triggerChange(event.target, { preserveFocus: true });
                }
                break;
            case KeyCodes.SPACE:
            case KeyCodes.ENTER:
                // Bug 28528: ENTER key must be handled explicitly, <a> elements
                // without 'href' attribute do not trigger click events. The 'href'
                // attribute has been removed from the buttons to prevent useless
                // tooltips with the link address.
                if (keyup) {
                    self.triggerChange(event.target, { preserveFocus: event.keyCode === KeyCodes.SPACE });
                    self.hideMenu();
                }
                return false;
            }
        }

        // initialization -----------------------------------------------------

        // initialize the button control and embed it in the drop-down menu
        gridButton.append($('<div>').append($('<div>')));
        this.getMenuNode().addClass('table-size-chooser').append(gridButton, sizeLabel);

        // register event handlers
        this.registerChangeHandler('click', { node: gridButton, valueResolver: function () { return gridSize; } })
            .on({ 'menu:open': menuOpenHandler, 'menu:close': menuCloseHandler, 'menu:refresh': menuRefreshHandler });
        gridButton.on('keydown keypress keyup', gridKeyHandler);

    }}); // class TableSizeChooser

    // class AcquireEditButton ================================================

    /**
     * Standard button for the 'Acquire edit rights' action.
     */
    EditControls.AcquireEditButton = Button.extend({ constructor: function (app, options) {

        var // the node containing the pulsing highlight color
            pulseNode = $('<div>').addClass('abs pulse'),
            // the pulse timer while the button is enabled
            timer = null;

        // base constructor ---------------------------------------------------

        Button.call(this, Utils.extendOptions({
            icon: 'icon-pencil',
            tooltip: gt('Acquire edit rights'),
            visibility: 'readonly'
        }, options));

        // private methods ----------------------------------------------------

        function startPulse() {
            timer = app.repeatDelayed(function () {
                pulseNode.toggleClass('active');
            }, { delay: 1000 });
        }

        function stopPulse() {
            timer.abort();
            timer = null;
        }

        // initialization -----------------------------------------------------

        this.getNode().addClass('acquire-edit');
        this.getButtonNode().append(pulseNode);
        this.on('group:enable', function (event, enabled) {
            if (enabled && !timer) {
                startPulse();
            } else if (!enabled && timer) {
                stopPulse();
            }
        });
        startPulse();

    }}); // class AcquireEditButton

    // class AccessRightsChooser ==============================================

    EditControls.AccessRightsChooser = RadioList.extend({ constructor: function (app, options) {

        // base constructor ---------------------------------------------------

        RadioList.call(this, Utils.extendOptions(options, {
            label: _.noI18n('Access Rights'),
            tooltip: _.noI18n('Access rights'),
            sorted: true
        }));

        // initialization -----------------------------------------------------

        _(ACCESS_RIGHTS_DEFINITIONS).each(function (definition) {
            this.createOptionButton(definition.mail, { label: _.noI18n(definition.name), tooltip: _.noI18n(definition.mail) });
        }, this);

    }}); // class AccessRightsChooser

    // exports ================================================================

    return EditControls;

});
