/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/operationsgenerator',
    ['io.ox/office/tk/utils',
     'io.ox/office/editframework/model/format/stylesheets'
    ], function (Utils, StyleSheets) {

    'use strict';

    // class OperationsGenerator ==============================================

    /**
     * An instance of this class contains an array of operations and provides
     * methods to generate new operations.
     *
     * @constructor
     *
     * @param {DocumentStyles} documentStyles
     *  Global collection with the style sheet containers and custom formatting
     *  containers of a document.
     */
    function OperationsGenerator(documentStyles) {

        var // the operations buffer
            operations = [];

        // methods ------------------------------------------------------------

        /**
         * Returns the array with all operations that have been generated so
         * far. Does not remove the operations from this generator instance.
         *
         * @returns {Array}
         *  The array with all generated operations.
         */
        this.getOperations = function () {
            return operations;
        };

        /**
         * Reverses the entire operations array.
         *
         * @returns {OperationsGenerator}
         *  A reference to this instance.
         */
        this.reverseOperations = function () {
            operations.reverse();
            return this;
        };

        /**
         * Creates and appends a new operation to the operations array.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Object} [operationOptions]
         *  Additional options that will be stored in the operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateOperation = function (name, operationOptions) {
            var operation = _.extend({ name: name }, operationOptions);
            operations.push(operation);
            return operation;
        };

        /**
         * Creates and appends a new operation to the operations array. Adds
         * explicit attributes of the passed node to the 'attrs' option of the
         * new operation.
         *
         * @param {HTMLElement|jQuery} element
         *  The element node that may contain explicit formatting attributes.
         *  If this object is a jQuery collection, uses the first node it
         *  contains.
         *
         * @param {String} name
         *  The name of the operation.
         *
         * @param {Object} [operationOptions]
         *  Additional options that will be stored in the operation.
         *
         * @returns {Object}
         *  The created operation object.
         */
        this.generateOperationWithAttributes = function (element, name, operationOptions) {

            var // explicit attributes of the passed node
                attributes = StyleSheets.getExplicitAttributes(element);

            // add the 'attrs' entry if there are attributes
            if (!_.isEmpty(attributes)) {
                operationOptions = _.extend({ attrs: attributes }, operationOptions);
            }

            // push the operation
            return this.generateOperation(name, operationOptions);
        };

        /**
         * Generates the 'setAttributes' operation needed to set the explicit
         * formatting attributes of the passed element node. If the passed node
         * does not contain any explicit attributes, no operation will be
         * generated.
         *
         * @param {HTMLElement|jQuery} element
         *  The element node whose formatting attributes will be converted to
         *  an operation. If this object is a jQuery collection, uses the first
         *  node it contains.
         *
         * @param {Object} position
         *  All operation attributes describing the logical position of the
         *  passed element in the document model. All attributes will be added
         *  to the generated operation.
         *
         * @param {Object} [options]
         *  A map with options controlling the operation generation process.
         *  Supports the following options:
         *  @param {String} [options.clearFamily]
         *      If specified, a style family for which additional formatting
         *      attributes with null values will be inserted into the
         *      'setAttributes' operation.
         *
         * @returns {OperationsGenerator}
         *  A reference to this instance.
         */
        this.generateSetAttributesOperation = function (element, position, options) {

            var // explicit attributes of the passed node
                elementAttributes = StyleSheets.getExplicitAttributes(element),
                // the operation options
                operationOptions = _.clone(position),
                // the style families for generated null attributes
                clearFamily = Utils.getStringOption(options, 'clearFamily', '');

            // insert null values for all attributes registered for the specified style family
            if (clearFamily.length > 0) {
                operationOptions.attrs = documentStyles.buildNullAttributes(clearFamily, { supportedFamilies: true });
            } else {
                operationOptions.attrs = {};
            }

            // merge the explicit attributes of the passed element
            documentStyles.extendAttributes(operationOptions.attrs, elementAttributes);

            // no attributes, no operation
            if (!_.isEmpty(operationOptions.attrs)) {
                this.generateOperation(OperationsGenerator.SET_ATTRIBUTES, operationOptions);
            }

            return this;
        };

    } // class OperationsGenerator

    // constants --------------------------------------------------------------

    // define names of generic operations
    _(OperationsGenerator).extend({

        /**
         * Name of the initial operation while importing a document, sets global
         * attributes of the model, including default formatting attributes.
         */
        SET_DOCUMENT_ATTRIBUTES: 'setDocumentAttributes',

        /**
         * Name of the operation that inserts a new theme into the theme container.
         */
        INSERT_THEME: 'insertTheme',

        /**
         * Name of the operation that inserts a new font description into the font
         * container.
         */
        INSERT_FONT_DESCRIPTION: 'insertFontDescription',

        /**
         * Name of the operation that sets attributes for a specific element or
         * range in the document model.
         */
        SET_ATTRIBUTES: 'setAttributes',

        /**
         * Name of the operation that inserts a new style sheet into the style
         * sheet container of a specific style family.
         */
        INSERT_STYLESHEET: 'insertStyleSheet',

        /**
         * Name of the operation that removes a style sheet from the style sheet
         * container of a specific style family.
         */
        DELETE_STYLESHEET: 'deleteStyleSheet',

        /**
         * Name of the operation that does nothing (internal use).
         */
        NOOP: 'noOp'

    });

    // exports ================================================================

    return _.makeExtendable(OperationsGenerator);

});
