/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2011 Open-Xchange Inc., Tarrytown, NY, USA. info@open-xchange.com
 *
 * @author Viktor Pracht <viktor.pracht@open-xchange.com>
 */

var fs = require("fs");
var path = require("path");
var child_process = require("child_process");
var _ = require("../underscore");

/**
 * Tests for a boolean Jake parameter.
 * Any of the following is interpreted as true: on, yes, true, 1
 */
exports.envBoolean = function (name) {
    return /^\s*(?:on|yes|true|1)/i.test(process.env[name]);
};

/**
 * Default destination directory.
 * @type String
 */
exports.builddir = process.env.builddir || "build";

/**
 * Resolves a filename relative to the build directory.
 * @param {String} name The filename to resolve
 * @type String
 * @return The filename in the build directory.
 */
exports.dest = function(name) { return path.join(exports.builddir, name); };

/**
 * Resolves a filename relative to the source directory.
 * When building an external app this is not the same as the current directory.
 * @param {String} name The filename to resolve
 * @type String
 * @return The filename in the source directory.
 */
exports.source = function(name) {
    if (process.env.BASEDIR) return process.env.BASEDIR + "/" + name;
    return name;
};

var hasJakefile = path.existsSync(exports.source('Jakefile.js'));

/**
 * Number of generated files.
 * @type Number
 */
var counter = 0;

exports.startTime = new Date;

function FileType() {}

/**
 * Returns all applicable hooks of a specific hook type.
 * @param {String} type The hook type to return.
 * @param {Function or Array} prepend An optional hook or array of hooks to
 * prepend before all other hooks.
 * @type Array
 * @returns An array of hooks.
 */
FileType.prototype.getHooks = function(type, prepend) {
    return [].concat(prepend || [], this[type] || [], types["*"][type] || []);
};

/**
 * Adds a hook to this file type.
 * @param {String} type The type of the added hook.
 * @param {Function} hook The hook function to add.
 * @type FileType
 * @return Itself, for chaining.
 */
FileType.prototype.addHook = function(type, hook) {
    var hooks = this[type] || (this[type] = []);
    hooks.push(hook);
    return this;
};

/**
 * Types of files which are processed with the same settings.
 * Each type consists of an array of handlers for dependency-building and
 * an array of filters to process the file contents. Optionally, other types
 * of hooks can be used by individual handlers and filters.
 * The special type "*" applies to all files and is applied after
 * the type-specific handlers and filters.
 */
var types = { "*": new FileType() };

exports.fileType = function(type) {
    return types[type] || (types[type] = new FileType());
};

/**
 * The name of the current top level task, if any.
 */
var topLevelTaskName = null;

/**
 * Defines a new top-level task.
 * Any subsequent file utility functions will add their target files to this
 * task as dependencies.
 * @param {String} name An optional name of the new task. If not specified,
 * no new task is created and automatic dependencies won't be created anymore.
 * All parameters are passed unmodified to the task() function.
 */
exports.topLevelTask = function(name) {
    topLevelTaskName = name;
    if (name) return task.apply(this, arguments);
};

exports.fileType("*").addHook("handler", function(filename) {
    if (topLevelTaskName) task(topLevelTaskName, [filename]);
});

/**
 * Callback for top-level tasks to report the number of generated files and the
 * build time.
 */
exports.summary = function(name) {
    return function() {
        var ms = (new Date).getTime() - exports.startTime.getTime();
        console.log("Generated " + counter + (counter == 1 ? " file" : " files")
            + " in " + (ms / 1000).toFixed(3) + "s by " + name);
    };
};

/**
 * Copies one or more files.
 * Any missing directories are created automatically.
 * @param {Array} files An array of strings specifying filenames to copy.
 * @param {String} files.dir An optional common parent directory. All filenames
 * in files are relative to it. Defaults to the project root.
 * @param {Object} options An optional object containing various options.
 * @param {String} options.to An optional target directory. The target
 * filenames are generated by resolving each filename from files relative to
 * options.to instead of files.dir. Defaults to the build directory.
 * @param {Function} options.filter An optional filter function which takes
 * the contents of a file as parameter and returns the filtered contents.
 * @param {Function} options.mapper An optional file name mapper.
 * It's a function which takes the original target file name (as computed by
 * files.dir and options.to) as parameter and returns the mapped file name.
 * @param {Number} options.mode Optional file permissions as UNIX file mode.
 * If present and non-zero, chmod is used to set the file mode of each target.
 */
exports.copy = function(files, options) {
    var srcDir = files.dir || "";
    var destDir = options && options.to || exports.builddir;
    var mapper = options && options.mapper || _.identity;
    for (var i = 0; i < files.length; i++) {
        exports.copyFile(path.join(srcDir, files[i]),
                         mapper(path.join(destDir, files[i])), options);
    }
};

/**
 * Returns a combined handler and a combined filter function for a combination
 * of filename and options.
 * @param {String} filename The name of the target file.
 * @param {Object} options An optional object with options for copy or concat.
 * @param {Function} options.filter An optional filter function which takes
 * the contents of a file as parameter and returns the filtered contents.
 * @param {String} options.type An optional file type. Defaults to the file
 * extension of the destination.
 * @type Object
 * @returns An object with two methods: handler and filter.
 * handler should be called to generate dependencies. If filter is not null,
 * It should be called with the contents of the file as a string parameter.
 * It will then return the filtered file contents as a string.
 */
function getType(filename, options) {
    if (!options) options = {};
    var type = exports.fileType(options.type || path.extname(filename));
    var handlers = type.getHooks("handler");
    var filters = type.getHooks("filter", options.filter);
    return {
        handler: function(filename) {
            var obj = { type: type };
            for (var i = 0; i < handlers.length; i++) {
                handlers[i].call(obj, filename);
            }
        },
        filter: filters.length ? function(task, data, getSrc) {
            var obj = { type: type, task: task, getSrc: getSrc };
            for (var i = 0; i < filters.length; i++) {
                data = filters[i].call(obj, data);
            }
            return data;
        } : null
    };
}

/**
 * Wrapper around Jake's file() function.
 * @param {String} dest The destination file name.
 * @param {Array} deps An array with dependency names.
 * @param {Function} callback The callback parameter for file().
 * @param {Object} options An optional object with options for file().
 * @param {String} type An optional file type. Defaults to the file
 * extension of the destination.
 */
exports.file = function(dest, deps, callback, options, type) {
    if (typeof options !== "object") {
        type = options;
        options = {};
    }
    dest = dest.replace(/\\/g, '/');
    var dir = path.dirname(dest);
    directory(dir);
    file(dest, deps, function() {
        callback.apply(this, arguments);
        counter++;
    }, options);
    if (hasJakefile) {
        file(dest, [dir, exports.source('Jakefile.js')]);
    } else {
        file(dest, [dir]);
    }
    var obj = { type: exports.fileType(type || path.extname(dest)) };
    var handlers = obj.type.getHooks("handler");
    for (var i = 0; i < handlers.length; i++) handlers[i].call(obj, dest);
};

/**
 * Copies a single file.
 * Any missing directories are created automatically.
 * @param {String} src The filename of the source file.
 * @param {String} dest The filename of the target file.
 * @param {Object} options An optional object containing various options.
 * @param {Function} options.filter An optional filter function which takes
 * the contents of a file as parameter and returns the filtered contents.
 * @param {String} options.type An optional file type. Defaults to the file
 * extension of the destination.
 * @param {Number} options.mode Optional file permissions as UNIX file mode.
 * If present and non-zero, chmod is used to set the file mode of the target.
 */
exports.copyFile = function(src, dest, options) {
    var type = getType(dest, options);
    var callback = type.filter ?
        function() {
            fs.writeFileSync(dest,
                type.filter(this, fs.readFileSync(src, "utf8"),
                    function(line) { return { name: src, line: line }; }));
            if (options && options.mode !== undefined) {
                fs.chmodSync(dest, options.mode);
            }
        } : function() {
            var data = fs.readFileSync(src);
            fs.writeFileSync(dest, data, 0, data.length, null);
            if (options && options.mode !== undefined) {
                fs.chmodSync(dest, options.mode);
            }
        };
    exports.file(dest, [src], callback, options && options.type);
};

/**
 * Helper function for concat and merge.
 * @private
 * @ignore
 * @param merge The part doing the actual merging of files. Its parameters are
 * this, the type from getType, the array of strings with contents of input
 * files and the files and options parameters to merge/concat.
 * @returns {Function} The merge or concat function for the API.
 */
function makeMerge(merge) {
    return function (name, files, options) {
        var srcDir = files.dir || "";
        var dest = path.join(options && options.to || exports.builddir, name);
        var destDir = path.dirname(dest);
        var deps = [];
        var type = getType(dest, options);
        for (var i = 0; i < files.length; i++) {
            if (typeof files[i] == "string") deps.push(path.join(srcDir, files[i]));
        }
        deps.push(destDir);
        deps.push(exports.source("Jakefile.js"));
        directory(destDir);
        file(dest, deps, function() {
            var data = [];
            for (var i = 0; i < files.length; i++) {
                var contents = typeof files[i] == "string" ?
                    fs.readFileSync(path.join(srcDir, files[i]), "utf8") :
                    files[i].getData();
                var last = contents.charAt(contents.length - 1);
                if (last != "\r" && last != "\n") contents += "\n";
                data.push(contents);
            }
            fs.writeFileSync(dest, merge(this, type, data, files, options));
            counter++;
        });
        type.handler(dest);
    };
}

/**
 * Concatenates one or more files and strings to a single file.
 * Any missing directories are created automatically.
 * @param {String} name The name of the destination file relative to the build
 * directory.
 * @param {Array} files An array of things to concatenate.
 * Plain strings are interpreted as filenames relative to files.dir,
 * objects having a method getData should return the contents as a string.
 * @param {String} files.dir An optional common parent directory. All filenames
 * in files are relative to it. Defaults to the project root.
 * @param {Object} options An optional object containing various options.
 * @param {String} options.to An optional target directory. The target
 * filenames are generated by resolving each filename from files relative to
 * options.to instead of files.dir. Defaults to the build directory.
 * @param {Function} options.filter An optional filter function which takes
 * the concatenated contents as parameter and returns the filtered contents.
 * @param {String} options.type An optional file type. Defaults to the file
 * extension of the destination.
 */
exports.concat = makeMerge(function (self, type, data, files) {
    var srcDir = files.dir || "";
    return type.filter ? type.filter(self, data.join(''), getSrc)
                       : data.join('');
    function getSrc(line) {
        var defs = fileDefs();
        var def = defs[_.sortedIndex(defs, line, getStart) - 1];
        function getStart(x) {
            return typeof x == 'number' ? x : x.start;
        }
        return { name: def.name, line: line - def.start };
    }
    function fileDefs() {
        if (fileDefs.value) return fileDefs.value;
        fileDefs.value = [];
        var start = 0;
        for (var i = 0; i < data.length; i++) {
            fileDefs.value.push({
                name: typeof files[i] !== 'string' ? '' :
                    path.join(srcDir, files[i]),
                start: start
            });
            start += data[i].split(/\r?\n|\r/g).length - 1;
        }
        return fileDefs.value;
    }
});

/**
 * Converts a string to a pseudo-file for use by concat().
 * @param {String} s The string which should be inserted.
 * @type Object
 * @return An object which can be used as an element of the second parameter to
 * concat(). It has one method: getData(), which returns the string s.
 */
exports.string = function(s) { return { getData: function() { return s; } }; };

/**
 * Merges one or more files and strings to a single file using a custom merge
 * function. The filters are applied to the contents of the input files
 * individually.
 * Any missing directories are created automatically.
 * @param {String} name The name of the destination file relative to the build
 * directory.
 * @param {Array} files An array of things to merge.
 * Plain strings are interpreted as filenames relative to files.dir,
 * objects having a method getData should return the contents as a string.
 * @param {String} files.dir An optional common parent directory. All filenames
 * in files are relative to it. Defaults to the project root.
 * @param {Object} options An optional object containing various options.
 * @param {String} options.to An optional target directory. The target
 * filenames are generated by resolving each filename from files relative to
 * options.to instead of files.dir. Defaults to the build directory.
 * @param {Function} options.filter An optional filter function which takes
 * the contents of each input file as parameter and returns the filtered
 * contents.
 * @param {String} options.type An optional file type. Defaults to the file
 * extension of the destination.
 * @param {Function} options.merge The merge function which receives
 * the filtered contents of all input files as an array of strings and
 * the files parameter as parameters and should return the merged contents as
 * a single string.
 */
exports.merge = makeMerge(function (self, type, data, files, options) {
    if (type.filter) {
        data = _.map(data, function (input, i) {
            return type.filter(self, input, getSrc);
            function getSrc(line) {
                return {
                    name: typeof files[i] === 'string' ? files[i] : '',
                    line: line
                };
            }
        });
    }
    return options.merge(data, files);
});

/**
 * Returns a list of filenames specified by a root directory and one or more
 * glob patterns.
 * @param {String} dir Optional root directory. Defaults to the project root.
 * @param {String or Array of String} globs One or more glob patterns.
 * @type Array of String
 * @returns An array of file names relative to dir, which match the specified
 * patterns.
 * The property dir is set to the parameter dir for use with copy and concat.
 */
exports.list = function(dir, globs) {
    if (globs === undefined) {
        globs = dir;
        dir = "";
    }
    if (typeof globs == "string") globs = [globs];
    globs = _.chain(globs).map(function (s) {
        return path.join(dir, s.replace(/\/$/, '/**/*'));
    }).filter(function (s) {
        return path.existsSync(jake.basedir(s));
    }).value();
    var retval = new jake.FileList(globs);
    retval.exclude(function(name) {
        try {
            return fs.statSync(name).isDirectory();
        } catch (e) {
            return true;
        }
    });
    retval = retval.toArray();
    retval = _.map(retval, function (s) { return path.relative(dir, s); });
    retval.dir = dir;
    return retval;
};

/**
 * Asynchronously executes an external command.
 * stdin, stdout and stderr are passed through to the parent process.
 * @param {Array} The command to execute and its arguments.
 * @param {Object} options Options for child_process.spawn.
 * @param {Function} callback A callback which is called when the command
 * returns.
 */
exports.exec = function(command, options, callback) {
    if (!callback) {
        callback = options;
        options = undefined;
    };
    var child = child_process.spawn("/usr/bin/env", command, options);
    child.stdout.on("data", function(data) { process.stdout.write(data); });
    child.stderr.on("data", function(data) { process.stderr.write(data); });
    child.on("exit", callback);
};

exports.gzip = function(src, dest, callback) {
    var child = child_process.spawn("gzip", ["-nc", src]);
    child.stdout.pipe(fs.createWriteStream(dest));
    child.on("exit", callback);
};

/**
 * Merges two sorted arrays based on an optional comparison function
 * (like Array.prototype.sort).
 * @param {Array} a The first array.
 * @param {Array} b The second array.
 * @param {Function} cmp An optional comparison function like in Array.sort().
 * @type Array
 * @return A sorted array with elements from a and b, except for duplicates
 * from b. All entries from a are included.
 */
exports.mergeArrays = function(a, b, cmp) {
    if (!cmp) cmp = function(x, y) { return x < y ? -1 : x > y ? 1 : 0; };
    var c = Array(a.length + b.length);
    var ai = 0, bi = 0, ci = 0;
    while (ai < a.length && bi < b.length) {
        var diff = cmp(a[ai], b[bi]);
        c[ci++] = diff > 0 ? b[bi++] : a[ai++];
        if (!diff) bi++;
    }
    while (ai < a.length) c[ci++] = a[ai++];
    while (bi < b.length) c[ci++] = b[bi++];
    c.length = ci;
    return c;
};

var includes = {};
var includesFile;

exports.includes = {

    /**
     * Specifies a file which stores include information between builds, and
     * loads it if it already exists.
     * @param {String} filename The file which stores include information
     * between builds.
     */
    load: function(filename) {
        includesFile = filename;
        if (path.existsSync(filename)) {
            includes = JSON.parse(fs.readFileSync(filename, "utf8"));
            for (var target in includes) {
                var inc = includes[target];
                file(target, inc.list);
                if (inc.type) {
                    getType(target, { type: inc.type }).handler(target);
                }
            }
        }
    },

    /**
     * Specifies which includes were found in a source file.
     * @param {String} file The target file which contains the results of
     * the inclusion.
     * @param {Array} includedFiles An array with names of included files.
     * @param {String} type An optional file type which can be used to
     * handle loaded files.
     */
    set: function(file, includedFiles, type) {
        includes[file] = { list: includedFiles };
        if (type) includes[file].type = type;
    },

    /**
     * Adds an include found in a source file.
     * @param {String} file The target file which contains the results of
     * the inclusion.
     * @param {String} include Name of the included file.
     */
    add: function(file, include) {
        if (!(file in includes)) includes[file] = { list: [] };
        includes[file].list.push(include);
    },

    /**
     * Saves the list of inlcudes to the file previously specified by
     * includes.load.
     */
    save: function() {
        for (var i in includes) {
            if (!includes[i].list.length) delete includes[i];
        }
        fs.writeFileSync(includesFile, JSON.stringify(includes));
    }

};

/**
 * A filter which processes //@include directives and takes care of dependencies
 */
exports.includeFilter = function (data) {
    var dest = this.task.name;
    exports.includes.set(dest, []);
    var self = this, line = 1;
    return data.replace(/(\/\/@include\s+(.*?)(\S*)(;?))?\r?\n/g,
        function(m, include, prefix, name, semicolon) {
            if (!include) {
                line++;
                return m;
            }
            var dir = path.dirname(self.getSrc(line).name);
            return (prefix || "") + exports.list(dir, name).map(function(file) {
                var include = path.join(dir, file);
                exports.includes.add(dest, include);
                return fs.readFileSync(include, "utf8");
            }).join("\n") + (semicolon + "\n");
        });
};
