/* vim:ts=4:sts=4:sw=4:
 * ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Ajax.org Code Editor (ACE).
 *
 * The Initial Developer of the Original Code is
 * Ajax.org B.V.
 * Portions created by the Initial Developer are Copyright (C) 2010
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *      Fabian Jakobs <fabian AT ajax DOT org>
 *      Mihai Sucan <mihai DOT sucan AT gmail DOT com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

define('ace/undomanager', function(require, exports, module) {
"use strict";

/**
 * class UndoManager
 *
 * This object maintains the undo stack for an [[EditSession `EditSession`]].
 *
 **/

/**
 * new UndoManager()
 * 
 * Resets the current undo state and creates a new `UndoManager`.
 **/
var UndoManager = function() {
    this.reset();
};

(function() {

    /**
    * UndoManager.execute(options) -> Void
    * - options (Object): Contains additional properties
    *
    * Provides a means for implementing your own undo manager. `options` has one property, `args`, an [[Array `Array`]], with two elements:
    *
    * * `args[0]` is an array of deltas
    * * `args[1]` is the document to associate with
    *
    **/
    this.execute = function(options) {
        var deltas = options.args[0];
        this.$doc  = options.args[1];
        this.$undoStack.push(deltas);
        this.$redoStack = [];
    };

    /**
    * UndoManager.undo(dontSelect) -> Range
    * - dontSelect (Boolean): {:dontSelect}
    *
    * [Perform an undo operation on the document, reverting the last change. Returns the range of the undo.]{: #UndoManager.undo}
    **/
    this.undo = function(dontSelect) {
        var deltas = this.$undoStack.pop();
        var undoSelectionRange = null;
        if (deltas) {
            undoSelectionRange =
                this.$doc.undoChanges(deltas, dontSelect);
            this.$redoStack.push(deltas);
        }
        return undoSelectionRange;
    };

    /**
    * UndoManager.redo(dontSelect) -> Void
    * - dontSelect (Boolean): {:dontSelect}
    *
    * [Perform a redo operation on the document, reimplementing the last change.]{: #UndoManager.redo}
    **/
    this.redo = function(dontSelect) {
        var deltas = this.$redoStack.pop();
        var redoSelectionRange = null;
        if (deltas) {
            redoSelectionRange =
                this.$doc.redoChanges(deltas, dontSelect);
            this.$undoStack.push(deltas);
        }
        return redoSelectionRange;
    };

    /**
    * UndoManager.reset() -> Void
    *
    * Destroys the stack of undo and redo redo operations.
    **/
    this.reset = function() {
        this.$undoStack = [];
        this.$redoStack = [];
    };

    /**
    * UndoManager.hasUndo() -> Boolean
    *
    * Returns `true` if there are undo operations left to perform.
    **/
    this.hasUndo = function() {
        return this.$undoStack.length > 0;
    };

    /**
    * UndoManager.hasRedo() -> Boolean
    *
    * Returns `true` if there are redo operations left to perform.
    **/
    this.hasRedo = function() {
        return this.$redoStack.length > 0;
    };

}).call(UndoManager.prototype);

exports.UndoManager = UndoManager;
});
