/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.api.contenttypes;

import java.util.BitSet;
import com.openexchange.usm.api.datatypes.PIMAttachment;
import com.openexchange.usm.api.datatypes.PIMAttachments;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.Folder;

/**
 * This handler must be specified for each ContentType that is registered. It
 * provides all required functionality for the interaction (i.e. creation, deletion,
 * update of objects) with the OX server. It also provides a method to resolve
 * synchronization conflicts.
 *
 * @author afe
 *
 */
public interface ContentTypeTransferHandler {

    /**
     * This method is called to send a creation to the OX server.
     * The fields of the DataObject are populated with the current values.
     * This method must store the server ID for the DataObject as its current value.
     *
     * @param object contains the given change
     * @throws USMException if the change fails
     */
    void writeNewDataObject(DataObject object) throws USMException;

    /**
     * This method is called to send an update to the OX server.
     * The current values of the (modified) fields of the DataObject contain the changes.
     *
     * @param object contains the given change
     * @param timestamp timestamp of the updated object. If the object was modified after the specified timestamp, then the update must fail.
     * @throws USMException if the change fails
     */
    void writeUpdatedDataObject(DataObject object, long timestamp) throws USMException;

    /**
     * This method is called to send a deletion to the OX server.
     * From the fields of the DataObject for deletion only current id, folder and timestamp are
     * guaranteed to be present.
     *
     * @param object contains the given change
     * @throws USMException if the change fails
     */
    void writeDeletedDataObject(DataObject object) throws USMException;

    /**
     * This method is called to receive the current state from the OX server. All fields
     * of interest must be retrieved. The DataObject given as parameter is guaranteed to contain
     * the required fields (id, folder), but may also contain other (old) information.
     * This method should overwrite all specified fields (their current value) and must set
     * the current timestamp.
     *
     * @param object element to completely read from server
     * @param requestedFields BitSet containing the field indexes that are of interest to the client, all other field will be ignored by the client
     * @throws USMException if the read fails (e.g. object not found on server)
     */
    void readDataObject(DataObject object, BitSet requestedFields) throws USMException;

    /**
     * This method reads the current content of the provided Folder object from the OX server.
     * It reads the complete folder content from the OX server. All fields of interest as set in
     * the USM session referenced by the folder of interest must
     * be retrieved. It may use information from previous synchronizations (retrieved from the
     * associated Session) to reduce the amount of data that is read from the OX server.
     * The DataObjects returned should only contain CREATED-changes.
     *
     * @param folder Folder for which the (updated) content shall be retrieved
     * @return Complete current content of given Folder
     * @throws USMException if the read fails (e.g. Folder not found on server)
     */
    DataObject[] readFolderContent(Folder folder) throws USMException;

    /**
     * This method reads the current content of the provided Folder object from the OX server.
     * It reads all updated folder content and at least limit new objects from the OX server (or all if fewer).
     * All fields of interest as set in the USM session referenced by the folder of interest must
     * be retrieved. It may use information from previous synchronizations (retrieved from the
     * associated Session) to reduce the amount of data that is read from the OX server.
     * The DataObjects returned should only contain CREATED-changes.
     *
     * @param folder Folder for which the (updated) content shall be retrieved
     * @param limit max. number of new elements read
     * @return Current content of given Folder, with information if the read was incomplete or not
     * @throws USMException if the read fails (e.g. Folder not found on server)
     */
    OXFolderContent readFolderContent(Folder folder, int limit) throws USMException;

    /**
     * This method reads the current content of the provided Folder object from the OX server.
     * It reads the complete folder content from the OX server. All requested fields of interest
     * must be retrieved.
     * The DataObjects returned should only contain CREATED-changes.
     *
     * @param folder Folder for which the (updated) content shall be retrieved
     * @return Complete current content of given Folder
     * @throws USMException if the read fails (e.g. Folder not found on server)
     */
    DataObject[] readFolderContent(Folder folder, BitSet requestedFields) throws USMException;

    /**
     * Based on the current ContentSync timestamp of the provided Folder object, this method must
     * perform the following:
     * timestamp != 0: read all updated elements since the specified timestamp
     *      Read all updated(created/modified/deleted) elements for the given Folder.
     *      All fields of interest must be retrieved for creation/update. For deletion only
     *      the required fields folder, id, changestate have to be set. The DataObjects
     *      returned may contain any ChangeState (depending on the type of change). This
     *      method must correctly set the ChangeState of the returned DataObjects,
     *      unchanged DataObjects need not be returned.
     *
     * @param folder Folder for which the (updated) content shall be retrieved
     * @param requestedFields
     * @param timestamp - the timestamp for last content synchronization
     * @return All updated content of given Folder
     * @throws USMException if the read fails (e.g. Folder not found on server)
     */
    DataObject[] readUpdatedFolderContent(Folder folder, BitSet requestedFields, long timestamp) throws USMException;

    /**
     * Retrieves all attachments to a given object. The content of every attachment is not loaded (use getAttachmentData() to retrieve it,
     * when necessary)
     *
     * @param object
     * @return object containing all attachments and the last modified timestamp
     * @throws USMException
     */
    PIMAttachments getAllAttachments(DataObject object) throws USMException;

    /**
     * Creates new attachment to a given DataObject from this content type.
     *
     * @param object
     * @param attachment
     * @throws USMException
     * @return the timestamp of the new created attachment
     */
    long createNewAttachment(DataObject object, PIMAttachment attachment) throws USMException;

    /**
     * Deletes a given list of attachments from a given object.
     *
     * @param object the object this attachments are attached to
     * @param attachmentsToDelete list of attachments to delete
     * @return the timestamp of this modification
     * @throws USMException
     */
    long deleteAttachments(DataObject object, PIMAttachment... attachmentsToDelete) throws USMException;

    /**
     * Retrieves the raw byte data of the attachment.
     *
     * @param object the object this attachment is attached to
     * @param attachmentId the attachment id
     * @return the raw byte data of the attachment
     * @throws USMException
     */
    byte[] getAttachmentData(DataObject object, int attachmentId) throws USMException;

    /**
     * Retrieves the raw byte data of the attachment as InputStream. As long as this stream is open, no other OX calls can be made for the
     * session associated with the specified DataObject. It is required that the provided InputStream gets closed after usage to free any
     * associated resources.
     *
     * @param object DataObject to which the attachment is attached to
     * @param attachmentId attachment id
     * @param offset offset in bytes from the start of the attachment data, specifies where the reading should start
     * @param length maximum number of bytes that should be read, starting from offset. If offset + length > the filesize, fewer bytes may
     *            be read
     * @return ResourceInputStream that provides the data and information about the content-length for the given attachment
     * @throws USMException
     */
    ResourceInputStream getAttachmentDataStream(DataObject object, int attachmentId, long offset, long length) throws USMException;
}
