/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml;

import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;
import javax.xml.bind.JAXBException;
import org.apache.commons.logging.Log;
import org.docx4j.dml.Theme;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.io.SaveToZipFile;
import org.docx4j.openpackaging.packages.OpcPackage;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.ThemePart;
import org.json.JSONObject;
import com.openexchange.config.ConfigurationService;
import com.openexchange.log.LogFactory;
import com.openexchange.office.FilterException;
import com.openexchange.office.FilterException.ErrorCode;
import com.openexchange.office.tools.ResourceManager;
import com.openexchange.server.ServiceLookup;

abstract public class OperationDocument {

    private final static Log LOG = com.openexchange.log.Log.valueOf(LogFactory.getLog(OperationDocument.class));
    private final ResourceManager   resourceManager;
    private final InputStream       inputDocumentStream;
    private final String            userLanguage;
    private final ServiceLookup     services;
    protected final boolean         saveDebugOperations;

    static final int DEFAULT_BUFFER_SIZE = 8192;

    protected OperationDocument(ServiceLookup _services, InputStream _inputDocumentStream, ResourceManager _resourceManager, String _userLanguage) {
        services = _services;
        inputDocumentStream = _inputDocumentStream;
        resourceManager = _resourceManager;
        userLanguage = _userLanguage;
        saveDebugOperations = getConfigurationService().getBoolProperty("io.ox/office//module/debugoperations", false);
    }

    public InputStream createDocument(String applyOperations)
        throws FilterException {

        try {
            applyOperations(applyOperations);
            InputStream debugDocument = null;
            if(saveDebugOperations&&inputDocumentStream instanceof ByteArrayInputStream) {
                try {
                    inputDocumentStream.reset();
                    debugDocument = inputDocumentStream;
                }
                catch(IOException e) {
                    //
                }
            }
            return debugSave(applyOperations, debugDocument, resourceManager);
        }
        catch(Throwable e) {
            LOG.error("OOXML Filter could not save the given document:", e);
            throw (e instanceof FilterException) ? (FilterException)e : new FilterException(e, ErrorCode.CRITICAL_ERROR);
        }
    }

    public JSONObject createOperations()
        throws FilterException {

        try {
            return getOperations();
        }
        catch (Throwable e) {
            LOG.error("OOXML Filter could not receive operations:", e);
            throw (e instanceof FilterException) ? (FilterException)e : new FilterException(e, ErrorCode.CRITICAL_ERROR);
        }
    }

    abstract public void applyOperations(String applyOperations)
        throws FilterException;

    abstract protected JSONObject getOperations()
        throws Exception;

    public InputStream save() throws FilterException {
        return debugSave(null, null, null);
    }

    /* For debug purposes we create a folder (debug) within the destination document
     * that stores each information necessary to repeat the last save operation that was done.
     * (original document, the operations just applied and the content of the resource manager. */

    private InputStream debugSave(String debugOperations, java.io.InputStream debugDocument, ResourceManager debugResourceManager)
        throws FilterException {

        try {

            SaveToZipFile saver = new SaveToZipFile(getPackage());
            if(saveDebugOperations) {
                if(debugOperations!=null) {
                    getPackage().getContentTypeManager().addNonPersistContentFile("debug/operationUpdates.dbg",
                        debugOperations.getBytes("UTF-8"),"dbg", "application/octet-stream");
                }
                if(debugDocument!=null) {
                    ZipInputStream zipInputStream = new ZipInputStream(debugDocument);
                    ByteArrayOutputStream outStream = new ByteArrayOutputStream();
                    ZipOutputStream zipOutputStream = new ZipOutputStream(outStream);
                    ZipEntry zipEntry = null;
                    byte[] buffer = new byte[0x10000];  // 64kb buffer
                    while((zipEntry = zipInputStream.getNextEntry())!=null) {

                        // if the source document is containing a debug folder, we will
                        // skip the old debug information and not include it within our copy
                        if(!zipEntry.getName().startsWith("debug/")) {
                            ZipEntry newEntry = new ZipEntry(zipEntry.getName());
                            zipOutputStream.putNextEntry(newEntry);
                            int read;
                            while ((read = zipInputStream.read(buffer))>=0){
                                zipOutputStream.write(buffer,0,read);
                            }
                            zipOutputStream.closeEntry();
                        }
                        zipInputStream.closeEntry();
                    }
                    zipInputStream.close();
                    zipOutputStream.close();

                    getPackage().getContentTypeManager().addNonPersistContentFile("debug/documentStream.dbg",
                        outStream.toByteArray(), "dbg", "application/octet-stream");
                }
                if(debugResourceManager!=null) {

// TODO:  the resourceManager needs to be able to serialize the resources, so we can store these data also within or debug folder
//
//                  getPackage().getContentTypeManager().addNonPersistContentFile("debug/debugResources.dbg",
//                      debugResourceManager., "application/octet-stream");
                }
            }
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            saver.save(out);
            return new ByteArrayInputStream(out.toByteArray());

        } catch (Exception e) {
            LOG.error("OOXML Filter could not save the given document:", e);
            throw new FilterException(e, ErrorCode.CRITICAL_ERROR);
        }
    }

    public void writeFile(String path) throws FilterException {

        try {

            final InputStream  inputStream = save();
            final OutputStream outputStream = new BufferedOutputStream(new FileOutputStream(new File(path)), DEFAULT_BUFFER_SIZE);

            int len;
            final byte[] buf = new byte[DEFAULT_BUFFER_SIZE];
            while ((len = inputStream.read(buf)) != -1) {
                outputStream.write(buf, 0, len);
            }
            outputStream.flush();
            outputStream.close();

        } catch (Exception e) {
            LOG.error("OOXML Filter could not save the given document:", e);
            throw new FilterException(e, ErrorCode.CRITICAL_ERROR);
        }
    }

    abstract public OpcPackage getPackage();

    abstract public Theme getTheme();

    public ResourceManager getResourceManager() {
        return resourceManager;
    }

    public String getUserLanguage() {
        return userLanguage;
    }

    public ConfigurationService getConfigurationService() {
        return services.getService(com.openexchange.config.ConfigurationService.class);
    }

    protected ThemePart getDefaultThemePart(PartName partName) {

        final String themeStr =

            "<a:theme xmlns:a=\"http://schemas.openxmlformats.org/drawingml/2006/main\"" +
            "    name=\"${themeName}\">" +
            "    <a:themeElements>" +
            "        <a:clrScheme name=\"${themeName}\">" +
            "            <a:dk1>" + "${dk1}" + " </a:dk1>" +
            "            <a:lt1>" + "${lt1}" + " </a:lt1>" +
            "            <a:dk2>" + "${dk2}" + " </a:dk2>" +
            "            <a:lt2>" + "${lt2}" + " </a:lt2>" +
            "            <a:accent1>" + "${accent1}" + " </a:accent1>" +
            "            <a:accent2>" + "${accent2}" + " </a:accent2>" +
            "            <a:accent3>" + "${accent3}" + " </a:accent3>" +
            "            <a:accent4>" + "${accent4}" + " </a:accent4>" +
            "            <a:accent5>" + "${accent5}" + " </a:accent5>" +
            "            <a:accent6>" + "${accent6}" + " </a:accent6>" +
            "            <a:hlink>"   + "${hlink}"   + " </a:hlink>"   +
            "            <a:folHlink>"+ "${folHlink}"+ " </a:folHlink>"+
            "        </a:clrScheme>" +
            "        <a:fontScheme name=\"${themeName}\">" +
            "            <a:majorFont>" +
            "                <a:latin typeface=\"${majorFont}\" />" +
            "                <a:ea typeface=\"\" />" +
            "                <a:cs typeface=\"\" />" +
            "            </a:majorFont>" +
            "            <a:minorFont>" +
            "                <a:latin typeface=\"${minorFont}\" />" +
            "                <a:ea typeface=\"\" />" +
            "                <a:cs typeface=\"\" />" +
            "            </a:minorFont>" +
            "        </a:fontScheme>" +
            "        <a:fmtScheme name=\"${themeName}\">" +
            "            <a:fillStyleLst>" +
            "                <a:solidFill>" +
            "                    <a:schemeClr val=\"phClr\" />" +
            "                </a:solidFill>" +
            "                <a:gradFill rotWithShape=\"1\">" +
            "                    <a:gsLst>" +
            "                        <a:gs pos=\"0\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"50000\" />" +
            "                                <a:satMod val=\"300000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"35000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"37000\" />" +
            "                                <a:satMod val=\"300000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"100000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"15000\" />" +
            "                                <a:satMod val=\"350000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                    </a:gsLst>" +
            "                    <a:lin ang=\"16200000\" scaled=\"1\" />" +
            "                </a:gradFill>" +
            "                <a:gradFill rotWithShape=\"1\">" +
            "                    <a:gsLst>" +
            "                        <a:gs pos=\"0\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"51000\" />" +
            "                                <a:satMod val=\"130000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"80000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"93000\" />" +
            "                                <a:satMod val=\"130000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"100000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"94000\" />" +
            "                                <a:satMod val=\"135000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                    </a:gsLst>" +
            "                    <a:lin ang=\"16200000\" scaled=\"0\" />" +
            "                </a:gradFill>" +
            "            </a:fillStyleLst>" +
            "            <a:lnStyleLst>" +
            "                <a:ln w=\"9525\" cap=\"flat\" cmpd=\"sng\" algn=\"ctr\">" +
            "                    <a:solidFill>" +
            "                        <a:schemeClr val=\"phClr\">" +
            "                            <a:shade val=\"95000\" />" +
            "                            <a:satMod val=\"105000\" />" +
            "                        </a:schemeClr>" +
            "                    </a:solidFill>" +
            "                    <a:prstDash val=\"solid\" />" +
            "                </a:ln>" +
            "                <a:ln w=\"25400\" cap=\"flat\" cmpd=\"sng\" algn=\"ctr\">" +
            "                    <a:solidFill>" +
            "                        <a:schemeClr val=\"phClr\" />" +
            "                    </a:solidFill>" +
            "                    <a:prstDash val=\"solid\" />" +
            "                </a:ln>" +
            "                <a:ln w=\"38100\" cap=\"flat\" cmpd=\"sng\" algn=\"ctr\">" +
            "                    <a:solidFill>" +
            "                        <a:schemeClr val=\"phClr\" />" +
            "                    </a:solidFill>" +
            "                    <a:prstDash val=\"solid\" />" +
            "                </a:ln>" +
            "            </a:lnStyleLst>" +
            "            <a:effectStyleLst>" +
            "                <a:effectStyle>" +
            "                    <a:effectLst>" +
            "                        <a:outerShdw blurRad=\"40000\" dist=\"20000\" dir=\"5400000\"" +
            "                            rotWithShape=\"0\">" +
            "                            <a:srgbClr val=\"000000\">" +
            "                                <a:alpha val=\"38000\" />" +
            "                            </a:srgbClr>" +
            "                        </a:outerShdw>" +
            "                    </a:effectLst>" +
            "                </a:effectStyle>" +
            "                <a:effectStyle>" +
            "                    <a:effectLst>" +
            "                        <a:outerShdw blurRad=\"40000\" dist=\"23000\" dir=\"5400000\"" +
            "                            rotWithShape=\"0\">" +
            "                            <a:srgbClr val=\"000000\">" +
            "                                <a:alpha val=\"35000\" />" +
            "                            </a:srgbClr>" +
            "                        </a:outerShdw>" +
            "                    </a:effectLst>" +
            "                </a:effectStyle>" +
            "                <a:effectStyle>" +
            "                    <a:effectLst>" +
            "                        <a:outerShdw blurRad=\"40000\" dist=\"23000\" dir=\"5400000\"" +
            "                            rotWithShape=\"0\">" +
            "                            <a:srgbClr val=\"000000\">" +
            "                                <a:alpha val=\"35000\" />" +
            "                            </a:srgbClr>" +
            "                        </a:outerShdw>" +
            "                    </a:effectLst>" +
            "                    <a:scene3d>" +
            "                        <a:camera prst=\"orthographicFront\">" +
            "                            <a:rot lat=\"0\" lon=\"0\" rev=\"0\" />" +
            "                        </a:camera>" +
            "                        <a:lightRig rig=\"threePt\" dir=\"t\">" +
            "                            <a:rot lat=\"0\" lon=\"0\" rev=\"1200000\" />" +
            "                        </a:lightRig>" +
            "                    </a:scene3d>" +
            "                    <a:sp3d>" +
            "                        <a:bevelT w=\"63500\" h=\"25400\" />" +
            "                    </a:sp3d>" +
            "                </a:effectStyle>" +
            "            </a:effectStyleLst>" +
            "            <a:bgFillStyleLst>" +
            "                <a:solidFill>" +
            "                    <a:schemeClr val=\"phClr\" />" +
            "                </a:solidFill>" +
            "                <a:gradFill rotWithShape=\"1\">" +
            "                    <a:gsLst>" +
            "                        <a:gs pos=\"0\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"40000\" />" +
            "                                <a:satMod val=\"350000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"40000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"45000\" />" +
            "                                <a:shade val=\"99000\" />" +
            "                                <a:satMod val=\"350000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"100000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"20000\" />" +
            "                                <a:satMod val=\"255000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                    </a:gsLst>" +
            "                    <a:path path=\"circle\">" +
            "                        <a:fillToRect l=\"50000\" t=\"-80000\" r=\"50000\" b=\"180000\" />" +
            "                    </a:path>" +
            "                </a:gradFill>" +
            "                <a:gradFill rotWithShape=\"1\">" +
            "                    <a:gsLst>" +
            "                        <a:gs pos=\"0\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:tint val=\"80000\" />" +
            "                                <a:satMod val=\"300000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                        <a:gs pos=\"100000\">" +
            "                            <a:schemeClr val=\"phClr\">" +
            "                                <a:shade val=\"30000\" />" +
            "                                <a:satMod val=\"200000\" />" +
            "                            </a:schemeClr>" +
            "                        </a:gs>" +
            "                    </a:gsLst>" +
            "                    <a:path path=\"circle\">" +
            "                        <a:fillToRect l=\"50000\" t=\"50000\" r=\"50000\" b=\"50000\" />" +
            "                    </a:path>" +
            "                </a:gradFill>" +
            "            </a:bgFillStyleLst>" +
            "        </a:fmtScheme>" +
            "    </a:themeElements>" +
            "    <a:objectDefaults />" +
            "    <a:extraClrSchemeLst />" +
            "</a:theme>";

        final String defaultSchemeName = "Larissa";

        ThemePart themePart = null;
        try {
            themePart = new ThemePart(partName);
            java.util.HashMap<String, String> mappings = new java.util.HashMap<String, String>();
            mappings.put("themeName", defaultSchemeName);
            mappings.put("dk1", "<a:sysClr val=\"windowText\" lastClr=\"000000\" />");
            mappings.put("lt1", "<a:sysClr val=\"window\" lastClr=\"FFFFFF\" />");
            mappings.put("dk2", "<a:srgbClr val=\"1F497D\" />");
            mappings.put("lt2", "<a:srgbClr val=\"EEECE1\" />");
            mappings.put("accent1", "<a:srgbClr val=\"4F81BD\" />");
            mappings.put("accent2", "<a:srgbClr val=\"C0504D\" />");
            mappings.put("accent3", "<a:srgbClr val=\"9BBB59\" />");
            mappings.put("accent4", "<a:srgbClr val=\"8064A2\" />");
            mappings.put("accent5", "<a:srgbClr val=\"4BACC6\" />");
            mappings.put("accent6", "<a:srgbClr val=\"F79646\" />");
            mappings.put("hlink",   "<a:srgbClr val=\"0000FF\" />");
            mappings.put("folHlink","<a:srgbClr val=\"800080\" />");
            mappings.put("majorFont", "Times New Roman");
            mappings.put("minorFont", "Arial");
            Theme theme = (Theme)org.docx4j.XmlUtils.unmarshallFromTemplate(themeStr, mappings);
            themePart.setJaxbElement(theme);
        }
        catch (JAXBException e) {
            e.printStackTrace();
        }
        catch (InvalidFormatException e) {
            e.printStackTrace();
        }
        return themePart;
    }
}
