/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.docx.operations;

import java.math.BigInteger;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.xml.bind.JAXBElement;
import org.docx4j.openpackaging.URIHelper;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.openpackaging.parts.WordprocessingML.DocumentSettingsPart;
import org.docx4j.openpackaging.parts.WordprocessingML.FontTablePart;
import org.docx4j.openpackaging.parts.WordprocessingML.NumberingDefinitionsPart;
import org.docx4j.openpackaging.parts.relationships.Namespaces;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.docx4j.relationships.Relationship;
import org.docx4j.vml.CTImageData;
import org.docx4j.vml.CTShape;
import org.docx4j.wml.BooleanDefaultTrue;
import org.docx4j.wml.CTSettings;
import org.docx4j.wml.CTTabStop;
import org.docx4j.wml.CTTblPrBase;
import org.docx4j.wml.CTTblPrBase.TblStyle;
import org.docx4j.wml.CTTblStylePr;
import org.docx4j.wml.CTTwipsMeasure;
import org.docx4j.wml.CTZoom;
import org.docx4j.wml.Color;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.DocDefaults;
import org.docx4j.wml.DocDefaults.PPrDefault;
import org.docx4j.wml.DocDefaults.RPrDefault;
import org.docx4j.wml.FontFamily;
import org.docx4j.wml.FontPanose;
import org.docx4j.wml.FontPitch;
import org.docx4j.wml.Fonts;
import org.docx4j.wml.Fonts.Font.AltName;
import org.docx4j.wml.Jc;
import org.docx4j.wml.Lvl;
import org.docx4j.wml.Lvl.LvlText;
import org.docx4j.wml.Lvl.Start;
import org.docx4j.wml.Lvl.Suff;
import org.docx4j.wml.NumFmt;
import org.docx4j.wml.NumberFormat;
import org.docx4j.wml.Numbering;
import org.docx4j.wml.Numbering.AbstractNum;
import org.docx4j.wml.Numbering.Num;
import org.docx4j.wml.Numbering.Num.LvlOverride;
import org.docx4j.wml.Numbering.Num.LvlOverride.StartOverride;
import org.docx4j.wml.Numbering.NumPicBullet;
import org.docx4j.wml.P;
import org.docx4j.wml.PPr;
import org.docx4j.wml.PPrBase;
import org.docx4j.wml.PPrBase.Ind;
import org.docx4j.wml.Pict;
import org.docx4j.wml.R;
import org.docx4j.wml.RFonts;
import org.docx4j.wml.RPr;
import org.docx4j.wml.RStyle;
import org.docx4j.wml.STPitch;
import org.docx4j.wml.STTblStyleOverrideType;
import org.docx4j.wml.SectPr;
import org.docx4j.wml.SectPr.PgMar;
import org.docx4j.wml.SectPr.PgSz;
import org.docx4j.wml.Style;
import org.docx4j.wml.Style.Name;
import org.docx4j.wml.Style.Next;
import org.docx4j.wml.Style.UiPriority;
import org.docx4j.wml.Styles;
import org.docx4j.wml.Tabs;
import org.docx4j.wml.Tbl;
import org.docx4j.wml.TblPr;
import org.docx4j.wml.TblWidth;
import org.docx4j.wml.Tc;
import org.docx4j.wml.TcPr;
import org.docx4j.wml.Text;
import org.docx4j.wml.Tr;
import org.docx4j.wml.TrPr;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.config.ConfigurationService;
import com.openexchange.office.ooxml.docx.OperationDocument;
import com.openexchange.office.ooxml.docx.tools.Character;
import com.openexchange.office.ooxml.docx.tools.Component;
import com.openexchange.office.ooxml.docx.tools.Component.DrawingComponent;
import com.openexchange.office.ooxml.docx.tools.Component.FldSimpleComponent;
import com.openexchange.office.ooxml.docx.tools.Component.HardBreakComponent;
import com.openexchange.office.ooxml.docx.tools.Component.PictVMLComponent;
import com.openexchange.office.ooxml.docx.tools.Component.Placeholder_Base;
import com.openexchange.office.ooxml.docx.tools.Component.Placeholder_Base.GraphicType;
import com.openexchange.office.ooxml.docx.tools.Component.TabComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TextComponent;
import com.openexchange.office.ooxml.docx.tools.Component.TextRun_Base;
import com.openexchange.office.ooxml.docx.tools.Drawings;
import com.openexchange.office.ooxml.docx.tools.Paragraph;
import com.openexchange.office.ooxml.docx.tools.PictVML;
import com.openexchange.office.ooxml.docx.tools.Table;
import com.openexchange.office.ooxml.docx.tools.Utils;
import com.openexchange.office.ooxml.tools.Commons;


public class CreateOperationHelper extends com.openexchange.office.ooxml.operations.CreateOperationHelper {

    private final WordprocessingMLPackage wordprocessingMLPackage;
    private final Map<String, String> conditionalTableAttrNames;

    public CreateOperationHelper(OperationDocument _operationDocument, JSONArray mOperationsArray)
    {
        super(_operationDocument, mOperationsArray);
        wordprocessingMLPackage = _operationDocument.getPackage();

        conditionalTableAttrNames = new HashMap<String, String>();
        conditionalTableAttrNames.put("firstRow", "firstRow");
        conditionalTableAttrNames.put("lastRow", "lastRow");
        conditionalTableAttrNames.put("firstCol", "firstCol");
        conditionalTableAttrNames.put("lastCol", "lastCol");
        conditionalTableAttrNames.put("band1Vert", "band1Vert");
        conditionalTableAttrNames.put("band2Vert", "band2Vert");
        conditionalTableAttrNames.put("band1Horz", "band1Hor");
        conditionalTableAttrNames.put("band2Horz", "band2Hor");
        conditionalTableAttrNames.put("neCell", "northEastCell");
        conditionalTableAttrNames.put("nwCell", "northWestCell");
        conditionalTableAttrNames.put("seCell", "southEastCell");
        conditionalTableAttrNames.put("swCell", "southWestCell");
    }

    @Override
    public OperationDocument getOperationDocument() {
        return (OperationDocument)operationDocument;
    }

    @Override
    public void createOperations()
        throws JSONException, ParseException {

        createOperations(wordprocessingMLPackage.getMainDocumentPart(), new ArrayList<Integer>());
    }

    @Override
    public void createDocumentDefaults(String userLanguage)
        throws JSONException {

        JSONObject documentAttributes = new JSONObject();
        Relationship r = wordprocessingMLPackage.getMainDocumentPart().getRelationshipsPart().getRelationshipByType(Namespaces.SETTINGS);
        if (r != null) {
            DocumentSettingsPart settingsPart = (DocumentSettingsPart)wordprocessingMLPackage.getMainDocumentPart().getRelationshipsPart().getPart(r);
            CTSettings settings = settingsPart.getJaxbElement();
            if (settings != null) {
                JSONObject jsonDocumentSettings = new JSONObject();
                CTTwipsMeasure defaultTabStop = settings.getDefaultTabStop();
                if (defaultTabStop != null) {
                    jsonDocumentSettings.put("defaultTabStop", Utils.mapTwipTo100THMM(defaultTabStop.getVal()));
                }
                CTZoom zoom = settings.getZoom();
                if (zoom != null) {
                    boolean addZoomAttr = false;
                    JSONObject zoomAttributes = new JSONObject();
                    if (zoom.getVal() != null) {
                        zoomAttributes.put("type", zoom.getVal().toString());
                        addZoomAttr = true;
                    }
                    if (zoom.getPercent() != null) {
                        zoomAttributes.put("value", zoom.getPercent().intValue());
                        addZoomAttr = true;
                    }
                    if (addZoomAttr) {
                        jsonDocumentSettings.put("zoom", zoomAttributes);
                    }
                }
                if(jsonDocumentSettings.length()>0)
                    documentAttributes.put("document", jsonDocumentSettings);
            }
        }
        SectPr sectPr = Utils.getDocumentProperties(wordprocessingMLPackage.getMainDocumentPart());
        if(sectPr!=null) {
            JSONObject jsonPageSettings = new JSONObject();
            PgSz pgSz = sectPr.getPgSz();
            if(pgSz!=null) {
                jsonPageSettings.put("width", Utils.mapTwipTo100THMM(pgSz.getW()));
                jsonPageSettings.put("height", Utils.mapTwipTo100THMM(pgSz.getH()));
            }
            PgMar pgMar = sectPr.getPgMar();
            if(pgMar!=null) {
                jsonPageSettings.put("marginLeft", Utils.mapTwipTo100THMM(pgMar.getLeft()));
                jsonPageSettings.put("marginTop", Utils.mapTwipTo100THMM(pgMar.getTop()));
                jsonPageSettings.put("marginRight", Utils.mapTwipTo100THMM(pgMar.getRight()));
                jsonPageSettings.put("marginBottom", Utils.mapTwipTo100THMM(pgMar.getBottom()));
            }
            if(jsonPageSettings.length()>0)
                documentAttributes.put("page", jsonPageSettings);
        }
        Styles styles = wordprocessingMLPackage.getMainDocumentPart().getStyleDefinitionsPart().getJaxbElement();
        DocDefaults docDefaults = styles.getDocDefaults();
        if(docDefaults!=null) {
            RPrDefault rDefault = docDefaults.getRPrDefault();
            if(rDefault!=null) {
                RPr rPrDefault = rDefault.getRPr();
                if(rPrDefault!=null) {
                    JSONObject jsonCharacterSettings = Character.createCharacterProperties(themeFonts, rPrDefault);
                    if(jsonCharacterSettings!=null&&jsonCharacterSettings.length()>0) {

                        // set user language if it doesn't exists within the document
                        String value = jsonCharacterSettings.optString("language");
                        if (value == null || value.length() == 0) {
                            if (userLanguage != null && userLanguage.length() > 0)
                                jsonCharacterSettings.put("language", userLanguage);
                            else
                                jsonCharacterSettings.put("language", "en-US"); // set default
                        }
                        value = jsonCharacterSettings.optString("languageEa");
                        if (value == null || value.length() == 0) {
                            jsonCharacterSettings.put("languageEa", "en-US"); // set default
                        }
                        value = jsonCharacterSettings.optString("languageBidi");
                        if (value == null || value.length() == 0) {
                            jsonCharacterSettings.put("languageBidi", "ar-SA"); // set default
                        }
                        documentAttributes.put("character", jsonCharacterSettings);
                    }
                }
            }
            PPrDefault pDefault = docDefaults.getPPrDefault();
            if(pDefault!=null) {
                PPr pPrDefault = pDefault.getPPr();
                if(pPrDefault!=null) {
                    JSONObject jsonParagraphSettings = Paragraph.createParagraphProperties(pPrDefault);
                    if(jsonParagraphSettings!=null&&jsonParagraphSettings.length()>0)
                        documentAttributes.put("paragraph", jsonParagraphSettings);
                }
            }
        }
        if(documentAttributes.length()>0)
            addSetDocumentAttributesOperation(documentAttributes);
    }

    /**
     * @param aOperationsArray
     * @param nPara
     */
    public void AddInsertParagraphOperation(final List<Integer> start)
        throws JSONException {

        final JSONObject insertParagraphObject = new JSONObject();
        insertParagraphObject.put("name", "insertParagraph");
        insertParagraphObject.put("start", start);
        operationsArray.put(insertParagraphObject);
    }

    /**
     * @param aOperationsArray
     * @param nPara
     * @param nPos
     * @param aText
     */
    public void AddInsertTextOperation(final List<Integer> paragraphPosition, final int textPosition, final String text)
        throws JSONException {

        final JSONObject insertTextObject = new JSONObject();
        List<Integer> startArray = new ArrayList<Integer>(paragraphPosition);
        startArray.add(textPosition);

        insertTextObject.put("name", "insertText");
        insertTextObject.put("start", startArray);
        insertTextObject.put("text", text);
        operationsArray.put(insertTextObject);
    }

    public void AddInsertTabOperation(final List<Integer> start)
        throws JSONException {

        final JSONObject insertTabObject = new JSONObject();
        insertTabObject.put("name", "insertTab");
        insertTabObject.put("start", start);
        operationsArray.put(insertTabObject);
    }

    public void AddInsertHardBreakOperation(final List<Integer> start)
        throws JSONException {

        final JSONObject insertHardBreakObject = new JSONObject();
        insertHardBreakObject.put("name", "insertHardBreak");
        insertHardBreakObject.put("start", start);
        operationsArray.put(insertHardBreakObject);
    }

    public void AddInsertTableOperation(final List<Integer> start, final JSONObject tableProperties, final JSONObject sizeExceeded)
        throws JSONException {

        final JSONObject insertTableObject = new JSONObject();
        insertTableObject.put("name", "insertTable");
        insertTableObject.put("start", start);
        if(tableProperties!=null&&tableProperties.length()>0)
            insertTableObject.put("attrs", tableProperties);
        if(sizeExceeded!=null)
            insertTableObject.put("sizeExceeded", sizeExceeded);
        operationsArray.put(insertTableObject);
    }

    public void AddInsertRowsOperation(final List<Integer> start, final JSONObject rowProperties)
        throws JSONException {

        final JSONObject insertRowObject = new JSONObject();
        insertRowObject.put("name", "insertRows");
        insertRowObject.put("start", start);
        if(rowProperties!=null&&rowProperties.length()>0)
            insertRowObject.put("attrs", rowProperties);
        operationsArray.put(insertRowObject);
    }

    public void AddInsertCellsOperation(final List<Integer> start, final JSONObject cellProperties)
        throws JSONException {

        final JSONObject insertCellObject = new JSONObject();
        insertCellObject.put("name", "insertCells");
        insertCellObject.put("start", start);
        if(cellProperties!=null&&cellProperties.length()>0)
            insertCellObject.put("attrs", cellProperties);
        operationsArray.put(insertCellObject);
    }

    public void AddInsertFieldOperation(final List<Integer> start, final String type, final String representation)
        throws JSONException {

        final JSONObject insertFieldObject = new JSONObject();
        insertFieldObject.put("name", "insertField");
        insertFieldObject.put("start", start);
        if(type!=null)
            insertFieldObject.put("type", type);
        if(representation!=null)
            insertFieldObject.put("representation", representation);
        operationsArray.put(insertFieldObject);
    }

    public void AddInsertDrawingOperation(final List<Integer> start, GraphicType graphicType, final JSONObject attrs)
        throws JSONException {

        final JSONObject insertDrawingObject = new JSONObject();
        insertDrawingObject.put("name", "insertDrawing");
        insertDrawingObject.put("start", start);
        insertDrawingObject.put("type", graphicType.name().toLowerCase());
        if(attrs!=null&&attrs.has("drawing"))
            insertDrawingObject.put("attrs", attrs);
        operationsArray.put(insertDrawingObject);
    }

    public void AddSetAttributesOperation(final JSONObject attrs,
        final List<Integer> startPosition, final List<Integer> endPosition)
        throws JSONException {

        final JSONObject setAttributeObject = new JSONObject();
        setAttributeObject.put("name", "setAttributes");
        setAttributeObject.put("attrs", attrs );
        setAttributeObject.put("start", startPosition);
        if(endPosition!=null)
            setAttributeObject.put("end", endPosition);
        operationsArray.put(setAttributeObject);
    }

    public void AddInsertStyleSheetOperation(String type, String styleId, String styleName, JSONObject attrs, String parentId, Boolean hidden, Integer uipriority, Boolean isDefault)
        throws JSONException {

        final JSONObject insertStylesheetObject = new JSONObject();
        insertStylesheetObject.put("name", "insertStyleSheet");
        insertStylesheetObject.put("type", type);
        insertStylesheetObject.put("styleId", styleId);
        insertStylesheetObject.put("styleName", styleName);
        if(attrs!=null)
            insertStylesheetObject.put("attrs", attrs);
        if(parentId!=null&&parentId.length()>0)
            insertStylesheetObject.put("parent", parentId);
        if(hidden!=null)
            insertStylesheetObject.put("hidden", hidden);
        if(uipriority!=null)
            insertStylesheetObject.put("uiPriority", uipriority);
        if(isDefault!=null&&isDefault)
            insertStylesheetObject.put("default", true);
        operationsArray.put(insertStylesheetObject);
    }

    public void AddInsertFontDescription(String name, JSONObject attrs)
        throws JSONException {

        if(name!=null&&name.length()>0&&attrs!=null&&attrs.length()>0) {
            final JSONObject fontDescription = new JSONObject();
            fontDescription.put("name", "insertFontDescription");
            fontDescription.put("fontName", name);
            fontDescription.put("attrs", attrs);
            operationsArray.put(fontDescription);
        }
    }

    public void CreateFontDescriptions()
        throws JSONException {

        FontTablePart fontTablePart = wordprocessingMLPackage.getMainDocumentPart().getFontTablePart();
        if(fontTablePart!=null) {
            org.docx4j.wml.Fonts fonts = fontTablePart.getJaxbElement();
            List<Fonts.Font> fontList = fonts.getFont();
            for(Fonts.Font font: fontList) {
                JSONObject attrs = new JSONObject();
                AltName altNames = font.getAltName();
                if(altNames!=null) {
                    String names = altNames.getVal();
                    if(names!=null) {
                        JSONArray jsonAltNames = new JSONArray();
                        String[] tokens = names.split(",");
                        for(String token : tokens)
                            Utils.jsonPut(jsonAltNames, token);
                        Utils.jsonPut(attrs, "altNames", jsonAltNames);
                    }
                }
                FontFamily fontFamily = font.getFamily();
                if(fontFamily!=null)
                    Utils.jsonPut(attrs, "family", fontFamily.getVal());
                FontPitch fontPitch = font.getPitch();
                if(fontPitch!=null) {
                    STPitch stPitch = fontPitch.getVal();
                    if(stPitch!=STPitch.DEFAULT)
                        Utils.jsonPut(attrs, "pitch", stPitch.toString().toLowerCase());
                }
                FontPanose fontPanose = font.getPanose1();
                if(fontPanose!=null) {
                    byte[] panose = fontPanose.getVal();
                    if(panose.length==10) {
                        JSONArray jsonPanose = new JSONArray();
                        for(byte entry : panose)
                            jsonPanose.put(entry);
                        Utils.jsonPut(attrs, "panose1", jsonPanose);
                    }
                }
                AddInsertFontDescription(font.getName(), attrs);
            }
        }
    }

    private void CreateConditionalTableStyle(JSONObject attrs, String conditionalTableStyle, CTTblPrBase tblPr, TrPr trPr, TcPr tcPr, PPr pPr, RPr rPr)
        throws JSONException {

        JSONObject tableStyle = new JSONObject();
        Utils.mergeJsonObjectIfUsed("table", tableStyle, Table.createTableProperties(wordprocessingMLPackage, tblPr, null, true, 0));
        Utils.mergeJsonObjectIfUsed("row", tableStyle, Table.createRowProperties(trPr));
        Utils.mergeJsonObjectIfUsed("cell", tableStyle, Table.createCellProperties(tcPr, null));
        Utils.mergeJsonObjectIfUsed("paragraph", tableStyle, Paragraph.createParagraphProperties(pPr));
        Utils.mergeJsonObjectIfUsed("character", tableStyle, Character.createCharacterProperties(themeFonts, rPr));
        if(tableStyle.length()>0)
            attrs.put(conditionalTableStyle, tableStyle);
    }

    public void CreateStyleOperations()
        throws JSONException {

        Styles styles = wordprocessingMLPackage.getMainDocumentPart().getStyleDefinitionsPart().getJaxbElement();
        List< Style > lStyle = styles.getStyle();
        for(Style style : lStyle) {
            String styleId = null;
            String styleName = null;
            String type = style.getType();
            Name name = style.getName();
            if(name!=null)
                styleName = name.getVal();
            styleId = style.getStyleId();
            if(type!=null&&styleId!=null&&styleId.length()>0) {
                String parentId = null;
                Style.BasedOn basedOn = style.getBasedOn();
                if (basedOn!=null)
                    parentId = basedOn.getVal();

                Integer uipriority = null;
                Boolean hidden = null;
                Boolean isDefault = style.isDefault();
                BooleanDefaultTrue defaultTrueHidden = style.getHidden();
                if (defaultTrueHidden!=null)
                    hidden = defaultTrueHidden.isVal();
                if(hidden==null||hidden==false)
                {
                    UiPriority prio = style.getUiPriority();
                    if(prio!=null)
                        uipriority = prio.getVal().intValue();
                }

                RPr rPr = null;
                PPr pPr = null;
                if(type.equals("character")) {
                    rPr = style.getRPr();
                    JSONObject attrs = new JSONObject();
                    if (rPr!=null) {
                        JSONObject character = Character.createCharacterProperties(themeFonts, rPr);
                        attrs.put("character", character);
                    }
                    AddInsertStyleSheetOperation("character", styleId, styleName, attrs, parentId, hidden, uipriority, isDefault);
                }
                else if (type.equals("paragraph")) {
                    rPr = style.getRPr();
                    pPr = style.getPPr();
                    JSONObject attrs = new JSONObject();

                    if(rPr!=null) {
                        JSONObject character = Character.createCharacterProperties(themeFonts, rPr);
                        attrs.put("character", character);
                    }
                    JSONObject paragraph = null;
                    if(pPr!=null) {
                        paragraph = Paragraph.createParagraphProperties(pPr);
                    }
                    Next nextStyle = style.getNext();
                    if(nextStyle!=null) {
                        String nextStyleId = nextStyle.getVal();
                        if(nextStyleId!=null&&nextStyleId.length()>0) {
                            if (paragraph==null) {
                                paragraph = new JSONObject();
                            }
                            paragraph.put("nextStyleId", nextStyleId);
                        }
                    }
                    if (paragraph!=null) {
                        attrs.put("paragraph", paragraph);
                    }
                    AddInsertStyleSheetOperation("paragraph", styleId, styleName, attrs, parentId, hidden, uipriority, isDefault);
                }
                else if (type.equals("table")) {
                    JSONObject attrs = new JSONObject();
                    CreateConditionalTableStyle(attrs, "wholeTable", style.getTblPr(), style.getTrPr(), style.getTcPr(), style.getPPr(), style.getRPr());
                    List<CTTblStylePr> conditionalStyles = style.getTblStylePr();
                    for(CTTblStylePr conditionalStyle : conditionalStyles) {
                        STTblStyleOverrideType conditionalType = conditionalStyle.getType();
                        String attrName = conditionalType.value();
                        if(conditionalTableAttrNames.containsKey(attrName))
                            attrName = conditionalTableAttrNames.get(attrName);
                        CreateConditionalTableStyle(attrs, attrName, conditionalStyle.getTblPr(), conditionalStyle.getTrPr(), conditionalStyle.getTcPr(), conditionalStyle.getPPr(), conditionalStyle.getRPr());
                    }
                    AddInsertStyleSheetOperation("table", styleId, styleName, attrs, parentId, hidden, uipriority, isDefault);
                }
            }
        }
    }

    private void createOperations(Object parent, List<Integer> parentPosition)
        throws JSONException, ParseException {

        if(parent instanceof ContentAccessor) {

            List<Object> rootContent = ((ContentAccessor)parent).getContent();

            int position = 0;
            int insertedComponents = 0;
            for (int i=0; i<rootContent.size(); i++) {

                Object o = Component.getContentModel(rootContent.get(i), parent, i);
                if (o instanceof P) {
                    CreateParagraphOperations((P)o, parentPosition, position++);
                    insertedComponents++;
                }
                else if(o instanceof Tbl) {
                    CreateTableOperations((Tbl)o, parentPosition, position++);
                    insertedComponents++;
                }
            }
            if((parent instanceof Tc)&&(insertedComponents>1)) {
                // removing empty paragraph within tables if it comes at last position and if there are more than one component
                JSONObject object = operationsArray.getJSONObject(operationsArray.length()-1);
                if(object.getString("name").equals("insertParagraph")) {
                    JSONObject nopObject = new JSONObject();
                    nopObject.put("name", "noOp");
                    operationsArray.put(operationsArray.length() - 1, nopObject);
                }
            }
        }
    }

    private void CreateParagraphOperations(P paragraph, List<Integer> parentPosition, int position)
        throws JSONException, ParseException {

        List<Integer> paragraphPosition = new ArrayList<Integer>(parentPosition);
        paragraphPosition.add(position);

        // insert paragraph and apply paragraph attributes
        AddInsertParagraphOperation(paragraphPosition);
        PPr pPr = paragraph.getPPr();
        if(pPr!=null) {
            JSONObject attrs = new JSONObject();
            PPrBase.PStyle pStyle = pPr.getPStyle();
            if(pStyle!=null) {
                attrs.put("styleId", pStyle.getVal());
            }
            JSONObject paragraphProperties = Paragraph.createParagraphProperties(pPr);
            if(paragraphProperties!=null) {
                attrs.put("paragraph", paragraphProperties);
            }
            if(attrs.length()>0) {
                AddSetAttributesOperation(attrs, paragraphPosition, null);
            }
        }

        // insert components (text and or other drawing objects)
        Component.ParagraphContext paragraphContext = new Component.ParagraphContext(paragraph);
        while(paragraphContext.hasNext()) {
            Component component = paragraphContext.next();
            if (component instanceof TextComponent)
                AddInsertTextOperation(paragraphPosition, component.getComponentNumber(), ((Text)component.getObject()).getValue());
            else if (component instanceof Placeholder_Base){
                List<Integer> startPosition = new ArrayList<Integer>(paragraphPosition);
                startPosition.add(component.getComponentNumber());
                JSONObject jsonDrawingProperties = null;
                if(component instanceof DrawingComponent) {
                    jsonDrawingProperties = Commons.surroundJSONObject("drawing", Drawings.createProperties(wordprocessingMLPackage, (DrawingComponent)component));
                    AddInsertDrawingOperation(startPosition, ((DrawingComponent) component).getType(), jsonDrawingProperties);
                }
                else if (component instanceof PictVMLComponent){
                    jsonDrawingProperties = Commons.surroundJSONObject("drawing", PictVML.createProperties(wordprocessingMLPackage, (PictVMLComponent)component));
                    AddInsertDrawingOperation(startPosition, ((PictVMLComponent) component).getType(), jsonDrawingProperties);
                }
                else
                    AddInsertDrawingOperation(startPosition, ((Placeholder_Base)component).getType(), null);
            }
            else if (component instanceof TabComponent) {
                List<Integer> tabPosition = new ArrayList<Integer>(paragraphPosition);
                tabPosition.add(component.getComponentNumber());
                AddInsertTabOperation(tabPosition);
            }
            else if (component instanceof HardBreakComponent) {
                List<Integer> hardBreakPosition = new ArrayList<Integer>(paragraphPosition);
                hardBreakPosition.add(component.getComponentNumber());
                AddInsertHardBreakOperation(hardBreakPosition);
            }
            else if (component instanceof FldSimpleComponent) {
                List<Integer> fieldPosition = new ArrayList<Integer>(paragraphPosition);
                fieldPosition.add(component.getComponentNumber());
                AddInsertFieldOperation(fieldPosition, ((FldSimpleComponent)component).getInstr(), ((FldSimpleComponent)component).getRepresentation());
            }
            else {
                // we never should come here...
                List<Integer> startPosition = new ArrayList<Integer>(paragraphPosition);
                startPosition.add(component.getComponentNumber());
                AddInsertDrawingOperation(startPosition, GraphicType.UNDEFINED, null);
            }
        }

        paragraphContext = new Component.ParagraphContext(paragraph);
        while(paragraphContext.hasNext()) {
            Component component = paragraphContext.next();

            if(component instanceof TextRun_Base) {
                JSONObject attrs = new JSONObject();
                JSONObject jsonCharacterProperties = null;

                R textRun = ((TextRun_Base)component).getTextRun();
                RPr rPr = textRun.getRPr();
                if(rPr!=null) {
                    RStyle rStyle = rPr.getRStyle();
                    if(rStyle!=null)
                        attrs.put("styleId", rStyle.getVal());
                    jsonCharacterProperties = Character.createCharacterProperties(themeFonts, rPr);
                }
                if(paragraph.getContent().get(component.getContextIndex()) instanceof P.Hyperlink) {
                    String url = Utils.getUrl(wordprocessingMLPackage.getMainDocumentPart(), ((P.Hyperlink)textRun.getParent()).getId());
                    if (url!=null&&url.length()>0) {
                        if(jsonCharacterProperties==null)
                            jsonCharacterProperties = new JSONObject();
                        jsonCharacterProperties.put("url", url);
                    }
                }
                if ((attrs.length()>0)||(jsonCharacterProperties!=null)) {
                    if(jsonCharacterProperties!=null) {
                        attrs.put("character", jsonCharacterProperties);
                    }
                    int startComponent = component.getComponentNumber();
                    int endComponent   = component.getNextComponentNumber()-1;
                    List<Integer> startPosition = new ArrayList<Integer>(paragraphPosition);
                    startPosition.add(startComponent);
                    if(startComponent==endComponent)
                        AddSetAttributesOperation(attrs, startPosition, null);
                    else {
                        List<Integer> endPosition = new ArrayList<Integer>(paragraphPosition);
                        endPosition.add(endComponent);
                        AddSetAttributesOperation(attrs, startPosition, endPosition);
                    }
                }
            }
        }
    }

    private void CreateTableOperations(Tbl tbl, List<Integer> parentPosition, int position)
        throws JSONException, ParseException {

        final ConfigurationService configuration = getOperationDocument().getConfigurationService();
        int maxTableColumns = configuration.getIntProperty("io.ox/office//module/maxTableColumns", 16);
        int maxTableRows = configuration.getIntProperty("io.ox/office//module/maxTableRows", 256);
        int maxTableCells = configuration.getIntProperty("io.ox/office//module/maxTableCells", 256);

        JSONArray tableGrid = Table.getTableGrid(tbl);
        int gridCount = tableGrid != null ? tableGrid.length() : 0;
        int columns = gridCount;
        int rows = 0;
        int rowWidth = 0;
        boolean useCellWidth = true;
        Component.TableContext tableContext = new Component.TableContext(tbl);
        List<Integer> tablePosition = new ArrayList<Integer>(parentPosition);

        while(tableContext.hasNext()) {
            Component rowComponent = tableContext.next();
            rows++;
            // check, if table cells in the first row have a width defined (in "dxa")
            if (rows == 1) {
                List<Integer> rowPosition = new ArrayList<Integer>(tablePosition);
                rowPosition.add(rowComponent.getComponentNumber());
                Component.TableRowContext tableRowContext = new Component.TableRowContext((Tr)rowComponent.getObject());
                while(tableRowContext.hasNext()) {
                    Component cellComponent = tableRowContext.next();
                    TcPr cellProperties = ((Tc)cellComponent.getObject()).getTcPr();
                    if (cellProperties != null) {
                        int cellWidth = 0;
                        TblWidth cellWidthObject = cellProperties.getTcW();
                        if (cellWidthObject != null) {
                            String cellWidthType = cellWidthObject.getType();
                            BigInteger bigInt = cellWidthObject.getW();
                            if (bigInt != null) {
                                cellWidth = bigInt.intValue();
                                if ((cellWidthType != null) && (cellWidthType.equals("dxa"))) {
                                    rowWidth = rowWidth + cellWidth * 2540 / 72 / 20;
                                } else {
                                    useCellWidth = false;
                                }
                            } else {
                                useCellWidth = false;
                            }
                        } else {
                            useCellWidth = false;
                        }
                    } else {
                        useCellWidth = false;
                    }
                }
            }
        }

        if (useCellWidth == false) {
            rowWidth = 0;  // <- do not use the value
        }

        boolean maxTableSizeExceeded = (columns>maxTableColumns)||(rows>maxTableRows)||((columns*rows)>maxTableCells);

        JSONObject tableAttrs = new JSONObject();
        TblPr tblPr = tbl.getTblPr();
        if(tblPr!=null) {
            // setting the table style that is used...
            TblStyle tblStyle = tblPr.getTblStyle();
            if(tblStyle!=null) {
                String style = tblStyle.getVal();
                if(style!=null&&style.length()>0) {
                    tableAttrs.put("styleId", style);
                }
            }
        }
        JSONObject jsonTableProperties = Table.createTableProperties(wordprocessingMLPackage, tbl.getTblPr(), tableGrid, parentPosition.size()==0, rowWidth);
        if(jsonTableProperties!=null) {
            tableAttrs.put("table", jsonTableProperties);
        }

        JSONObject sizeExceeded = null;
        if(maxTableSizeExceeded) {
            sizeExceeded = new JSONObject();
            sizeExceeded.put("rows", rows);
            sizeExceeded.put("columns", columns);
        }
        tablePosition.add(position);
        AddInsertTableOperation(tablePosition, tableAttrs, sizeExceeded);

        if(!maxTableSizeExceeded&&gridCount>0) {
            tableContext = new Component.TableContext(tbl);
            while(tableContext.hasNext()) {
                Component rowComponent = tableContext.next();
                List<Integer> rowPosition = new ArrayList<Integer>(tablePosition);
                rowPosition.add(rowComponent.getComponentNumber());
                AddInsertRowsOperation(rowPosition, Commons.surroundJSONObject("row", Table.createRowProperties(((Tr)rowComponent.getObject()).getTrPr())));
                Component.TableRowContext tableRowContext = new Component.TableRowContext((Tr)rowComponent.getObject());
                while(tableRowContext.hasNext()) {
                    Component cellComponent = tableRowContext.next();
                    List<Integer> cellPosition = new ArrayList<Integer>(rowPosition);
                    cellPosition.add(cellComponent.getComponentNumber());
                    AddInsertCellsOperation(cellPosition, Commons.surroundJSONObject("cell", Table.createCellProperties(((Tc)cellComponent.getObject()).getTcPr(), tbl)));
                    createOperations(cellComponent.getObject(), cellPosition);
                }
            }
        }
    }

    public void AddInsertListOperation( long listIdentifier, JSONObject listDefinition )
        throws JSONException {

        final JSONObject insertListObject = new JSONObject();
        insertListObject.put("name", "insertListStyle");
        insertListObject.put("listStyleId", 'L' + String.valueOf(listIdentifier));
        insertListObject.put("listDefinition", listDefinition);
        operationsArray.put(insertListObject);
    }

    /**
     * Puts all list level attributes into the level's JSONObject
     * Is used to load the levels of the abstract numbering as well
     * as for the levels of the LvlOverride
     */
    private void createLevel( JSONObject levelObject, Lvl level, final HashMap<Long, String> picMap )
        throws JSONException {

        Jc jc = level.getLvlJc();//start, end, center, both, ....
        if( jc != null)
            levelObject.put( "textAlign", jc.getVal().value() );
        Start start = level.getStart();
        if( start != null )
            levelObject.put("listStartValue", start.getVal().longValue());
        NumFmt numFmt = level.getNumFmt();
        // contains the numbering type like
        // decimal, upperRoman, lowerRoman, upperLetter, lowerLetter, ordinal, cardinalText, ordinalText
        // hex, ...
        if( numFmt != null && numFmt.getVal() != null){
            NumberFormat numberFormat = numFmt.getVal();
            levelObject.put("numberFormat", numberFormat.value());
        }
        LvlText lvlText = level.getLvlText();
        if( lvlText != null )
            levelObject.put("levelText", lvlText.getVal() );

        if( level.getLvlRestart() != null ) // determines whether a sub-level restarts on a 'higher' than the parent level
            levelObject.put("listStartValue", level.getLvlRestart().getVal().longValue());
        Lvl.PStyle pstyle = level.getPStyle();
        if( pstyle != null ){
            levelObject.put("paraStyle", pstyle.getVal());
        }
        Lvl.LvlPicBulletId lvlPicBulletId = level.getLvlPicBulletId();
        if( lvlPicBulletId != null ) {
            long bulletId = lvlPicBulletId.getVal().longValue();
            String picURI = picMap.get( new Long( bulletId ));
            levelObject.put( "levelPicBulletUri", picURI );
        }

        Suff suff = level.getSuff();
        if( suff != null){ //number suffix - could be 'tab', 'space' or 'nothing'
            levelObject.put( "labelFollowedBy", suff.getVal().equals("tab") ? "listtab" : suff.getVal());
        }
        PPr paraProperties = level.getPPr();//paragraph properties - indent, tab, ...
        if( paraProperties != null ){
            Ind ind = paraProperties.getInd();
            if( ind != null ){
                BigInteger hanging = ind.getHanging();
                if( hanging != null && hanging.longValue() > 0) //either hanging or firstLine can be set, if hanging is set then firstLine is ignored
                    levelObject.put("indentFirstLine", - Utils.mapTwipTo100THMM(hanging) );
                else{
                    BigInteger firstLine = ind.getFirstLine();
                    if( firstLine != null && firstLine.longValue() > 0 )
                        levelObject.put("indentFirstLine", Utils.mapTwipTo100THMM(firstLine));
                }
                levelObject.put("indentLeft", Utils.mapTwipTo100THMM(ind.getLeft()));
            }
            Tabs tabs = paraProperties.getTabs();
            if( tabs != null ){
                List<CTTabStop> tabList = tabs.getTab();
                if( tabList != null && !tabList.isEmpty()){
                    // there is only one tab stop
                    levelObject.put( "tabStopPosition", Utils.mapTwipTo100THMM(tabList.get(0).getPos()) );
                }
            }
        }
        RPr rPr = level.getRPr();
        if( rPr != null ){
            /*
            <w:rFonts w:ascii="Symbol" w:hAnsi="Symbol" w:hint="default"/><w:color w:val="auto"/>
            */
            RFonts rFonts = rPr.getRFonts();
            if( rFonts != null){
                levelObject.put( "fontName", rFonts.getAscii());
                //rFonts.getHAnsi();
                //rFonts.getHint();
            }
            Color color = rPr.getColor();
            if( color != null ){
                Utils.jsonPut(levelObject, "color", Utils.createColor(color));
            }
        }
    }

    /**
     * OOXML numbering parts contains two related arrays. One is the array of abstract numberings and on is the mapping from
     * numId to abstractNumId.
     * A numId of zero (assigned to a paragraph) switches the numbering off.
     *
     * The list operations create list definitions for a numId rather than the mapping numId->abstractNumId->definition
     *
     * The mapping also allows for overriding certain levels of the abstract numbering.
     *
     */
    public void CreateListOperations()
        throws JSONException {

        NumberingDefinitionsPart numDefPart = wordprocessingMLPackage.getMainDocumentPart().getNumberingDefinitionsPart();
        if( numDefPart == null )
            return;
        Numbering numbering = numDefPart.getJaxbElement();
        // load bullet picture mapping
        HashMap<Long, String> picMap = new HashMap<Long, String>();
        List<NumPicBullet> numPicBulletList = numbering.getNumPicBullet();
        if( numPicBulletList != null ){
            Iterator<NumPicBullet> numPicBulletIter = numPicBulletList.iterator();
            while( numPicBulletIter.hasNext() ){
                NumPicBullet numPicBullet = numPicBulletIter.next();
                Pict pict = numPicBullet.getPict();
                if( pict != null ){
                    //
                    CTShape shape = null;
                    Iterator<Object> anyAndAnyIter = pict.getAnyAndAny().iterator();
                     while( anyAndAnyIter.hasNext() ){
                         Object element = anyAndAnyIter.next();
                         if( element instanceof JAXBElement ) {
                             if( ((JAXBElement<?>)element).getValue().getClass().getName() == "org.docx4j.vml.CTShape"){
                                 shape = (CTShape)((JAXBElement<?>)element).getValue();
                                 break;
                            }
                         }
                         else if( element instanceof CTShape ) {
                             //
                             shape = (CTShape)element;
                             break;
                         }
                     }
                     if( shape != null ){
                         List<JAXBElement<?>> shapeElements = shape.getPathOrFormulasOrHandles();
                         JAXBElement<?> imageData = Commons.findElement(shapeElements, "imagedata");
                         if( imageData != null){
                             CTImageData ctImageData = (CTImageData)imageData.getValue();
                             String pictRelId = ctImageData.getId();
                             RelationshipsPart relPart = numDefPart.getRelationshipsPart();
                             Relationship rel = numDefPart.getRelationshipsPart().getRelationshipByID(pictRelId);
                             String imageUrl;
                             try {
                                 imageUrl = URIHelper. resolvePartUri(relPart.getSourceURI(), new URI(rel.getTarget())).getPath();
                                 if(imageUrl.charAt(0)=='/') {
                                     imageUrl = imageUrl.substring(1);
                                 }
                                 picMap.put( new Long( numPicBullet.getNumPicBulletId().longValue() ), imageUrl );
                             } catch (URISyntaxException e) {
                                // TODO Auto-generated catch block
                             }
                         }
                     }
                }
            }
        }


        //abstract numbering container
        HashMap<Long, JSONObject> abstractNumMap = new HashMap<Long, JSONObject>();
        //iterator for abstract definitions
        Iterator<AbstractNum> abstractIter = numbering.getAbstractNum().iterator();
        while( abstractIter.hasNext() ){
            AbstractNum abstractNum = abstractIter.next();
            long abstractNumId = abstractNum.getAbstractNumId().longValue();
            final JSONObject listObject = new JSONObject();
            //now create an operation
            abstractNum.getMultiLevelType();//hybridMultilevel, multilevel, singleLevel
            Iterator<Lvl> levelIter = abstractNum.getLvl().iterator();
            for( int index = 0; index < 9; ++index ){
                final JSONObject levelObject = new JSONObject();
                if( levelIter.hasNext() ){
                    Lvl level = levelIter.next();
                    createLevel( levelObject, level, picMap );
                }
                listObject.put( "listLevel" + index, levelObject );
            }
            abstractNumMap.put( new Long(abstractNumId), listObject );
        }
        //
        Iterator<Num> numListIter = numbering.getNum().iterator();
        while( numListIter.hasNext() ){
            Num num = numListIter.next();

            JSONObject listObject = abstractNumMap.get(num.getAbstractNumId().getVal().longValue());
            List<LvlOverride> lvlOverride = num.getLvlOverride();

            if( lvlOverride != null && !lvlOverride.isEmpty()){
                listObject = new JSONObject(listObject.toString());
                Iterator<LvlOverride> lvlIter = lvlOverride.iterator();
                while( lvlIter.hasNext() ){
                    LvlOverride override = lvlIter.next();
                    long ilvlOverride = override.getIlvl().longValue();
                    JSONObject levelObject = listObject.getJSONObject("listLevel" + ilvlOverride);
                    StartOverride startOverride = override.getStartOverride();
                    if( startOverride != null )
                        levelObject.put("listStartValue", startOverride.getVal().longValue());
                    Lvl level = override.getLvl();
                    if( level != null ){
                        createLevel( levelObject, level, picMap );
                    }
                }
            }
            AddInsertListOperation( num.getNumId().longValue(), listObject);
        }
    }
}
