/**
 * **********************************************************************
 *
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER
 *
 * Copyright 2008, 2010 Oracle and/or its affiliates. All rights reserved.
 *
 * Use is subject to license terms.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at http://www.apache.org/licenses/LICENSE-2.0. You can also
 * obtain a copy of the License at http://odftoolkit.org/docs/license.txt
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ***********************************************************************
 */
package org.odftoolkit.odfdom.pkg;

import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.apache.xerces.dom.ElementNSImpl;
import org.apache.xerces.dom.ParentNode;
import org.odftoolkit.odfdom.component.Component;
import org.odftoolkit.odfdom.component.OdfFileSaxHandler;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.dom.element.OdfStylableElement;
import org.odftoolkit.odfdom.dom.element.office.OfficeAnnotationElement;
import org.odftoolkit.odfdom.dom.element.office.OfficeBodyElement;
import org.odftoolkit.odfdom.dom.element.text.TextAElement;
import org.odftoolkit.odfdom.dom.element.text.TextNoteElement;
import org.odftoolkit.odfdom.dom.element.text.TextSElement;
import org.odftoolkit.odfdom.dom.element.text.TextSpanElement;
import org.odftoolkit.odfdom.incubator.doc.style.OdfStyle;
import org.w3c.dom.Attr;
import org.w3c.dom.DOMException;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.Text;

abstract public class OdfElement extends ElementNSImpl {

	private static final long serialVersionUID = -4939293285696678939L;
	private boolean isComponentRoot = false;
	private boolean mIsIngoredComponent = false;
	private Component mComponent = null;
	// ToDo: Only on component roots
	private int mComponentSize = 0;

	/**
	 * Creates a new instance of OdfElement
	 */
	public OdfElement(OdfFileDom ownerDocument, String namespaceURI,
			String qualifiedName) throws DOMException {
		super(ownerDocument, namespaceURI, qualifiedName);
	}

	/**
	 * Creates a new instance of OdfElement
	 */
	public OdfElement(OdfFileDom ownerDocument, OdfName aName)
			throws DOMException {
		super(ownerDocument, aName.getUri(), aName.getQName());
	}

	abstract public OdfName getOdfName();

	protected <T extends OdfElement> T getParentAs(Class<T> clazz) {
		Node parent = getParentNode();
		if (parent != null && clazz.isInstance(parent)) {
			return clazz.cast(parent);
		} else {
			return null;
		}
	}

	protected <T extends OdfElement> T getAncestorAs(Class<T> clazz) {
		Node node = getParentNode();
		while (node != null) {
			if (clazz.isInstance(node)) {
				return clazz.cast(node);
			}
			node = node.getParentNode();
		}
		return null;
	}

	@Override
	public String toString() {
		return mapNode(this, new StringBuilder()).toString();
	}

	/**
	 * Only Siblings will be traversed by this method as Children
	 */
	static private StringBuilder mapNodeTree(Node node, StringBuilder xml) {
		while (node != null) {
			// mapping node and this mapping include always all descendants
			xml = mapNode(node, xml);
			// next sibling will be mapped to XML
			node = node.getNextSibling();
		}
		return xml;
	}

	private static StringBuilder mapNode(Node node, StringBuilder xml) {
		if (node instanceof Element) {
			xml = mapElementNode(node, xml);
		} else if (node instanceof Text) {
			xml = mapTextNode(node, xml);
		}
		return xml;
	}

	private static StringBuilder mapTextNode(Node node, StringBuilder xml) {
		if (node != null) {
			xml = xml.append(node.getTextContent());
		}
		return xml;
	}

	private static StringBuilder mapElementNode(Node node, StringBuilder xml) {
		if (node != null) {
			xml = xml.append("<");
			xml = xml.append(node.getNodeName());
			xml = mapAttributeNode(node, xml);
			xml = xml.append(">");
			xml = mapNodeTree(node.getFirstChild(), xml);
			xml = xml.append("</");
			xml = xml.append(node.getNodeName());
			xml = xml.append(">");
		}
		return xml;
	}

	private static StringBuilder mapAttributeNode(Node node, StringBuilder xml) {
		NamedNodeMap attrs = null;
		int length;
		if ((attrs = node.getAttributes()) != null
				&& (length = attrs.getLength()) > 0) {
			for (int i = 0; length > i; i++) {
				xml = xml.append(" ");
				xml = xml.append(attrs.item(i).getNodeName());
				xml = xml.append("=\"");
				xml = xml.append(attrs.item(i).getNodeValue());
				xml = xml.append("\"");
			}
		}
		return xml;
	}

	/**
	 * Set the value of an ODF attribute by
	 * <code>OdfName</code>.
	 *
	 * @param name The qualified name of the ODF attribute.
	 * @param value The value to be set in <code>String</code> form
	 */
	public void setOdfAttributeValue(OdfName name, String value) {
		setAttributeNS(name.getUri(), name.getQName(), value);
	}

	/**
	 * Set an ODF attribute to this element
	 *
	 * @param attribute the attribute to be set
	 */
	public void setOdfAttribute(OdfAttribute attribute) {
		setAttributeNodeNS(attribute);
	}

	/**
	 * Retrieves a value of an ODF attribute by
	 * <code>OdfName</code>.
	 *
	 * @param name The qualified name of the ODF attribute.
	 * @return The value of the attribute as <code>String</code> or
	 * <code>null</code> if the attribute does not exist.
	 */
	public String getOdfAttributeValue(OdfName name) {
		return getAttributeNS(name.getUri(), name.getLocalName());
	}

	/**
	 * Retrieves an ODF attribute by
	 * <code>OdfName</code>.
	 *
	 * @param name The qualified name of the ODF attribute.
	 * @return The <code>OdfAttribute</code> or <code>null</code> if the
	 * attribute does not exist.
	 */
	public OdfAttribute getOdfAttribute(OdfName name) {
		return (OdfAttribute) getAttributeNodeNS(name.getUri(), name.getLocalName());
	}

	/**
	 * Retrieves an ODF attribute by
	 * <code>NamespaceName</code>, and local name.
	 *
	 * @param namespace The namespace of the ODF attribute.
	 * @param localname The local name of the ODF attribute.
	 * @return The <code>OdfAttribute</code> or <code>null</code> if the
	 * attribute does not exist.
	 */
	public OdfAttribute getOdfAttribute(NamespaceName namespace, String localname) {
		return (OdfAttribute) getAttributeNodeNS(namespace.getUri(),
				localname);
	}

	/**
	 * Determines if an ODF attribute exists.
	 *
	 * @param name The qualified name of the ODF attribute.
	 * @return True if the attribute exists.
	 */
	public boolean hasOdfAttribute(OdfName name) {
		return hasAttributeNS(name.getUri(), name.getLocalName());
	}

	/**
	 * returns the first child node that implements the given class.
	 *
	 * @param <T> The type of the ODF element to be found.
	 * @param clazz is a class that extends OdfElement.
	 * @param parentNode is the parent O of the children to be found.
	 * @return the first child node of the given parentNode that is a clazz or
	 * null if none is found.
	 */
	@SuppressWarnings("unchecked")
	static public <T extends OdfElement> T findFirstChildNode(Class<T> clazz,
			Node parentNode) {
		if (parentNode != null && parentNode instanceof ParentNode) {
			Node node = ((ParentNode) parentNode).getFirstChild();
			while ((node != null) && !clazz.isInstance(node)) {
				node = node.getNextSibling();
			}

			if (node != null) {
				return (T) node;
			}
		}

		return null;
	}

	/**
	 * returns the first sibling after the given reference node that implements
	 * the given class.
	 *
	 * @param <T> The type of the ODF element to be found.
	 * @param clazz is a class that extends OdfElement.
	 * @param refNode the reference node of the siblings to be found.
	 * @return the first sibling of the given reference node that is a class or
	 * null if none is found.
	 */
	@SuppressWarnings("unchecked")
	static public <T extends OdfElement> T findNextChildNode(Class<T> clazz,
			Node refNode) {
		if (refNode != null) {
			Node node = refNode.getNextSibling();
			while (node != null && !clazz.isInstance(node)) {
				node = node.getNextSibling();
			}

			if (node != null) {
				return (T) node;
			}
		}

		return null;
	}

	/**
	 * returns the first previous sibling before the given reference node that
	 * implements the given class.
	 *
	 * @param clazz is a class that extends OdfElement.
	 * @param refNode the reference node which siblings are to be searched.
	 * @return the first previous sibling of the given reference node that is a
	 * class or null if none is found.
	 */
	@SuppressWarnings("unchecked")
	static public <T extends OdfElement> T findPreviousChildNode(
			Class<T> clazz, Node refNode) {
		if (refNode != null) {
			Node node = refNode.getPreviousSibling();
			while (node != null && !clazz.isInstance(node)) {
				node = node.getPreviousSibling();
			}

			if (node != null) {
				return (T) node;
			}
		}

		return null;
	}

	@Override
	/**
	 * Clones this element but without cloning xml:id attributes
	 *
	 * @param deep if a deep copy should happen, otherwise only the element with
	 * attributes and content is copied
	 */
	public Node cloneNode(boolean deep) {
		OdfElement cloneElement = ((OdfFileDom) this.ownerDocument).newOdfElement(this.getClass());
		// if it is an unknown ODF element
		if (cloneElement == null) {
			cloneElement = new OdfAlienElement((OdfFileDom) this.getOwnerDocument(), OdfName.newName(OdfNamespace.getNamespace(this.getNamespaceURI()), this.getTagName()));
		}

		if (attributes != null) {
			for (int i = 0; i < attributes.getLength(); i++) {
				Node item = attributes.item(i);
				String qname = null;
				String prefix = item.getPrefix();
				if (prefix == null) {
					qname = item.getLocalName();
					cloneElement.setAttribute(qname, item.getNodeValue());
				} else {
					qname = prefix + ":" + item.getLocalName();
					if (!qname.equals("xml:id")) {
						cloneElement.setAttributeNS(item.getNamespaceURI(), qname, item.getNodeValue());
					}
				}
			}
		}

		// aside of the XML the flag of being a component root have to be copied
		if (this.isComponentRoot) {
			cloneElement.markAsComponentRoot(true);
		}


		if (deep) {
			Node childNode = getFirstChild();
			while (childNode != null) {
				cloneElement.appendChild(childNode.cloneNode(true));
				childNode = childNode.getNextSibling();
			}
		}
		// ToDo: There should be an easier - more obvious - way than this... 
		if (this.selfAndDescendantTextIgnoredAsComponent()) {
			cloneElement.ignoredComponent(true);
		}
		cloneElement.mComponentSize = this.mComponentSize;
		return cloneElement;
	}

	/**
	 * @depth level of children to be cloned All attributes except xml:id
	 * attributes will not be cloned.
	 */
	public Node cloneNode(int depth) {
		OdfElement cloneElement = ((OdfFileDom) this.ownerDocument).newOdfElement(this.getClass());
		// if it is an unknown ODF element
		if (cloneElement == null) {
			cloneElement = new OdfAlienElement((OdfFileDom) this.getOwnerDocument(), OdfName.newName(OdfNamespace.getNamespace(this.getNamespaceURI()), this.getTagName()));
		}

		if (attributes != null) {
			for (int i = 0; i < attributes.getLength(); i++) {
				Node item = attributes.item(i);
				String qname = null;
				String prefix = item.getPrefix();
				if (prefix == null) {
					qname = item.getLocalName();
					cloneElement.setAttribute(qname, item.getNodeValue());
				} else {
					qname = prefix + ":" + item.getLocalName();
					if (!qname.equals("xml:id")) {
						cloneElement.setAttributeNS(item.getNamespaceURI(), qname, item.getNodeValue());
					}
				}
			}
		}

		if (depth > 0) {
			Node childNode = getFirstChild();
			while (childNode != null) {
				if (childNode instanceof OdfElement) {
					cloneElement.appendChild(((OdfElement) childNode).cloneNode(depth - 1));
				}
				childNode = childNode.getNextSibling();
			}
		}

		return cloneElement;
	}

	/**
	 * Clones the content of the source element including attributes even xml:id
	 * to the target element. Helpful when changing a <text:h> to a <text:p> and
	 * vice versa, when outline attribute changes.
	 *
	 * @param deep if a deep copy should happen.
	 */
	public static OdfElement cloneNode(OdfElement source, OdfElement target, boolean deep) {
		NamedNodeMap attributes = source.getAttributes();
		if (attributes != null) {
			for (int i = 0; i < attributes.getLength(); i++) {
				Node item = attributes.item(i);
				String qname = null;
				String prefix = item.getPrefix();
				if (prefix == null) {
					qname = item.getLocalName();
					target.setAttribute(qname, item.getNodeValue());
				} else {
					qname = prefix + ":" + item.getLocalName();
					//if (!qname.equals("xml:id")) {
					target.setAttributeNS(item.getNamespaceURI(), qname, item.getNodeValue());
					//}
				}
			}
		}

		// aside of the XML the flag of being a component root have to be copied
		if (source.isComponentRoot) {
			target.markAsComponentRoot(true);
		}


		if (deep) {
			Node childNode = source.getFirstChild();
			while (childNode != null) {
				target.appendChild(childNode.cloneNode(true));
				childNode = childNode.getNextSibling();
			}
		}
		// ToDo: There should be an easier - more obvious - way than this... 
		if (source.selfAndDescendantTextIgnoredAsComponent()) {
			target.ignoredComponent(true);
		}
		target.mComponentSize = source.mComponentSize;
		return target;
	}

	@Override
	public Node appendChild(Node node) {
		// No Counting necessary as appendChild() will call insertBefore()
		return super.appendChild(node);
	}

	/**
	 * Recursive traverse the potential text container and count its content
	 * size
	 */
	private int descendentsCount(Node parent, int size) {
		if (!isIgnoredElement((Element) parent)) {
			NodeList children = parent.getChildNodes();
			Node child;
			for (int i = 0; i < children.getLength(); i++) {
				child = children.item(i);
				if (child instanceof Text) {
					size += ((Text) child).getLength();
				} else if (child instanceof OdfElement) {
					OdfElement element = (OdfElement) child;
					if (OdfFileSaxHandler.isTextSelection(element)) {
						size += descendentsCount(element, size);
					} else if (((OdfElement) element).isComponentRoot()) {
						size += element.componentLength();
					}
				}
			}
		}
		return size;
	}

	public int getRepetition() {
		return 1;
	}

	/**
	 * Recursive traverse the text container and count the size of the content
	 */
	public int componentLength() {
//////SVANTE CLEAN ME		
//		if(mComponentSize == null){
//			if(isComponentRoot()){
//				mComponentSize = 1;
//			}else{
//				mComponentSize = 0;
//			}
//		} 		
		return mComponentSize;
	}

	/**
	 * indicates if some other object is equal to this one.
	 *
	 * @param obj - the reference object with which to compare.
	 * @return true if this object is the same as the obj argument; false
	 * otherwise.
	 */
	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}

		if ((obj == null) || !(obj instanceof OdfElement)) {
			return false;
		}

		OdfElement compare = (OdfElement) obj;

		// compare node name
		if (!localName.equals(compare.localName)) {
			return false;
		}

		if (!this.namespaceURI.equals(compare.namespaceURI)) {
			return false;
		}

		// compare node attributes
		if (attributes == compare.attributes) {
			return true;
		}

		if ((attributes == null) || (compare.attributes == null)) {
			return false;
		}

		int attr_count1 = attributes.getLength();
		int attr_count2 = compare.attributes.getLength();

		List<Node> attr1 = new ArrayList<Node>();
		for (int i = 0; i < attr_count1; i++) {
			Node node = attributes.item(i);
			if (node.getNodeValue().length() == 0) {
				continue;
			}
			attr1.add(node);
		}

		List<Node> attr2 = new ArrayList<Node>();
		for (int i = 0; i < attr_count2; i++) {
			Node node = compare.attributes.item(i);
			if (node.getNodeValue().length() == 0) {
				continue;
			}
			attr2.add(node);
		}

		if (attr1.size() != attr2.size()) {
			return false;
		}

		for (int i = 0; i < attr1.size(); i++) {
			Node n1 = attr1.get(i);
			if (n1.getLocalName().equals("name")
					&& n1.getNamespaceURI().equals(
					OdfDocumentNamespace.STYLE.getUri())) {
				continue; // do not compare style names
			}
			Node n2 = null;
			int j = 0;
			for (j = 0; j < attr2.size(); j++) {
				n2 = attr2.get(j);
				if (n1.getLocalName().equals(n2.getLocalName())) {
					String ns1 = n1.getNamespaceURI();
					String ns2 = n2.getNamespaceURI();
					if (ns1 != null && ns2 != null && ns1.equals(ns2)) {
						break;
					}
				}
			}
			if (j == attr2.size()) {
				return false;
			}

			if (!n1.getTextContent().equals(n2.getTextContent())) {
				return false;
			}
		}

		// now compare child elements
		NodeList childs1 = this.getChildNodes();
		NodeList childs2 = compare.getChildNodes();

		int child_count1 = childs1.getLength();
		int child_count2 = childs2.getLength();
		if ((child_count1 == 0) && (child_count2 == 0)) {
			return true;
		}

		List<Node> nodes1 = new ArrayList<Node>();
		for (int i = 0; i < child_count1; i++) {
			Node node = childs1.item(i);
			if (node.getNodeType() == Node.TEXT_NODE) {
				if (node.getNodeValue().trim().length() == 0) {
					continue; // skip whitespace text nodes
				}
			}
			nodes1.add(node);
		}

		List<Node> nodes2 = new ArrayList<Node>();
		for (int i = 0; i < child_count2; i++) {
			Node node = childs2.item(i);
			if (node.getNodeType() == Node.TEXT_NODE) {
				if (node.getNodeValue().trim().length() == 0) {
					continue; // skip whitespace text nodes
				}
			}
			nodes2.add(node);
		}

		if (nodes1.size() != nodes2.size()) {
			return false;
		}

		for (int i = 0; i < nodes1.size(); i++) {
			Node n1 = nodes1.get(i);
			Node n2 = nodes2.get(i);
			if (!n1.equals(n2)) {
				return false;
			}
		}
		return true;
	}

	protected void onRemoveNode(Node node) {
		Node child = node.getFirstChild();
		while (child != null) {
			this.onRemoveNode(child);
			child = child.getNextSibling();
		}

		if (OdfElement.class.isInstance(node)) {
			((OdfElement) node).onRemoveNode();
		}
	}

	protected void onInsertNode(Node node) {
		Node child = node.getFirstChild();
		while (child != null) {
			this.onInsertNode(child);
			child = child.getNextSibling();
		}

		if (OdfElement.class.isInstance(node)) {
			((OdfElement) node).onInsertNode();
		}
	}

	protected void onRemoveNode() {
	}

	protected void onInsertNode() {
	}

	@Override
	public Node insertBefore(Node newChild, Node refChild) throws DOMException {
		Node n = null;
		onInsertNode(newChild);
		n = super.insertBefore(newChild, refChild);
		raiseComponentSize(newChild);
		return n;
	}


	/* Removes the element from the DOM tree, but keeping its ancestors by moving its children in its place */
	public static Element removeElement(Element oldElement) throws DOMException {
		Element parent = (Element) oldElement.getParentNode();
		if (parent != null) {
			NodeList children = oldElement.getChildNodes();
			int childCount = children.getLength();
			Node lastChild = children.item(childCount - 1);
			parent.replaceChild(lastChild, oldElement);
			Node newChild;
			for (int i = childCount - 2; i >= 0; i--) {
				newChild = children.item(i);
				parent.insertBefore(newChild, lastChild);
				lastChild = newChild;
			}
		}
		return parent;
	}

	@Override
	public Node removeChild(Node oldChild) throws DOMException {
		onRemoveNode(oldChild);
		reduceComponentSize(oldChild);
		return super.removeChild(oldChild);
	}

	/**
	 * Component size is being reduced by the size of the child
	 */
	private void reduceComponentSize(Node child) {
		if (child instanceof Text) {
			changeSize(-1 * ((Text) child).getLength());
		} else if (child instanceof Element) {
			if (OdfFileSaxHandler.isTextComponentRoot(child)) {
				changeSize(-1);
			} else {
				changeSize(-1 * descendentsCount((Element) child, 0));
			}
		}
	}

	/**
	 * Returns if the text should be returned or is under a nested paragraph or
	 * ignored text element (e.g. text:note-citation).
	 */
	private boolean isIgnoredText(OdfElement parent) {
		boolean isIgnored = true;
		while (parent != null) {
			if (parent.isComponentRoot() || parent.selfAndDescendantTextIgnoredAsComponent()) {
				if (parent.selfAndDescendantTextIgnoredAsComponent()) {
					isIgnored = true;
				} else {
					isIgnored = false;
				}
				break;
			}
			parent = parent.getParentAs(OdfElement.class);
		}
		return isIgnored;
	}

	/**
	 * Component size is being reduced by the size of the child
	 */
	private void raiseComponentSize(Node child) {
		if (child instanceof Text) {
			if (!isIgnoredText((OdfElement) child.getParentNode())) {
				changeSize(((Text) child).getLength());
			}
		} else if (child instanceof Element) {
			if (child instanceof OdfElement && ((OdfElement) child).isComponentRoot()) {
				if (!((OdfElement) child).selfAndDescendantTextIgnoredAsComponent()) {
					// in theory 1 is the default and repeated factors and space count factors should be applied
					// it is something different to size (which returns the content size, instead it is like a width?)

					// Elements with repetition are: text:s, table:table-cell and table:table-row
					int repetition = ((OdfElement) child).getRepetition();
					if (repetition != 1) {
						changeSize(repetition);
					} else {
						changeSize(1);
					}
				}
//				// SPECIAL HANDLING FOR FRAME/IMAGE COMPONENT
//			} else if (child instanceof DrawImageElement) {
//				Node precedingImageSibling = null;
//				precedingImageSibling = child;
//				boolean frameAlreadyCount = false;
//				while ((precedingImageSibling = precedingImageSibling.getPreviousSibling()) != null) {
//					if (precedingImageSibling instanceof DrawImageElement) {
//						frameAlreadyCount = true;
//					}
//				}
//				if (!frameAlreadyCount) {
//					Node parent = child.getParentNode();
//					if (parent != null) {
//						Node grandParent = parent.getParentNode();
//						if (grandParent != null) {
//							((OdfElement) grandParent).changeSize(1);
//						}
//					}
//				}
			} else {
				// ToDo: Improvement: we may limit to elements that may be text delimiter (span) or inbetween a text container and text delimiter (for ongoing recursion)
				changeSize(descendentsCount((Element) child, 0));
			}
		}
	}

	/**
	 * A change of the element will be raised to the top till a component is
	 * found
	 */
	// ToBeMoved to TEXT CONTINER CHILDREN only!
	private int changeSize(int sizeDifference) {
		int size = 0;
		if (sizeDifference != 0) {
			OdfElement element = this;
			if (!element.isComponentRoot() && !isIgnoredElement(element)) {
				do {
					element = element.getParentAs(OdfElement.class);
				} while (element != null && !element.isComponentRoot() && !isIgnoredElement(element));
			}
			if (element != null && element.isComponentRoot()) {
				element.mComponentSize = element.componentLength() + sizeDifference;
				size = element.mComponentSize;
//				element.mComponentSize += sizeDifference;
//				size = element.mComponentSize;


			}
		}
		return size;
	}

	/**
	 * Removes all the content from the element
	 */
	public void removeContent() {
		NodeList children = this.getChildNodes();
		Node child;
		for (int i = 0; i < children.getLength(); i++) {
			child = children.item(i);
			this.removeChild(child);
		}
		mComponentSize = 0;
	}

	@Override
	public Node replaceChild(Node newChild, Node oldChild) throws DOMException {
		raiseComponentSize(newChild);
		onRemoveNode(oldChild);
		onInsertNode(newChild);
		//Currently the replace only reduces
		//reduceComponentSize(oldChild);
		return super.replaceChild(newChild, oldChild);
	}

	/**
	 * Accept an visitor instance to allow the visitor to do some operations.
	 * Refer to visitor design pattern to get a better understanding.
	 *
	 * @param visitor	an instance of DefaultElementVisitor
	 */
	public void accept(ElementVisitor visitor) {
		visitor.visit(this);
	}

	/**
	 * Returns the component reference
	 */
	public OdfElement getComponentRoot() {
		OdfElement element = this;
		if (!element.isComponentRoot()) {
			do {
				element = element.getParentAs(OdfElement.class);
			} while (element != null && !element.isComponentRoot());
		}
		return element;
	}

	public void markAsComponentRoot(boolean isRoot) {
		isComponentRoot = true;
		OdfElement parent = (OdfElement) getParentNode();
//		// if the component was already added, add the size now
//		if(parent != null && this instanceof DrawFrameElement){
//			parent.raiseComponentSize(this);
//		}
	}

	public Component getComponent() {
		return mComponent;
	}

	public void setComponent(Component component) {
		mComponent = component;
	}

	public boolean isComponentRoot() {
		return isComponentRoot;
	}

	/**
	 * @return true if the text should not count as for component path nor the
	 * element root itself. This might occur for nested paragraphs or ignored
	 * text element (e.g. text:note-citation).
	 */
	public boolean selfAndDescendantTextIgnoredAsComponent() {
		return mIsIngoredComponent;
	}

	/**
	 * @param true if the text should not count as for component path nor the
	 * element root itself. This might occur for nested paragraphs or ignored
	 * text element (e.g. text:note-citation). For instance called by a SAX
	 * Component parser, * *
	 * see <code>org.odftoolkit.odfdom.component.OdfFileSaxHandler</code>
	 */
	public void ignoredComponent(boolean isIngoredComponent) {
		mIsIngoredComponent = isIngoredComponent;
	}

	/**
	 * If the string is inserted into a text:p/text:h element and it will be
	 * inserted in the start/end all spaces are replaced by <text:s/>
	 * element(s). tabulator and linefeeds are being removed.
	 *
	 * If both the previous text node ends with a space and newString starts
	 * with a space, we would need to encode the single leading space as an
	 * element, otherwise it would be stripped. Same occurs for the next text
	 * node and an ending space. For Example:
	 * <span> text </span><text:s c="7"/><span> text2 </span> <== SAVE when
	 * starting ending a span as well with space element independent of
	 * preceding
	 */
	static protected void appendUsingWhitespaceHandling(Node precedingNode, OdfElement parent, Node followingNode, String newString) {
//		addTextNode(precedingNode, parent, followingNode, newString);
		// only add text, if parent exists 
		if (parent != null) {
			int spaceCount = 0;
			// Note: The delta between startPosition and endPosition marks the text to be written out
			// startPosition will only be raised to endposition, when characters have to be skipped!			
			int startPos = 0;
			int endPos = 0;
			// check if first character is a white space			
			for (int i = 0; i < newString.length(); i++) {
				char c = newString.charAt(i);
				if (c == '\u0020' // space
						|| c == '\t' // \t (tabulator = 0x09) 
						// \r (carriage return = 0x0D) 
						|| c == '\r'
						// \n (line feed =' 0x0A)
						|| c == '\n') {
					spaceCount++;

					if (spaceCount > 1) {
						// if there are more than one space a space element have to be inserted, write out the previous spaces
						if (endPos - startPos > 0) {
							precedingNode = addTextNode(precedingNode, parent, followingNode, newString.substring(startPos, endPos));
							// for the single whitespace not written
						}
						// NOT including the additional whitespace character
						startPos = endPos;
					}
				} else { // else if there was no whitespace character found or at the beginning
					if (spaceCount > 1 || i == 1 && spaceCount == 1) {
						TextSElement s = new TextSElement((OdfFileDom) parent.getOwnerDocument());
						// if there were multiple preceing whitespace, write out a space Element
						if (spaceCount > 1) {
							s.setTextCAttribute(spaceCount);
						}

						// write out space element
						precedingNode = addElementNode(precedingNode, parent, followingNode, s);
						endPos += spaceCount;
						startPos = endPos;
						spaceCount = 0;

						// reset space count to zero as now a character was found						
					} else if (spaceCount == 1) {
						endPos++;
						spaceCount = 0;
					}

					// write out character
					endPos++;// including this character					
				}
			}
			// reset space count to zero as now a character was found
			if (spaceCount > 1) {
				TextSElement s = new TextSElement((OdfFileDom) parent.getOwnerDocument());
				// if there were multiple preceing whitespace, write out a space Element
				if (spaceCount > 1) {
					s.setTextCAttribute(spaceCount);
				}
				// write out space element
				precedingNode = addElementNode(precedingNode, parent, followingNode, s);
				endPos += spaceCount;
				startPos = endPos;
				spaceCount = 0;
			}
			if (endPos - startPos > 0) {
				precedingNode = addTextNode(precedingNode, parent, followingNode, newString.substring(startPos, endPos));
			}
			if (spaceCount == 1) {
				TextSElement s = new TextSElement((OdfFileDom) parent.getOwnerDocument());
				precedingNode = addElementNode(precedingNode, parent, followingNode, s);
			}
		} else {
			Logger.getLogger(OdfElement.class.getName()).log(Level.SEVERE, "Node parent should not be NULL!");
		}
	}

	private static Node addElementNode(Node precedingNode, OdfElement parent, Node followingNode, Element newElement) {
		Node newNode = null;
		// APPEND: if there is no text node to expand and no element that follows
		if (followingNode == null) {
			newNode = parent.appendChild(newElement);
		} else if (followingNode != null) {
			// insert before the given element
			newNode = parent.insertBefore(newElement, followingNode);
		}
		return newNode;

	}

	private static Node addTextNode(Node precedingNode, OdfElement parent, Node followingNode, String newString) {
		Node newNode = null;
		// APPEND: if there is no text node to expand and no element that follows
		if (precedingNode == null && followingNode == null || precedingNode != null && precedingNode instanceof Element && followingNode == null) {
			newNode = parent.appendChild(parent.getOwnerDocument().createTextNode(newString));
		} else {
			// INSERT:
			if (precedingNode != null && precedingNode instanceof Text) {
				// insert at the end of the given text
				((Text) precedingNode).appendData(newString);
				newNode = precedingNode;
			} else if (followingNode != null) {
				// insert before the given element
				newNode = parent.insertBefore(parent.getOwnerDocument().createTextNode(newString), followingNode);
			}
		}
		return newNode;
	}

	private static Node addElementNode(Node precedingNode, OdfElement parent, Node followingNode, OdfElement newElement) {
		Node newNode = null;
		// APPEND: if there is no text node to expand and no element that follows
		if (followingNode == null) {
			newNode = parent.appendChild(newElement);
		} else {
			// INSERT:
			// insert before the given following-node
			newNode = parent.insertBefore(newElement, followingNode);
		}
		return newNode;
	}

	/**
	 * Splitting the element at the given position into two halves
	 *
	 * @param textPosStart The logical position of the first character (or other
	 * paragraph child component) that will be moved to the beginning of the new
	 * paragraph.
	 * @return the new created second text container
	 */
	public OdfElement split(int posStart) {
		OdfElement newSecondElement = (OdfElement) this.cloneNode(true);
		int size = OdfElement.getContentSize(this);

		// This will become the first paragraph
		// Only delete if the start node is within the component length
		// Do NOT do a a 
		// if there is only one character the size is 1 and the textPosStart would be 0
//TODO FIXME: Why was the parent DELETEd, when next line was after the condition??!?
		Element parent = (Element) this.getParentNode();
		if (size > posStart) {
			this.delete(posStart, size);
		}
		Node _nextSibling = this.getNextSibling();
		if (_nextSibling != null) {
			parent.insertBefore(newSecondElement, _nextSibling);
		} else {
			parent.appendChild(newSecondElement);
		}

		// only delete if the start position is not before the first component
		if (posStart != 0) {
			// minus one, as the textPosStart was already in the first Element
			newSecondElement.delete(0, posStart - 1);
		}

		return newSecondElement;
		// FIXME: There is a better way than cloning and deleting two halves, but it works for POC
		//splitNodes(this.getFirstChild(), textPosStart, 0);
	}

	/**
	 * ********************************************************
	 */
	/**
	 * Receives node from this text container element.
	 *
	 * @param textPosEnd The end delimiter for the deletion. To delete text to
	 * the end of the paragraph, as represent for the end of the paragraph
	 * Integer.MAX_VALUE can be used.
	 */
	public Node receiveNode(int textPosStart) {
		if (textPosStart < 0) {
			Logger.getLogger(OdfElement.class.getName()).warning("A negative index " + textPosStart + " was given to insert text into the paragraph!");
		}
		// start recrusion
		ArrayList<Node> nodeContainer = new ArrayList(1);
		TextContentTraverser.traverseSiblings(this.getFirstChild(), 0, textPosStart, textPosStart + 1, TextContentTraverser.Algorithm.RECEIVE, nodeContainer);

		Node receivedNode = null;
		if (nodeContainer.size() == 1) {
			receivedNode = nodeContainer.get(0);
		}
		return receivedNode;
	}

	/**
	 * @param textPosStart the first text level component to be marked, start
	 * counting with 0
	 * @param textPosEnd the last text level component to be marked, start
	 * counting with 0
	 * @param textPosEnd the last text level component to be marked, start
	 * counting with 0
	 * @param newSelection the element that should embrace the text defined by
	 * the positions provided
	 */
	public void markText(int textPosStart, int textPosEnd, OdfElement newSelection) {
		if (textPosStart < 0) {
			Logger.getLogger(OdfElement.class.getName()).warning("A negative index " + textPosStart + " was given to insert text into the paragraph!");
		}
		if (textPosEnd < textPosStart) {
			// might be caused by invalid span around whitespace that is being eleminated by ODF whitespacehandling
			Logger.getLogger(OdfElement.class.getName()).warning("The start index " + textPosStart + " shall not be higher than the end index " + textPosEnd + "!");
		}
		// incrementing textPosEnd to get in sync with string counting
		TextContentTraverser.traverseSiblings(this.getFirstChild(), 0, textPosStart, textPosEnd + 1, TextContentTraverser.Algorithm.MARK, newSelection);
	}

	/**
	 * Counts the number of descendant components
	 */
	public int countDescendantComponents() {
//		if (textPosStart < 0) {
//			Logger.getLogger(OdfElement.class.getName()).warning("A negative index " + textPosStart + " was given to insert text into the paragraph!");
//		}
//		if (textPosEnd < textPosStart) {
//			// might be caused by invalid span around whitespace that is being eleminated by ODF whitespacehandling
//			Logger.getLogger(OdfElement.class.getName()).warning("The start index " + textPosStart + " shall not be higher than the end index " + textPosEnd + "!");
//		}

		// incrementing textPosEnd to get in sync with string counting
		//return TextContentTraverser.traverseSiblings(this.getFirstChild(), 0, textPosStart, textPosEnd + 1, TextContentTraverser.Algorithm.MARK, newSelection);
		return TextContentTraverser.traverseSiblings(this.getFirstChild(), 0, 0, Integer.MAX_VALUE, TextContentTraverser.Algorithm.COUNT, Integer.MAX_VALUE);
	}

// Below recursion works, the new one not yet..	
//	public void moveChildrenTo(Element newParent) {
//		// incrementing textPosEnd to get in sync with string counting
//		TextContentTraverser.traverseSiblings(this.getFirstChild(), 0, 0, Integer.MAX_VALUE, TextContentTraverser.Algorithm.MOVE, newParent);
//	}
	public void moveChildrenTo(Element newParent) {
		moveNodes(this.getFirstChild(), newParent);
	}

	private void moveNodes(Node node, Element newParent) {
		while (node != null) {
			// IMPORTANT: Get next sibling first, otherwise references get lost when appending to new parent
			Node _nextSibling = node.getNextSibling();
			if (node instanceof Element) {
				newParent.appendChild(node);
			} else if (node instanceof Text) {
				moveTextNode((Text) node, newParent);
			}
			node = _nextSibling;
		}
	}

	private void moveTextNode(Text node, Element newParent) {
		if (node != null) {
			newParent.appendChild(node);
		}
	}

	/**
	 * Insert text to a certain position. The text will be appended to the
	 * previous position text, so the span of the previous character will be
	 * expanded
	 *
	 * @param newString string to be inserted
	 * @param position text index of the new string
	 */
	public void insert(String newString, int textPosStart) {
		if (newString != null && !newString.isEmpty()) {
			insertContent(newString, textPosStart);
		}
	}

	public void insert(Element newNode, int textPosStart) {
		if (newNode != null) {
			insertContent(newNode, textPosStart);
		}
	}

	/**
	 * Insert text to a certain position. The text will be appended to the
	 * previous position text, so the span of the previous character will be
	 * expanded
	 *
	 * @param newString string to be inserted
	 * @param position text index of the new string
	 */
	private void insertContent(Object newContent, int textPosStart) { // parameter order?				
		if (textPosStart < 0) {
			Logger.getLogger(OdfElement.class.getName()).warning("A negative index " + textPosStart + " was given to insert text into the paragraph!");
		}
		Node firstChild = this.getFirstChild();
		if (firstChild == null) {
			// if there is no new node, simply exchange multiple whitespaces with <text:s>
			if (newContent instanceof String) {
				appendUsingWhitespaceHandling(null, (OdfElement) this, null, (String) newContent);
			} else if (newContent instanceof Element) {
				this.appendChild((Element) newContent);
			}
		} else {
			List newData = new ArrayList(2);
			newData.add(newContent);
			//int currentPos = insertNodes(node, newString, textPosStart, 0);
			int currentPos = TextContentTraverser.traverseSiblings(firstChild, 0, textPosStart, textPosStart, TextContentTraverser.Algorithm.INSERT, newData);
			if (newData.size() == 1) {
				// if there is were element(s), but no components within this element
				if (newContent instanceof String) {
					appendUsingWhitespaceHandling(null, (OdfElement) this, null, (String) newContent);
				} else if (newContent instanceof Element) {
					this.appendChild((Element) newContent);
				}
			}
//				if (currentPos > textPosStart) {
//					Logger.getLogger(OdfElement.class.getName()).warning("The index " + textPosStart + " is outside the existing text of the paragraph!");
//				}
		}
	}

	/**
	 * Deletes text from this paragraph element.
	 *
	 * @param textPosStart Counting starts with 0, which is the first character
	 * of the paragraph.
	 * @param textPosEnd The end delimiter for the deletion. To delete text to
	 * the end of the paragraph, as represent for the end of the paragraph
	 * Integer.MAX_VALUE can be used.
	 */
	public void delete(int textPosStart, int textPosEnd) {
		if (textPosStart < 0) {
			Logger.getLogger(OdfElement.class.getName()).warning("A negative index " + textPosStart + " was given to insert text into the paragraph!");
		}
		if (textPosEnd < textPosStart) {
			Logger.getLogger(OdfElement.class.getName()).warning("The start index " + textPosStart + " have to be higher than the end index " + textPosEnd + "!");
		}

		/**
		 * Deletion implementation: 1) The text position of the first deletion
		 * will be searched. 2) Afterwards all following-sibling, text and nodes
		 * will be deleted, until the end text position is found 3) Split the
		 * text take the returned and remove it from parent
		 */
		List deleteStatus = new ArrayList(1);

		// start recrusion
		TextContentTraverser.traverseSiblings(this.getFirstChild(), 0, textPosStart, textPosEnd + 1, TextContentTraverser.Algorithm.DELETE, deleteStatus);
	}

	private static class TextContentTraverser {

		/**
		 *
		 * @param node the element node will be checked if it is a text or an
		 * element. If an element it will be dispatched to check component than
		 * executed
		 * @param currentPos the current component position
		 * @param posStart the text position where the span starts
		 * @param posEnd the text position, where the span ends (one higher as
		 * the last component number to be included)
		 * @param algorithm dependent on this variable a different subroutine is
		 * being used after traversing the sub-tree, e.g. insert, delete, mark,
		 * count..
		 * @param data differs from the type of algorithm, e.g. for insert it
		 * contains the data to be inserted
		 * @return the current position after the node was processed
		 */
		static private int traverseSiblings(Node node, int currentPos, int posStart, int posEnd, Algorithm algorithm, Object data) {
			// loop to take over components into the span & split them if required, until final position was reached!
			if (algorithm.equals(Algorithm.DELETE)) {
				// position equal as after the last found even unknown components will be deleted
				while (node != null && currentPos <= posEnd) {
					// IMPORTANT: get next sibling first, otherwise references get lost by Xerces during splitting
					Node _nextSibling = node.getNextSibling();
					if (node instanceof Element) {
// ** THE COMMENTED CODE BELOW REMOVED A COMPONENT, WHICH BECAME EMPTY AFTER DELETING SOME CONTENT. 						
// ** THIS IS NOW CONSIDERED HARMFUL, AS PARAGRAPH/HEADING/TABLE ALWAYS REMAIN
//						int countBefore = ((List) data).size();
						currentPos = checkElementNode((Element) node, currentPos, posStart, posEnd, algorithm, data);
						// OPTIMIZATION: the data is being used as flag to realize, when the element being checked does not contain any component 
//						List deleteStatus = (List) data;
//						int countAfter = deleteStatus.size();
//						if (countBefore < countAfter && (Boolean) deleteStatus.get(0) && ((OdfElement) node).countDescendantComponents() == 0) {
//							Element parent = (Element) node.getParentNode();
//							if (parent != null) {
//								// delete the empty boilerplate
//								parent.removeChild(node);
//								// reset the status
//								deleteStatus.clear();
//							}
//						}
					} else if (node instanceof Text) {
						currentPos = algorithm.execute(node, currentPos, posStart, posEnd, data);
					}
					// next sibling will be checked
					node = _nextSibling;
				}
			} else if (algorithm.equals(Algorithm.INSERT)) {

				// position equal as it could be inserted before on 0th (first) place
				while (node != null && (currentPos < posEnd || (posEnd == 0 && currentPos == 0))) {
					// IMPORTANT: get next sibling first, otherwise references get lost by Xerces during splitting
					Node _nextSibling = node.getNextSibling();
					if (node instanceof Element) {
						currentPos = checkElementNode((Element) node, currentPos, posStart, posEnd, algorithm, data);
					} else if (node instanceof Text) {
						currentPos = algorithm.execute(node, currentPos, posStart, posEnd, data);
					}
					// next sibling will be checked
					node = _nextSibling;
				}
			} else if (algorithm.equals(Algorithm.COUNT)) {
				while (node != null && (currentPos < posEnd || (posEnd == 0 && currentPos == 0))) {
					// IMPORTANT: get next sibling first, otherwise references get lost by Xerces during splitting
					Node _nextSibling = node.getNextSibling();
					if (node instanceof Element) {
						currentPos = checkElementNode((Element) node, currentPos, posStart, posEnd, algorithm, data);
					} else if (node instanceof Text) {
						currentPos = algorithm.execute(node, currentPos, posStart, posEnd, data);
					}
					// next sibling will be checked
					node = _nextSibling;
				}
			} else if (algorithm.equals(Algorithm.MARK)) {
				while (node != null && currentPos < posEnd) {
					// IMPORTANT: get next sibling first, otherwise references get lost by Xerces during splitting
					Node _nextSibling = node.getNextSibling();
					currentPos = algorithm.execute(node, currentPos, posStart, posEnd, data);
					node = _nextSibling;
				}
			} else {
				while (node != null && currentPos < posEnd) {
					// IMPORTANT: get next sibling first, otherwise references get lost by Xerces during splitting
					Node _nextSibling = node.getNextSibling();
					// ToDo: || algorithm.equals(Algorithm.MOVE)
					// if (algorithm.equals(Algorithm.DELETE) && algorithm.equals(Algorithm.MARK)) {
					if (algorithm.equals(Algorithm.MARK)) {
						currentPos = algorithm.execute(node, currentPos, posStart, posEnd, data);
					} else {
						if (node instanceof Element) {
							currentPos = checkElementNode((Element) node, currentPos, posStart, posEnd, algorithm, data);
						} else if (node instanceof Text) {
							currentPos = algorithm.execute(node, currentPos, posStart, posEnd, data);
						}
					}
					// next sibling will be checked
					node = _nextSibling;
				}
			}
			return currentPos;
		}

		/**
		 *
		 * @param currentNode the element currentNode will be checked if it is a
		 * component. If a component it will be executed fully or partly moved
		 * into the span
		 * @param currentPos the current component position
		 * @param posStart the text position where the span starts
		 * @param posEnd the text position, where the span ends (one higher as
		 * the last component number to be included)
		 * @param newSpan the span collecting the marked components
		 * @return
		 */
		static private int checkElementNode(Element currentNode, int currentPos, int posStart, int posEnd, Algorithm algorithm, Object newContent) {
			if (currentNode != null) {
				if (currentNode instanceof OdfElement && ((OdfElement) currentNode).isComponentRoot() && !((OdfElement) currentNode).mIsIngoredComponent) {
					currentPos = algorithm.execute(currentNode, currentPos, posStart, posEnd, newContent);
				} else {
					// if element is no component, neglect the element (e.g. another <text:span>, but analyze its content
					Node firstChild = currentNode.getFirstChild();
					if (firstChild != null) {
						currentPos = traverseSiblings(firstChild, currentPos, posStart, posEnd, algorithm, newContent);
					}
				}
			}
			return currentPos;
		}

		/**
		 * @param newSpanContent the content will be moved from its former
		 * parent into the span
		 * @param containerElement the span the content will be moved to
		 * @param nextSibling the span will be added in front of the sibling or
		 * appended to the parent if the sibling is NULL
		 */
		static private void addSpanAndContent(Node newSpanContent, TextSpanElement containerElement, Node _nextSibling) {
			Node parent = newSpanContent.getParentNode();
			parent.removeChild(newSpanContent);
			containerElement.appendChild(newSpanContent);
			if (_nextSibling != null) {
				parent.insertBefore(containerElement, _nextSibling);
			} else {
				parent.appendChild(containerElement);
			}
			if (newSpanContent instanceof TextSpanElement && containerElement instanceof TextSpanElement) {
				// Merge the styles of containerElement to existing innerSpan
				OdfStyle.mergeSelectionWithSameRange((TextSpanElement) containerElement, (OdfStylableElement) newSpanContent);
			}
			// check if there are any descendants spans
			mergeWithDescendantSpans(containerElement, (TextSpanElement) containerElement);
		}

		/*  */
		static enum Algorithm {

			/**
			 */
			INSERT(1),
			/**
			 */
			DELETE(2),
			/**
			 */
			MARK(3),
			/**
			 */
			RECEIVE(4),
			/**
			 */
			MOVE(5),
			/**
			 */
			COUNT(6);
			private int mId;

			Algorithm(int id) {
				mId = id;
			}

			int execute(Node currentNode, int currentPos, int posStart, int posEnd, Object newContent) {
				switch (mId) {
					case 1:
						currentPos = insert(currentNode, currentPos, posStart, posEnd, (List) newContent);
						break;
					case 2:
						currentPos = delete(currentNode, currentPos, posStart, posEnd, (List) newContent);
						break;
					case 3:
						currentPos = mark(currentNode, currentPos, posStart, posEnd, (TextSpanElement) newContent);
						break;
					case 4:
						currentPos = receive(currentNode, currentPos, posStart, posEnd, (ArrayList) newContent);
						break;
					case 5:
						moveChildrenTo(currentNode, currentPos, posStart, posEnd, (Element) newContent);
						break;
					case 6:
						currentPos = count(currentNode, currentPos);
						break;

				}
				return currentPos;
			}

			/**
			 * This function takes a node and moves it fully or partly into the
			 * given
			 * <text:span/>, if the node components (e.g. character) are within
			 * the given posStart and posEnd.
			 *
			 * @param currentNode	The current child node to
			 * @param currentPos the current component position to be checked.
			 * starting with 0
			 * @param posStart the start position of the span.
			 * @param posEnd the end position of the span as string position.
			 * NOTE: One higher than the originally given component position.
			 * Therefore always one higher than posStart.
			 * @param newNode
			 * @return the position of the next component to be checked
			 */
			int mark(Node currentNode, int currentPos, int posStart, int posEnd, TextSpanElement newNode) {

				if (currentNode != null) {
					Integer nextSplitPos;
					// if the start is equal to the first position avoid the first split
					if (currentPos >= posStart) {
						// we are about to gather content in our span
						nextSplitPos = posEnd;
					} else {
						// we have not reached the area to mark
						nextSplitPos = posStart;
					}

					// ** GET NODE SIZE
					Integer contentLength = getNodeWidth(currentNode);

					// add the full currentNode ==> if the end of the span is equal to the end of the currentNode 
					boolean isTotalSelection = (currentPos + contentLength == nextSplitPos) && nextSplitPos == posEnd;
					// add the middle part of the currentNode ==> if the currentNode already starts within the span, but the end is not within the span
					boolean isFirstPart = (currentPos >= posStart && currentPos + contentLength < posEnd);
					// split the currentNode ==> if the next split position is within the currentNode 
					boolean needsSplit = currentPos + contentLength > nextSplitPos;

					// MOVE PARTS INTO THE SELECTION ELEMENT				
					if (isTotalSelection || isFirstPart) {
						Node _nextSibling = currentNode.getNextSibling();
						addSpanAndContent(currentNode, newNode, _nextSibling);
						currentPos += contentLength;

						// SPLIT THE NODE
					} else if (needsSplit) {
						int secondPartLength = 1;
						Node secondPart = null;
						if (currentNode instanceof Text) {
							// splitCursor is the first character of second part (counting starts with 0)
							secondPart = ((Text) currentNode).splitText(nextSplitPos - currentPos);
							secondPartLength = ((Text) secondPart).getLength();
						} else {
							//handle component split...
							secondPart = ((OdfElement) currentNode).split(nextSplitPos - currentPos);
							secondPartLength = OdfElement.getNodeWidth(secondPart);
						}
						boolean reachedStartPosition = nextSplitPos != posEnd;
						if (reachedStartPosition) {
							// if the second split is still in the same currentNode..
							// if the end of the cut is equal to the end of the span
							if (currentPos + (contentLength - secondPartLength) == posEnd) {
								addSpanAndContent(secondPart, newNode, secondPart.getNextSibling());
								currentPos += contentLength;
							} else {
								// position changed based on the first cut part
								currentPos = currentPos + (contentLength - secondPartLength);
								currentPos = mark(secondPart, currentPos, posStart, posEnd, newNode);
							}
						} else {
							addSpanAndContent(currentNode, newNode, secondPart);
							currentPos += contentLength - secondPartLength;
						}
						// SKIP THE NODE AS SELECTION NOT STARTED
					} else {
						currentPos += contentLength;
					}
				}
				return currentPos;
			}

			int insert(Node currentNode, int currentPos, int posStart, int posEnd, List newContent) {
				if (currentNode != null) {
					// // ** GET NODE SIZE
					// Integer contentLength = getNodeWidth(currentNode);
					// ** GET NODE SIZE
					Integer contentLength = 1; // component default is 1
					if (currentNode instanceof Text) {
						contentLength = ((Text) currentNode).getLength();
					} else {
						// get size from component
						contentLength = ((OdfElement) currentNode).getRepetition();
					}
					// ** There is content to be inserted..
					if (currentPos == posStart) { // if we are already at the right place, insert content before the currentNode
						Node parent = currentNode.getParentNode();
						Object newData = newContent.get(0);
						if (newData instanceof String) {
							OdfElement.appendUsingWhitespaceHandling(null, (OdfElement) parent, currentNode, (String) newData);
						} else if (newContent.get(0) instanceof Element) {
							OdfElement.addElementNode(null, (OdfElement) parent, currentNode, (Element) newContent.get(0));
						}
						// Mark that the content has been added
						newContent.add(Boolean.TRUE);
						currentPos += contentLength;
					} else if (currentPos + contentLength >= posStart) {
						// if the complete text node is selected, append behind..
						if (currentPos + contentLength == posStart) {
							Object newData = newContent.get(0);
							if (newData instanceof String) {
								OdfElement.appendUsingWhitespaceHandling(currentNode, (OdfElement) currentNode.getParentNode(), currentNode.getNextSibling(), (String) newData);
							} else if (newData instanceof Element) {
								OdfElement.addElementNode(currentNode, (OdfElement) currentNode.getParentNode(), currentNode.getNextSibling(), (Element) newData);
							}
							// Mark that the content has been added
							newContent.add(Boolean.TRUE);
							currentPos = posStart;
						} else { // else if only a part of the text node is selected
							Node secondPart = null;
							if (currentNode instanceof Text) {
								// splitCursor is the first character of second part (counting starts with 0)
								secondPart = ((Text) currentNode).splitText(posStart - currentPos);
								Object newData = newContent.get(0);
								if (newData instanceof String) {
									OdfElement.appendUsingWhitespaceHandling(currentNode, (OdfElement) currentNode.getParentNode(), secondPart, (String) newData);
								} else if (newContent instanceof Element) {
									OdfElement.addElementNode(currentNode, (OdfElement) currentNode.getParentNode(), secondPart, (Element) newContent);
								}
								// Mark that the content has been added
								newContent.add(Boolean.TRUE);
							} else {
								//handle component split...
								secondPart = ((OdfElement) currentNode).split(posStart - currentPos);
								Node parent = currentNode.getParentNode();
								Object newData = newContent.get(0);
								if (newData instanceof String) {
									OdfElement.appendUsingWhitespaceHandling(currentNode, (OdfElement) parent, secondPart, (String) newData);
								} else if (newContent instanceof Element) {
									OdfElement.addElementNode(currentNode, (OdfElement) parent, secondPart, (Element) newContent);
								}
								// Mark that the content has been added
								newContent.add(Boolean.TRUE);
							}
							currentPos += contentLength;
						}
						// ** SKIP THE NODE AS SELECTION NOT STARTED
					} else {
						currentPos += contentLength;
					}
				}
				return currentPos;
			}

			// Within the text node, either the first or the last part have to be deleted (best selectable)
			int delete(Node currentNode, int currentPos, int posStart, int posEnd, List deleteStatus) {

				if (currentNode != null) {
					Integer nextSplitPos;
					// if the start is equal to the first position avoid the first split
					if (currentPos >= posStart) {
						// we are about to gather content in our span
						nextSplitPos = posEnd;
					} else {
						// we have not reached the area to mark
						nextSplitPos = posStart;
					}
					// ** GET NODE SIZE			
					Integer contentLength = getNodeWidth(currentNode);

					// add the middle part of the currentNode ==> if the currentNode already starts within the span, but the end is not within the span
					boolean inExecutionMode = currentPos >= posStart && contentLength + currentPos > posStart && currentPos + contentLength <= posEnd;
					// split the currentNode ==> if the next split position is within the currentNode
					// We need to search to the next found component, as all unknown components will be deleted after the known
					boolean needsSplit = currentPos + contentLength > nextSplitPos && nextSplitPos - currentPos != 0;

					// APPLY ACTION TO PARTS (here delete them)
					if (inExecutionMode) {
						OdfElement parent = (OdfElement) currentNode.getParentNode();
						parent.removeChild(currentNode);
						deleteStatus.add(0, Boolean.TRUE);
						// remove empty boilderplate
						while (!parent.isComponentRoot() && parent.countDescendantComponents() == 0) {
							OdfElement grandParent = (OdfElement) parent.getParentNode();
							if (grandParent instanceof OfficeBodyElement) {
								break;
							}
							grandParent.removeChild(parent);
							parent = grandParent;
						}
						currentPos += contentLength;

						// SPLIT THE NODE
					} else if (needsSplit) {
						int secondPartLength = 1;
						Node secondPart = null;
						if (currentNode instanceof Text) {
							// splitCursor is the first character of second part (counting starts with 0)
							secondPart = ((Text) currentNode).splitText(nextSplitPos - currentPos);
							secondPartLength = ((Text) secondPart).getLength();
						} else {
							//handle span split...
							secondPart = ((OdfElement) currentNode).split(nextSplitPos - currentPos);
							secondPartLength = OdfElement.getNodeWidth(secondPart);
						}
						boolean reachedStartPosition = nextSplitPos != posEnd;
						if (reachedStartPosition) {
							// if the second split is still in the same currentNode..
							// if the end of the cut is equal to the end of the span
							if (currentPos + (contentLength - secondPartLength) == posEnd) {
								OdfElement parent = (OdfElement) currentNode.getParentNode();
								parent.removeChild(secondPart);
								deleteStatus.add(0, Boolean.TRUE);
								// remove empty boilderplate
								while (!parent.isComponentRoot() && parent.countDescendantComponents() == 0) {
									OdfElement grandParent = (OdfElement) parent.getParentNode();
									if (grandParent instanceof OfficeBodyElement) {
										break;
									}
									grandParent.removeChild(parent);
									parent = grandParent;
								}
								currentPos += contentLength;
							} else {
								// position changed based on the first cut part
								currentPos = currentPos + (contentLength - secondPartLength);
								currentPos = delete(secondPart, currentPos, posStart, posEnd, deleteStatus);
							}
						} else {
							OdfElement parent = (OdfElement) currentNode.getParentNode();
							parent.removeChild(currentNode);
							deleteStatus.add(0, Boolean.TRUE);
							// remove empty boilderplate
							while (!parent.isComponentRoot() && parent.countDescendantComponents() == 0) {
								OdfElement grandParent = (OdfElement) parent.getParentNode();
								if (grandParent instanceof OfficeBodyElement) {
									break;
								}
								grandParent.removeChild(parent);
								parent = grandParent;
							}
							currentPos += contentLength - secondPartLength;
						}
						// SKIP THE NODE AS SELECTION NOT STARTED
					} else {
						currentPos += contentLength;
					}
				}
				return currentPos;


//				Integer contentLength = node.getLength();
//				// see if targetposition (either start or end) is within this text node
//				if (currentPos + contentLength >= currentTargetPos) {
//					// If the delete pos is this complete text node and it is already past startPos, delete node 
//					if (currentPos + contentLength == currentTargetPos && textPosEnd == currentTargetPos) {
//						node.getParentNode().removeChild(node);
//					} else { // if delete position is within the text node
//						Integer splitPosition = null;
//						splitPosition = currentTargetPos - currentPos;
//						Text secondPart = node.splitText(splitPosition);
//						//if position within text was the startNode, only delete second part
//						if (currentTargetPos == textPosStart) {
//							// if the end position is in the the same text node as the start position
//							if (currentPos + contentLength >= textPosEnd) {
//								// split the string once more
//								secondPart.splitText(textPosEnd - splitPosition - currentPos);
//							}
//							node.getParentNode().removeChild(secondPart);
//						} else {//if we had been in the delete mode, only delete first part					
//							// node has become the firstNode after split
//							node.getParentNode().removeChild(node);
//						}
//					}
//				} else {
//					// if we are in the deletion mode
//					if (currentTargetPos == textPosEnd) {
//						// delete the complete node
//						node.getParentNode().removeChild(node);
//					}
//				}
//				return currentPos += contentLength;
			}

			private int receive(Node currentNode, int currentPos, int posStart, int posEnd, ArrayList newNodeContainer) {
				if (currentNode != null) {

					// ** GET NODE SIZE
					Integer contentLength = 1; // component default is 1
					if (currentNode instanceof Text) {
						contentLength = ((Text) currentNode).getLength();
					} else {
						// get size from component
						contentLength = ((OdfElement) currentNode).getRepetition();
					}

					// if the current node is selected
					if (currentPos == posStart) {
						newNodeContainer.add(currentNode);
						currentPos = posEnd;
					} else if (currentPos + contentLength > posStart) { // else if only a part of the text node is selected
						Node secondPart = null;
						if (currentNode instanceof Text) {
							// splitCursor is the first character of second part (counting starts with 0)
							secondPart = ((Text) currentNode).splitText(posStart - currentPos);
							newNodeContainer.add(secondPart);
						} else {
							//handle component split...
							secondPart = ((OdfElement) currentNode).split(posStart - currentPos);
							newNodeContainer.add(secondPart);
						}
						currentPos = posEnd;

						// ** SKIP THE NODE AS SELECTION NOT STARTED
					} else {
						currentPos += contentLength;
					}
				}
				return currentPos;
			}

			private void moveChildrenTo(Node currentNode, int currentPos, int posStart, int posEnd, Element newParent) {
				if (currentNode != null) {
					newParent.appendChild(currentNode);
				}
			}

			/**
			 * @return in opposite of all other algorithms the currentPos is
			 * being used to count the number of components
			 */
			private int count(Node currentNode, int currentPos) {
				if (currentNode != null) {

					// ** GET NODE SIZE
					Integer contentLength = 1; // component default is 1
					if (!(currentNode instanceof Text)) {
						// get size from component
						contentLength = ((OdfElement) currentNode).getRepetition();
					} else {
						if (((Text) currentNode).toString().trim().length() > 0) {
							contentLength = ((Text) currentNode).getLength();
						}
					}
					currentPos += contentLength;
				}
				return currentPos;
			}
		}
	}

	/**
	 * Returns if the text should be returned or is under a nested paragraph or
	 * ignored text element (e.g. text:note-citation).
	 */
	private static boolean isIgnoredText(Text text) {
		boolean isIgnored = true;
		Node parentNode = text.getParentNode();
		if (parentNode instanceof OdfElement) {
			isIgnored = isIgnoredElement((OdfElement) parentNode);
		}
		return isIgnored;
	}

	private static boolean isIgnoredElement(OdfElement element) {
		boolean isIgnored = true;
		if (!element.mIsIngoredComponent) {
			while (element != null) {
				if (element.isComponentRoot() && !element.selfAndDescendantTextIgnoredAsComponent()) {
					isIgnored = false;
					break;
				} else {
					if (element.selfAndDescendantTextIgnoredAsComponent()) {
						isIgnored = true;
						break;
					} else {
						Node parent = element.getParentAs(OdfElement.class);
						if (parent instanceof OdfElement) {
							isIgnored = isIgnoredElement((OdfElement) parent);
						}
						break;
					}
				}
			}
		}
		return isIgnored;
	}

	private static int getContentSize(Node currentNode) {
		int contentLength = 0; // by default there is no length

		Node nextChild = currentNode.getFirstChild();
		while (nextChild != null) {
			contentLength += getNodeWidth(nextChild);
			nextChild = nextChild.getNextSibling();
		}
		return contentLength;
	}

	private static void mergeWithDescendantSpans(Element currentElement, TextSpanElement highPrioSpan) {
		// here starts the new function
		if (currentElement.hasChildNodes()) {
			NodeList children = currentElement.getChildNodes();
			int childCount = children.getLength();
			Node lastChild = children.item(childCount - 1);
			// if there is only one child node
			if (childCount == 1) {
				// and this child node is a text span
				if (lastChild instanceof TextAElement) {
					// call again to check for a span
					mergeWithDescendantSpans((Element) lastChild , highPrioSpan);
				} else if (lastChild instanceof TextSpanElement) {
					// merge it with our high prio span
					OdfStyle.mergeSelectionWithSameRange((TextSpanElement) highPrioSpan, (OdfStylableElement) lastChild);
				} 
//				else if (lastChild instanceof OdfElement) {
//					// call this function again..
//					mergeWithDescendantSpans((OdfElement) lastChild, highPrioSpan);
//				}
			} else { // go through all children and see if one of them is a span
//						11
//				Node newChild;
//				for (int i = childCount - 2; i >= 0; i--) {
//					newChild = children.item(i);
//					
//				}
				// if this is the case, split the important span (can't we reuse sources from the list handling?)
			}
		}
	}

	private static int getNodeWidth(Node currentNode) {
		// ** GET NODE SIZE					
		int contentLength = 0; // by default there is no length
		// if a text
		if (currentNode instanceof Text) {
			if (!isIgnoredText((Text) currentNode)) {
				contentLength = ((Text) currentNode).getLength();
			}
			// if a component element and NOT ignored
		} else if (currentNode instanceof OdfElement) {
			if (OdfFileSaxHandler.isComponentRoot((OdfElement) currentNode) || currentNode instanceof TextSElement) {
				if (isIgnoredElement((OdfElement) currentNode)) {
					contentLength = 0;
				} else {
					// get size from component
					contentLength = ((OdfElement) currentNode).getRepetition();
				}
			} else {
				Node nextChild = ((OdfElement) currentNode).firstChild;
				while (nextChild != null) {
					contentLength += getNodeWidth(nextChild);
					nextChild = nextChild.getNextSibling();
				}
			}
		}
		return contentLength;
	}

	/**
	 * Copy attributes from one element to another, existing attributes will be
	 * overwritten
	 */
	public static void copyAttributes(OdfElement from, OdfElement to) {
		NamedNodeMap attributes = from.getAttributes();
		for (int i = 0; i < attributes.getLength(); i++) {
			Attr node = (Attr) attributes.item(i);
			to.setAttributeNS(node.getNamespaceURI(), node.getNodeName(), node.getValue());
		}
	}

	/**
	 * @return the child element of a given parent from a given position
	 */
	public Element getChildElement(String uri, String localName, int position) {
		NodeList childList = this.getElementsByTagNameNS(uri, localName);
		return (Element) childList.item(position);
	}

	public static boolean isIgnoredElement(Element element) {
		return isIgnoredElement(element.getNamespaceURI(), element.getLocalName());
	}

	public static boolean isIgnoredElement(String uri, String localName) {
		boolean isIgnored = false;
		if (uri != null && uri.equals(TextNoteElement.ELEMENT_NAME.getUri())) {
			// text:note
			if (localName.equals(TextNoteElement.ELEMENT_NAME.getLocalName())) {
				isIgnored = true;
			}
		}
		if (uri != null && uri.equals(OfficeAnnotationElement.ELEMENT_NAME.getUri())) {
			// office:annotation
			if (localName.equals(OfficeAnnotationElement.ELEMENT_NAME.getLocalName())) {
				isIgnored = true;
			}
		}
		return isIgnored;
	}

	/**
	 * @returns the next element sibling of the given node or null if none
	 * exists
	 */
	public static OdfElement getNextElementSibling(Node node) {
		OdfElement nextElement = null;
		Node _nextSibling = node.getNextSibling();
		if (_nextSibling instanceof OdfElement) {
			nextElement = (OdfElement) _nextSibling;
		} else if (_nextSibling instanceof Text) {
			nextElement = getNextElementSibling(_nextSibling);
		}
		return nextElement;
	}

	/**
	 * @returns the next element sibling of the given node or null if none
	 * exists
	 */
	public static OdfElement getPreviousElementSibling(Node node) {
		OdfElement previousElement = null;
		Node _previousElement = node.getPreviousSibling();
		if (_previousElement instanceof OdfElement) {
			previousElement = (OdfElement) _previousElement;
		} else if (_previousElement instanceof Text) {
			previousElement = getPreviousElementSibling(_previousElement);
		}
		return previousElement;
	}

	/**
	 * @returns the first element child of the this or null if none exists
	 */
	public OdfElement getFirstElementChild() {
		OdfElement firstElementChild = null;
		NodeList nodeList = this.getChildNodes();
		Node node = nodeList.item(0);
		if (node != null) {
			if (node instanceof OdfElement) {
				firstElementChild = (OdfElement) node;
			} else {
				firstElementChild = getNextElementSibling(node);
			}
		}
		return firstElementChild;
	}

	/**
	 * If the parent component root element is not the element parent of the
	 * child component's root element, there have to be boilerplate elements
	 * in-between that might have to be deleted as well, if the last sibling was
	 * deleted.
	 *
	 * @param targetElement the element to be deleted. Bottom-up will be all
	 * parents traversed and deleted in case the original element to be deleted
	 * had no other siblings being components.
	 */
	public static boolean removeComponentElementAndEmptyBoilerplate(OdfElement parentComponentElement, OdfElement targetElement) {
		OdfElement targetParent = (OdfElement) targetElement.getParentNode();
		targetParent.removeChild(targetElement);
		// if the parent component root element is not the parent of the child component, there have to be boilerplate elements inbetween that have to take care of
		return OdfElement.removeComponentElementAndEmptyBoilerplate(parentComponentElement, targetParent, targetParent.countDescendantComponents() == 0);
	}

	/**
	 * If the parent component root element is not the element parent of the
	 * child component's root element, there have to be boilerplate elements
	 * in-between that might have to be deleted as well, if the last sibling was
	 * deleted.
	 *
	 * @param targetElement the element to be deleted. Bottom-up will be all
	 * parents traversed and deleted in case the original element to be deleted
	 * had no other siblings being components.
	 */
	public static boolean removeComponentElementAndEmptyBoilerplate(OdfElement parent, OdfElement targetElement, boolean isLastComponent) {
		// if there is boilerplate elements between the targetElement and the parent
		if (!targetElement.getParentNode().equals(parent)) {
			OdfElement targetParent = (OdfElement) targetElement.getParentNode();
			// if there is only a single component below 
			if (isLastComponent) {
				// check if there are now component siblings..
				if (targetParent.countDescendantComponents() > 0) {
					isLastComponent = false;
				} else {
					// if there is still no other component, delete the boilerplate element
					targetParent.removeChild(targetElement);
				}
			}
			removeComponentElementAndEmptyBoilerplate(parent, targetParent, isLastComponent);
		} else if (isLastComponent) {
			parent.removeChild(targetElement);
		}
		return isLastComponent;
	}

	public int countPrecedingElementSiblings() {
		int i = 0;
		Node node = this.getPreviousSibling();
		while (node != null) {
			node = node.getPreviousSibling();
			if (node instanceof Element) {
				i++;
			}
		}
		return i;
	}
}
