/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.realtime.impl;

import java.util.Date;
import java.util.Enumeration;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import com.openexchange.realtime.Component;
import com.openexchange.realtime.ComponentHandle;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.IDEventHandler;
import com.openexchange.server.ServiceLookup;
import com.openexchange.timer.ScheduledTimerTask;
import com.openexchange.timer.TimerService;

/**
 * {@link ConnectionComponent}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class ConnectionComponent implements Component {

    private static class FailSafeRunnable implements Runnable {

        private final ConnectionComponent m_connectionPool;

        public FailSafeRunnable(final ConnectionComponent connectionPool) {
            super();
            m_connectionPool = connectionPool;
        }

        @Override
        public void run() {
            m_connectionPool.checkForFailSaveDocument();
        }
    }

    /**
     * Initializes a new {@link ConnectionComponent}.
     * @param services
     */
    public ConnectionComponent(ServiceLookup services) {
        super();
        m_services = services;
        m_saveDataTimer = m_services.getService(TimerService.class);
    }

    /**
     * @return
     */
    ConcurrentHashMap<ID, Connection> getOpenConnections() {
        return m_openConnections;
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.realtime.Component#create(com.openexchange.realtime.packet.ID)
     */
    @Override
    public ComponentHandle create(ID id) {
        // Create operations queue with layout: synthetic.office://operations/folderId.fileId~fileVersion_fileName
        Connection connection = new Connection(m_services, id);
        m_openConnections.put(id, connection);
        if (m_openConnections.size() == 1) {
            this.startFailSafeTimer();
        }
        id.on("dispose", new IDEventHandler() {

            @Override
            public void handle(String event, ID id2, Object source, Map<String, Object> properties) {
                ConcurrentHashMap<ID, Connection> connections = getOpenConnections();
                connections.remove(id2);
                if (connections.size() == 0) {
                    stopFailSafeTimer();
                }
            }
        });

        return connection;
    }

    /* (non-Javadoc)
     * @see com.openexchange.realtime.Component#getId()
     */
    @Override
    public String getId() {
        // Address operations queue with layout: synthetic.office://operations/folderId.fileId~fileVersion_fileName
        return "office";
    }

    /* (non-Javadoc)
     * @see com.openexchange.realtime.Component#getEvictionPolicy()
     */
    @Override
    public EvictionPolicy getEvictionPolicy() {
        // No automatic shut down of operations queue
        return com.openexchange.realtime.Component.NONE;
    }

    public void shutdown() {
        if (m_saveDataTimer != null) {
            m_saveDataTimer.purge();
        }
        for(ID id: m_openConnections.keySet()) {
            Connection connection = m_openConnections.get(id);
            if (null != connection) {
                connection.saveDocumentOnDispose();
            }
            id.trigger("dispose", this);
        }
    }

    /**
     * Checks if there are documents which need to be saved
     * for better fail safe handling.
     */
    void checkForFailSaveDocument() {
        Enumeration<Connection> connections = m_openConnections.elements();
        while (connections.hasMoreElements()) {
            try {
                Date currentDate = new Date();
                Connection connection = connections.nextElement();
                if (connection != null && !connection.savingDocumentInProgress()) {

                    long diff = getTimeDifference(connection.getSaveTimeStamp(), currentDate);
                    long ops = connection.getNumberOfPendingOperations();

                    // check if we have to do a fail safe document save
                    if ((diff >= 60000) && (ops >= 100)) {
                        // The first save action should always store the document with
                        // a new revision. Check if we already saved with revision or
                        // not.
                        boolean saveWithoutRevision = connection.savedWithRevision();
                        connection.failSafeSaveDocument(saveWithoutRevision);
                    } else if ((diff >= 900000) && ops > 0) {
                        connection.failSafeSaveDocument(false); // with new revision
                    }
                }
            } catch (Exception e) {
                // Ignore if saving doesn't work
            }
        }
    }

    void startFailSafeTimer() {
        if (m_saveDataTimer != null) {
            m_failSafeTimerTask = m_saveDataTimer.scheduleAtFixedRate(new FailSafeRunnable(this), 30000, 30000);
        } else {
            m_failSafeTimerTask = null;
        }
    }

    void stopFailSafeTimer() {
        if (m_failSafeTimerTask != null) {
            m_failSafeTimerTask.cancel();
            m_failSafeTimerTask = null;
        }
    }

    /**
     * Calculates the absolute difference between two Dates
     *
     * @param d1 Date1
     * @param d2 Date2
     * @return
     *  The difference in milliseconds.
     */
    private long getTimeDifference(Date d1, Date d2) {
        return Math.abs(d1.getTime() - d2.getTime());
    }

    // - Members ---------------------------------------------------------------

    private final ServiceLookup m_services;

    private final ConcurrentHashMap<ID, Connection> m_openConnections = new ConcurrentHashMap<ID, Connection>();

    private final TimerService m_saveDataTimer;

    private ScheduledTimerTask m_failSafeTimerTask = null;
}
