/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.tools;

import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import javax.xml.bind.DatatypeConverter;
import javax.xml.bind.JAXBElement;
import org.docx4j.dml.CTColor;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;


final public class Commons {
    private Commons() {}

    final static String hexArray = "0123456789ABCDEF";

    public static String bytesToHexString(byte[] bytes, int offset, int size) {
        final StringBuilder hex = new StringBuilder(2 * size);
        for (int i=0; i<size; i++) {
            final byte b = bytes[offset + i];
            hex.append(hexArray.charAt((b & 0xF0) >> 4))
               .append(hexArray.charAt(b&0x0F));
        }
        return hex.toString();
    }

    public static  String ctColorToString( CTColor color ){
        final String fillString = "000000";
        byte[] rgbVal = color.getSrgbClr() != null ? color.getSrgbClr().getVal() : color.getSysClr().getLastClr();
        String ret = Integer.toHexString((((rgbVal[0]& 0xFF)<< 16) + ((rgbVal[1]& 0xFF)<< 8) + (rgbVal[2]& 0xFF)));
        if( ret.length() < 6 )
            ret = fillString.substring( ret.length() ) + ret;
        return ret;
    }

    public static byte[] ctColorToBytes(CTColor color){
        return color.getSrgbClr() != null ? color.getSrgbClr().getVal() : color.getSysClr().getLastClr();
    }

    public static byte[] hexStringToBytes(String color) {
        return DatatypeConverter.parseHexBinary(color);
    }

    public static JAXBElement<?> findElement(List<JAXBElement<?>> jaxbElements, String elementName) {
        for (int i=0; i<jaxbElements.size(); i++) {
            if (elementName.equals(jaxbElements.get(i).getName().getLocalPart())) {
                return jaxbElements.get(i);
            }
        }
        return null;
    }

    public static void removeElement(List<JAXBElement<?>> jaxbElements, String elementName) {
        for(int i=0; i<jaxbElements.size(); i++) {
            if(elementName.equals(jaxbElements.get(i).getName().getLocalPart())) {
                jaxbElements.remove(i);
            }
        }
    }

    public static void insertElement(List<JAXBElement<?>> jaxbElements, String elementName, JAXBElement<?> jaxbInsert) {
        JAXBElement<?> jaxb = findElement(jaxbElements, elementName);
        if(jaxb!=null)
            jaxb = jaxbInsert;
        else
            jaxbElements.add(jaxbInsert);
    }

    public static JSONObject surroundJSONObject(String type, JSONObject source) {
        if(source==null||source.length()==0)
            return null;
        JSONObject newJSON = new JSONObject();
        try {
            newJSON.put(type,  source);
        } catch (JSONException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return newJSON;
    }

    public static void mergeJsonObjectIfUsed(String destinationPropertyName, JSONObject destination, JSONObject source)
        throws JSONException {

        if(source!=null&&source.length()>0) {
            if(destination.has(destinationPropertyName)) {
                Iterator<String> keys = source.keys();
                while(keys.hasNext()) {
                    String key = keys.next();
                    Object val = source.get(key);
                    destination.put(key, val);
                }
           } else
                destination.put(destinationPropertyName, source);
        }
    }

    public static void jsonPut(JSONObject destination, String destinationPropertyName, JSONObject source)
        throws JSONException {
        if(source!=null&&source.length()>0) {
            destination.put(destinationPropertyName, source);
        }
    }

    public static void jsonPut(JSONObject destination, String destinationPropertyName, JSONArray source)
        throws JSONException {
        if(source!=null&&source.length()>0) {
            destination.put(destinationPropertyName, source);
        }
    }

    public static void jsonPut(JSONObject destination, String destinationPropertyName, String source)
        throws JSONException {
        if(source!=null&&source.length()>0) {
            destination.put(destinationPropertyName, source);
        }
    }

    public static void jsonPut(JSONObject destination, String destinationPropertyName, Object source)
        throws JSONException {
        if(source!=null) {
            destination.put(destinationPropertyName, source);
        }
    }

    public static void jsonPut(JSONArray destination, String source) {
        if(source!=null&&source.length()>0) {
            destination.put(source);
        }
    }

    public static final Map<String, String> highlightColorToRgb = createHighlightColorsMap();
    public static final double themeTransformFactor = 100000.0 / 255.0;
    public static final double themeTransformDiv = 255.0 / 100000.0;

    private static Map<String, String> createHighlightColorsMap() {
        // highlight color map according to ooxml specification
        Map<String, String> result = new HashMap<String, String>();
        result.put( "black", "000000");
        result.put( "blue", "0000FF");
        result.put( "cyan", "00FFFF");
        result.put( "darkBlue", "00008B");
        result.put( "darkCyan", "008B8B");
        result.put( "darkGray", "A9A9A9");
        result.put( "darkGreen", "006400");
        result.put( "darkMagenta", "800080");
        result.put( "darkRed", "8B0000");
        result.put( "darkYellow", "808000");
        result.put( "green", "00FF00");
        result.put( "lightGray", "D3D3D3");
        result.put( "magenta", "FF00FF");
        result.put( "none", "");
        result.put( "red", "FF0000");
        result.put( "white", "FFFFFF");
        result.put( "yellow", "FFFF00");
        return Collections.unmodifiableMap(result);
    }

    private static int rgbStringToInt(String rgbColor) {
        return Integer.parseInt(rgbColor, 16);
    }

    public static String mapHightlightColorToRgb(String highlightColor) {
        String rgbColor = null;

        Map<String, String> map = Commons.highlightColorToRgb;
        rgbColor = map.get(highlightColor);

        return rgbColor;
    }

    private static int colorDiff(int rgbColor1, int rgbColor2) {
        int r1 = ((rgbColor1 & 0x00ff0000) >> 16);
        int g1 = ((rgbColor1 & 0x0000ff00) >> 8);
        int y1 = (rgbColor1 & 0x000000ff);
        int r2 = ((rgbColor2 & 0x00ff0000) >> 16);
        int g2 = ((rgbColor2 & 0x0000ff00) >> 8);
        int y2 = (rgbColor2 & 0x000000ff);

        return Math.abs(r1-r2) + Math.abs(g1-g2) + Math.abs(y1-y2);
    }

    public static String mapRgbToPredefined(String rgbColor, Map<String, String> mappings) {
        String predefinedColor = null;
        int diff = Integer.MAX_VALUE;;
        int rgbColorValue = Integer.parseInt(rgbColor);

        if (mappings != null) {
            Set<String> keys = mappings.keySet();
            for (String color : keys) {
                int colorValue = Commons.rgbStringToInt(color);
                int currDiff = Commons.colorDiff(rgbColorValue, colorValue);
                if (currDiff < diff) {
                    diff = currDiff;
                    predefinedColor = color;
                    if (diff == 0)
                        return predefinedColor;
                }
            }
        }

        return predefinedColor;
    }

    public static class Pair<T, E> {

        private T first;
        private E second;

        public Pair(T first, E second) {
            this.first = first;
            this.second = second;
        }
        public T getFirst() {
            return first;
        }
        public E getSecond() {
            return second;
        }
        public void setFirst(T first) {
            this.first = first;
        }
        public void setSecond(E second) {
            this.second = second;
        }
    }
}
