/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.tools;

import java.util.Iterator;
import org.apache.commons.logging.Log;
import org.docx4j.XmlUtils;
import org.docx4j.jaxb.Context;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.CTBooleanProperty;
import org.xlsx4j.sml.CTBorder;
import org.xlsx4j.sml.CTBorderPr;
import org.xlsx4j.sml.CTCellAlignment;
import org.xlsx4j.sml.CTColor;
import org.xlsx4j.sml.CTFill;
import org.xlsx4j.sml.CTFont;
import org.xlsx4j.sml.CTFontName;
import org.xlsx4j.sml.CTFontSize;
import org.xlsx4j.sml.CTNumFmt;
import org.xlsx4j.sml.CTPatternFill;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTUnderlineProperty;
import org.xlsx4j.sml.CTVerticalAlignFontProperty;
import org.xlsx4j.sml.CTXf;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.ObjectFactory;
import org.xlsx4j.sml.STBorderStyle;
import org.xlsx4j.sml.STHorizontalAlignment;
import org.xlsx4j.sml.STPatternType;
import org.xlsx4j.sml.STUnderlineValues;
import org.xlsx4j.sml.STVerticalAlignRun;
import org.xlsx4j.sml.STVerticalAlignment;
import com.openexchange.log.LogFactory;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.xlsx.OperationDocument;


public class CellUtils {

    protected static Log log = LogFactory.getLog(CellUtils.class);

    public static void applyCellProperties(OperationDocument operationDocument, JSONObject attrs, Cell cell, CTStylesheet stylesheet)
        throws JSONException {

        if(attrs==null)
            return;

        final CTXf xfSource = stylesheet.getCellXfByIndex(cell.getS());
        CTXf detachedXf = xfSource==null?Context.getsmlObjectFactory().createCTXf():XmlUtils.deepCopy(xfSource, Context.getJcSML());

        // character properties
        JSONObject characterProperties = attrs.optJSONObject("character");
        if (characterProperties != null) {
            applyCharacterProperties(operationDocument, characterProperties, detachedXf, stylesheet, false);
        }
        JSONObject cellProperties = attrs.optJSONObject("cell");
        if(cellProperties!=null) {
            applyCellProperties(operationDocument, cellProperties, detachedXf, stylesheet, false);
        }
        long index = stylesheet.getOrApplyCellXf(detachedXf);
        if(index>0) {
            cell.setS(index);
        }
    }

    public static JSONObject createCellProperties(Cell cell, CTStylesheet stylesheet)
        throws JSONException {

        JSONObject jsonCellProperties = null;
        if(cell!=null) {
            CTXf xf = stylesheet.getCellXfByIndex(cell.getS());
            if(xf!=null) {
                jsonCellProperties = new JSONObject();
                // creating style attribute (0 is default and though not created)
                Long xfId = xf.getXfId();
                if(xfId!=null&&xfId!=0) {
                    jsonCellProperties.put("styleId", Long.toString(xfId));
                }
                Commons.mergeJsonObjectIfUsed("character", jsonCellProperties, createCharacterProperties(xf, stylesheet, false));
                Commons.mergeJsonObjectIfUsed("cell", jsonCellProperties, createCellProperties(xf, stylesheet, false));
            }
        }
        return jsonCellProperties!=null&&jsonCellProperties.length()>0 ? jsonCellProperties : null;
    }

    static public Long getAttributeIndex(Long index, Boolean isApply, boolean createStyle) {
        if(createStyle) {
            if((isApply==null||isApply.booleanValue())&&index!=null) {
                return index;
            }
            return Long.valueOf(0);
        }
        return isApply!=null&&isApply.booleanValue()?index:null;
    }

    static private CTFill createDetachedFill(CTStylesheet stylesheet, CTXf xf) {

        Long fillId = xf.getFillId();
        if(fillId==null||fillId.longValue()==0) {
            return Context.getsmlObjectFactory().createCTFill();
        }
        CTFill ctFill = stylesheet.getFillByIndex(fillId);
        if(ctFill!=null) {
            ctFill = XmlUtils.deepCopy(ctFill, Context.getJcSML());
            if(ctFill==null) {
                throw new RuntimeException("xlsx export: CTFill deepCopy fails");
            }
            return ctFill;
        }
        log.warn("xlsx export: could not get CTFill for Id");
        return Context.getsmlObjectFactory().createCTFill();
    }

    static private CTBorder createDetachedBorder(CTStylesheet stylesheet, CTXf xf) {

        Long borderId = xf.getBorderId();
        if(borderId==null||borderId.longValue()==0) {
            return Context.getsmlObjectFactory().createCTBorder();
        }
        CTBorder ctBorder = stylesheet.getBorderByIndex(borderId);
        if(ctBorder!=null) {
            ctBorder = XmlUtils.deepCopy(ctBorder, Context.getJcSML());
            if(ctBorder==null) {
                throw new RuntimeException("xlsx export: CTBorder deepCopy fails");
            }
            return ctBorder;
        }
        log.warn("xlsx export: could not get CTBorder for Id");
        return Context.getsmlObjectFactory().createCTBorder();
    }

    private static CTBorderPr applyBorder(OperationDocument operationDocument, JSONObject borderProperties, CTStylesheet stylesheet, CTBorderPr _borderPr)
        throws JSONException {

        CTBorderPr borderPr = _borderPr!=null?_borderPr:Context.getsmlObjectFactory().createCTBorderPr();

        JSONObject jsonColor = borderProperties.optJSONObject("color");
        if(jsonColor!=null) {
            CTColor color = Context.getsmlObjectFactory().createCTColor();
            Utils.applyColor(operationDocument, color, jsonColor);
            borderPr.setColor(color);
        }

        int width = borderProperties.optInt("width", 26);
        int pixelWidth = 2;
        if(width < 20) {
            pixelWidth = 1;
        }
        else if(width < 40) {
            pixelWidth = 2;
        }
        else if(width < 60) {
            pixelWidth = 3;
        }
        else {
            pixelWidth = 4;
        }

        String style = borderProperties.optString("style", null);
        if(style!=null) {

            STBorderStyle borderStyle = STBorderStyle.THIN;
            if(style.equals("none")) {
                borderStyle = STBorderStyle.NONE;
            }
            else if(style.equals("single")) {
                borderStyle = STBorderStyle.HAIR;
                if(pixelWidth==2) {
                    borderStyle = STBorderStyle.THIN;
                }
                else if(pixelWidth==3) {
                    borderStyle = STBorderStyle.MEDIUM;
                }
                else if(pixelWidth==4) {
                    borderStyle = STBorderStyle.THICK;
                }
            }
            else if(style.equals("double")) {
                borderStyle = STBorderStyle.DOUBLE;
            }
            else if(style.equals("dotted")) {
                borderStyle = STBorderStyle.DOTTED;
                if(pixelWidth>2) {
                    borderStyle = STBorderStyle.MEDIUM_DASH_DOT_DOT;
                }
            }
            else if(style.equals("dashed")) {
                borderStyle = STBorderStyle.DASHED;
                if(pixelWidth>2) {
                    borderStyle = STBorderStyle.MEDIUM_DASHED;
                }
            }
            borderPr.setStyle(borderStyle);
        }
        return borderPr;
    }

    public static void applyCellProperties(OperationDocument operationDocument, JSONObject cellProperties, CTXf xf, CTStylesheet stylesheet, boolean createStyle)
        throws JSONException {

        final ObjectFactory objectFactory = Context.getsmlObjectFactory();

        // detached property sets are only created if needed
        CTFill          detachedFill = null;
        CTBorder        detachedBorder = null;

        Iterator<String> keys = cellProperties.keys();
        while(keys.hasNext()) {
            String attr = keys.next();
            Object value = cellProperties.get(attr);

            // CTNumFmt
            if(attr.equals("numberFormat")) {

                Long    destNumFmtId = null;
                if(value instanceof JSONObject) {

                    long numberFormatId = ((JSONObject)value).optLong("id", -1);
                    final String numberFormatCode = ((JSONObject)value).optString("code", null);

                    if(numberFormatId>=0) {
                        if(numberFormatCode==null) {
                            // numberFormatId without numberFormatCode is only valid if this Id is already existing or it is a predefined Id (<=49)
                            CTNumFmt numFmt = stylesheet.getNumberFormatById(numberFormatId);
                            if(numFmt!=null) {
                                destNumFmtId = numberFormatId;
                            }
                            else if(numberFormatId<=49) {
                                numFmt = objectFactory.createCTNumFmt();
                                numFmt.setNumFmtId(numberFormatId);
                                stylesheet.applyNumberFormat(numFmt);
                                destNumFmtId = numberFormatId;
                            }
                            else {
                                log.warn("xlsx export: trying to apply a non predefined numberFormat without format code");
                            }
                        }
                        else {
                            destNumFmtId = stylesheet.getOrApplyNumberFormatByFormatCode(numberFormatCode).getNumFmtId();
                        }
                    }
                    else if (numberFormatCode!=null) {
                        destNumFmtId = stylesheet.getOrApplyNumberFormatByFormatCode(numberFormatCode).getNumFmtId();
                    }
                }
                xf.setNumFmtId(destNumFmtId);
                if(createStyle==false) {
                    xf.setApplyNumberFormat(destNumFmtId!=null);
                }
            }
            // CTFill
            else if(attr.equals("fillColor")) {
                if(detachedFill==null) {
                    detachedFill = createDetachedFill(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedFill.setGradientFill(null);
                    CTPatternFill patternFill = objectFactory.createCTPatternFill();
                    detachedFill.setPatternFill(patternFill);
                    patternFill.setPatternType(STPatternType.SOLID);
                    patternFill.setBgColor(null);
                    CTColor color = objectFactory.createCTColor();
                    Utils.applyColor(operationDocument, color, (JSONObject)value);
                    patternFill.setFgColor(color);
                }
                else {
                    xf.setFillId(null);
                }
            }
            // CTBorder
            else if(attr.equals("borderLeft")) {
                if(detachedBorder==null) {
                    detachedBorder = createDetachedBorder(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedBorder.setLeft(applyBorder(operationDocument, (JSONObject)value, stylesheet, detachedBorder.getLeft()));
                }
                else {
                    detachedBorder.setLeft(null);
                }

            }
            else if(attr.equals("borderRight")) {
                if(detachedBorder==null) {
                    detachedBorder = createDetachedBorder(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedBorder.setRight(applyBorder(operationDocument, (JSONObject)value, stylesheet, detachedBorder.getRight()));
                }
                else {
                    detachedBorder.setRight(null);
                }
            }
            else if(attr.equals("borderTop")) {
                if(detachedBorder==null) {
                    detachedBorder = createDetachedBorder(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedBorder.setTop(applyBorder(operationDocument, (JSONObject)value, stylesheet, detachedBorder.getTop()));
                }
                else {
                    detachedBorder.setTop(null);
                }
            }
            else if(attr.equals("borderBottom")) {
                if(detachedBorder==null) {
                    detachedBorder = createDetachedBorder(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedBorder.setBottom(applyBorder(operationDocument, (JSONObject)value, stylesheet, detachedBorder.getBottom()));
                }
                else {
                    detachedBorder.setBottom(null);
                }
            }
            else if(attr.equals("borderInsideHor")) {
                if(detachedBorder==null) {
                    detachedBorder = createDetachedBorder(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedBorder.setHorizontal(applyBorder(operationDocument, (JSONObject)value, stylesheet, detachedBorder.getHorizontal()));
                }
                else {
                    detachedBorder.setHorizontal(null);
                }
            }
            else if(attr.equals("borderInsideVert")) {
                if(detachedBorder==null) {
                    detachedBorder = createDetachedBorder(stylesheet, xf);
                }
                if(value instanceof JSONObject) {
                    detachedBorder.setVertical(applyBorder(operationDocument, (JSONObject)value, stylesheet, detachedBorder.getVertical()));
                }
                else {
                    detachedBorder.setVertical(null);
                }
            }
            // CTCellAlignment
            else if(attr.equals("alignHor")) {
                CTCellAlignment cellAlignment = xf.getAlignment();
                if(cellAlignment==null) {
                    cellAlignment = objectFactory.createCTCellAlignment();
                    xf.setAlignment(cellAlignment);
                }
                if(value instanceof String) {
                    String alignment = (String)value;
                    if(alignment.equals("left")) {
                        cellAlignment.setHorizontal(STHorizontalAlignment.LEFT);
                    }
                    else if(alignment.equals("center")) {
                        cellAlignment.setHorizontal(STHorizontalAlignment.CENTER);
                    }
                    else if(alignment.equals("right")) {
                        cellAlignment.setHorizontal(STHorizontalAlignment.RIGHT);
                    }
                    else if(alignment.equals("justify")) {
                        cellAlignment.setHorizontal(STHorizontalAlignment.JUSTIFY);
                    }
                    else {
                        cellAlignment.setHorizontal(STHorizontalAlignment.GENERAL);
                    }
                }
                else {
                    cellAlignment.setHorizontal(STHorizontalAlignment.GENERAL);
                }
                if(createStyle==false) {
                    xf.setApplyAlignment(true);
                }
            }
            else if(attr.equals("alignVert")) {
                CTCellAlignment cellAlignment = xf.getAlignment();
                if(cellAlignment==null) {
                    cellAlignment = objectFactory.createCTCellAlignment();
                    xf.setAlignment(cellAlignment);
                }
                if(value instanceof String) {
                    String alignment = (String)value;
                    if(alignment.equals("top")) {
                        cellAlignment.setVertical(STVerticalAlignment.TOP);
                    }
                    else if(alignment.equals("middle")) {
                        cellAlignment.setVertical(STVerticalAlignment.CENTER);
                    }
                    else if(alignment.equals("justify")) {
                        cellAlignment.setVertical(STVerticalAlignment.JUSTIFY);
                    }
                    else {
                        cellAlignment.setVertical(STVerticalAlignment.BOTTOM);
                    }
                }
                else {
                    cellAlignment.setVertical(STVerticalAlignment.BOTTOM);
                }
                if(createStyle==false) {
                    xf.setApplyAlignment(true);
                }
            }
            else if(attr.equals("wrapText")) {
                CTCellAlignment cellAlignment = xf.getAlignment();
                if(cellAlignment==null) {
                    cellAlignment = objectFactory.createCTCellAlignment();
                    xf.setAlignment(cellAlignment);
                }
                if(value instanceof Boolean) {
                    cellAlignment.setWrapText((Boolean)value);
                }
                else {
                    cellAlignment.setWrapText(null);
                }
                if(createStyle==false) {
                    xf.setApplyAlignment(true);
                }
            }
        }

        // CTFill
        if(detachedFill!=null) {
            long fillIndex = stylesheet.getOrApplyFill(detachedFill);
            xf.setFillId(fillIndex);
            if(createStyle==false) {
                xf.setApplyFill(true);
            }
        }
        if(detachedBorder!=null) {
            long borderIndex = stylesheet.getOrApplyBorder(detachedBorder);
            xf.setBorderId(borderIndex);
            if(createStyle==false) {
                xf.setApplyBorder(true);
            }
        }

        // CTCellAlignment
    }

    public static JSONObject createCellProperties(CTXf xf, CTStylesheet stylesheet, boolean createStyle)
        throws JSONException {

        JSONObject jsonCellProperties = new JSONObject();
        // CTNumFmt
        Long _numberFormatId = getAttributeIndex(xf.getNumFmtId(), xf.isApplyNumberFormat(), createStyle);
        long numberFormatId = _numberFormatId==null?0:_numberFormatId.longValue();
        Utils.createNumberFormat(jsonCellProperties, numberFormatId, stylesheet.getNumberFormatById(numberFormatId));

        // CTFILL
        Utils.createFill(jsonCellProperties, stylesheet, stylesheet.getFillByIndex(getAttributeIndex(xf.getFillId(), xf.isApplyFill(), createStyle)));

        // CTBorders
        Utils.createBorders(jsonCellProperties, stylesheet, stylesheet.getBorderByIndex(getAttributeIndex(xf.getBorderId(), xf.isApplyBorder(), createStyle)));

        // CTCellAlignment
        if(xf.isApplyAlignment()!=null&&xf.isApplyAlignment().booleanValue()) {
            Utils.createAlignment(jsonCellProperties, xf.getAlignment());
        }
        return jsonCellProperties.length()>0?jsonCellProperties:null;
    }

    public static void applyCharacterProperties(OperationDocument operationDocument, JSONObject characterProperties, CTXf xf, CTStylesheet stylesheet, boolean createStyle)
        throws JSONException {

        final ObjectFactory objectFactory = Context.getsmlObjectFactory();

        CTFont font = stylesheet.getFontByIndex(xf.getFontId());
        if(font==null) {
            log.warn("xlsx export: could not get font for Id");
            font = objectFactory.createCTFont();
        }
        CTFont detachedFont = XmlUtils.deepCopy(font, Context.getJcSML());
        if(detachedFont==null) {
            throw new RuntimeException("xlsx export: CTFont deepCopy fails");
        }

        Iterator<String> keys = characterProperties.keys();
        while(keys.hasNext()) {
            String attr = keys.next();
            Object value = characterProperties.get(attr);
            if(attr.equals("fontName")) {
                if(value instanceof String) {
                    Object fontName = detachedFont.getProperty("name");
                    if(fontName==null) {
                        fontName = detachedFont.addProperty("name", objectFactory.createCTFontName(objectFactory.createCTFontName()));
                    }
                    if(fontName instanceof CTFontName) {
                        ((CTFontName)fontName).setVal((String)value);
                    }
                }
                else {
                    detachedFont.deleteProperty("name");
                }
            }
            else if (attr.equals("fontSize")) {
                if(value instanceof Integer) {
                    Object fontSize = detachedFont.getProperty("sz");
                    if(fontSize==null) {
                        fontSize = detachedFont.addProperty("sz", objectFactory.createCTFontSz(objectFactory.createCTFontSize()));
                    }
                    if(fontSize instanceof CTFontSize) {
                        ((CTFontSize) fontSize).setVal((Integer)value);
                    }
                }
                else {
                    detachedFont.deleteProperty("sz");
                }
            }
            else if(attr.equals("bold")) {
                if(value instanceof Boolean) {
                    Object b = detachedFont.getProperty("b");
                    if(b==null) {
                        b = detachedFont.addProperty("b", objectFactory.createCTFontB(objectFactory.createCTBooleanProperty()));
                    }
                    if(b instanceof CTBooleanProperty) {
                        ((CTBooleanProperty) b).setVal((Boolean)value);
                    }
                }
                else {
                    detachedFont.deleteProperty("b");
                }
            }
            else if(attr.equals("italic")) {
                if(value instanceof Boolean) {
                    Object i = detachedFont.getProperty("i");
                    if(i==null) {
                        i = detachedFont.addProperty("i", objectFactory.createCTFontI(objectFactory.createCTBooleanProperty()));
                    }
                    if(i instanceof CTBooleanProperty) {
                        ((CTBooleanProperty) i).setVal((Boolean)value);
                    }
                }
                else {
                    detachedFont.deleteProperty("i");
                }
            }
            else if(attr.equals("strike")) {
                if(value instanceof String) {
                    Object strike = detachedFont.getProperty("strike");
                    if(strike==null) {
                        strike = detachedFont.addProperty("strike", objectFactory.createCTFontStrike(objectFactory.createCTBooleanProperty()));
                    }
                    if(strike instanceof CTBooleanProperty) {
                        ((CTBooleanProperty)strike).setVal(((String)value).equals("none")==false);
                    }
                }
                else {
                    detachedFont.deleteProperty("strike");
                }

            }
            else if(attr.equals("color")) {
                if(value instanceof JSONObject) {
                    Object color = detachedFont.getProperty("color");
                    if(color==null) {
                        color = detachedFont.addProperty("color", objectFactory.createCTFontColor(objectFactory.createCTColor()));
                    }
                    if(color instanceof CTColor) {
                        Utils.applyColor(operationDocument, (CTColor)color, (JSONObject)value);
                    }
                }
                else {
                    detachedFont.deleteProperty("color");
                }
            }
            else if(attr.equals("vertAlign")) {
                if(value instanceof String) {
                    Object vertAlign = detachedFont.getProperty("vertAlign");
                    if(vertAlign==null) {
                        vertAlign = detachedFont.addProperty("vertAlign", objectFactory.createCTFontVertAlign(objectFactory.createCTVerticalAlignFontProperty()));
                    }
                    if(vertAlign instanceof CTVerticalAlignFontProperty) {
                        STVerticalAlignRun align = STVerticalAlignRun.BASELINE;
                        if(((String)value).equals("sup")) {
                            align = STVerticalAlignRun.SUPERSCRIPT;
                        }
                        else if(((String)value).equals("sub")) {
                            align = STVerticalAlignRun.SUBSCRIPT;
                        }
                        ((CTVerticalAlignFontProperty) vertAlign).setVal(align);
                    }
                }
                else {
                    detachedFont.deleteProperty("vertAlign");
                }
            }
            else if(attr.equals("underline")) {
                if(value instanceof Boolean) {
                    Object u = detachedFont.getProperty("u");
                    if(u==null) {
                        u = detachedFont.addProperty("u", objectFactory.createCTFontU(objectFactory.createCTUnderlineProperty()));
                    }
                    if(u instanceof CTUnderlineProperty) {
                        ((CTUnderlineProperty)u).setVal(((Boolean)value).booleanValue() ? STUnderlineValues.SINGLE : STUnderlineValues.NONE);
                    }
                }
                else {
                    detachedFont.deleteProperty("u");
                }
            }
        }
        long fontIndex = stylesheet.getOrApplyFont(detachedFont);
        xf.setFontId(fontIndex);
        if(createStyle==false) {
            xf.setApplyFont(true);
        }
    }

    public static JSONObject createCharacterProperties(CTXf xf, CTStylesheet stylesheet, boolean createStyle)
        throws JSONException {

        JSONObject jsonCharacterProperties = new JSONObject();
        CTFont font = stylesheet.getFontByIndex(getAttributeIndex(xf.getFontId(), xf.isApplyFont(), createStyle));
        if(font!=null) {
            Object fontName = font.getProperty("name");
            if(fontName instanceof CTFontName) {
                Commons.jsonPut(jsonCharacterProperties, "fontName", ((CTFontName)fontName).getVal());
            }
            Object fontSize = font.getProperty("sz");
            if(fontSize instanceof CTFontSize) {
                Commons.jsonPut(jsonCharacterProperties, "fontSize", ((CTFontSize)fontSize).getVal());
            }
            Object b = font.getProperty("b");
            if(b instanceof CTBooleanProperty) {
                Commons.jsonPut(jsonCharacterProperties, "bold", ((CTBooleanProperty)b).isVal());
            }
            Object i = font.getProperty("i");
            if(i instanceof CTBooleanProperty) {
                Commons.jsonPut(jsonCharacterProperties, "italic", ((CTBooleanProperty)i).isVal());
            }
            Object strike = font.getProperty("strike");
            if(strike instanceof CTBooleanProperty) {
                Commons.jsonPut(jsonCharacterProperties, "strike", ((CTBooleanProperty)strike).isVal()?"single":"none");
            }
            Object color = font.getProperty("color");
            if(color instanceof CTColor) {
                Commons.jsonPut(jsonCharacterProperties, "color", Utils.createColor(stylesheet, (CTColor)color));
            }
            Object vertAlign = font.getProperty("vertAlign");
            if(vertAlign instanceof CTVerticalAlignFontProperty) {
                String align = "baseline";
                switch(((CTVerticalAlignFontProperty)vertAlign).getVal()) {
                    case BASELINE :    break;
                    case SUPERSCRIPT : align = "sup"; break;
                    case SUBSCRIPT :   align = "sub"; break;
                }
                Commons.jsonPut(jsonCharacterProperties, "vertAlign", align);
            }
            Object u = font.getProperty("u");
            if(u instanceof CTUnderlineProperty) {
                STUnderlineValues underline = ((CTUnderlineProperty)u).getVal();
                if(underline!=null) {
                    Commons.jsonPut(jsonCharacterProperties, "underline", underline!=STUnderlineValues.NONE);
                }
            }
        }
        return jsonCharacterProperties.length()>0?jsonCharacterProperties:null;
    }
}
