/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ooxml.xlsx.operations;

import java.util.Iterator;
import java.util.List;
import org.docx4j.openpackaging.packages.SpreadsheetMLPackage;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorkbookPart;
import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.docx4j.openpackaging.parts.relationships.RelationshipsPart;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xlsx4j.sml.CTCellFormula;
import org.xlsx4j.sml.CTCellStyle;
import org.xlsx4j.sml.CTCellStyleXfs;
import org.xlsx4j.sml.CTCellStyles;
import org.xlsx4j.sml.CTMergeCell;
import org.xlsx4j.sml.CTMergeCells;
import org.xlsx4j.sml.CTRst;
import org.xlsx4j.sml.CTStylesheet;
import org.xlsx4j.sml.CTXf;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Cell.CellRefRange;
import org.xlsx4j.sml.Col;
import org.xlsx4j.sml.Cols;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.STCellFormulaType;
import org.xlsx4j.sml.STCellType;
import org.xlsx4j.sml.Sheet;
import org.xlsx4j.sml.SheetData;
import org.xlsx4j.sml.Sheets;
import org.xlsx4j.sml.Workbook;
import org.xlsx4j.sml.Worksheet;
import com.openexchange.office.ooxml.tools.Commons;
import com.openexchange.office.ooxml.xlsx.OperationDocument;
import com.openexchange.office.ooxml.xlsx.tools.CellUtils;
import com.openexchange.office.ooxml.xlsx.tools.ColumnUtils;
import com.openexchange.office.ooxml.xlsx.tools.RowUtils;
import com.openexchange.office.ooxml.xlsx.tools.SheetUtils;

public class CreateOperationHelper extends com.openexchange.office.ooxml.operations.CreateOperationHelper {

    private final SpreadsheetMLPackage spreadsheetMLPackage;

    public CreateOperationHelper(OperationDocument _operationDocument, JSONArray mOperationsArray) {
        super(_operationDocument, mOperationsArray);
        spreadsheetMLPackage = _operationDocument.getPackage();
    }

    @Override
    public OperationDocument getOperationDocument() {
        return (OperationDocument)operationDocument;
    }

    @Override
    public void createDocumentDefaults(String userLanguage) {

    }

    public void CreateStyleOperations()
        throws Exception {

        CTStylesheet stylesheet = getOperationDocument().getStylesheet(false);
        if(stylesheet!=null) {
            CTCellStyleXfs cellStyleXfs = stylesheet.getCellStyleXfs();
            if(cellStyleXfs!=null) {
                List<CTXf> xfs = cellStyleXfs.getXf();
                for(int i=0;i<xfs.size();i++) {
                    String styleId = Integer.toString(i);
                    String styleName = "";
                    JSONObject jsonCellProperties = new JSONObject();
                    Commons.mergeJsonObjectIfUsed("cell", jsonCellProperties, CellUtils.createCellProperties(xfs.get(i), stylesheet, true));
                    Commons.mergeJsonObjectIfUsed("character", jsonCellProperties, CellUtils.createCharacterProperties(xfs.get(i), stylesheet, true));
                    JSONObject attrs = jsonCellProperties.length()>0?jsonCellProperties:null;
                    String parentId = null;
                    Boolean hidden = null;
                    Integer uiPriority = null;
                    Boolean isDefault = null;

                    // check for proper UI styleName
                    CTCellStyles cellStyles = stylesheet.getCellStyles();
                    if(cellStyles!=null) {
                        List<CTCellStyle> cellStyleList = cellStyles.getCellStyle();
                        for(int is=0; is<cellStyleList.size();is++) {
                            CTCellStyle cellStyle = cellStyleList.get(is);
                            long xfId = cellStyle.getXfId();
                            if(xfId==i) {
                                styleName = cellStyle.getName();
                                hidden = cellStyle.isHidden();
                                break;
                            }
                        }
                    }
                    addInsertStyleSheetOperation("cell", styleId, styleName, attrs, parentId, hidden, uiPriority, isDefault);
                }
            }
        }
    }

    @Override
    public void createOperations()
        throws Exception {

        WorkbookPart workbookPart = spreadsheetMLPackage.getWorkbookPart();
        Workbook workbook = workbookPart.getJaxbElement();
        Sheets sheets = workbook.getSheets();
        List<Sheet> sheetList = sheets.getSheet();
        for(int sheetIndex = 0; sheetIndex < sheetList.size(); sheetIndex++) {
            createSheetOperations(sheetList.get(sheetIndex), sheetIndex);
        }
    }

    private JSONObject getCellObject(Cell cell)
        throws Exception {

        JSONObject cellObject = new JSONObject();
        CTCellFormula cellFormula = cell.getF();
        if(cellFormula!=null&&(cellFormula.getT()==STCellFormulaType.NORMAL||cellFormula.getT()==STCellFormulaType.SHARED)) {
            if(cellFormula.getT()==STCellFormulaType.SHARED) {
                cellObject.put("shared", cellFormula.getSi());
                Cell.CellRef cellRef = Cell.createCellRef(cellFormula.getRef());
                if(cellRef!=null) {
                    cellObject.put("ref", cellRef.getJSONArray());
                }
            }
            if(cellFormula.getValue()!=null&&cellFormula.getValue().length()>0)
                cellObject.put("value", "=" + cellFormula.getValue());
        }
        else if (cell.getT()==STCellType.INLINE_STR) {
            CTRst is = cell.getIs();
            if(is!=null) {
                // TODO: CTRst can include formatted text runs, but we will support simple t first.
                if(is.getT()!=null) {
                    cellObject.put("value",  is.getT());
                }
            }
        }
        else if(cell.getV()!=null) {
            if(cell.getT()==STCellType.N) {
                try {

                    double number = Double.parseDouble(cell.getV());
                    cellObject.put("value", number);
                }
                catch(NumberFormatException e) {
                    // TODO: ups
                }
            }
            else if(cell.getT()==STCellType.S) {
                String sharedString = getOperationDocument().getSharedString(cell.getV());
                if(sharedString!=null) {
                    cellObject.put("value", getOperationDocument().escapeString(sharedString));
                }
            }
            else if(cell.getT()==STCellType.B) {
                if(cell.getV().equals("0")||cell.getV().toLowerCase().equals("false")) {
                    cellObject.put("value", false);
                }
                else if(cell.getV().equals("1")||cell.getV().toLowerCase().equals("true")) {
                    cellObject.put("value", true);
                }
            }
            else {  // STCellType.STR, STCellType.E or STCellType.F (ARRAY or DATA_TABLE format)
                cellObject.put("value",  cell.getV());
            }
        }
       JSONObject jsonCellAttrs = CellUtils.createCellProperties(cell, getOperationDocument().getStylesheet(true));
       if(jsonCellAttrs!=null) {
           cellObject.put("attrs", jsonCellAttrs );
       }
        return cellObject;
    }

    private void createSheetOperations(final Sheet sheet, int sheetIndex)
        throws Exception {

        addInsertSheetOperation(sheetIndex, sheet.getName(), Commons.surroundJSONObject("sheet", SheetUtils.createSheetProperties(sheet)));

        RelationshipsPart relationshipPart = spreadsheetMLPackage.getWorkbookPart().getRelationshipsPart();
        WorksheetPart worksheetPart = (WorksheetPart)relationshipPart.getPart(sheet.getId());
        Worksheet worksheet = worksheetPart.getJaxbElement();
        SheetData sheetData = worksheet.getSheetData();
        if(sheetData!=null) {
            createRowOperations(sheetData.getRowIter(), sheetIndex);
            CTMergeCells mergeCells = worksheet.getMergeCells();
            if(mergeCells!=null) {
                List<CTMergeCell> mergeCellList = mergeCells.getMergeCell();
                if(mergeCellList!=null) {
                    for(CTMergeCell mergeCell:mergeCellList) {
                        addMergeCells(sheetIndex, Cell.createCellRefRange(mergeCell.getRef()));
                    }
                }
            }
        }
        createColumnOperations(worksheet.getCols(), sheetIndex);
    }

    private void createRowOperations(Iterator<Row> rowIter, int sheetIndex)
        throws Exception {

        if(rowIter!=null) {
            while(rowIter.hasNext()) {
                Row row = rowIter.next();

                Long _rwIx = row.getR();
                if(_rwIx!=null) {

                    int rowIndex = _rwIx.intValue() - 1;
                    // row index is necessary to apply our attributes...
                    // otherwise we are in big ...

                    JSONObject attrs = RowUtils.createRowProperties(row, getOperationDocument().getStylesheet(true));
                    if(attrs!=null) {
                        addSetRowAttributes(sheetIndex, rowIndex, attrs);
                    }
                }

                JSONArray rowArray = new JSONArray();
                Cell.CellRef topLeftCellRef = null;
                int lastCellX = -1;
                JSONArray cellArray = new JSONArray();
                Iterator<Cell> cellIter = row.getCellIter();
                while(cellIter.hasNext()) {
                    Cell cell = cellIter.next();
                    final Cell.CellRef cellRef = Cell.createCellRef(cell.getR().toUpperCase());     // TODO: cellRef is optional...
                    if(cellRef!=null) {
                        JSONObject cellObject = getCellObject(cell);
                        if(topLeftCellRef==null) {
                            // first cell in our array
                            topLeftCellRef = cellRef;
                        }
                        else {
                            // check if empty cells are to be written and start with a new setCellContentOperation if needed
                            int emptyCells = cellRef.getColumn() - lastCellX;
                            if(emptyCells>3) {
                                // this are too many empty cells, we will create an additional setContent operation
                                if(cellArray.length()>0) {
                                    // we have to write back our current content and start a new collection
                                    rowArray.put(cellArray);
                                    addSetCellContentsOperation(sheetIndex, topLeftCellRef, rowArray);
                                    rowArray = new JSONArray();
                                    cellArray = new JSONArray();
                                }
                                topLeftCellRef = cellRef;
                            }
                            else {
                                while(--emptyCells>0) {
                                    cellArray.put((Object)null);
                                }
                            }
                        }
                        cellArray.put(cellObject);
                        lastCellX = cellRef.getColumn();
                    }
                }
                rowArray.put(cellArray);
                if(topLeftCellRef!=null)
                    addSetCellContentsOperation(sheetIndex, topLeftCellRef, rowArray);
            }
        }
    }

    private void createColumnOperations(List<Cols> colsIter, int sheetIndex)
        throws Exception {

        if(colsIter==null)
            return;

        for(Cols cols:colsIter) {
            List<Col> colIter = cols.getCol();
            if(colIter!=null) {
                for(Col col:colIter) {
                    JSONObject attrs = ColumnUtils.createColumnProperties(col, getOperationDocument().getStylesheet(true));
                    if(attrs!=null) {
                        addSetColumnAttributes(sheetIndex, col.getMin()-1, col.getMax()-1, attrs);
                    }
                }
            }
        }
    }

    public void addInsertSheetOperation(int sheetIndex, final String sheetName, final JSONObject attrs)
        throws JSONException {

        final JSONObject insertSheetObject = new JSONObject();
        insertSheetObject.put("name", "insertSheet");
        insertSheetObject.put("sheet", sheetIndex);
        insertSheetObject.put("sheetName", sheetName);
        if(attrs!=null)
            insertSheetObject.put("attrs", attrs);

        operationsArray.put(insertSheetObject);
    }

    public void addSetCellContentsOperation(int sheetIndex, Cell.CellRef cellRef, JSONArray cellContent)
        throws JSONException {

        final JSONObject insertCellsObject = new JSONObject();
        insertCellsObject.put("name", "setCellContents");
        insertCellsObject.put("sheet", sheetIndex);
        insertCellsObject.put("start", cellRef.getJSONArray());
        insertCellsObject.put("contents", cellContent);

        operationsArray.put(insertCellsObject);
    }

    public void addSetRowAttributes(int sheetIndex, int rowIndex, JSONObject attrs)
        throws JSONException {

        if(attrs==null||attrs.length()==0)
            return;

        final JSONObject addSetRowAttributesObject = new JSONObject();
        addSetRowAttributesObject.put("name", "setRowAttributes");
        addSetRowAttributesObject.put("sheet", sheetIndex);
        addSetRowAttributesObject.put("start", rowIndex);
        addSetRowAttributesObject.put("attrs", attrs);

        operationsArray.put(addSetRowAttributesObject);
    }

    public void addSetColumnAttributes(int sheetIndex, long start, long end, JSONObject attrs)
        throws JSONException {

        if(attrs==null||attrs.length()==0)
            return;

        final JSONObject addSetColumnAttributesObject = new JSONObject();
        addSetColumnAttributesObject.put("name", "setColumnAttributes");
        addSetColumnAttributesObject.put("sheet", sheetIndex);
        addSetColumnAttributesObject.put("start", start);
        if(end>start) {
            addSetColumnAttributesObject.put("end", end);
        }
        addSetColumnAttributesObject.put("attrs", attrs);

        operationsArray.put(addSetColumnAttributesObject);
    }

    public void addMergeCells(int sheetIndex, CellRefRange cellRefRange)
        throws JSONException {

        if(cellRefRange==null)
            return;

        final JSONObject addMergeCellsObject = new JSONObject();
        addMergeCellsObject.put("name", "mergeCells");
        addMergeCellsObject.put("sheet", sheetIndex);
        addMergeCellsObject.put("start", cellRefRange.getStart().getJSONArray());
        addMergeCellsObject.put("end", cellRefRange.getEnd().getJSONArray());

        operationsArray.put(addMergeCellsObject);
    }
}
