/*
 * Copyright 2012 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.odftoolkit.odfdom.component;

import org.odftoolkit.odfdom.component.OdfOperationDocument;
import java.io.File;
import java.io.FileInputStream;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.json.JSONObject;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Ignore;
import org.junit.Test;
import org.odftoolkit.odfdom.doc.LoadSaveTest;
import org.odftoolkit.odfdom.utils.ResourceUtilities;

/**
 * Loads a document with nested paragraphs and gathers its operations. Gathered
 * operations will be applied to an empty text document. The changed text
 * document will be saved and reloaded. New gathered operations will be compared
 * with the original ones, expected to be identical!
 *
 * @author svanteschubert
 */
public class RoundtripNestedParagraphTest extends RoundtripTestTemplate{

	private static final Logger LOG = Logger.getLogger(RoundtripNestedParagraphTest.class.getName());
	private static final String OUTPUT_DIRECTORY = "regression-tests" + File.separatorChar;

	@BeforeClass
	public static void setUpBeforeClass() throws Exception {
		// Creating the output directory for the tests
		File outputDir = ResourceUtilities.newTestOutputFile(OUTPUT_DIRECTORY);
		outputDir.mkdir();
	}

	public RoundtripNestedParagraphTest() {
	}

	@Test
	@Ignore // Is nowadays redundant to overall roundtrip test!!
	public void nestedParagraphRoundtrip() {

		final String SOURCE_FILE_NAME_TRUNC = "NestedParagraphs";
		final String INITIAL_OPS_SUFFIX = "_odt_inital-ops-ascii.txt";		
		final String RELOADED_OPS_SUFFIX = "_odt_reloaded-ops-ascii.txt";
		final String TARGET_FILE_NAME_SUFFIX = "_OUT.odt";
		final String SOURCE_FILE_NAME_SUFFIX = ".odt";		
		
		try {
			//**********LOAD
			OdfOperationDocument doc = new OdfOperationDocument(new FileInputStream(ResourceUtilities.newTestOutputFile(SOURCE_FILE_NAME_TRUNC + SOURCE_FILE_NAME_SUFFIX)));
			// read the (known) operations from the test document			
			JSONObject ops = doc.getOperations();
			LOG.log(Level.INFO, "The Operation in ASCII are: {0}", ops.toString(true).replace("},{", "},\n{"));
			// FOR REGRESSION TEST REFERENCE CREATION - THE OPERATION OF THE ORIGINAL DOCUMENT
			ResourceUtilities.saveStringToFile(ResourceUtilities.newTestOutputFile(OUTPUT_DIRECTORY + SOURCE_FILE_NAME_TRUNC + SOURCE_FILE_NAME_SUFFIX + INITIAL_OPS_SUFFIX), ops.toString(true).replace("},{", "},\n{"));

			String initalOpsFromFile = ResourceUtilities.loadFileAsString(ResourceUtilities.getReferenceFile(OUTPUT_DIRECTORY + SOURCE_FILE_NAME_TRUNC + SOURCE_FILE_NAME_SUFFIX + INITIAL_OPS_SUFFIX));
			LOG.log(Level.INFO, "The ops from file :{0}", initalOpsFromFile);
			Assert.assertTrue(ops.toString(true).replace("},{", "},\n{").equals(initalOpsFromFile));
		
			//**********APPLY
			OdfOperationDocument doc2 = new OdfOperationDocument();
			int opCount = doc2.applyOperations(ops);
			LOG.log(Level.INFO, "opCount: {0}", opCount);

			//**********SAVE		
			File testFile = ResourceUtilities.newTestOutputFile(OUTPUT_DIRECTORY + SOURCE_FILE_NAME_TRUNC + TARGET_FILE_NAME_SUFFIX);
			doc2.getDocument().save(testFile);
			LOG.log(Level.INFO, "***Saved:\n\t{0}", testFile.getAbsolutePath());

			//**********RELOAD
			OdfOperationDocument doc3 = new OdfOperationDocument(new FileInputStream(ResourceUtilities.newTestOutputFile(OUTPUT_DIRECTORY + SOURCE_FILE_NAME_TRUNC + TARGET_FILE_NAME_SUFFIX)));
			// read the (known) operation from the test document			
			JSONObject ops2 = doc3.getOperations();
			LOG.log(Level.INFO, "The reloaded Ops are:{0}", ops2.toString());

			// FOR REGRESSION TEST REFERENCE CREATION - THE OPERATION OF THE RELOADED DOCUMENT			
			ResourceUtilities.saveStringToFile(ResourceUtilities.newTestOutputFile(OUTPUT_DIRECTORY + SOURCE_FILE_NAME_TRUNC +  SOURCE_FILE_NAME_SUFFIX +  RELOADED_OPS_SUFFIX), ops2.toString(true).replace("},{", "},\n{"));

			String reloadedOpsFromFile = ResourceUtilities.loadFileAsString(ResourceUtilities.getReferenceFile(OUTPUT_DIRECTORY + SOURCE_FILE_NAME_TRUNC + SOURCE_FILE_NAME_SUFFIX + RELOADED_OPS_SUFFIX));
			LOG.log(Level.INFO, "The ops from file :{0}", reloadedOpsFromFile);
	
			
			// Test the known read operations with the above one - using ASCII-compare, e.g. UTF-8 character for list bullets 		
			Assert.assertTrue(ops2.toString(true).replace("},{", "},\n{").equals(reloadedOpsFromFile));
		} catch (Throwable e) {
			Logger.getLogger(LoadSaveTest.class.getName()).log(Level.SEVERE, e.getMessage(), e);
			Assert.fail(e.getMessage());
		}
	}
}