package com.openexchange.office.calcengine.client;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;

import com.openexchange.config.ConfigurationService;
import com.openexchange.server.ServiceLookup;


//=============================================================================
public class CalcEngineConfig
{
    //-------------------------------------------------------------------------
    public static final String PROP_PREFIX = "calcengine";
    
    //-------------------------------------------------------------------------
    public static final String PROP_PLACEHOLDER = "%%s%%";

    //-------------------------------------------------------------------------
    /** define the mode how calc engine has to be used.
     *  possible modes are : 
     *  - inline
     *  - remote
     *  - simulator
     *  
     *  @see {@link ECalcEngineMode.from}
     */
    public static final String PROP_MODE = PROP_PREFIX+".mode";

    //-------------------------------------------------------------------------
    public static final String PROP_JOB_TIMEOUT = PROP_PREFIX+".job.timeout";
    
    //-------------------------------------------------------------------------
    public static final String PROP_WORKER_JAR = PROP_PREFIX+".worker.jar";
    public static final String PROP_WORKER_PORT_RANGE_MIN = PROP_PREFIX+".worker.port_range.min";
    public static final String PROP_WORKER_PORT_RANGE_MAX = PROP_PREFIX+".worker.port_range.max";

    //-------------------------------------------------------------------------
    /**
	calcengine.server.count = 2

	calcengine.server.1.host                 = ces-01.oxhh.int
	calcengine.server.1.port                 = 4711
	calcengine.server.1.worker.ip-range.low  = 8000
	calcengine.server.1.worker.ip-range.high = 8010
    
	calcengine.server.2.host                 = ces-01.oxhh.int
	calcengine.server.2.port                 = 4712
	calcengine.server.2.worker.ip-range.low  = 8020
	calcengine.server.2.worker.ip-range.high = 8030
     */
    public static final String PROP_SERVER_COUNT = PROP_PREFIX+".server.count";
    
    public static final String PROP_SERVER_X_HOST                 = PROP_PREFIX+".server."+PROP_PLACEHOLDER+".host";
    public static final String PROP_SERVER_X_PORT                 = PROP_PREFIX+".server."+PROP_PLACEHOLDER+".port";
    public static final String PROP_SERVER_X_WORKER_PORTRANGE_MIN = PROP_PREFIX+".server."+PROP_PLACEHOLDER+".worker.port_range.min";
    public static final String PROP_SERVER_X_WORKER_PORTRANGE_MAX = PROP_PREFIX+".server."+PROP_PLACEHOLDER+".worker.port_range.max";
    
    //-------------------------------------------------------------------------
    public static final ECalcEngineMode DEFAULT_MODE = ECalcEngineMode.E_INLINE;
    
    //-------------------------------------------------------------------------
    /** Use factory method {@link get()} to create new instances.
     *  We want to force using our singleton .-)
     */
    private CalcEngineConfig ()
        throws Exception
    {}
    
    //-------------------------------------------------------------------------
    public synchronized static CalcEngineConfig get ()
        throws Exception
    {
        if (m_gSingleton == null)
            m_gSingleton = new CalcEngineConfig ();
        return m_gSingleton;
    }

    //-------------------------------------------------------------------------
    public void setContext (ServiceLookup aSmgr)
        throws Exception
    {
        m_aSmgr = aSmgr;
    }

    //-------------------------------------------------------------------------
    public ECalcEngineMode getMode ()
        throws Exception
    {
        final String          sMode = mem_CfgSrv ().getProperty(PROP_MODE);
        final ECalcEngineMode eMode = ECalcEngineMode.from(sMode, DEFAULT_MODE);
        return eMode;
    }

    //-------------------------------------------------------------------------
    public int getJobTimeout ()
        throws Exception
    {
        final String sTimeout = mem_CfgSrv ().getProperty(PROP_JOB_TIMEOUT);
        Validate.notEmpty(sTimeout, "Invalid configuration value : "+PROP_JOB_TIMEOUT+" is missing or empty.");
        final int    nTimeout = Integer.parseInt(sTimeout);
        Validate.isTrue(nTimeout>=0, "Invalid configuration value : "+PROP_JOB_TIMEOUT+" needs to >= 0.");
        return nTimeout;
    }

    //-------------------------------------------------------------------------
    public String getWorkerJar ()
        throws Exception
    {
        final String sJar = mem_CfgSrv ().getProperty(PROP_WORKER_JAR);
        Validate.notEmpty(sJar, "Invalid configuration value : "+PROP_WORKER_JAR+" is missing or empty.");
        return sJar;
    }

    //-------------------------------------------------------------------------
    public int getServerCount ()
        throws Exception
    {
    	final String sCount = mem_CfgSrv ().getProperty(PROP_SERVER_COUNT);
        Validate.notEmpty(sCount, "Invalid configuration value : "+PROP_SERVER_COUNT+" is missing or empty.");
        final int    nCount = Integer.parseInt(sCount);
        Validate.isTrue(nCount>0, "Invalid configuration value : "+PROP_SERVER_COUNT+" needs to > 0.");
        return nCount;
    }

    //-------------------------------------------------------------------------
    public String getServerHost (final int nNr)
        throws Exception
    {
    	final String sKey  = StringUtils.replace(PROP_SERVER_X_HOST, PROP_PLACEHOLDER, Integer.toString(nNr));
    	final String sHost = mem_CfgSrv ().getProperty(sKey);
        Validate.notEmpty(sHost, "Invalid configuration value : "+sKey+" is missing or empty.");
        return sHost;
    }

    //-------------------------------------------------------------------------
    public int getServerPort (final int nNr)
        throws Exception
    {
    	final String sKey  = StringUtils.replace(PROP_SERVER_X_PORT, PROP_PLACEHOLDER, Integer.toString(nNr));
    	final String sPort = mem_CfgSrv ().getProperty(sKey);
        Validate.notEmpty(sPort, "Invalid configuration value : "+sKey+" is missing or empty.");
        final int    nPort = Integer.parseInt(sPort);
        Validate.isTrue(nPort>0, "Invalid configuration value : "+sKey+" needs to > 0.");
        return nPort;
    }

    //-------------------------------------------------------------------------
    public int getWorkerPortRangeMin ()
        throws Exception
    {
    	final String sPort = mem_CfgSrv ().getProperty(PROP_WORKER_PORT_RANGE_MIN);
        Validate.notEmpty(sPort, "Invalid configuration value : "+PROP_WORKER_PORT_RANGE_MIN+" is missing or empty.");
        final int    nPort = Integer.parseInt(sPort);
        Validate.isTrue(nPort>0, "Invalid configuration value : "+PROP_WORKER_PORT_RANGE_MIN+" needs to > 0.");
        return nPort;
    }

    //-------------------------------------------------------------------------
    public int getWorkerPortRangeMax ()
        throws Exception
    {
    	final String sPort = mem_CfgSrv ().getProperty(PROP_WORKER_PORT_RANGE_MAX);
        Validate.notEmpty(sPort, "Invalid configuration value : "+PROP_WORKER_PORT_RANGE_MAX+" is missing or empty.");
        final int    nPort = Integer.parseInt(sPort);
        Validate.isTrue(nPort>0, "Invalid configuration value : "+PROP_WORKER_PORT_RANGE_MAX+" needs to > 0.");
        return nPort;
    }

    //-------------------------------------------------------------------------
    @Deprecated
    public int getWorkerPortRangeMin (final int nNr)
        throws Exception
    {
    	final String sKey  = StringUtils.replace(PROP_SERVER_X_WORKER_PORTRANGE_MIN, PROP_PLACEHOLDER, Integer.toString(nNr));
    	final String sPort = mem_CfgSrv ().getProperty(sKey);
        Validate.notEmpty(sPort, "Invalid configuration value : "+sKey+" is missing or empty.");
        final int    nPort = Integer.parseInt(sPort);
        Validate.isTrue(nPort>0, "Invalid configuration value : "+sKey+" needs to > 0.");
        return nPort;
    }

    //-------------------------------------------------------------------------
    @Deprecated
    public int getWorkerPortRangeMax (final int nNr)
        throws Exception
    {
    	final String sKey  = StringUtils.replace(PROP_SERVER_X_WORKER_PORTRANGE_MAX, PROP_PLACEHOLDER, Integer.toString(nNr));
    	final String sPort = mem_CfgSrv ().getProperty(sKey);
        Validate.notEmpty(sPort, "Invalid configuration value : "+sKey+" is missing or empty.");
        final int    nPort = Integer.parseInt(sPort);
        Validate.isTrue(nPort>0, "Invalid configuration value : "+sKey+" needs to > 0.");
        return nPort;
    }

    //-------------------------------------------------------------------------
    private ConfigurationService mem_CfgSrv ()
        throws Exception
    {
        if (m_aCfgSrv == null)
            m_aCfgSrv = m_aSmgr.getService(ConfigurationService.class);
        return m_aCfgSrv;
    }

    //-------------------------------------------------------------------------
    private static CalcEngineConfig m_gSingleton = null;
    
    //-------------------------------------------------------------------------
    private ServiceLookup m_aSmgr = null;
    
    //-------------------------------------------------------------------------
    private ConfigurationService m_aCfgSrv = null;
}
