/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2012 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.calcengine.client.impl;

import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.Validate;

import com.openexchange.office.calcengine.client.CalcEngineClientFactory;
import com.openexchange.office.calcengine.client.CalcEngineDescriptor;
import com.openexchange.office.calcengine.client.ECalcEngineMode;
import com.openexchange.office.calcengine.client.ICalcEngineClient;
import com.openexchange.office.tools.executablewrapper.Executable;
import com.openexchange.office.tools.executablewrapper.IExecutableWatcher;
import com.openexchange.office.tools.executablewrapper.SimpleExecutableWatcher;
import com.openexchange.office.tools.jobpool.JobArguments;
import com.openexchange.office.tools.jobpool.JobBase;
import com.openexchange.office.tools.jobpool.JobEnvironment;
import com.openexchange.office.tools.jobpool.JobResults;

//=============================================================================
public class CalcEngineJob extends JobBase
{
	//-------------------------------------------------------------------------
	public static boolean SIMULATION_MODE = false;

	//-------------------------------------------------------------------------
	public static final String ENV_JAR  = "env.jar";
	public static final String ENV_PORT = "env.port";

	//-------------------------------------------------------------------------
	public static final String ARG_OPERATION          = "arg.operation"      ;
	public static final String ARG_DOCUMENT_HANDLE    = "arg.document.handle";
	public static final String ARG_OPERATIONLIST      = "arg.operationlist"  ;
	
	//-------------------------------------------------------------------------
	public static final String OP_CREATE_DOCUMENT     = "createDocument"  ;
	public static final String OP_DESTROY_DOCUMENT    = "destroyDocument" ;
	public static final String OP_EXECUTE_OPERATION   = "executeOperation";
    public static final String OP_GET_VERSION         = "getVersion";

	//-------------------------------------------------------------------------
	public static final String RESULT_DOCUMENT_HANDLE = "result.document.handle";
	public static final String RESULT_4_EXECUTION     = "result.4.execution";
	
	//-------------------------------------------------------------------------
	public CalcEngineJob ()
		throws Exception
	{}

	//-------------------------------------------------------------------------
	@Override
	protected void activateImpl(final JobEnvironment aEnv)
		throws Exception
	{
		// read configuration first to know the connection details
		
		final String  sJar  = aEnv.get (ENV_JAR , null);
		final Integer nPort = aEnv.get (ENV_PORT, 0   );
		
		Validate.notEmpty (sJar   , "Miss environment var '"+ENV_JAR +"'.");
		Validate.isTrue   (nPort>0, "Miss environment var '"+ENV_PORT+"'.");
		
		// start worker process (running in background)
		
		final Executable aExe = new Executable ();

		aExe.setExecutable ("java"                 );
		aExe.addArgument   ("-jar"                 );
		aExe.addArgument   (sJar                   );
		aExe.addArgument   ("-p"                   );
		aExe.addArgument   (Integer.toString(nPort));
		aExe.addArgument   ("-i"                   );
		aExe.addArgument   ("localhost"            );
		
		if (SIMULATION_MODE) // for testing purposes only !
			aExe.addArgument ("-s");
		
		m_aProcess = aExe;

		final String         sWatchPoint4PortOpen = "INFO: Started SelectChannelConnector@localhost:"+nPort;
		final CountDownLatch aWatchPointSync      = new CountDownLatch (1);
		aExe.registerWatcher(SimpleExecutableWatcher.create(aWatchPointSync, sWatchPoint4PortOpen));

		System.out.println ("#### START ..."  );
		m_aProcessSync = new CountDownLatch (1);
		m_aProcess.runAsync (m_aProcessSync);
		System.out.println ("#### STARTED ...");
		
		final boolean bStarted = aWatchPointSync.await(5000, TimeUnit.MILLISECONDS); // TODO read from config
		if ( ! bStarted)
			throw new Exception ("Worker not started in time.");
		
		// connect to that worker
		
		CalcEngineDescriptor aDesc         = new CalcEngineDescriptor ();
		                     aDesc.m_eMode = ECalcEngineMode.E_REMOTE;
		                     aDesc.m_sHost = "localhost";
		                     aDesc.m_nPort = nPort;
		ICalcEngineClient    iAPI          = CalcEngineClientFactory.getDirect(aDesc);
		m_iAPI = iAPI;
	}

	//-------------------------------------------------------------------------
	@Override
	protected void validateImpl(final JobEnvironment aEnv)
		throws Exception
	{
		// TODO implement me
	}

	//-------------------------------------------------------------------------
	@Override
	protected void deactivateImpl(final JobEnvironment aEnv)
		throws Exception
	{
		if (m_aProcess != null)
		{
			System.out.println ("Kill worker ....");
			m_aProcess.kill();
			System.out.println ("Done.");
		}
	}

	//-------------------------------------------------------------------------
	@Override
	protected void executeImpl(final JobEnvironment aEnv      ,
							   final JobArguments   lArguments,
							   final JobResults     lResults  )
        throws Exception
    {
		Validate.notNull(m_iAPI, "No API / no process ? Did you defined an environment for this job ?");
		
		String sOp = lArguments.get (ARG_OPERATION);
		Validate.notEmpty(sOp, "Invalid argument '"+ARG_OPERATION+"'.");
		
		if (StringUtils.equalsIgnoreCase(sOp, OP_CREATE_DOCUMENT))
			impl_createDocument (lArguments, lResults);
		else
		if (StringUtils.equalsIgnoreCase(sOp, OP_DESTROY_DOCUMENT))
			impl_destroyDocument (lArguments, lResults);
		else
		if (StringUtils.equalsIgnoreCase(sOp, OP_EXECUTE_OPERATION))
			impl_executeOperation (lArguments, lResults);
		else
			throw new UnsupportedOperationException ("No support for operation '"+sOp+"' yet. Please implement.");
	}

	//-------------------------------------------------------------------------
	private void impl_createDocument(final JobArguments lArguments,
			   						 final JobResults   lResults  )
        throws Exception
    {
		String sId = m_iAPI.createDocument();
		lResults.set(RESULT_DOCUMENT_HANDLE, sId);
    }
	
	//-------------------------------------------------------------------------
	private void impl_destroyDocument(final JobArguments lArguments,
			   						  final JobResults   lResults  )
        throws Exception
    {
		final String sDocHandle = lArguments.get(ARG_DOCUMENT_HANDLE);
		m_iAPI.destroyDocument(sDocHandle);
    }

	//-------------------------------------------------------------------------
	private void impl_executeOperation(final JobArguments lArguments,
			   					       final JobResults   lResults  )
        throws Exception
    {
		final String sDocHandle  = lArguments.get(ARG_DOCUMENT_HANDLE);
		final String lOperations = lArguments.get(ARG_OPERATIONLIST  );
		final String sResult     = m_iAPI.executeOperation(sDocHandle, lOperations);
		lResults.set (RESULT_4_EXECUTION, sResult);
    }

	//-------------------------------------------------------------------------
	private Executable m_aProcess = null;
	
	//-------------------------------------------------------------------------
	private CountDownLatch m_aProcessSync = null;
	
	//-------------------------------------------------------------------------
	private ICalcEngineClient m_iAPI = null;
}
