/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/text/format/drawingstyles',
    ['io.ox/office/tk/utils',
     'io.ox/office/framework/model/format/stylesheets',
     'io.ox/office/framework/model/format/border',
     'io.ox/office/framework/view/drawingframe',
     'io.ox/office/text/dom',
     'io.ox/office/text/drawingResize',
     'gettext!io.ox/office/text'
    ], function (Utils, StyleSheets, Border, DrawingFrame, DOM, DrawingResize, gt) {

    'use strict';

    var // definitions for common drawing attributes
        DEFINITIONS = {

            /**
             * Width of the drawing, as number in 1/100 of millimeters.
             */
            width: {
                def: 0,
                format: function (element, width) {
                    if (width === 0) {
                        element.width('100%');
                    } else {
                        element.width(Utils.convertHmmToLength(width, 'px', 1));
                    }
                }
            },

            /**
             * Height of the drawing, as number in 1/100 of millimeters.
             */
            height: {
                def: 0,
                format: function (element, height) {
                    element.height(Utils.convertHmmToLength(height, 'px', 1));
                }
            },

            /**
             * Margin from top border of the drawing to text contents, in 1/100
             * of millimeters.
             */
            marginTop: { def: 0 },

            /**
             * Margin from bottom border of the drawing to text contents, in
             * 1/100 of millimeters.
             */
            marginBottom: { def: 0 },

            /**
             * Margin from left border of the drawing to text contents, in 1/100
             * of millimeters.
             */
            marginLeft: { def: 0 },

            /**
             * Margin from right border of the drawing to text contents, in
             * 1/100 of millimeters.
             */
            marginRight: { def: 0 },

            /**
             * If set to true, the drawing is rendered as inline element ('as
             * character'), otherwise it is anchored relative to another
             * element (page, paragraph, table cell, ...).
             */
            inline: { def: true },

            anchorHorBase: { def: 'margin' },

            anchorHorAlign: { def: 'left' },

            anchorHorOffset: { def: 0 },

            anchorVertBase: { def: 'margin' },

            anchorVertAlign: { def: 'top' },

            anchorVertOffset: { def: 0 },

            /**
             * Specifies how text floats around the drawing.
             * - 'none': Text does not float around the drawing.
             * - 'square': Text floats around the bounding box of the drawing.
             * - 'tight': Text aligns to the left/right outline of the drawing.
             * - 'through': Text aligns to the entire outline of the drawing.
             * - 'topAndBottom': Text floats above and below the drawing only.
             */
            textWrapMode: { def: 'none' },

            /**
             * Specifies on which side text floats around the drawing. Effective
             * only if the attribute 'textWrapMode' is either 'square',
             * 'tight', or 'through'.
             * - 'both': Text floats at the left and right side.
             * - 'left': Text floats at the left side of the drawing only.
             * - 'right': Text floats at the right side of the drawing only.
             * - 'largest': Text floats at the larger side of the drawing only.
             */
            textWrapSide: { def: 'both' },

            /**
             * Image Data. The string contains either base64 image data, or svg.
             * If base64 encoded image data is used, the string begins with "data:"
             * otherwise if svg is used it begins with "<svg"
             */
            replacementData: {
                def: '',
                scope: 'element'
            },

            // Image specific attributes

            /**
             * URL pointing to the image data. If the image was embedded in the
             * document archive, the URL will be relativ to the document (image specific style).
             */
            imageUrl: {
                def: '',
                scope: 'element'
            },

            /**
             * Image data (image specific style).
             */
            imageData: {
                def: '',
                scope: 'element'
            },

            /**
             * Amount of left part of the image cropped outside the object
             * border, in percent (image specific style).
             */
            cropLeft: { def: 0 },

            /**
             * Amount of right part of the image cropped outside the object
             * border, in percent (image specific style).
             */
            cropRight: { def: 0 },

            /**
             * Amount of top part of the image cropped outside the object
             * border, in percent (image specific style).
             */
            cropTop: { def: 0 },

            /**
             * Amount of bottom part of the image cropped outside the object
             * border, in percent (image specific style).
             */
            cropBottom: { def: 0 },

            /**
             * Style, width and color of the left drawing border.
             */
            borderLeft: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-left', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the right drawing border.
             */
            borderRight: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-right', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the top drawing border.
             */
            borderTop: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-top', this.getCssBorder(border));
                }
            },

            /**
             * Style, width and color of the bottom drawing border.
             */
            borderBottom: {
                def: Border.NONE,
                format: function (element, border) {
                    element.css('border-bottom', this.getCssBorder(border));
                }
            },

            /**
             * Flip the drawing horizontally (swap left and right side).
             */
            flipH: { def: false },

            /**
             * Flip the drawing vertically (swap top and bottom side).
             */
            flipV: { def: false },

            name: { def: ''},

            description: { def: '' }

        },

        // predefined drawing attributes for floating modes used in GUI
        FLOAT_MODE_ATTRIBUTES = {
            inline:       { inline: true },
            leftFloated:  { inline: false, anchorHorBase: 'column', anchorHorAlign: 'left',   textWrapMode: 'square', textWrapSide: 'right', anchorHorOffset: 0 },
            rightFloated: { inline: false, anchorHorBase: 'column', anchorHorAlign: 'right',  textWrapMode: 'square', textWrapSide: 'left',  anchorHorOffset: 0 },
            noneFloated:  { inline: false, anchorHorBase: 'column', anchorHorAlign: 'center', textWrapMode: 'topAndBottom', anchorHorOffset: 0 }
        },

        // values for the 'textWrapMode' attribute allowing to wrap the text around the drawing
        WRAPPED_TEXT_VALUES = /^(square|tight|through)$/;

    // private global functions ===============================================

    /**
     * Returns whether the passed 'textWrapMode' attribute allows to wrap the
     * text around the drawing.
     */
    function isTextWrapped(textWrapMode) {
        return WRAPPED_TEXT_VALUES.test(textWrapMode);
    }

    /**
     * Tries to find a preceding text span for the passed drawing. Leading
     * floating drawings in a paragraph do not have a preceding text span; in
     * this case, the first text span following the drawing will be returned.
     */
    function findRelatedTextSpan(drawing) {
        var span = null;

        // the closest preceding text span
        if (drawing[0].previousSibling) {
            span = Utils.findPreviousSiblingNode(drawing, function () { return DOM.isTextSpan(this); });
        }

        // no preceding span found: find following span
        if (!span) {
            span = Utils.findNextSiblingNode(drawing, function () { return DOM.isTextSpan(this); });
        }

        return span;
    }

    /**
     * Calculates the offset and size of the bitmap in an image object for one
     * dimension (either horizontally or vertically), according to the passed
     * cropping settings.
     *
     * @param {Number} objectSize
     *  With/height of the object node, in 1/100 of millimeters.
     *
     * @param {Number} leadingCrop
     *  The leading cropping value (left/top), in percent.
     *
     * @param {Number} trailingCrop
     *  The trailing cropping value (right/bottom), in percent.
     *
     * @returns {Object}
     *  An object containing 'offset' and 'size' CSS attributes specifying how
     *  to format the bitmap (in pixels with 'px' unit name).
     */
    function calculateBitmapSettings(objectSize, leadingCrop, trailingCrop) {

        var // sum of leading and trailing cropping (must not exceed a specific amount)
            totalCrop = leadingCrop + trailingCrop,
            // resulting settings for the bitmap
            size = 0, offset = 0;

        // do not crop more than 90% of the bitmap
        if (totalCrop > 90) {
            leadingCrop *= (90 / totalCrop);
            trailingCrop *= (90 / totalCrop);
        }

        // bitmap size and offset, according to object size and cropping
        size = objectSize * 100 / (100 - leadingCrop - trailingCrop);
        offset = (size * leadingCrop) / 100;

        // convert to CSS pixels
        return {
            offset: Utils.convertHmmToCssLength(-offset, 'px', 1),
            size: Utils.convertHmmToCssLength(size, 'px', 1)
        };
    }

    // class DrawingStyles =====================================================

    /**
     * Contains the style sheets for drawing formatting attributes. The CSS
     * formatting will be read from and written to drawing elements of any type.
     *
     * @constructor
     *
     * @extends StyleSheets
     *
     * @param {TextApplication} app
     *  The root application instance.
     *
     * @param {DocumentStyles} documentStyles
     *  Collection with the style containers of all style families.
     */
    function DrawingStyles(app, documentStyles) {

        // base constructor ---------------------------------------------------

        StyleSheets.call(this, app, documentStyles, 'drawing', DEFINITIONS, {
            formatHandler: updateDrawingFormatting
        });

        // private methods ----------------------------------------------------

        /**
         * Repaints the selection boxes for the specified drawing node, if it
         * is selected. Updates the move and resize handling after the drawing
         * has changed between inline and floating mode.
         */
        function repaintDrawingSelection(drawing) {
            if (DrawingFrame.isSelected(drawing)) {
                DrawingFrame.clearSelection(drawing);
                DrawingResize.drawDrawingSelection(app, drawing);
            }
        }

        /**
         * Calculating the largest line height of a span inside a paragraph.
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose spans will be investigated.
         *
         * @returns {Number}
         *  The largest line height of the spans in the paragraph in 1/100 mm.
         */
        function getMaximumLineHeightInParagraph(paragraph) {

            var // the largest value for the line height of a span in 1/100 mm
                maxCssLineHeight = 0;

            paragraph.children('span').each(function (index, span) {

                var lineHeight = Utils.convertCssLengthToHmm($(span).css('line-height'));

                if (lineHeight > maxCssLineHeight) {
                    maxCssLineHeight = lineHeight;
                }
            });

            return maxCssLineHeight;
        }

        /**
         * Modifying the offset for all floated drawings, that are positioned behind
         * the moved drawing.
         *
         * @param {jQuery} drawingNode
         *  The moved drawing node.
         *
         * @param {Number} addOffset
         *  The change of the offset node for the drawings, that are located behind the
         *  moved drawing node (in px).
         */
        function modifyFollowingOffsetNodes(drawingNode, addOffset) {

            var node = Utils.getDomNode(drawingNode),
                localDrawing = null;

            while (node) {
                if (DOM.isOffsetNode(node)) {
                    localDrawing = node.nextSibling;
                    $(node).height($(node).height() + addOffset);
                    $(node).remove();
                    $(node).insertBefore(localDrawing);
                }

                node = node.nextSibling;
            }
        }

        /**
         * Will be called for every drawing node whose attributes have been
         * changed. Repositions and reformats the drawing according to the
         * passed attributes.
         *
         * @param {jQuery} drawing
         *  The drawing node whose attributes have been changed, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateDrawingFormatting(drawing, mergedAttributes) {

            var // the drawing attributes of the passed attribute map
                drawingAttributes = mergedAttributes.drawing,
                // the paragraph element containing the drawing node
                paragraph = drawing.parent(),
                // total width of the paragraph, in 1/100 mm
                paraWidth = Utils.convertLengthToHmm(paragraph.width(), 'px'),
                // total width of the page, in 1/100 mm
                pageWidth = null,
                // preceding node used for vertical offset
                verticalOffsetNode = drawing.prev(DOM.OFFSET_NODE_SELECTOR),
                // current drawing width, in 1/100 mm
                drawingWidth = Utils.convertLengthToHmm(drawing.width(), 'px'),
                // current drawing height, in 1/100 mm
                drawingHeight = Utils.convertLengthToHmm(drawing.height(), 'px'),
                // offset from top/left/right margin of paragraph element, in 1/100 mm
                topOffset = 0, leftOffset = 0, rightOffset = 0,
                // margins to be applied at the drawing
                topMargin = 0, bottomMargin = 0, leftMargin = 0, rightMargin = 0,
                // text wrapping side (left/right/none)
                wrapMode = 'none',
                // value of the 'transform' CSS attribute, e.g. for flipped drawings
                transformations = '',
                // type of the drawing: 'image', ...
                type = drawing.data('type'),
                // the content node inside the drawing
                contentNode = DrawingFrame.getContentNode(drawing),
                // image data string. if Base64 image, string starts with 'data:'
                base64String = 'data:',
                // image data string. if SVG image, string starts with '<svg'
                svgString = '<svg',
                // the height of the offset node before moving the drawing
                oldVerticalOffsetNodeHeight = 0,
                // the height of the offset node after moving the drawing
                newVerticalOffsetNodeHeight = 0,
                // specifies if the drawing has replacement data
                hasReplacementData = drawingAttributes.replacementData && drawingAttributes.replacementData.length;

            // position

            if (drawingAttributes.inline) {

                // switch from floating to inline mode
                if (!drawing.hasClass('inline')) {

                    // remove leading node used for positioning
                    verticalOffsetNode.remove();

                    // TODO: Word uses fixed predefined margins in inline mode, we too?
                    drawing.removeClass('float left right').addClass('inline').css('margin', '0 1mm');
                    // ignore other attributes in inline mode

                    // repaint the selection, convert it to a non-movable selection
                    repaintDrawingSelection(drawing);
                }

            } else {

                // switch from inline to floating mode
                if (!drawing.hasClass('float')) {
                    drawing.removeClass('inline').addClass('float');
                    // repaint the selection, convert it to a movable selection
                    repaintDrawingSelection(drawing);
                }

                // calculate top offset (only if drawing is anchored to paragraph or to line)
                // anchor to line means the base line of the character position. so it is not correctly calculated
                // using the paragraph, but it is better to use it, than to ignore the value. the calculation is
                // nearly correct for characters in the first row inside the paragraph.
                if ((drawingAttributes.anchorVertBase === 'paragraph') || (drawingAttributes.anchorVertBase === 'line')) {
                    if (drawingAttributes.anchorVertAlign === 'offset') {
                        topOffset = Math.max(drawingAttributes.anchorVertOffset, 0);
                        // For an offset node it is necessary to subtract the height of the text position in
                        // the paragraph in which the drawing is located. The drawing cannot be located higher
                        // than its position allows it.
                        topOffset -= Utils.convertLengthToHmm(drawing.offset().top - drawing.parent().offset().top, 'px');
                        // adding the height of an already existing offset node
                        if (($(drawing).prev().length) && (DOM.isOffsetNode($(drawing).prev()))) {
                            topOffset += Utils.convertLengthToHmm($(drawing).prev().height(), 'px');
                        }

                        if (topOffset < 0) { topOffset = 0; }
                        // But this is also no final solution, because the drawing is shifted
                        // downwards, if text is included before the drawing position in the paragraph.
                    } else {
                        // TODO: automatic alignment (top/bottom/center/...)
                        topOffset = 0;
                    }
                }

                // calculate top offset (only if drawing is anchored to line)
                // if (drawingAttributes.anchorVertBase === 'line') {
                //     if (drawingAttributes.anchorVertAlign === 'offset') {
                //         topOffset = Math.max(drawingAttributes.anchorVertOffset, 0);
                //     } else {
                //         // TODO: automatic alignment (top/bottom/center/...)
                //         topOffset = 0;
                //     }
                // }

                // saving the old height of the vertical offset node
                if (verticalOffsetNode) {
                    oldVerticalOffsetNodeHeight = verticalOffsetNode.height();  // saving the height (in pixel)
                }

                // calculate top/bottom drawing margins
                topMargin = Utils.minMax(drawingAttributes.marginTop, 0, topOffset);
                bottomMargin = Math.max(drawingAttributes.marginBottom, 0);

                // add or remove leading offset node used for positioning
                // TODO: support for multiple drawings (also overlapping) per side
                topOffset -= topMargin;
                if (topOffset < 700) {
                    // offset less than 7mm: expand top margin to top of paragraph,
                    // otherwise the first text line overwrites the drawing
                    topMargin += topOffset;
                    // remove offset node
                    verticalOffsetNode.remove();
                } else {
                    // create offset node if not existing yet
                    if (verticalOffsetNode.length === 0) {
                        verticalOffsetNode = $('<div>', { contenteditable: false }).addClass('float offset').width(1).insertBefore(drawing);
                    }
                    // set height of the offset node
                    verticalOffsetNode.height(Utils.convertHmmToLength(topOffset, 'px', 1));
                    // Sometimes the browser ignores the change of height of the vertical offset node. This happens often, if there are only
                    // small movements of the drawing upwards. The browser does not ignore the vertical offset node, if it is removed from
                    // the dom and immediately inserted again before the drawing (28312).
                    verticalOffsetNode.remove();
                    verticalOffsetNode.insertBefore(drawing);
                    newVerticalOffsetNodeHeight = verticalOffsetNode.height();
                }

                // modifying the vertical offset for all following floated drawings in the same paragraph
                if (newVerticalOffsetNodeHeight !== oldVerticalOffsetNodeHeight) {
                    // the drawing was moved upwards -> all following floated drawings in the same paragraph need an increased offset node
                    modifyFollowingOffsetNodes(drawing, oldVerticalOffsetNodeHeight - newVerticalOffsetNodeHeight);
                }

                // calculate left/right offset (only if drawing is anchored to column)
                if (drawingAttributes.anchorHorBase === 'column') {
                    switch (drawingAttributes.anchorHorAlign) {
                    case 'center':
                        leftOffset = (paraWidth - drawingWidth) / 2;
                        break;
                    case 'right':
                        leftOffset = paraWidth - drawingWidth;
                        break;
                    case 'offset':
                        leftOffset = drawingAttributes.anchorHorOffset;
                        if (drawingAttributes.anchorHorOffset + drawingWidth > paraWidth) {
                            leftOffset = paraWidth - drawingWidth;
                        }
                        break;
                    default:
                        leftOffset = 0;
                    }
                } else if (drawingAttributes.anchorHorBase === 'page') {
                    // 'page' looks better if it is handled like column, not like the page node 'div.page'
                    // pageWidth = Utils.convertLengthToHmm(drawing.closest(DOM.PAGE_NODE_SELECTOR).width(), 'px');
                    switch (drawingAttributes.anchorHorAlign) {
                    case 'center':
                        leftOffset = (paraWidth - drawingWidth) / 2;
                        break;
                    case 'right':
                        leftOffset = paraWidth - drawingWidth;
                        break;
                    case 'offset':
                        leftOffset = drawingAttributes.anchorHorOffset;
                        break;
                    default:
                        leftOffset = 0;
                    }
                } else {
                    // TODO: other anchor bases (character/margins/...)
                    leftOffset = 0;
                }
                rightOffset = paraWidth - leftOffset - drawingWidth;

                // determine text wrapping side
                if (isTextWrapped(drawingAttributes.textWrapMode)) {
                    switch (drawingAttributes.textWrapSide) {
                    case 'left':
                        wrapMode = 'left';
                        break;
                    case 'right':
                        wrapMode = 'right';
                        break;
                    case 'both':
                    case 'largest':
                        // no support for 'wrap both sides' in CSS, default to 'largest'
                        wrapMode = (leftOffset > rightOffset) ? 'left' : 'right';
                        break;
                    default:
                        Utils.warn('updateDrawingFormatting(): invalid text wrap side: ' + drawingAttributes.textWrapSide);
                        wrapMode = 'none';
                    }
                } else {
                    // text does not float beside drawing
                    wrapMode = 'none';
                }

                // calculate left/right drawing margins
                switch (wrapMode) {

                case 'left':
                    // drawing floats at right paragraph margin
                    rightMargin = rightOffset;
                    leftMargin = Math.max(drawingAttributes.marginLeft, 0);
                    // if there is less than 6mm space available for text, occupy all space (no wrapping)
                    if (leftOffset - leftMargin < 600) { leftMargin = Math.max(leftOffset, 0); }
                    break;

                case 'right':
                    // drawing floats at left paragraph margin
                    leftMargin = leftOffset;
                    rightMargin = Math.max(drawingAttributes.marginRight, 0);
                    // if there is less than 6mm space available for text, occupy all space (no wrapping)
                    if (rightOffset - rightMargin < 600) { rightMargin = Math.max(rightOffset, 0); }
                    break;

                default:
                    // no wrapping: will be modeled by left-floated with large CSS margins
                    wrapMode = 'right';
                    leftMargin = leftOffset;
                    rightMargin = Math.max(rightOffset, 0);
                }

                // set text wrap mode to drawing and offset node
                drawing.add(verticalOffsetNode).removeClass('left right').addClass((wrapMode === 'left') ? 'right' : 'left');

                // in Webkit browser there is a bug, that a floated drawing can be moved above the surrounding text at the top
                // of the drawing. So if the drawing is not located in the first line of the paragraph (then it has an offset
                // node), the drawing should get a top margin, so that it does not overlap the text (fix for 28011).
                if (($.browser.webkit) && (topOffset >= 700)) {
                    // receiving the largest 'line-height' of the paragraph.
                    topMargin += getMaximumLineHeightInParagraph(paragraph);
                }

                // apply CSS formatting to drawing node
                drawing.css({
                    marginTop: Utils.convertHmmToCssLength(topMargin, 'px', 1),
                    marginBottom: Utils.convertHmmToCssLength(bottomMargin, 'px', 1),
                    marginLeft: Utils.convertHmmToCssLength(leftMargin, 'px', 1),
                    marginRight: Utils.convertHmmToCssLength(rightMargin, 'px', 1)
                });
            }

            // vertically or horizontally flipped drawings
            if (drawingAttributes.flipH && drawingAttributes.flipV) {
                transformations += ' scale(-1)';
            } else if (drawingAttributes.flipH) {
                transformations += ' scaleX(-1)';
            } else if (drawingAttributes.flipV) {
                transformations += ' scaleY(-1)';
            }

            // always set the CSS attribute, to be able to remove existing transformations
            Utils.setCssAttributeWithPrefixes(contentNode, 'transform', transformations);

            // use replacement data, if available (valid for all drawing types)
            if (hasReplacementData) {
                if (drawingAttributes.replacementData.indexOf(base64String) === 0) {
                    imageNode = $('<img>', { src: drawingAttributes.replacementData });
                    contentNode.append(imageNode);
                } else if (drawingAttributes.replacementData.indexOf(svgString) === 0) {
                    Utils.getDomNode(contentNode).appendChild($(drawingAttributes.replacementData)[0]);
                }
            }

            // set text attribute to support tooltip for the drawing
            Utils.setControlTooltip(drawing, drawingAttributes.name);

            // some attributes are specific to the drawing type
            if ((type === 'image') || (type === 'ole')) {

                var // horizontal offset/size of the bitmap, as CSS attributes
                    horizontalSettings = null,
                    // vertical offset/size of the bitmap, as CSS attributes
                    verticalSettings = null,
                    // the image node inside the drawing node
                    imageNode = contentNode.find('img'),
                    // the source data or url for the image
                    imgSrc = null,
                    // an <img> node can be used for urls or image sources starting with 'data:'
                    useImageNode = false,
                    // an <svg> node can be used directly for image sources starting with '<svg'
                    useSvgNode = false,
                    // varibles for saving cropping attribute values
                    topCrop = 0,
                    bottomCrop = 0,
                    leftCrop = 0,
                    rightCrop = 0,
                    tempCrop = 0;

                if (imageNode.length === 0) {
                    // inserting the image
                    if (drawingAttributes.imageData && drawingAttributes.imageData.length) {
                        imgSrc = drawingAttributes.imageData;
                        if (imgSrc.indexOf(base64String) === 0) {
                            useImageNode = true;
                        } else if (imgSrc.indexOf(svgString) === 0) {
                            useSvgNode = true;
                        }
                    } else {
                        imgSrc = drawing.data('absoluteURL');
                        useImageNode = true;
                    }

                    if (useImageNode) {
                        imageNode = $('<img>', { src: imgSrc });
                        contentNode.append(imageNode);
                    } else if (useSvgNode) {
                        contentNode[0].innerHTML = imgSrc;
                    }
                }

                if ((drawingWidth > 0) && (drawingHeight > 0)) {

                    topCrop = drawingAttributes.cropTop;
                    bottomCrop = drawingAttributes.cropBottom;
                    leftCrop = drawingAttributes.cropLeft;
                    rightCrop = drawingAttributes.cropRight;

                    // IE with fliph at image -> switch cropTop and cropBottom
                    // IE with flipv at image -> switch cropLeft and cropRight
                    if ($.browser.msie) {
                        if (drawingAttributes.flipH) {
                            tempCrop = topCrop;
                            topCrop = bottomCrop;
                            bottomCrop = tempCrop;
                        }

                        if (drawingAttributes.flipV) {
                            tempCrop = leftCrop;
                            leftCrop = rightCrop;
                            rightCrop = tempCrop;
                        }
                    }

                    horizontalSettings = calculateBitmapSettings(drawingWidth, leftCrop, rightCrop);
                    verticalSettings = calculateBitmapSettings(drawingHeight, topCrop, bottomCrop);

                    // set CSS formatting at the <img> element
                    imageNode.css({
                        left: horizontalSettings.offset,
                        top: verticalSettings.offset,
                        width: horizontalSettings.size,
                        height: verticalSettings.size
                    });
                }
            } else if (!hasReplacementData && (/^(undefined|shape|diagram|chart)$/).test(type)) {

                DrawingFrame.insertReplacementNodes(drawing, drawingAttributes.name, drawingAttributes.description);
            }
        }

    } // class ImageStyles

    // static methods ---------------------------------------------------------

    /**
     * Returns the drawing position identifier representing the passed drawing
     * attributes.
     *
     * @param {Object} attributes
     *  A map with drawing attributes, as name/value pairs.
     *
     * @returns {String|Null}
     *  The GUI drawing position identifier; or null, if any of the passed
     *  attributes is ambiguous.
     */
    DrawingStyles.getPositionFromAttributes = function (attributes) {

        // inline mode overrules floating attributes
        if (_.isNull(attributes.inline)) { return null; }
        if (attributes.inline === true) { return 'inline'; }

        // filter unsupported position anchor modes
        if (!/^(column)$/.test(attributes.anchorHorBase) || !/^(paragraph|margin|line)$/.test(attributes.anchorVertBase)) { return null; }

        // filter ambiguous or unsupported horizontal alignment types
        if (!/^(left|center|right)$/.test(attributes.anchorHorAlign)) { return null; }

        // filter ambiguous text wrapping modes
        if (_.isNull(attributes.textWrapMode) || _.isNull(attributes.textWrapSide)) { return null; }

        // build the resulting drawing position
        return attributes.anchorHorAlign + ':' + (isTextWrapped(attributes.textWrapMode) ? attributes.textWrapSide : 'none');
    };

    /**
     * Returns the drawing attributes that are needed to represent the passed
     * GUI drawing position.
     *
     * @param {String} position
     *  The GUI drawing position.
     *
     * @param {Object} [origAttributes]
     *  The original attributes of the target drawing object.
     *
     * @param {Object}
     *  A map with drawing attributes, as name/value pairs.
     */
    DrawingStyles.getAttributesFromPosition = function (position, origAttributes) {

        var // extract alignment and text wrap mode
            matches = position.match(/^([a-z]+):([a-z]+)$/),
            // resulting drawing attributes
            drawingAttributes = {};

        // inline flag overrules all other attributes, no need to return any other (unfortunately it is still necessary for undo reasons)
        if (position === 'inline') {
            return { inline: true,
                     anchorHorBase: null,
                     anchorHorAlign: null,
                     anchorHorOffset: null,
                     anchorVertBase: null,
                     anchorVertAlign: null,
                     anchorVertOffset: null,
                     textWrapMode: null,
                     textWrapSide: null };
        }

        // check that passed position contains alignment and text wrapping mode
        if (!_.isArray(matches) || (matches.length < 3)) {
            Utils.warn('DrawingStyles.getAttributesFromPosition(): invalid drawing position: "' + position + '"');
            return {};
        }

        // set horizontal alignment
        _(drawingAttributes).extend({ inline: false, anchorHorBase: 'column', anchorHorAlign: matches[1], anchorHorOffset: 0 });

        // set text wrapping mode
        if (matches[2] === 'none') {
            _(drawingAttributes).extend({ textWrapMode: 'topAndBottom', textWrapSide: null });
        } else {
            _(drawingAttributes).extend({ textWrapMode: 'square', textWrapSide: matches[2] });
        }

        // reset vertical alignments for unsupported positions ('page' and 'margin')
        if (_.isObject(origAttributes) && /^(page|margin)$/.test(origAttributes.anchorVertBase)) {
            _(drawingAttributes).extend({ anchorVertBase: 'paragraph', anchorVertOffset: 0 });
        }

        return drawingAttributes;
    };

    // exports ================================================================

    // derive this class from class StyleSheets
    return StyleSheets.extend({ constructor: DrawingStyles });

});
