/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/text/format/characterstyles',
    ['io.ox/office/tk/utils',
     'io.ox/office/framework/model/format/color',
     'io.ox/office/framework/model/format/fonts',
     'io.ox/office/framework/model/format/lineheight',
     'io.ox/office/framework/model/format/stylesheets',
     'io.ox/office/text/dom'
    ], function (Utils, Color, Fonts, LineHeight, StyleSheets, DOM) {

    'use strict';

    var // definitions for character attributes
        DEFINITIONS = {

            fontName: {
                def: 'Arial',
                format: function (element, fontName) {
                    element.css('font-family', this.getCssFontFamily(fontName));
                },
                preview: true // 'true' calls the own 'format' method
            },

            fontSize: {
                def: 11,
                // format: done together with escapement in the format handler
                preview: function (element, fontSize) {
                    fontSize = Utils.round(10 + (fontSize - 10) / 1.5, 0.1);
                    element.css('font-size', Utils.minMax(fontSize, 6, 22) + 'pt');
                }
            },

            bold: {
                def: false,
                format: function (element, state) {
                    element.css('font-weight', state ? 'bold' : 'normal');
                },
                preview: true // 'true' calls the own 'format' method
            },

            italic: {
                def: false,
                format: function (element, state) {
                    element.css('font-style', state ? 'italic' : 'normal');
                },
                preview: true // 'true' calls the own 'format' method
            },

            underline: {
                def: false,
                format: function (element, state) {
                    var value = element.css('text-decoration');
                    element.css('text-decoration', Utils.toggleToken(value, 'underline', state, 'none'));
                },
                preview: true // 'true' calls the own 'format' method
            },

            strike: {
                def: 'none',
                format: function (element, strike) {
                    var value = element.css('text-decoration');
                    element.css('text-decoration', Utils.toggleToken(value, 'line-through', strike !== 'none', 'none'));
                },
                preview: true // 'true' calls the own 'format' method
            },

            vertAlign: { def: 'baseline' },

            color: {
                def: Color.AUTO,
                // format: color will be set in format handler, depending on fill colors
                preview: function (element, color) {
                    element.css('color', this.getCssColor(color, 'text'));
                }
            },

            fillColor: {
                def: Color.AUTO,
                format: function (element, color) {
                    element.css('background-color', this.getCssColor(color, 'fill'));
                }
            },

            language: {
                def: '',
                format: function (element, value) {
                    element.attr('lang', value);
                }
            },

            url: {
                def: '',
                scope: 'element',
                format: function (element, url) {
                    if (_.isString(url) && (url.length > 0)) {
                        element.attr('title', url);
                    } else {
                        element.removeAttr('title');
                    }
                }
            },

            // special attributes

            spellerror: {
                def: false,
                scope: 'element',
                format: function (element, state) {
                    element.toggleClass('spellerror', state);
                },
                special: true
            },

            highlight: {
                def: false,
                scope: 'element',
                format: function (element, state) {
                    element.toggleClass('highlight', state);
                },
                special: true
            }

        },

        PARENT_FAMILIES = {
            paragraph: function (span) { return span.closest(DOM.PARAGRAPH_NODE_SELECTOR); }
        };

    // class CharacterStyles ==================================================

    /**
     * Contains the style sheets for character formatting attributes. The CSS
     * formatting will be written to text span elements contained somewhere in
     * the paragraph elements.
     *
     * @constructor
     *
     * @extends StyleSheets
     *
     * @param {TextApplication} app
     *  The root application instance.
     *
     * @param {TextDocumentStyles} documentStyles
     *  Collection with the style containers of all style families.
     */
    function CharacterStyles(app, documentStyles) {

        var // self reference
            self = this;

        // base constructor ---------------------------------------------------

        StyleSheets.call(this, app, documentStyles, 'character', DEFINITIONS, {
            parentFamilies: PARENT_FAMILIES,
            formatHandler: updateCharacterFormatting
        });

        // private methods ----------------------------------------------------

        /**
         * Will be called for every text span whose character attributes have
         * been changed.
         *
         * @param {jQuery} textSpan
         *  The text span whose character attributes have been changed, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateCharacterFormatting(textSpan, mergedAttributes) {

            var // the parent paragraph of the node (may be a grandparent)
                paragraph = $(textSpan).closest(DOM.PARAGRAPH_NODE_SELECTOR),
                // the paragraph style container
                paragraphStyles = documentStyles.getStyleSheets('paragraph'),
                // the merged attributes of the paragraph
                paragraphAttributes = paragraphStyles.getElementAttributes(paragraph).paragraph,

                // the character attributes of the passed attribute map
                characterAttributes = mergedAttributes.character,
                // effective text color, font size, and vertical offset
                textColor = characterAttributes.color,
                fontSize = characterAttributes.fontSize,
                verticalAlign = 'baseline';

            // calculate effective text color, according to fill colors
            textColor = self.getCssTextColor(textColor, [paragraphAttributes.fillColor, characterAttributes.fillColor]);

            // calculate font height and vertical alignment (according to escapement)
            switch (characterAttributes.vertAlign) {
            case 'sub':
                fontSize = Utils.round(fontSize * 0.66, 0.1);
                verticalAlign = '-30%';
                break;
            case 'super':
                fontSize = Utils.round(fontSize * 0.66, 0.1);
                verticalAlign = '50%';
                break;
            }

            // update CSS of the text span
            textSpan.css({
                color: textColor,
                fontSize: fontSize + 'pt',
                verticalAlign: verticalAlign
            });

            // update calculated line height due to changed font settings
            LineHeight.updateElementLineHeight(textSpan, paragraphAttributes.lineHeight);

            // TODO: set bullet character formatting according to paragraph attributes
            $(paragraph).find('> ' + DOM.LIST_LABEL_NODE_SELECTOR + ' > span').css('font-size', characterAttributes.fontSize + 'pt');
        }

    } // class CharacterStyles

    // static methods ---------------------------------------------------------

    /**
     * Tries to merge the passed text span with its next or previous sibling
     * text span. To be able to merge two text spans, they must contain equal
     * formatting attributes. If merging was successful, the sibling span will
     * be removed from the DOM.
     *
     * @param {HTMLElement|jQuery} node
     *  The DOM node to be merged with its sibling text span. If this object is
     *  a jQuery object, uses the first DOM node it contains.
     *
     * @param {Boolean} next
     *  If set to true, will try to merge with the next span, otherwise with
     *  the previous text span.
     */
    CharacterStyles.mergeSiblingTextSpans = function (node, next) {

        var // the sibling text span, depending on the passed direction
            sibling = null,
            // text in the passed and in the sibling node
            text = null, siblingText = null;

        // passed node and sibling node, as DOM nodes
        node = Utils.getDomNode(node);
        sibling = node[next ? 'nextSibling' : 'previousSibling'];

        // both nodes must be text spans with the same attributes
        if (DOM.isTextSpan(node) && DOM.isTextSpan(sibling) && StyleSheets.hasEqualElementAttributes(node, sibling)) {

            // add text of the sibling text node to the passed text node
            text = node.firstChild.nodeValue;
            siblingText = sibling.firstChild.nodeValue;
            node.firstChild.nodeValue = next ? (text + siblingText) : (siblingText + text);

            // remove the entire sibling span element
            $(sibling).remove();
        }
    };

    // exports ================================================================

    // derive this class from class StyleSheets
    return StyleSheets.extend({ constructor: CharacterStyles });

});
