/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/view',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/config',
     'io.ox/office/tk/control/button',
     'io.ox/office/tk/control/label',
     'io.ox/office/tk/control/textfield',
     'io.ox/office/tk/control/unitfield',
     'io.ox/office/tk/control/radiogroup',
     'io.ox/office/tk/control/radiolist',
     'io.ox/office/framework/view/toolbox',
     'io.ox/office/framework/view/editview',
     'io.ox/office/framework/view/editcontrols',
     'io.ox/office/framework/model/format/border',
     'io.ox/office/framework/model/format/color',
     'io.ox/office/spreadsheet/utils/sheetutils',
     'io.ox/office/spreadsheet/utils/colrowintervals',
     'io.ox/office/spreadsheet/utils/colrowcollection',
     'io.ox/office/spreadsheet/utils/cellcollection',
     'io.ox/office/spreadsheet/view/statusbar',
     'io.ox/office/spreadsheet/view/gridpane',
     'io.ox/office/spreadsheet/view/headerpane',
     'io.ox/office/spreadsheet/view/cornerpane',
     'gettext!io.ox/office/spreadsheet',
     'less!io.ox/office/spreadsheet/view/style.less'
    ], function (Utils, Config, Button, Label, TextField, UnitField, RadioGroup, RadioList, ToolBox, EditView, EditControls, Border, Color, SheetUtils, ColRowIntervals, ColRowCollection, CellCollection, StatusBar, GridPane, HeaderPane, CornerPane, gt) {

    'use strict';

    var // map pointing to neighbor positions for all grid pane positions
        GRID_PANE_INFOS = {
            topLeft:     { hor: 'left',  vert: 'top',    nextHor: 'topRight',    nextVert: 'bottomLeft'  },
            topRight:    { hor: 'right', vert: 'top',    nextHor: 'topLeft',     nextVert: 'bottomRight' },
            bottomLeft:  { hor: 'left',  vert: 'bottom', nextHor: 'bottomRight', nextVert: 'topLeft'     },
            bottomRight: { hor: 'right', vert: 'bottom', nextHor: 'bottomLeft',  nextVert: 'topRight'    }
        },

        // the size of the freeze separator nodes, in pixels
        FREEZE_SIZE = 1,

        // the size of the split separator nodes, in pixels
        SPLIT_SIZE = 2,

        // the highlighted inner size of the split tracking node, in pixels
        TRACKING_SIZE = 4,

        // the additional margin of the split tracking nodes, in pixels
        TRACKING_MARGIN = 3,

        // the position offset of tracking nodes compared to split lines
        TRACKING_OFFSET = (TRACKING_SIZE - SPLIT_SIZE) / 2 + TRACKING_MARGIN,

        // the minimum size of grid/header panes, in pixels
        MIN_PANE_SIZE = 40,

        // the attribute names of the cell borders
        CELL_BORDER_ATTRIBUTES = ['borderLeft', 'borderRight', 'borderTop', 'borderBottom', 'borderInsideHor', 'borderInsideVert' ],

        // default settings for a pane side
        DEFAULT_PANE_SIDE_SETTINGS = { visible: false, offset: 0, size: 0, scrollable: false, showOppositeScroll: false, hiddenSize: 0 },

        // dummy sheet layout data, used before first view update message for a sheet
        DEFAULT_SHEET_LAYOUT_DATA = { sheet: -1, cols: 0, rows: 0, width: 0, height: 0, used: { cols: 0, rows: 0, width: 0, height: 0 } },

        // the default character attributes
        DEFAULT_CHARACTER_ATTRIBUTES = {
            fontName: 'Arial',
            fontSize: 10,
            bold: false,
            italic: false,
            underline: false,
            strike: 'none',
            color: Color.AUTO
        },

        // the default cell attributes
        DEFAULT_CELL_ATTRIBUTES = {
            alignHor: 'auto',
            alignVert: 'bottom',
            wrapText: false,
            numberFormat: 0,
            fillColor: Color.AUTO,
            borderTop: Border.NONE,
            borderBottom: Border.NONE,
            borderLeft: Border.NONE,
            borderRight: Border.NONE,
            borderInsideHor: Border.NONE,
            borderInsideVert: Border.NONE
        },

        // the default column attributes
        DEFAULT_COLUMN_ATTRIBUTES = {
            visible: true,
            width: 2000 // TODO
        },

        // the default row attributes
        DEFAULT_ROW_ATTRIBUTES = {
            visible: true,
            height: 800, // TODO
            autoHeight: true
        },

        // default layout data for the active cell
        DEFAULT_CELL_LAYOUT_DATA = {
            display: '',
            result: null,
            formula: null,
            attrs: {
                character: DEFAULT_CHARACTER_ATTRIBUTES,
                cell:      DEFAULT_CELL_ATTRIBUTES,
                column:    DEFAULT_COLUMN_ATTRIBUTES,
                row:       DEFAULT_ROW_ATTRIBUTES
            }
        },

        // default selection layout data, containing active cell data
        DEFAULT_SELECTION_LAYOUT_DATA = {
            active: DEFAULT_CELL_LAYOUT_DATA,
            subtotals: {
                sum: 0,
                min: 0,
                max: 0
            }
        },

        // default options for unit fields (column width and row height)
        DEFAULT_UNIT_FIELD_OPTIONS = {
            width: 70,
            css: { textAlign: 'right' },
            min: 0,
            precision: 10,
            smallStep: 10,
            largeStep: 500,
            roundStep: true
        };

    // private global functions ===============================================

    /**
     * Returns the horizontal position of the passed grid pane position.
     *
     * @param {String} panePos
     *  The grid pane position.
     *
     * @returns {String}
     *  The position 'left' for the pane positions 'topLeft' and 'bottomLeft',
     *  or 'right' for the pane positions 'topRight' and 'bottomRight'.
     */
    function getHorizontalPaneSide(panePos) {
        return GRID_PANE_INFOS[panePos].hor;
    }

    /**
     * Returns the horizontal position of the passed grid pane position.
     *
     * @param {String} panePos
     *  The grid pane position.
     *
     * @returns {String}
     *  The position 'top' for the pane positions 'topLeft' and 'topRight', or
     *  'bottom' for the pane positions 'bottomLeft' and 'bottomRight'.
     */
    function getVerticalPaneSide(panePos) {
        return GRID_PANE_INFOS[panePos].vert;
    }

    /**
     * Returns the pane position of the horizontal neighbor of the passed grid
     * pane position.
     *
     * @param {String} panePos
     *  The grid pane position.
     *
     * @returns {String}
     *  The grid pane position of the horizontal neighbor (for example, returns
     *  'bottomLeft' for the pane position 'bottomRight').
     */
    function getNextHorizontalPanePos(panePos) {
        return GRID_PANE_INFOS[panePos].nextHor;
    }

    /**
     * Returns the pane position of the vertical neighbor of the passed grid
     * pane position.
     *
     * @param {String} panePos
     *  The grid pane position.
     *
     * @returns {String}
     *  The grid pane position of the vertical neighbor (for example, returns
     *  'topRight' for the pane position 'bottomRight').
     */
    function getNextVerticalPanePos(panePos) {
        return GRID_PANE_INFOS[panePos].nextVert;
    }

    /**
     * Generates a localized sheet name with the specified index.
     *
     * @param {Number} index
     *  The sheet index to be inserted into the sheet name.
     *
     * @returns {String}
     *  The generated sheet name (e.g. "Sheet1", "Sheet2", etc.).
     */
    function generateSheetName(index) {
        return (
            //#. %1$d is the index in a sheet name (e.g. "Sheet1", "Sheet2", etc.)
            //#, c-format
            gt('Sheet%1$d', _.noI18n(index)));
    }

    // class SpreadsheetView ==================================================

    /**
     * Represents the entire view of a spreadsheet document. Contains the view
     * panes of the sheet currently shown (the 'active sheet'), which contain
     * the scrollable cell grids (there will be several view panes, if the
     * sheet view is split or frozen); and the selections of all existing
     * sheets.
     *
     * @constructor
     *
     * @extends EditView
     */
    function SpreadsheetView(app) {

        var // self reference
            self = this,

            // the spreadsheet document model
            model = null,

            // the scrollable root DOM node containing the headers and panes
            rootNode = $('<div>').addClass('pane-root'),

            // the row/column header panes, mapped by position keys
            headerPanes = {},

            // the top-left corner pane
            cornerPane = null,

            // the grid panes, mapped by position keys
            gridPanes = {},

            // the split line separating the left and right grid panes
            horSplitLine = $('<div>').addClass('split-line horizontal'),

            // the split line separating the top and bottom grid panes
            vertSplitLine = $('<div>').addClass('split-line vertical'),

            // the split tracking node separating left and right grid panes
            horTrackingNode = $('<div>')
                .addClass('split-tracking horizontal')
                .css({ top: 0, bottom: 0, width: TRACKING_SIZE + 'px', padding: '0 ' + TRACKING_MARGIN + 'px' }),

            // the split tracking node separating top and bottom grid panes
            vertTrackingNode = $('<div>')
                .addClass('split-tracking vertical')
                .css({ left: 0, right: 0, height: TRACKING_SIZE + 'px', padding: TRACKING_MARGIN + 'px 0' }),

            // the split tracking node covering the intersection of the other tracking points
            freeTrackingNode = $('<div>')
                .addClass('split-tracking horizontal vertical')
                .css({ width: TRACKING_SIZE + 'px', height: TRACKING_SIZE + 'px', padding: TRACKING_MARGIN + 'px' }),

            // all tracking nodes, as jQuery collection
            allTrackingNodes = horTrackingNode.add(vertTrackingNode).add(freeTrackingNode),

            // tracking nodes for column/row resizing, mapped by position
            resizeNodes = {
                left: $('<div>').addClass('resize-tracking left'),
                right: $('<div>').addClass('resize-tracking right'),
                top: $('<div>').addClass('resize-tracking top'),
                bottom: $('<div>').addClass('resize-tracking bottom')
            },

            // layout data of the active sheet
            sheetLayoutData = _.copy(DEFAULT_SHEET_LAYOUT_DATA, true),

            // column/row collections for all pane sides
            colRowCollections = {
                left:   new ColRowCollection('columns'),
                right:  new ColRowCollection('columns'),
                top:    new ColRowCollection('rows'),
                bottom: new ColRowCollection('rows')
            },

            // settings for all pane sides
            paneSideSettings = {
                left:   Utils.extendOptions(DEFAULT_PANE_SIDE_SETTINGS, { columns: true,  next: 'right',  panePos1: 'topLeft',     panePos2: 'bottomLeft' }),
                right:  Utils.extendOptions(DEFAULT_PANE_SIDE_SETTINGS, { columns: true,  next: 'left',   panePos1: 'topRight',    panePos2: 'bottomRight' }),
                top:    Utils.extendOptions(DEFAULT_PANE_SIDE_SETTINGS, { columns: false, next: 'bottom', panePos1: 'topLeft',     panePos2: 'topRight' }),
                bottom: Utils.extendOptions(DEFAULT_PANE_SIDE_SETTINGS, { columns: false, next: 'top',    panePos1: 'bottomLeft',  panePos2: 'bottomRight' })
            },

            // cell collections for all grid panes
            cellCollections = {
                topLeft: new CellCollection(),
                topRight: new CellCollection(),
                bottomLeft: new CellCollection(),
                bottomRight: new CellCollection()
            },

            // the contents and formatting of the selection and active cell
            selectionLayoutData = _.copy(DEFAULT_SELECTION_LAYOUT_DATA, true),

            // view settings of the active sheet
            sheetViewSettings = {
                split: false,
                freeze: false,
                splitLeft: 0,
                splitTop: 0,
                activePane: 'bottomRight'
            },

            // the size of the header corner node (and thus of the row/column header nodes)
            headerWidth = 0,
            headerHeight = 0;

        // base constructor ---------------------------------------------------

        EditView.call(this, app, {
            initHandler: initHandler,
            deferredInitHandler: deferredInitHandler,
            grabFocusHandler: grabFocusHandler,
            overlayMargin: { left: 8, right: Utils.SCROLLBAR_WIDTH + 8, top: 20, bottom: Utils.SCROLLBAR_HEIGHT }
        });

        // private methods ----------------------------------------------------

        /**
         * Returns the layout data of the specified column. Tries to receive
         * the layout data from the left and right column collection.
         *
         * @param {Number} col
         *  The zero-based column index.
         *
         * @returns {Object|Null}
         *  The collection entry with the column layout data, if available;
         *  otherwise null.
         */
        function getColEntry(col) {
            return colRowCollections.right.getEntryByIndex(col) ||
                colRowCollections.left.getEntryByIndex(col);
        }

        /**
         * Returns the layout data of the specified row. Tries to receive the
         * layout data from the top and bottom row collection.
         *
         * @param {Number} row
         *  The zero-based row index.
         *
         * @returns {Object|Null}
         *  The collection entry with the row layout data, if available;
         *  otherwise null.
         */
        function getRowEntry(row) {
            return colRowCollections.bottom.getEntryByIndex(row) ||
                colRowCollections.top.getEntryByIndex(row);
        }

        /**
         * Returns the layout data of the specified cell. Tries to receive the
         * layout data from all cell collections.
         *
         * @param {Number} col
         *  The zero-based column index.
         *
         * @param {Number} row
         *  The zero-based row index.
         *
         * @returns {Object|Null}
         *  The collection entry with the cell layout data, if available;
         *  otherwise null.
         */
        function getCellEntry(col, row) {
            return cellCollections.bottomRight.getEntry(col, row) ||
                cellCollections.bottomLeft.getEntry(col, row) ||
                cellCollections.topRight.getEntry(col, row) ||
                cellCollections.topLeft.getEntry(col, row);
        }

        /**
         * Requests an update of the selection part of the view layout data.
         */
        function requestSelectionUpdate(selection) {
            self.requestUpdate({ selection: { ranges: selection.ranges, active: selection.activeCell } });
        }

        /**
         * Processes a changed sheet selection. Updates the selection part of
         * the view layout data, and triggers a 'change:selection' event.
         */
        function triggerSelectionChanged(selection) {

            var // collection entry of the active column
                colEntry = getColEntry(selection.activeCell[0]),
                // collection entry of the active row
                rowEntry = getRowEntry(selection.activeCell[1]),
                // collection entry of the active cell
                cellEntry = getCellEntry(selection.activeCell[0], selection.activeCell[1]);

            if (colEntry && rowEntry && cellEntry) {
                // deep copy of the data, will be changed dynamically when modifying the cell
                selectionLayoutData.active = Utils.extendOptions(DEFAULT_CELL_LAYOUT_DATA, cellEntry);
                // add column attributes
                if (_.isObject(colEntry.attrs) && _.isObject(colEntry.attrs.column)) {
                    selectionLayoutData.active.attrs.column = Utils.extendOptions(selectionLayoutData.active.attrs.column, colEntry.attrs.column);
                }
                // add row attributes
                if (_.isObject(rowEntry.attrs) && _.isObject(rowEntry.attrs.row)) {
                    selectionLayoutData.active.attrs.row = Utils.extendOptions(selectionLayoutData.active.attrs.row, rowEntry.attrs.row);
                }
            } else {
                // initialization from known data failed, request new data from server
                requestSelectionUpdate(selection);
            }

            // notify all listeners
            self.trigger('change:selection', selection);
        }

        /**
         * Selects a sheet in this spreadsheet view instance.
         *
         * @param {Number} sheet
         *  The zero-based index of the new current sheet.
         *
         * @param {Boolean} [force]
         *  If set to true, the view will be updated even if the active sheet
         *  index has not changed.
         */
        function setActiveSheet(sheet, force) {

            // do nothing, if sheet index does not change
            if (!force && (sheetLayoutData.sheet === sheet)) { return; }

            // initialize the sheet layout data
            sheetLayoutData = Utils.extendOptions(DEFAULT_SHEET_LAYOUT_DATA, { sheet: sheet });

            // initialize the column/row/cell collections
            _(colRowCollections).invoke('clear');
            _(cellCollections).invoke('clear');

            // notify listeners
            self.trigger('change:activesheet', sheet).refreshGridPanes();

            // initialize the selection of the new sheet
            triggerSelectionChanged(self.getCellSelection());
        }

        /**
         * Handles a new sheet inserted into the document model.
         */
        function insertSheetHandler(event, sheet) {

            // listen to selection changes, notify selection listeners of the view
            model.getSheetModel(sheet).on('change:selection', function (event, sheetModel, selection) {
                if (sheetModel === model.getSheetModel(self.getActiveSheet())) {
                    triggerSelectionChanged(selection);
                }
            });

            // update active sheet index
            if (sheet <= self.getActiveSheet()) {
                setActiveSheet(self.getActiveSheet() + 1);
            }
        }

        /**
         * Handles a sheet that has been deleted from the document model.
         */
        function deleteSheetHandler(event, sheet) {

            // update active sheet index
            if (sheet < self.getActiveSheet()) {
                setActiveSheet(self.getActiveSheet() - 1);
            } else if (self.getActiveSheet() >= model.getSheetCount()) {
                setActiveSheet(model.getSheetCount() - 1);
            } else if (sheet === self.getActiveSheet()) {
                setActiveSheet(self.getActiveSheet(), true);
            }
        }

        /**
         * Handles 'update' push messages. If the message data contains a list
         * of changed cells for the active sheet, updates the visible cells in
         * all grid panes.
         *
         * @param {Object} data
         *  The message data of the 'update' push message.
         */
        function updateCellsHandler(data) {

            var // complete changed data
                changedData = Utils.getObjectOption(data, 'changed'),
                // array of changed cells per sheet
                sheets = null,
                // array of changed cells in active sheet
                cells = null;

            // update message may not contain any change data
            if (!_.isObject(changedData)) { return; }

            // process by update type (specific "cells", or "all")
            switch (Utils.getStringOption(changedData, 'type')) {
            case 'cells':
                if ((sheets = Utils.getArrayOption(changedData, 'sheets'))) {
                    _(sheets).any(function (sheetData) {
                        if (self.getActiveSheet() === sheetData.sheet) {
                            cells = sheetData.cells;
                            return true; // early exit of the _.any() method
                        }
                    });
                } else {
                    Utils.warn('SpreadsheetView.updateCellsHandler(): no sheets defined');
                }
                break;
            case 'all':
                break;
            default:
                Utils.error('SpreadsheetView.updateCellsHandler(): invalid type in cell update data');
                return;
            }

            // trigger forced update of all grid panes in the active sheet
            _(gridPanes).each(function (gridPane) {
                gridPane.requestUpdate(cells);
            });

            // update selection layout data
            requestSelectionUpdate(self.getCellSelection());
        }

        /**
         * Handles all tracking events for the split separators.
         */
        function splitTrackingHandler(event) {

            var // the event source node
                sourceNode = $(this),
                // whether horizontal and/or vertical tracking is active
                horizontal = sourceNode.hasClass('horizontal'),
                vertical = sourceNode.hasClass('vertical'),
                // minimum and maximum position of split lines
                minLeft = headerWidth, maxLeft = rootNode.width() - Utils.SCROLLBAR_WIDTH - SPLIT_SIZE,
                minTop = headerHeight, maxTop = rootNode.height() - Utils.SCROLLBAR_HEIGHT - SPLIT_SIZE;

            // returns the X position of a split line according to the current event
            function getSplitLineLeft() {
                return Utils.minMax(paneSideSettings.left.offset + paneSideSettings.left.size + (horizontal ? event.offsetX : 0), minLeft, maxLeft);
            }

            // returns the Y position of a split line according to the current event
            function getSplitLineTop() {
                return Utils.minMax(paneSideSettings.top.offset + paneSideSettings.top.size + (vertical ? event.offsetY : 0), minTop, maxTop);
            }

            switch (event.type) {

            case 'tracking:start':
                horTrackingNode.toggleClass('tracking-active', horizontal);
                vertTrackingNode.toggleClass('tracking-active', vertical);
                break;

            case 'tracking:move':
                if (horizontal) { horTrackingNode.add(freeTrackingNode).css({ left: getSplitLineLeft() - TRACKING_OFFSET }); }
                if (vertical) { vertTrackingNode.add(freeTrackingNode).css({ top: getSplitLineTop() - TRACKING_OFFSET }); }
                break;

            case 'tracking:end':
                var left = getSplitLineLeft(), top = getSplitLineTop();
                left = ((left <= minLeft) || (left >= maxLeft)) ? 0 : Utils.convertLengthToHmm(left - headerWidth, 'px');
                top = ((top <= minTop) || (top >= maxTop)) ? 0 : Utils.convertLengthToHmm(top - headerHeight, 'px');
                allTrackingNodes.removeClass('tracking-active');
                self.setSplitPosition(left, top);
                app.getController().update();
                break;

            case 'tracking:cancel':
                allTrackingNodes.removeClass('tracking-active');
                refreshGridPanes();
                break;
            }
        }

        /**
         * Shows the tracking overlay nodes while resizing a column or a row.
         *
         * @param {String} paneSide
         *  The identifier of the pane side with active resize tracking.
         *
         * @param {Number} offset
         *  The absolute start offset of the column or row, in pixels.
         *
         * @param {Number} size
         *  The current size of the column or row.
         */
        function showHeaderPaneResizer(paneSide, offset, size) {

            var // the passed offset, relative to the root node of the view
                relativeOffset = paneSideSettings[paneSide].offset + offset - headerPanes[paneSide].getFirstVisibleOffset(),
                // size of the leading overlay node
                leadingSize = Math.max(0, relativeOffset) + 10,
                // start offset of the trailing overlay node
                trailingOffset = relativeOffset + size;

            if (paneSideSettings[paneSide].columns) {
                resizeNodes.left.show().css({ left: -10, width: leadingSize });
                resizeNodes.right.show().css({ left: trailingOffset, right: -10 });
            } else {
                resizeNodes.top.show().css({ top: -10, height: leadingSize });
                resizeNodes.bottom.show().css({ top: trailingOffset, bottom: -10 });
            }
            resizeNodes.all.toggleClass('resize-hide', size === 0);
        }

        /**
         * Hides the tracking overlay nodes shown while resizing a column or a
         * row in a header pane.
         */
        function hideHeaderPaneResizer() {
            resizeNodes.all.hide();
        }

        /**
         * Initialization after construction.
         */
        function initHandler() {

            var // additional DOM nodes in debug pane
                debugNodes = {};

            // store reference to document model
            model = app.getModel();

            // insert the root node into the application pane
            self.insertContentNode(rootNode);

            // listen to sheet events
            model.on({ 'sheet:insert': insertSheetHandler, 'sheet:delete': deleteSheetHandler });

            // application notifies changed contents/results of cells
            app.on('docs:update', updateCellsHandler);

            // create the grid panes
            _(['topLeft', 'topRight', 'bottomLeft', 'bottomRight']).each(function (panePos) {
                gridPanes[panePos] = new GridPane(app, panePos);
                rootNode.append(gridPanes[panePos].getNode());
            });

            // create the header panes
            _(['left', 'right', 'top', 'bottom']).each(function (paneSide) {
                headerPanes[paneSide] = new HeaderPane(app, paneSide);
                rootNode.append(headerPanes[paneSide].getNode());
                // visualize resize tracking from header panes
                headerPanes[paneSide]
                    .on('resize:start', function (event, offset, size) {
                        showHeaderPaneResizer(paneSide, offset, size);
                        self.trigger('resize:start', paneSide, offset, size);
                    })
                    .on('resize:update', function (event, offset, size) {
                        showHeaderPaneResizer(paneSide, offset, size);
                        self.trigger('resize:update', paneSide, offset, size);
                    })
                    .on('resize:end', function () {
                        hideHeaderPaneResizer();
                        self.trigger('resize:end', paneSide);
                    });
            });

            // create the corner pane
            cornerPane = new CornerPane(app);
            rootNode.append(cornerPane.getNode());

            // append the split lines and split tracking nodes
            rootNode.append(horSplitLine, vertSplitLine, horTrackingNode, vertTrackingNode, freeTrackingNode);

            // append the resize tracking nodes
            resizeNodes.all = resizeNodes.left.add(resizeNodes.right).add(resizeNodes.top).add(resizeNodes.bottom);
            rootNode.append(resizeNodes.all.hide());

            // enable tracking, register event handlers
            allTrackingNodes
                .enableTracking()
                .on('tracking:start tracking:move tracking:end tracking:cancel', splitTrackingHandler);

            // update position of grid panes, when general view layout changes
            app.on('docs:import:after', function () {
                self.on('refresh:layout', function () { self.refreshGridPanes(); });
            });

            // disable browser context menu in entire application pane
            rootNode.on('contextmenu', function () {
                self.grabFocus();
                return false;
            });

            // log current selection in debug mode
            if (Config.isDebug()) {

                // create the DOM nodes in the debug pane
                self.addDebugInfoHeader('Sheet');
                debugNodes.all = self.addDebugInfoNode('all');
                debugNodes.used = self.addDebugInfoNode('used');

                self.addDebugInfoHeader('Selection');
                debugNodes.sheet = self.addDebugInfoNode('sheet');
                debugNodes.ranges = self.addDebugInfoNode('ranges');
                debugNodes.active = self.addDebugInfoNode('active');

                // log updated pane contents and layout
                self.on('update:grid', function (event, changedFlags) {

                    function dumpRange(node, size) {
                        var rangeText = ((size.cols > 0) && (size.rows > 0)) ? SheetUtils.getRangeName({ start: [0, 0], end: [size.cols - 1, size.rows - 1] }) : 'empty',
                            sizeText = size.width + '\xd7' + size.height;
                        node.text(_.noI18n('range=' + rangeText + ', size=' + sizeText));
                    }

                    if (changedFlags.sheet) {
                        dumpRange(debugNodes.all, sheetLayoutData);
                        dumpRange(debugNodes.used, sheetLayoutData.used);
                    }
                });

                // log information about the active sheet
                self.on('change:activesheet', function (event, sheet) {
                    debugNodes.sheet.text(_.noI18n('index=' + sheet + ', name="' + model.getSheetName(sheet) + '"'));
                });

                // log all selection events
                self.on('change:selection', function (event, selection) {
                    debugNodes.ranges.text(_.noI18n('count=' + selection.ranges.length + ', ranges=[' + _(selection.ranges).map(SheetUtils.getRangeName).join(',') + ']'));
                    debugNodes.active.text(_.noI18n('range=' + SheetUtils.getRangeName(selection.ranges[selection.activeRange]) + ', cell=' + SheetUtils.getCellName(selection.activeCell)));
                });
            }
        }

        /**
         * Initialization after importing the document. Creates all tool boxes
         * in the side pane and overlay pane. Needed to be executed after
         * import, to be able to hide specific GUI elements depending on the
         * file type.
         */
        function deferredInitHandler() {

            self.createToolBox('format', { label: gt('Format'), visible: 'document/editable' })
                .addGroup('character/fontname', new EditControls.FontFamilyChooser(app, { width: 117 }))
                .addGap(11)
                .addGroup('character/fontsize', new EditControls.FontHeightChooser({ width: 47 }))
                .newLine()
                .addGroup('character/bold',      new Button(EditControls.BOLD_OPTIONS))
                .addGroup('character/italic',    new Button(EditControls.ITALIC_OPTIONS))
                .addGroup('character/underline', new Button(EditControls.UNDERLINE_OPTIONS))
                .addGroup('character/strike',    new Button(EditControls.STRIKEOUT_OPTIONS))
                .newLine()
                .addGroup('character/color', new EditControls.ColorChooser(app, 'text', { icon: 'docs-font-color', tooltip: gt('Text color') }))
                .addGap()
                .addGroup('cell/fillcolor', new EditControls.ColorChooser(app, 'fill', { icon: 'docs-cell-fill-color', tooltip: gt('Fill color') }))
                .addGap()
                .addGroup('cell/borders', new EditControls.BorderChooser({ tooltip: gt('Cell borders') }))
                .addRightTab()
                .addGroup('cell/resetAttributes', new Button(EditControls.CLEAR_FORMAT_OPTIONS));

            self.createToolBox('borders', { label: gt('Borders'), visible: 'cell/border/enabled' })
                .addGroup('cell/borderstyle', new EditControls.BorderStyleChooser({ width: 96 }))
                .addGap()
                .addGroup('cell/borderwidth', new EditControls.CellBorderWidthChooser({ width: 35 }))
                .addGap()
                .addGroup('cell/bordercolor', new EditControls.ColorChooser(app, 'line', { icon: 'docs-cell-fill-color', tooltip: gt('Border color') }));

            self.createToolBox('alignment', { label: gt('Alignment'), visible: 'document/editable' })
                .addGroup('cell/alignhor', new RadioGroup({ toggleValue: 'auto' })
                    .createOptionButton('left',    { icon: 'docs-para-align-left',    tooltip: /*#. alignment of text in paragraphs or cells */ gt('Left') })
                    .createOptionButton('center',  { icon: 'docs-para-align-center',  tooltip: /*#. alignment of text in paragraphs or cells */ gt('Center') })
                    .createOptionButton('right',   { icon: 'docs-para-align-right',   tooltip: /*#. alignment of text in paragraphs or cells */ gt('Right') })
                    .createOptionButton('justify', { icon: 'docs-para-align-justify', tooltip: /*#. alignment of text in paragraphs or cells */ gt('Justify') })
                )
                .addRightTab()
                .addGroup('cell/alignvert', new RadioList({ icon: 'docs-cell-vertical-bottom', tooltip: /*#. text alignment in cells */ gt('Vertical alignment'), highlight: true, updateCaptionMode: 'icon' })
                    .createOptionButton('top',     { icon: 'docs-cell-vertical-top',    label: /*#. text alignment in cells */ gt('Top') })
                    .createOptionButton('middle',  { icon: 'docs-cell-vertical-middle', label: /*#. text alignment in cells */ gt('Middle') })
                    .createOptionButton('bottom',  { icon: 'docs-cell-vertical-bottom', label: /*#. text alignment in cells */ gt('Bottom') })
                );

            self.createToolBox('rowscols', { label: gt('Rows and Columns'), visible: 'document/editable' })
                .addGroup('row/insert', new Button({ icon: 'docs-table-insert-row', tooltip: gt('Insert row') }))
                .addGroup('row/delete', new Button({ icon: 'docs-table-delete-row', tooltip: gt('Delete selected rows') }))
                .addRightTab()
                .addPrivateGroup(new Label({ icon: 'icon-text-height', tooltip: gt('Row height') }))
                .addGap(0)
                .addGroup('row/height', new UnitField(Utils.extendOptions(DEFAULT_UNIT_FIELD_OPTIONS, { tooltip: gt('Row height'), max: SheetUtils.MAX_ROW_HEIGHT })))
                .newLine()
                .addGroup('column/insert', new Button({ icon: 'docs-table-insert-column', tooltip: gt('Insert column') }))
                .addGroup('column/delete', new Button({ icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') }))
                .addRightTab()
                .addPrivateGroup(new Label({ icon: 'icon-text-width', tooltip: gt('Column width') }))
                .addGap(0)
                .addGroup('column/width',  new UnitField(Utils.extendOptions(DEFAULT_UNIT_FIELD_OPTIONS, { tooltip: gt('Column width'), max: SheetUtils.MAX_COLUMN_WIDTH })));

            self.addPane(new StatusBar(app));

            self.getOverlayToolBox()
                .addGroup('character/bold',   new Button(EditControls.BOLD_OPTIONS))
                .addGroup('character/italic', new Button(EditControls.ITALIC_OPTIONS));

            if (Config.isDebug()) {
                self.createDebugToolBox()
                    .addGroup('sheet/name', new TextField({ tooltip: _.noI18n('Rename sheet'), width: 106 }))
                    .addRightTab()
                    .addGroup('sheet/delete', new Button({ label: _.noI18n('Delete sheet') }))
                    .newLine()
                    .addGroup('view/split', new Button({ label: _.noI18n('Split'), toggle: true }))
                    .addGroup('view/freeze', new Button({ label: _.noI18n('Freeze'), toggle: true }));
            }
        }

        /**
         * Moves the browser focus into the active sheet pane.
         */
        function grabFocusHandler() {
            var activePane = gridPanes[sheetViewSettings.activePane];
            if (activePane) {
                activePane.grabFocus();
            } else {
                Utils.error('SpreadsheetView.grabFocusHandler(): no active pane');
                self.getVisibleGridPane('bottomRight').grabFocus();
            }
        }

        /**
         * Refreshes the layout of the grid panes and tracking nodes, and the
         * contents of the grid areas currently visible, according to the view
         * settings of the active sheet.
         */
        function refreshGridPanes() {

            var // whether freeze mode is active (has higher priority than split mode)
                freezeActive = self.getFreezeMode(),
                // whether split mode is active, but not freeze mode
                splitActive = !freezeActive && self.getSplitMode(),
                // the size of the split lines
                splitSize = freezeActive ? FREEZE_SIZE : SPLIT_SIZE,
                // whether the split lines are visible
                horSplit = false, vertSplit = false,
                // start position of the split lines
                splitLineLeft = 0, splitLineTop = 0;

            // whether the left and top panes are visible (must have enough room in split mode, always in freeze mode)
            paneSideSettings.left.visible = (splitActive && (sheetViewSettings.splitLeft > MIN_PANE_SIZE)) || (freezeActive && (sheetViewSettings.splitLeft > 0));
            paneSideSettings.top.visible = (splitActive && (sheetViewSettings.splitTop > MIN_PANE_SIZE)) || (freezeActive && (sheetViewSettings.splitTop > 0));

            // calculate current size of header nodes
            // TODO: use current maximum row
            cornerPane.initialize(0xFFFFF);
            headerWidth = cornerPane.getNode().width();
            headerHeight = cornerPane.getNode().height();

            // calculate inner width and height of left and top panes, and position of split lines
            // TODO: convert freeze panes from rows/columns to pixels
            if (paneSideSettings.left.visible) {
                paneSideSettings.left.offset = headerWidth;
                paneSideSettings.left.size = Utils.convertHmmToLength(sheetViewSettings.splitLeft, 'px', 1);
                paneSideSettings.left.hiddenSize = 0;
                splitLineLeft = paneSideSettings.left.offset + paneSideSettings.left.size;
            }
            if (paneSideSettings.top.visible) {
                paneSideSettings.top.offset = headerHeight;
                paneSideSettings.top.size = Utils.convertHmmToLength(sheetViewSettings.splitTop, 'px', 1);
                paneSideSettings.top.hiddenSize = 0;
                splitLineTop = paneSideSettings.top.offset + paneSideSettings.top.size;
            }

            // determine whether right and bottom panes are visible (must have enough room in split and freeze mode)
            paneSideSettings.right.visible = !paneSideSettings.left.visible || (splitLineLeft + splitSize + MIN_PANE_SIZE + Utils.SCROLLBAR_WIDTH <= rootNode.width());
            paneSideSettings.bottom.visible = !paneSideSettings.top.visible || (splitLineTop + splitSize + MIN_PANE_SIZE + Utils.SCROLLBAR_HEIGHT <= rootNode.height());

            // visibility of the split lines
            horSplit = paneSideSettings.left.visible && paneSideSettings.right.visible;
            vertSplit = paneSideSettings.top.visible && paneSideSettings.bottom.visible;

            // calculate the resulting grid pane positions and sizes
            if (paneSideSettings.right.visible) {
                paneSideSettings.right.offset = horSplit ? (splitLineLeft + splitSize) : headerWidth;
                paneSideSettings.right.size = rootNode.width() - paneSideSettings.right.offset;
                paneSideSettings.right.hiddenSize = freezeActive ? paneSideSettings.left.size : 0;
            } else if (paneSideSettings.left.visible) {
                paneSideSettings.left.size = rootNode.width() - headerWidth;
            }
            if (paneSideSettings.bottom.visible) {
                paneSideSettings.bottom.offset = vertSplit ? (splitLineTop + splitSize) : headerHeight;
                paneSideSettings.bottom.size = rootNode.height() - paneSideSettings.bottom.offset;
                paneSideSettings.bottom.hiddenSize = freezeActive ? paneSideSettings.top.size : 0;
            } else if (paneSideSettings.top.visible) {
                paneSideSettings.top.size = rootNode.height() - headerHeight;
            }

            // set scroll mode (left/top panes are not scrollable in frozen mode in their
            // own direction, e.g. left frozen panes are not scrollable to left/right)
            paneSideSettings.left.scrollable = splitActive;
            paneSideSettings.right.scrollable = true;
            paneSideSettings.top.scrollable = splitActive;
            paneSideSettings.bottom.scrollable = true;

            // set up scroll bar visibility in the opposite direction (e.g. left/right
            // scroll bars of top panes are hidden, if bottom panes are visible)
            paneSideSettings.left.showOppositeScroll = !paneSideSettings.right.visible;
            paneSideSettings.right.showOppositeScroll = true;
            paneSideSettings.top.showOppositeScroll = !paneSideSettings.bottom.visible;
            paneSideSettings.bottom.showOppositeScroll = true;

            // initialize the header panes
            _(headerPanes).each(function (headerPane, paneSide) {
                headerPane.initialize(paneSideSettings[paneSide]);
            });

            // initialize the grid panes
            _(gridPanes).each(function (gridPane, panePos) {
                gridPane.initialize(paneSideSettings[getHorizontalPaneSide(panePos)], paneSideSettings[getVerticalPaneSide(panePos)]);
            });

            // visibility and position of the split lines
            horSplitLine.toggle(horSplit).css({ left: splitLineLeft, width: splitSize });
            vertSplitLine.toggle(vertSplit).css({ top: splitLineTop, height: splitSize });

            // visibility and position of the split tracking nodes
            horTrackingNode.toggle(splitActive && horSplit).css({ left: splitLineLeft - TRACKING_OFFSET });
            vertTrackingNode.toggle(splitActive && vertSplit).css({ top: splitLineTop - TRACKING_OFFSET });
            freeTrackingNode.toggle(splitActive && horSplit && vertSplit).css({ left: splitLineLeft - TRACKING_OFFSET, top: splitLineTop - TRACKING_OFFSET });
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the specified grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getGridPane = function (panePos) {
            return gridPanes[panePos];
        };

        /**
         * Returns the specified grid pane, if it is visible. Otherwise,
         * returns the nearest visible grid pane.
         *
         * @param {String} panePos
         *  The position identifier of the preferred grid pane. If only one
         *  grid pane is visible, it will be returned regardless of this
         *  parameter. If one of the pane sides contained in the passed pane
         *  position is not visible, returns the grid pane from the other
         *  visible pane side (for example, if requesting the top-left pane
         *  while only bottom panes are visible, returns the bottom-left pane).
         *
         * @returns {GridPane}
         *  The grid pane instance at the specified pane position.
         */
        this.getVisibleGridPane = function (panePos) {

            // jump to other horizontal pane, if the pane side is hidden
            if (!paneSideSettings[getHorizontalPaneSide(panePos)].visible) {
                panePos = getNextHorizontalPanePos(panePos);
            }

            // jump to other vertical pane, if the pane side is hidden
            if (!paneSideSettings[getVerticalPaneSide(panePos)].visible) {
                panePos = getNextVerticalPanePos(panePos);
            }

            // now, panePos points to a visible grid pane (at least one pane is always visible)
            return gridPanes[panePos];
        };

        /**
         * Returns the specified header pane.
         *
         * @param {String} paneSide
         *  The identifier of the pane side ('left', 'right', 'top', 'bottom').
         *
         * @returns {HeaderPane}
         *  The header pane instance at the specified pane side.
         */
        this.getHeaderPane = function (paneSide) {
            return headerPanes[paneSide];
        };

        /**
         * Returns the horizontal header pane (left or right) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The horizontal header pane instance at the corresponding pane side.
         */
        this.getHorizontalHeaderPane = function (panePos) {
            return headerPanes[getHorizontalPaneSide(panePos)];
        };

        /**
         * Returns the vertical header pane (top or bottom) for the specified
         * pane position.
         *
         * @param {String} panePos
         *  The position identifier of the grid pane.
         *
         * @returns {HeaderPane}
         *  The vertical header pane instance at the corresponding pane side.
         */
        this.getVerticalHeaderPane = function (panePos) {
            return headerPanes[getVerticalPaneSide(panePos)];
        };

        /**
         * Returns the first visible header pane in the specified direction
         * (for example, for columns returns the left header pane, if it is
         * visible, otherwise the right header pane).
         *
         * @param {Boolean} columns
         *  If set to true, returns the first visible horizontal header pane
         *  (left or right); otherwise returns the first visible vertical
         *  header pane (top or bottom).
         *
         * @returns {HeaderPane}
         *  The first visible header pane instance in the specified direction.
         */
        this.getVisibleHeaderPane = function (columns) {
            return columns ?
                (paneSideSettings.left.visible ? headerPanes.left : headerPanes.right) :
                (paneSideSettings.top.visible ? headerPanes.top : headerPanes.bottom);
        };

        /**
         * Returns the top-left corner pane.
         *
         * @returns {CornerPane}
         *  The corner pane instance.
         */
        this.getCornerPane = function () {
            return cornerPane;
        };

        /**
         * Cancels node tracking that may currently be active (split separator
         * nodes), and refreshes the layout of the grid panes.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.refreshGridPanes = function () {
            $.cancelTracking();
            refreshGridPanes();
            return this;
        };

        /**
         * Collects several update requests sends a single deferred server
         * request.
         *
         * @param {Object} requestProps
         *  The properties to be inserted into the current request data object.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.requestUpdate = (function () {

            var // the data object collecting all grid pane requests
                requestData = null,
                // the last running request
                runningDef = null,
                // the start time of the initial request (without other requests already running)
                startTime = 0,
                // all parts of the layout data that have been changed
                changedFlags = {},
                // counter for running requests
                requestIndex = 0;

            // notifies all listeners
            function triggerUpdateEvent() {
                self.trigger('update:grid', changedFlags);
                changedFlags = {};
            }

            // direct callback: register a new sheet rectangle to be updated in a grid pane
            function registerUpdateRequest(requestProps) {
                if (self.getActiveSheet() >= 0) {
                    if (!requestData || (requestData.sheet !== self.getActiveSheet())) {
                        requestData = { sheet: self.getActiveSheet() };
                    }
                    _(requestData).extend(requestProps);
                }
                return self;
            }

            // deferred callback: send single server request and notify all waiting grid panes
            function executeUpdateRequest() {

                var // the new server request
                    def = null;

                if (!requestData) { return; }

                // take start time, if no other request is running already
                if (!runningDef) {
                    startTime = _.now();
                }

                // send request and immediately delete the request data (new updates may be requested in the meantime)
                requestIndex += 1;
                Utils.info('SpreadsheetView.executeUpdateRequest(): sending update request #' + requestIndex + '...');
                def = runningDef = app.requestViewUpdate(requestData);
                def._DBG_INDEX = requestIndex;
                def._DBG_TIME = _.now();
                requestData = null;

                def.fail(function () {
                    Utils.error('SpreadsheetView.executeUpdateRequest(): update request #' + def._DBG_INDEX + ' failed');
                })
                .done(function (layoutData) {

                    Utils.info('SpreadsheetView.executeUpdateRequest(): update response #' + def._DBG_INDEX + ' received (' + (_.now() - def._DBG_TIME) + 'ms)');

                    // check validity of response object
                    if (!_.isObject(layoutData) || !_.isObject(layoutData.sheet) || !_.isObject(layoutData.sheet.used)) {
                        Utils.error('SpreadsheetView.executeUpdateRequest(): missing required layout data');
                        return;
                    }

                    // ignore this response if the active sheet has changed in the meantime
                    if (layoutData.sheet.sheet !== self.getActiveSheet()) { return; }

                    // copy sheet layout data to internal class member
                    sheetLayoutData = _.copy(layoutData.sheet, true);
                    changedFlags.sheet = true;

                    // convert sheet size and used area size to pixels
                    sheetLayoutData.width = Utils.convertHmmToLength(sheetLayoutData.width, 'px', 1);
                    sheetLayoutData.height = Utils.convertHmmToLength(sheetLayoutData.height, 'px', 1);
                    sheetLayoutData.used.width = Utils.convertHmmToLength(sheetLayoutData.used.width, 'px', 1);
                    sheetLayoutData.used.height = Utils.convertHmmToLength(sheetLayoutData.used.height, 'px', 1);

                    // initialize the column/row collections
                    _(colRowCollections).each(function (collection, paneSide) {
                        var paneSideData = Utils.getObjectOption(layoutData, paneSide);
                        if (paneSideData) {
                            collection.initialize(paneSideData);
                            changedFlags[paneSide] = true;
                        }
                    });

                    // enlarge total size of sheet according to column/row information, or
                    // shrink total size of sheet if last column/row information is available
                    // (sheet size may differ from position/size of last columns/rows due
                    // to rounding errors while converting from 1/100 mm to pixels)
                    _(['left', 'right']).each(function (paneSide) {
                        if (paneSideSettings[paneSide].visible) {
                            var collection = colRowCollections[paneSide],
                                endOffset = collection.getOffset() + collection.getSize();
                            sheetLayoutData.width = Math.max(sheetLayoutData.width, endOffset);
                            if (collection.containsIndex(sheetLayoutData.cols - 1)) {
                                sheetLayoutData.width = Math.min(sheetLayoutData.width, endOffset);
                            }
                        }
                    });
                    _(['top', 'bottom']).each(function (paneSide) {
                        if (paneSideSettings[paneSide].visible) {
                            var collection = colRowCollections[paneSide],
                                endOffset = collection.getOffset() + collection.getSize();
                            sheetLayoutData.height = Math.max(sheetLayoutData.height, endOffset);
                            if (collection.containsIndex(sheetLayoutData.rows - 1)) {
                                sheetLayoutData.height = Math.min(sheetLayoutData.height, endOffset);
                            }
                        }
                    });

                    // initialize the cell collections for all grid panes
                    _(cellCollections).each(function (collection, panePos) {
                        var gridPaneData = Utils.getObjectOption(layoutData, panePos);
                        if (gridPaneData) {
                            collection.initialize(gridPaneData);
                            changedFlags[panePos] = true;
                        }
                    });

                    // save the display string of the active selection
                    if (_.isObject(layoutData.selection)) {
                        selectionLayoutData = Utils.extendOptions(DEFAULT_SELECTION_LAYOUT_DATA, layoutData.selection);
                        changedFlags.selection = true;
                    }

                    // Check if this is the last request. Do not notify listeners about response
                    // data of previous requests, but collect the response data and all change
                    // flags of multiple requests, and notify listeners after last response.
                    if (def === runningDef) {
                        runningDef = null;
                        triggerUpdateEvent();
                    } else if (_.now() - startTime >= 3000) {
                        startTime = _.now();
                        triggerUpdateEvent();
                    }
                });
            }

            // create and return the debounced method SpreadsheetView.requestUpdate()
            return app.createDebouncedMethod(registerUpdateRequest, executeUpdateRequest, { delay: 50, maxDelay: 250 });

        }()); // SpreadsheetView.requestUpdate()

        /**
         * Collects update requests of several grid panes and sends a single
         * deferred server request.
         *
         * @internal
         *  Called from the implementation of the GridPane class.
         *
         * @param {String} panePos
         *  The position of the grid pane that wants to update its layout.
         *
         * @param {Object} updateRectangle
         *  The rectangle in the sheet to be updated, in 1/100 mm.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.requestGridPaneUpdate = function (panePos, updateRectangle) {
            var requestProps = {};
            Utils.log('SpreadsheetView.requestGridPaneUpdate(): sheet=' + this.getActiveSheet() + ' panePos=' + panePos + ' left=' + updateRectangle.left + ' top=' + updateRectangle.top + ' width=' + updateRectangle.width + ' height=' + updateRectangle.height);
            requestProps[getHorizontalPaneSide(panePos)] = { left: updateRectangle.left, width: updateRectangle.width };
            requestProps[getVerticalPaneSide(panePos)] = { top: updateRectangle.top, height: updateRectangle.height };
            return this.requestUpdate(requestProps);
        };

        /**
         * Returns the layout data of the active sheet.
         *
         * @returns {Object}
         *  The layout data of the active sheet, containing the grid size and
         *  pixel size of the sheet, and the used area in the sheet.
         */
        this.getSheetLayoutData = function () {
            return sheetLayoutData;
        };

        /**
         * Returns the column/row collection for the specified pane side.
         *
         * @param {String} paneSide
         *  The pane side identifier ('left', 'right', 'top', or 'bottom').
         *
         * @returns {ColRowCollection}
         *  The column/row collection for the specified pane side.
         */
        this.getColRowCollection = function (paneSide) {
            return colRowCollections[paneSide];
        };

        /**
         * Returns the column collection for the specified grid pane.
         *
         * @param {String} panePos
         *  The position of the grid pane.
         *
         * @returns {ColRowCollection}
         *  The column collection for the specified pane side.
         */
        this.getColumnCollection = function (panePos) {
            return colRowCollections[getHorizontalPaneSide(panePos)];
        };

        /**
         * Returns the row collection for the specified grid pane.
         *
         * @param {String} panePos
         *  The position of the grid pane.
         *
         * @returns {ColRowCollection}
         *  The row collection for the specified pane side.
         */
        this.getRowCollection = function (panePos) {
            return colRowCollections[getVerticalPaneSide(panePos)];
        };

        /**
         * Returns the cell collection for the specified grid pane.
         *
         * @param {String} panePos
         *  The position of the grid pane.
         *
         * @returns {CellCollection}
         *  The cell collection for the specified pane side.
         */
        this.getCellCollection = function (panePos) {
            return cellCollections[panePos];
        };

        /**
         * Sets the scroll position of the grid panes and header panes next to
         * the specified grid pane.
         *
         * @internal
         *  Called from the implementation of the GridPane class.
         *
         * @param {String} panePos
         *  The position of the grid pane with the modified scroll position.
         *
         * @param {Number} scrollLeft
         *  The new horizontal scroll position of the specified grid pane.
         *
         * @param {Number} scrollTop
         *  The new vertical scroll position of the specified grid pane.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.syncGridPaneScroll = function (panePos, scrollLeft, scrollTop) {

            // update scroll position of other grid panes
            gridPanes[getNextVerticalPanePos(panePos)].scrollLeft(scrollLeft);
            gridPanes[getNextHorizontalPanePos(panePos)].scrollTop(scrollTop);

            // update scroll position of header panes
            headerPanes[getHorizontalPaneSide(panePos)].scroll(scrollLeft);
            headerPanes[getVerticalPaneSide(panePos)].scroll(scrollTop);

            return this;
        };

        // selection and scrolling --------------------------------------------

        /**
         * Returns the zero-based index of the current sheet in this
         * spreadsheet view instance.
         *
         * @returns {Number}
         *  The zero-based index of the current sheet.
         */
        this.getActiveSheet = function () {
            return sheetLayoutData.sheet;
        };

        /**
         * Selects a sheet in this spreadsheet view instance.
         *
         * @param {Number} sheet
         *  The zero-based index of the new current sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setActiveSheet = function (sheet) {
            setActiveSheet(sheet);
            return this;
        };

        /**
         * Returns the logical range address of the entire area in the active
         * sheet from the top-left cell to the bottom-right cell.
         *
         * @returns {Object}
         *  The logical range address of the entire sheet area.
         */
        this.getSheetRange = function () {
            return { start: [0, 0], end: [sheetLayoutData.cols - 1, sheetLayoutData.rows - 1] };
        };

        /**
         * Returns the logical range address of the used area in the active
         * sheet.
         *
         * @returns {Object}
         *  The logical range address of the used area.
         */
        this.getUsedRange = function () {
            return { start: [0, 0], end: [sheetLayoutData.used.cols - 1, sheetLayoutData.used.rows - 1] };
        };

        /**
         * Returns an object containing all information about the cell
         * selection in the active sheet.
         *
         * @returns {Object}
         *  The cell selection data, in the properties 'ranges' (array of range
         *  addresses), 'activeRange' (array index of the active range in the
         *  'ranges' property), and 'activeCell' (logical cell address).
         */
        this.getCellSelection = function () {
            return model.getSheetModel(this.getActiveSheet()).getCellSelection();
        };

        /**
         * Returns all selected cell ranges in the active sheet.
         *
         * @returns {Array}
         *  The selected cell ranges, as array of logical range positions.
         */
        this.getSelectedRanges = function () {
            return this.getCellSelection().ranges;
        };

        /**
         * Returns the active cell of the selection in the active sheet (the
         * highlighted cell in the active range that will be edited when
         * switching to text edit mode).
         *
         * @returns {Number[]}
         *  The logical position of the active cell.
         */
        this.getActiveCell = function () {
            return this.getCellSelection().activeCell;
        };

        /**
         * Changes the cell selection in the active sheet.
         *
         * @param {Object} selection
         *  The cell selection data, in the properties 'ranges' (array of range
         *  addresses), 'activeRange' (array index of the active range in the
         *  'ranges' property), and 'activeCell' (logical cell address).
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCellSelection = function (selection) {
            model.getSheetModel(this.getActiveSheet()).setCellSelection(selection);
            return this;
        };

        /**
         * Selects a single cell in the active sheet. The cell will become the
         * active cell of the sheet.
         *
         * @param {Number[]} address
         *  The logical address of the cell to be selected.
         *
         * @param {Object} options
         *  A map of options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.append=false]
         *      If set to true, the new cell will be appended to the current
         *      selection. Otherwise, the old selection will be removed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.selectCell = function (address, options) {
            return this.selectRange({ start: address, end: address }, options);
        };

        /**
         * Selects a single cell range in the active sheet.
         *
         * @param {Object} range
         *  The logical address of the cell range to be selected.
         *
         * @param {Object} options
         *  A map of options to control the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.append=false]
         *      If set to true, the new range will be appended to the current
         *      selection. Otherwise, the old selection will be removed.
         *  @param {Number[]} [options.active]
         *      The logical address of the active cell in the selected range.
         *      If omitted, the top-left cell of the range will be activated.
         *      Must be located inside the passed range address.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.selectRange = function (range, options) {

            var // adjust the passed range address
                adjustedRange = SheetUtils.getAdjustedRange(range),
                // whether to append the range to the current selection
                append = Utils.getBooleanOption(options, 'append', false),
                // the initial selection to append the new range to
                selection = append ? this.getCellSelection() : { ranges: [] },
                // the new active cell
                activeCell = Utils.getArrayOption(options, 'active', []);

            // initialize and set the new selection
            selection.ranges.push(adjustedRange);
            selection.activeRange = selection.ranges.length - 1;
            // set active cell
            if ((activeCell.length === 2) && SheetUtils.rangeContainsCell(adjustedRange, activeCell)) {
                selection.activeCell = activeCell;
            } else {
                selection.activeCell = adjustedRange.start;
            }
            // commit new selection
            this.setCellSelection(selection);
            return this;
        };

        /**
         * Modifies the position of the current active selection range.
         *
         * @param {Object} range
         *  The new logical address of the active selection range. If the
         *  passed range does not contain the current active cell, the top-left
         *  cell of the range will be activated.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.changeActiveRange = function (range) {

            var // adjust the passed range address
                adjustedRange = SheetUtils.getAdjustedRange(range),
                // the initial selection to append the new range to
                selection = this.getCellSelection();

            // initialize and set the new active range
            selection.ranges[selection.activeRange] = adjustedRange;
            if (!SheetUtils.rangeContainsCell(adjustedRange, selection.activeCell)) {
                selection.activeCell = adjustedRange.start;
            }
            this.setCellSelection(selection);
            return this;
        };

        /**
         * Sets either the horizontal or vertical scroll position of the grid
         * panes and header pane associated with the specified pane side.
         *
         * @param {String} paneSide
         *  The pane side identifier ('left', 'right', 'top', or 'bottom').
         *
         * @param {Number} scrollPos
         *  The new scroll position of the specified grid and header panes.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.scrollPaneSide = function (paneSide, scrollPos) {

            var // the method name used to scroll the grid pane
                scrollMethod = paneSideSettings[paneSide].columns ? 'scrollLeft' : 'scrollTop';

            // update scroll position of the grid panes and header pane
            gridPanes[paneSideSettings[paneSide].panePos1][scrollMethod](scrollPos);
            gridPanes[paneSideSettings[paneSide].panePos2][scrollMethod](scrollPos);
            headerPanes[paneSide].scroll(scrollPos);

            return this;
        };

        /**
         * Scrolls the grid panes and header pane associated with the specified
         * pane side to make a column or row visible.
         *
         * @param {String} paneSide
         *  The pane side identifier ('left', 'right', 'top', or 'bottom').
         *
         * @param {Number} index
         *  The zero-based column or row index to be made visible.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.scrollPaneSideToEntry = function (paneSide, index) {

            var // the method name used to scroll the grid pane
                scrollMethod = paneSideSettings[paneSide].columns ? 'scrollToColumn' : 'scrollToRow';

            // update scroll position of the grid panes
            gridPanes[paneSideSettings[paneSide].panePos1][scrollMethod](index);
            gridPanes[paneSideSettings[paneSide].panePos2][scrollMethod](index);

            return this;
        };

        /**
         * Returns the value of the specified subtotal in the current
         * selection.
         *
         * @param {String} type
         *  The type identifier of the subtotal value.
         *
         * @returns {Number}
         *  The current value of the specified subtotal.
         */
        this.getSubtotalValue = function (type) {
            return selectionLayoutData.subtotals[type] || 0;
        };

        // split and freeze ---------------------------------------------------

        this.getSplitMode = function () {
            return sheetViewSettings.split;
        };

        this.setSplitMode = function (state) {
            if (self.getSplitMode() !== state) {
                sheetViewSettings.split = state;
                if (state && (sheetViewSettings.splitLeft === 0) && (sheetViewSettings.splitTop === 0)) {
                    sheetViewSettings.splitLeft = 7000;
                    sheetViewSettings.splitTop = 5000;
                }
                this.refreshGridPanes();
            }
            return this;
        };

        this.setSplitPosition = function (left, top) {
            left = Math.max(0, left);
            top = Math.max(0, top);
            // TODO: calculate split position according to cursor position
            if ((sheetViewSettings.splitLeft !== left) || (sheetViewSettings.splitTop !== top)) {
                sheetViewSettings.splitLeft = left;
                sheetViewSettings.splitTop = top;
                sheetViewSettings.split = sheetViewSettings.split && ((left > 0) || (top > 0));
                this.refreshGridPanes();
            }
            return this;
        };

        this.getFreezeMode = function () {
            return sheetViewSettings.freeze;
        };

        this.setFreezeMode = function (state) {
            if (self.getFreezeMode() !== state) {
                sheetViewSettings.freeze = state;
                // TODO: calculate split position according to cursor position
                if (state && (sheetViewSettings.splitLeft === 0) && (sheetViewSettings.splitTop === 0)) {
                    sheetViewSettings.splitLeft = 7000;
                    sheetViewSettings.splitTop = 5000;
                }
                this.refreshGridPanes();
            }
            return this;
        };

        // sheet operations ---------------------------------------------------

        /**
         * Inserts a new sheet in the spreadsheet document, and activates it in
         * the view.
         *
         * @param {String} [sheetName]
         *  The new name of the sheet. If missing, or if this name exists
         *  already, a new unused name will be generated.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.insertSheet = function (sheetName) {

            var insertIndex = this.getActiveSheet() + 1,
                nameIndex = model.getSheetCount();

            // validate sheet name, or generate a valid name
            sheetName = _.isString(sheetName) ? Utils.cleanString(sheetName) : '';
            while ((sheetName.length === 0) || model.hasSheet(sheetName)) {
                sheetName = generateSheetName(nameIndex);
                nameIndex += 1;
            }

            // insert the new sheet into the document, activate the sheet
            if (model.insertSheet(insertIndex, sheetName)) {
                setActiveSheet(insertIndex);
            }

            return this;
        };

        /**
         * Deletes the active sheet from the spreadsheet document, and
         * activates another sheet in the view. Does nothing, if the document
         * contains a single sheet only.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteSheet = function () {
            if (model.getSheetCount() > 1) {
                // event handler of the 'sheet:delete' will update active sheet
                model.deleteSheet(this.getActiveSheet());
            }
            return this;
        };

        /**
         * Renames the active sheet in the spreadsheet document, if possible.
         *
         * @param {String} sheetName
         *  The new name of the active sheet.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.renameSheet = function (sheetName) {
            model.setSheetName(this.getActiveSheet(), sheetName);
            return this;
        };

        // column operations --------------------------------------------------

        /**
         * Inserts new columns into the active sheet, according to the current
         * selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.insertColumns = function () {
            model.enterUndoGroup(function () {
                var intervals = new ColRowIntervals(this.getSelectedRanges(), 'columns');
                intervals.iterateIntervals(function (interval) {
                    model.insertCells(self.getActiveSheet(), [interval.first, 0], null, 'column', true, interval.count);
                }, { reverse: true });
            }, this);
            return this;
        };

        /**
         * Deletes existing columns from the active sheet, according to the
         * current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteColumns = function () {
            model.enterUndoGroup(function () {
                var intervals = new ColRowIntervals(this.getSelectedRanges(), 'columns');
                intervals.iterateIntervals(function (interval) {
                    model.deleteCells(self.getActiveSheet(), [interval.first, 0], null, 'column', true, interval.count);
                }, { reverse: true });
            }, this);
            return this;
        };

        /**
         * Changes the width of columns in the active sheet, according to the
         * current selection.
         *
         * @param {Number} width
         *  The column width in 1/100 mm. If this value is less than 1, the
         *  columns will be hidden, and their original width will not be
         *  changed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setColumnWidthToSelection = function (width) {
            model.enterUndoGroup(function () {
                var intervals = new ColRowIntervals(this.getSelectedRanges(), 'columns');
                intervals.iterateIntervals(function (interval) {
                    self.setColumnWidth(interval.first, interval.last, width);
                });
            }, this);
            return this;
        };

        /**
         * Changes the width of the columns in the specified column interval in
         * the active sheet.
         *
         * @param {Number} firstCol
         *  The zero-based index of the first column in the interval.
         *
         * @param {Number} lastCol
         *  The zero-based index of the last column in the interval.
         *
         * @param {Number} width
         *  The column width in 1/100 mm. If this value is less than 1, the
         *  columns will be hidden, and their original width will not be
         *  changed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setColumnWidth = function (firstCol, lastCol, width) {
            var colAttrs = (width < 1) ? { visible: false } : { width: width, visible: true },
                activeCellData = selectionLayoutData.active,
                activeCol = self.getActiveCell()[0];

            lastCol = lastCol || firstCol;

            // if active cell is between first and last column, merge new column attributes
            if ((firstCol <= activeCol) && (activeCol <= lastCol)) {
                activeCellData.attrs.column = Utils.extendOptions(activeCellData.attrs.column, colAttrs);
            }

            model.setColumnAttributes(self.getActiveSheet(), firstCol, lastCol, { column: colAttrs });
            return this;
        };

        // row operations -----------------------------------------------------

        /**
         * Inserts new rows into the active sheet, according to the current
         * selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.insertRows = function () {
            model.enterUndoGroup(function () {
                var intervals = new ColRowIntervals(this.getSelectedRanges(), 'rows');
                intervals.iterateIntervals(function (interval) {
                    model.insertCells(self.getActiveSheet(), [0, interval.first], null, 'row', true, interval.count);
                }, { reverse: true });
            }, this);
            return this;
        };

        /**
         * Deletes existing rows from the active sheet, according to the
         * current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.deleteRows = function () {
            model.enterUndoGroup(function () {
                var intervals = new ColRowIntervals(this.getSelectedRanges(), 'rows');
                intervals.iterateIntervals(function (interval) {
                    model.deleteCells(self.getActiveSheet(), [0, interval.first], null, 'row', true, interval.count);
                }, { reverse: true });
            }, this);
            return this;
        };

        /**
         * Changes the height of rows in the active sheet, according to the
         * current selection.
         *
         * @param {Number} height
         *  The row height in 1/100 mm. If this value is less than 1, the rows
         *  will be hidden, and their original height will not be changed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setRowHeightToSelection = function (height) {
            model.enterUndoGroup(function () {
                var intervals = new ColRowIntervals(this.getSelectedRanges(), 'rows');
                intervals.iterateIntervals(function (interval) {
                    self.setRowHeight(interval.first, interval.last, height);
                });
            }, this);
            return this;
        };

        /**
         * Changes the height of the rows in the specified row interval in the
         * active sheet.
         *
         * @param {Number} firstRow
         *  The zero-based index of the first row in the interval.
         *
         * @param {Number} lastRow
         *  The zero-based index of the last row in the interval.
         *
         * @param {Number} height
         *  The row height in 1/100 mm. If this value is less than 1, the rows
         *  will be hidden, and their original height will not be changed.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setRowHeight = function (firstRow, lastRow, height) {
            var rowAttrs = (height < 1) ? { visible: false } : { height: height, visible: true },
                activeCellData = selectionLayoutData.active,
                activeRow = self.getActiveCell()[1];

            lastRow = lastRow || firstRow;

            // if active cell is between first and last row, merge new row attributes
            if ((firstRow <= activeRow) && (activeRow <= lastRow)) {
                activeCellData.attrs.row = Utils.extendOptions(activeCellData.attrs.row, rowAttrs);
            }

            model.setRowAttributes(self.getActiveSheet(), firstRow, lastRow, { row: rowAttrs });
            return this;
        };

        // cell operations ----------------------------------------------------

        /**
         * Returns the contents of the current active cell.
         *
         * @returns {Object}
         *  The cell contents, in the properties 'display' and 'result', and
         *  the optional property 'formula' if the current cell is a formula
         *  cell.
         */
        this.getActiveCellContents = function () {

            var activeCellData = selectionLayoutData.active,
                cellContents = { display: activeCellData.display, result: activeCellData.result };

            if (_.isString(activeCellData.formula)) {
                cellContents.formula = activeCellData.formula;
            }
            return cellContents;
        };

        /**
         * Fills one or more cell ranges in the document with the same value,
         * and updates the view according to the cell value.
         *
         * @param {Number} sheet
         *  The zero-based sheet index the cell is located in.
         *
         * @param {Object} selection
         *  The selection to be filled.
         *
         * @param {String} value
         *  The new value of the cells in the ranges.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.fillCellContents = function (sheet, selection, value) {
            var operationOptions = { parse: Utils.LOCALE, shared: 0, ref: selection.activeCell };
            model.enterUndoGroup(function () {
                _(selection.ranges).each(function (range, index) {
                    model.fillCellRange(sheet, range, value, undefined, operationOptions);
                    value = undefined;
                    operationOptions = { shared: 0 };
                });
            }, this);
            return this;
        };

        /**
         * Sets the contents of a single cell in the document, and updates the
         * view according to the cell value.
         *
         * @param {Number} sheet
         *  The zero-based sheet index the cell is located in.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @param {String} value
         *  The new value of the cell.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCellContents = function (sheet, address, value) {
            model.setCellContents(sheet, address, { value: value }, { parse: Utils.LOCALE });
            return this;
        };

        /**
         * Returns the attribute set of the current active cell, together with
         * the column attributes and row attributes of the column/row the
         * active cell is located in.
         *
         * @returns {Object}
         *  The attribute set of the active cell with attribute maps keyed by
         *  attribute family.
         */
        this.getActiveCellAttributes = function () {
            return _.copy(selectionLayoutData.active.attrs, true);
        };

        /**
         * Changes multiple attributes for the cells in the current selection.
         *
         * @param {Object} [attributes]
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute families.
         *
         * @param {Object} [operationOptions]
         *  Additional properties that will be inserted into the generated
         *  'fillCellRange' operation. May contain any additional property
         *  supported by this operation.
         *
         * @param {Object} [options]
         *  A map with additional options controlling the behavior of this
         *  method. The following options are supported:
         *  @param {Boolean} [options.clear=false]
         *      If set to true, all existing explicit attributes will be
         *      removed from the selection while applying the new attributes.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCellAttributes = function (attributes, operationOptions, options) {

            var // the data object representing the active cell
                activeCellData = selectionLayoutData.active;

            // build character and cell null attributes
            function buildNullAttributes() {
                var attributes = { cell: {}, character: {} };
                _(DEFAULT_CELL_ATTRIBUTES).each(function (value, attr) { attributes.cell[attr] = null; });
                _(DEFAULT_CHARACTER_ATTRIBUTES).each(function (value, attr) { attributes.character[attr] = null; });
                return attributes;
            }

            // filter by supported attribute families, remove empty attribute maps
            attributes = _.isObject(attributes) ? attributes : {};
            _(attributes).each(function (attrs, family) {
                if (!/^(cell|character)$/.test(family) || _.isEmpty(attributes[family])) {
                    delete attributes[family];
                }
            });

            // add all attributes to be cleared
            if (Utils.getBooleanOption(options, 'clear', false)) {
                attributes = Utils.extendOptions(buildNullAttributes(), attributes);
            }

            // nothing to do if no attributes will be changed
            if (_.isEmpty(attributes)) { return this; }

            // apply the 'fillCellRange' operations for all selection ranges
            model.enterUndoGroup(function () {
                _(self.getSelectedRanges()).each(function (range) {
                    model.fillCellRange(self.getActiveSheet(), range, undefined, attributes, operationOptions);
                });
            });

            // update the layout data of the active cell
            activeCellData.attrs = _.copy(DEFAULT_CELL_LAYOUT_DATA.attrs, true);
            _(attributes).each(function (attrs, family) {
                _(attrs).each(function (value, name) {
                    var obj;
                    if (!_.isNull(value)) {
                        // for the border attributes, that can be set partially, it is necessary to set a style, if it only contains
                        // the width or the color. Otherwise the style is defaulted to 'none', so that 'cell/border/enabled' will be
                        // false and the 'Borders' toolbox collapses temporarely. Attention: Do not modify the original value, but
                        // only a clone.
                        if ((name.indexOf('border') === 0) && _.isObject(value) && (! value.style)) {
                            obj = _.clone(value);
                            obj.style = 'single';
                            activeCellData.attrs[family][name] = obj;
                        } else {
                            activeCellData.attrs[family][name] = value;
                        }
                    }
                });
            });

            return this;
        };

        /**
         * Changes a single attribute of type 'cell' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the cell attribute.
         *
         * @param {Any} value
         *  The new value of the cell attribute. If set to the value null, all
         *  attributes set explicitly will be removed from the cells.
         *
         * @param {Object} [operationOptions]
         *  Additional properties that will be inserted into the generated
         *  'fillCellRange' operation. May contain any additional property
         *  supported by this operation.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCellAttribute = function (name, value, operationOptions) {
            return this.setCellAttributes({ cell: Utils.makeSimpleObject(name, value) }, operationOptions);
        };

        /**
         * Changes a single attribute of type 'character' for all cells in the
         * current selection.
         *
         * @param {String} name
         *  The name of the character attribute.
         *
         * @param {Any} value
         *  The new value of the character attribute. If set to the value null,
         *  all attributes set explicitly will be removed from the cells.
         *
         * @param {Object} [operationOptions]
         *  Additional properties that will be inserted into the generated
         *  'fillCellRange' operation. May contain any additional property
         *  supported by this operation.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.setCharacterAttribute = function (name, value, operationOptions) {
            return this.setCellAttributes({ character: Utils.makeSimpleObject(name, value) }, operationOptions);
        };

        /**
         * removes all explicit formatting attributes from all cells in the
         * current selection.
         *
         * @returns {SpreadsheetView}
         *  A reference to this instance.
         */
        this.clearCellAttributes = function () {
            return this.setCellAttributes({}, undefined, { clear: true });
        };

        /**
         * Calculating the cell border width that will be displayed at the
         * side pane from the attributes of the cell. A number is only returned,
         * if there are borders at the cell and all borders have the same width.
         *
         * @param {Object} attributes
         *  The cell attributes object.
         *
         * @returns Number
         *  The value of the border width of the cell in 'pt'.
         */
        this.getCellBorderWidthFromAttributes = function (attributes) {
            var allWidths = [],
                oneWidth;

            _.each(CELL_BORDER_ATTRIBUTES, function (pos) {
                if ((attributes) && (attributes[pos]) && (attributes[pos].width)) {
                    allWidths.push(attributes[pos].width);
                    oneWidth = attributes[pos].width;
                }
            });

            return ((_.uniq(allWidths)).length === 1) ? Utils.convertHmmToLength(oneWidth, 'pt', 0.1) : '';
        };

        /**
         * Creating an attributes object for cells, that can be
         * used to assign a width to the cell borders.
         *
         * @param {Number} value
         *  The width in 1/100 mm that will be assigned to the cell borders.
         *
         * @returns {Object}
         *  The attribute object for the cell.
         */
        this.getAttributesFromCellBorderWidth = function (value) {

            var attrs = {};

            value = Utils.convertLengthToHmm(value, 'pt');

            // Always send the width for all borders. This is necessary, if multiple cells are selected.
            _.each(CELL_BORDER_ATTRIBUTES, function (name) {
                attrs[name] = { width: value };
            });

            return attrs;
        };

        /**
         * Calculating the cell border color that will be displayed at the
         * side pane from the attributes of the cell. A color is only returned,
         * if there are borders at the cell and all borders have the same color.
         *
         * @param {Object} attributes
         *  The cell attributes object.
         *
         * @returns {Color}
         *  The color object of the cell borders.
         */
        this.getCellBorderColorFromAttributes = function (attributes) {

            var allColors = [],
                oneColor;

            _.each(CELL_BORDER_ATTRIBUTES, function (pos) {
                if ((attributes) && (attributes[pos]) && (attributes[pos].color)) {
                    allColors.push(attributes[pos].color.value);
                    oneColor = attributes[pos].color;
                }
            });

            return ((_.uniq(allColors)).length === 1) ? oneColor : '';
        };

        /**
         * Creating an attributes object for cells, that can be
         * used to assign a color to the cell borders.
         *
         * @param {Color} value
         *  The color object that will be assigned to the cell borders.
         *
         * @returns {Object}
         *  The attribute object for the cell.
         */
        this.getAttributesFromCellBorderColor = function (value) {

            var attrs = {};

            // Always send the color for all borders. This is necessary, if multiple cells are selected.
            _.each(CELL_BORDER_ATTRIBUTES, function (name) {
                attrs[name] = { color: value };
            });

            return attrs;
        };

        /**
         * Calculating the cell border style that will be displayed at the
         * side pane from the attributes of the cell. A style is only returned,
         * if there are borders at the cell and all borders have the same style.
         *
         * @param {Object} attributes
         *  The cell attributes object.
         *
         * @returns {String}
         *  The style of the cell borders.
         */
        this.getCellBorderStyleFromAttributes = function (attributes) {

            var allStyles = [];

            _.each(CELL_BORDER_ATTRIBUTES, function (pos) {
                if ((attributes) && (attributes[pos]) && (attributes[pos].style) && (attributes[pos].style) !== 'none') {
                    allStyles.push(attributes[pos].style);
                }
            });

            return (_.uniq(allStyles).length === 1) ? allStyles[0] : '';
        };

        /**
         * Creating an attributes object for cells, that can be
         * used to assign a color to the cell borders.
         *
         * @param {Color} value
         *  The color object that will be assigned to the cell borders.
         *
         * @returns {Object}
         *  The attribute object for the cell.
         */
        this.getAttributesFromCellBorderStyle = function (value) {

            var attrs = {};

            // Always send the style for all borders. This is necessary, if multiple cells are selected.
            _.each(CELL_BORDER_ATTRIBUTES, function (name) {
                attrs[name] = { style: value };
            });

            return attrs;
        };

    } // class SpreadsheetView

    // exports ================================================================

    // derive this class from class EditView
    return EditView.extend({ constructor: SpreadsheetView });

});
