/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/gridpane',
    ['io.ox/office/tk/utils',
     'io.ox/office/tk/keycodes',
     'io.ox/office/framework/model/format/color',
     'io.ox/office/framework/model/format/lineheight',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Utils, KeyCodes, Color, LineHeight, SheetUtils) {

    'use strict';

    var // margin at right or bottom border of the entire sheet area, in pixels
        SCROLL_AREA_MARGIN = 30,

        // the names of all tracking events but 'tracking:start'
        TRACKING_EVENT_NAMES = 'tracking:move tracking:scroll tracking:end tracking:cancel';

    // private global functions ===============================================

    /**
     * Returns the CSS alignment of automatically aligned cells depending on
     * the result type of the cell. Numbers will be right aligned, strings will
     * be left aligned, and Booleans and error codes will be centered.
     *
     * @param {String|Number|Boolean|Null} result
     *  The result value of the cell.
     *
     * @returns {String}
     *  The value for the CSS 'text-align' property; or an empty string in case
     *  of an error.
     */
    function resolveAutoAlignment(result) {

        // Error or String
        if (_.isString(result)) {
            return (result[0] === '#' ? 'center' : 'left');
        }
        // Number
        if (_.isNumber(result)) {
            return 'right';
        }
        // Boolean
        if (_.isBoolean(result)) {
            return 'center';
        }
        // empty cells: left aligned
        return 'left';
    }


    // class GridPane =========================================================

    /**
     * Represents a single scrollable grid pane in the spreadsheet view. If the
     * view has been split or frozen at a specific cell position, the view will
     * consist of up to four panes (top-left, top-right, bottom-left, and
     * bottom-right pane).
     *
     * @constructor
     *
     * @param {SpreadsheetApplication} app
     *  The application that contains this grid pane.
     *
     * @param {String} panePos
     *  The position of this grid pane.
     */
    function GridPane(app, panePos) {

        var // self reference
            self = this,

            // the spreadsheet view
            view = app.getView(),

            // the container node of this grid pane (the 'tabindex' attribute makes the node focusable)
            rootNode = $('<div>').addClass('grid-pane unselectable').attr('tabindex', 0),

            // the content node with the cell grid and selection filled dynamically
            contentNode = $('<div>').addClass('grid-content').appendTo(rootNode),

            // the grid node inside the scrollable area containing the cell nodes
            gridNode = $('<div>').addClass('abs grid-cells noI18n').appendTo(contentNode),

            // the selection node inside the scrollable area containing the selected ranges
            selectionNode = $('<div>').addClass('abs grid-selection').appendTo(contentNode),

            // the text area control for in-place cell editing (will be inserted into the root node)
            textArea = Utils.createTextArea(),

            // the scrollable node of this pane (embed into rootNode to allow to hide scroll bars if needed)
            scrollNode = $('<div>').addClass('grid-scroll').appendTo(rootNode),

            // the node controlling the size and position of the scroll bars
            scrollSizeNode = $('<div>').addClass('grid-scroll-size').appendTo(scrollNode),

            // the busy node shown if the grid pane is not initialized yet
            busyNode = $('<div>').addClass('abs').busy(),

            // the document style sheets
            documentStyles = app.getModel().getDocumentStyles(),

            // the column and row collection for this grid pane
            colCollection = view.getColumnCollection(panePos),
            rowCollection = view.getRowCollection(panePos),

            // the cell collection for this grid pane
            cellCollection = view.getCellCollection(panePos),

            // the absolute scroll position represented by this grid pane (may differ from position of scroll bars)
            scrollLeft = 0, scrollTop = 0,

            // the ratio between (public) absolute scroll position, and internal scroll bars
            scrollLeftRatio = 1, scrollTopRatio = 1,

            // if this flag is set, scrolling has been triggered manually (no recalculation of scroll position)
            scrollManually = false,

            // distance between cell A1 and the top-left visible cell of this pane
            hiddenWidth = 0, hiddenHeight = 0,

            // current area containing up-to-date cell contents
            updatedRectangle = null,

            // the type of the current tracking mode ('select', 'move', 'fill', null)
            trackingType = null,

            // all settings for in-place edit mode
            editSettings = null;

        // private methods ----------------------------------------------------

        /**
         * Returns the effective offset and size of the visible area in the
         * scrollable node.
         *
         * @returns {Object}
         *  The offset and size of the visible area in the sheet, in pixels.
         */
        function getVisibleAreaPosition() {
            return {
                left: scrollLeft + hiddenWidth,
                top: scrollTop + hiddenHeight,
                width: scrollNode[0].clientWidth,
                height: scrollNode[0].clientHeight
            };
        }

        /**
         * Returns the absolute position of the specified cell in the sheet.
         *
         * @param {Number[]} address
         *  The logical address of the cell.
         *
         * @returns {Object|Null}
         *  The position of the specified cell in the sheet, in pixels; or null
         *  if the layout information of the cell is not available.
         */
        function getCellPosition(address) {

            var // column and row position of the cell
                colPos = colCollection.getEntryPosition(address[0]),
                rowPos = rowCollection.getEntryPosition(address[1]);

            // TODO: handle merged cells
            return (colPos && rowPos) ? { left: colPos.offset, top: rowPos.offset, width: colPos.size, height: rowPos.size } : null;
        }

        /**
         * Scrolls this grid pane to make the specified cell visible.
         *
         * @param {Number} [col]
         *  The zero-based index of the column to be made visible. If omitted,
         *  the horizontal scroll position will not be changed.
         *
         * @param {Number} [row]
         *  The zero-based index of the row to be made visible. If omitted, the
         *  vertical scroll position will not be changed.
         */
        function scrollToCell(col, row) {

            var // position and size of the passed cell
                colPosition = null, rowPosition = null,
                // the absolute offset and size of the visible area in the sheet
                visiblePosition = getVisibleAreaPosition(),
                // new scrolling position
                newScrollLeft = scrollLeft, newScrollTop = scrollTop;

            // calculate new horizontal scroll position
            if (_.isNumber(col)) {
                if ((colPosition = colCollection.getEntryPosition(col))) {
                    if (colPosition.offset < visiblePosition.left) {
                        newScrollLeft = colPosition.offset - hiddenWidth;
                    } else if (colPosition.offset + colPosition.size + 2 > visiblePosition.left + visiblePosition.width) {
                        newScrollLeft = colPosition.offset + colPosition.size + 2 - visiblePosition.width - hiddenWidth;
                    }
                } else {
                    // TODO: request view update for columns outside the known range
                }
            }

            // calculate new vertical scroll position
            if (_.isNumber(row)) {
                if ((rowPosition = rowCollection.getEntryPosition(row))) {
                    if (rowPosition.offset < visiblePosition.top) {
                        newScrollTop = rowPosition.offset - hiddenHeight;
                    } else if (rowPosition.offset + rowPosition.size + 2 > visiblePosition.top + visiblePosition.height) {
                        newScrollTop = rowPosition.offset + rowPosition.size + 2 - visiblePosition.height - hiddenHeight;
                    }
                } else {
                    // TODO: request view update for rows outside the known range
                }
            }

            // update scroll position
            if ((newScrollLeft !== scrollLeft) || (newScrollTop !== scrollTop)) {
                self.scroll(newScrollLeft, newScrollTop);
            }
        }

        /**
         * Requests an update for this grid pane from the view.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Boolean} [options.force=false]
         *      Whether to perform a forced update without checking the current
         *      scroll position and available layout data.
         *  @param {Array} [options.cells]
         *      A list of cell addresses with changed contents. Can be used to
         *      update the cell grid partly.
         */
        function requestUpdate(options) {

            var // the absolute offset and size of the visible area in the sheet
                visiblePosition = getVisibleAreaPosition(),
                // whether the update has to be requested anyway
                force = Utils.getBooleanOption(options, 'force', false);

            if ((rootNode.css('display') !== 'none') && (visiblePosition.width > 0) && (visiblePosition.height > 0)) {

                // convert all length properties to 1/100 mm
                _(visiblePosition).each(function (length, name) {
                    visiblePosition[name] = Utils.convertLengthToHmm(length, 'px');
                });

                // check if view position reaches the borders of the up-to-date area
                if (force || !updatedRectangle ||
                    ((updatedRectangle.left > 0) && (visiblePosition.left - updatedRectangle.left < visiblePosition.width / 3)) ||
                    ((updatedRectangle.top > 0) && (visiblePosition.top - updatedRectangle.top < visiblePosition.height / 3)) ||
                    (updatedRectangle.left + updatedRectangle.width - visiblePosition.left - visiblePosition.width < visiblePosition.width / 3) ||
                    (updatedRectangle.top + updatedRectangle.height - visiblePosition.top - visiblePosition.height < visiblePosition.height / 3)) {

                    // show the busy node behind the cells until the response has been sent
                    rootNode.prepend(busyNode);

                    // enlarge area to be updated in all directions, request update from server
                    updatedRectangle = {
                        left: Math.max(0, visiblePosition.left - visiblePosition.width - 30),
                        top: Math.max(0, visiblePosition.top - visiblePosition.height - 30)
                    };
                    updatedRectangle.width = visiblePosition.left + 2 * visiblePosition.width + 30 - updatedRectangle.left;
                    updatedRectangle.height = visiblePosition.top + 2 * visiblePosition.height + 30 - updatedRectangle.top;
                    view.requestGridPaneUpdate(panePos, updatedRectangle);
                }
            }
        }

        /**
         * Updates the size of the scrollable area according to the used area
         * in the sheet, and the current scroll position. Tries to reserve
         * additional space for the full width of this grid pane at the right
         * border, and the full height of this grid pane at the bottom border,
         * to be able to scroll page-by-page into the unused area of the sheet.
         */
        function updateScrollAreaSize() {

            var // the sheet layout data stored in the view
                sheetLayoutData = view.getSheetLayoutData(),
                // the absolute offset and size of the visible area in the sheet
                visiblePosition = getVisibleAreaPosition(),
                // the new absolute width and height of the scrollable area
                scrollableWidth = 0, scrollableHeight = 0,
                // the real size of the scroll size node (may be smaller due to browser limits)
                effectiveWidth = 0, effectiveHeight = 0;

            // scrollable area includes the used area, and the current (but extended) scroll position
            scrollableWidth = Math.max(sheetLayoutData.used.width, visiblePosition.left + 2 * visiblePosition.width);
            scrollableHeight = Math.max(sheetLayoutData.used.height, visiblePosition.top + 2 * visiblePosition.height);

            // restrict to the total size of the sheet, plus a little margin
            scrollableWidth = Math.min(scrollableWidth, sheetLayoutData.width + SCROLL_AREA_MARGIN);
            scrollableHeight = Math.min(scrollableHeight, sheetLayoutData.height + SCROLL_AREA_MARGIN);

            // remove the visible offset (the grid pane may not start at cell A1)
            scrollableWidth -= hiddenWidth;
            scrollableHeight -= hiddenHeight;

            // initialize the size of the scroll sizer node, the resulting size may be smaller
            Utils.setContainerNodeSize(scrollSizeNode, scrollableWidth, scrollableHeight);
            effectiveWidth = scrollSizeNode.width();
            effectiveHeight = scrollSizeNode.height();

            // recalculate ratio between absolute and effective scroll area size
            scrollLeftRatio = (effectiveWidth > 0) ? (scrollableWidth / effectiveWidth) : 0;
            scrollTopRatio = (effectiveHeight > 0) ? (scrollableHeight / effectiveHeight) : 0;
        }

        /**
         * Updates the position and size of the content node (containing the
         * cell grid and the selection ranges) according to the current scroll
         * position and pane layout data.
         */
        function updateContentPosition() {
            contentNode.css({
                left: colCollection.getOffset() - hiddenWidth - scrollLeft,
                top: rowCollection.getOffset() - hiddenHeight - scrollTop,
                width: colCollection.getSize(),
                height: rowCollection.getSize()
            });
        }

        /**
         * Sets the physical position of the scroll bars according to the
         * current absolute scroll position.
         */
        function updateScrollPosition() {
            scrollManually = true;
            scrollNode
                .scrollLeft((scrollLeftRatio > 0) ? Math.round(scrollLeft / scrollLeftRatio) : 0)
                .scrollTop((scrollTopRatio > 0) ? Math.round(scrollTop / scrollTopRatio) : 0);
            scrollManually = false;
        }

        /**
         * Updates the position and size of the text area control for the
         * in-place edit mode, according to the current length of the text in
         * the text area control, and the scroll position of the grid pane.
         */
        function updateTextAreaPosition() {

            var // the address of the edited cell
                address = null,
                // the position of the edited cell
                position = {},
                // the visible area of this grid pane
                visiblePosition = null,
                // effective position of the text area
                leftOffset = 0, topOffset = 0,
                // the minimum and maximum width for the edit area
                minWidth = 0, maxWidth = 0;

            // calculates maximum width according to alignment (distance to visible border, but at least one third of the visible area)
            function getMaxWidth() {

                var // the distance from left visible border to right cell border (for right alignment)
                    leftWidth = Utils.minMax(position.left + position.width - visiblePosition.left, 0, visiblePosition.width),
                    // the distance from left cell border to right visible border (for left alignment)
                    rightWidth = Utils.minMax(visiblePosition.left + visiblePosition.width - position.left, 0, visiblePosition.width),
                    // the effective maximum width
                    maxWidth = 0;

                switch (editSettings.align) {
                case 'right':
                    maxWidth = leftWidth;
                    break;
                case 'center':
                    maxWidth = Math.min(leftWidth, rightWidth) * 2 - position.width;
                    break;
                default:
                    maxWidth = rightWidth;
                }

                return Math.max(maxWidth, Math.round(visiblePosition.width / 3));
            }

            // binary search algorithm to find the optimal width of the text field
            function setOptimalTextAreaWidth() {

                var // the width at the beginning of an iteration
                    prevWidth = 0,
                    // last fitting width (in case iteration ends while width is too small)
                    fittingWidth = maxWidth;

                do {
                    prevWidth = textArea.outerWidth();
                    if (textArea[0].clientHeight < textArea[0].scrollHeight) {
                        // text does not fit: enlarge width (round to blocks of 16 pixels)
                        minWidth = prevWidth;
                        textArea.css({ width: Math.ceil((prevWidth + maxWidth) / 32) * 16 });
                    } else {
                        // text fits: reduce width (round to blocks of 16 pixels)
                        maxWidth = fittingWidth = prevWidth;
                        textArea.css({ width: Math.ceil((minWidth + prevWidth) / 32) * 16 });
                    }
                } while (prevWidth !== textArea.outerWidth());

                // set last known fitting width
                textArea.css({ width: fittingWidth });
            }

            // do nothing if in-place edit mode is not active
            if (!editSettings) { return; }

            // get cell position and visible area of the grid pane
            address = editSettings.selection.activeCell;
            visiblePosition = getVisibleAreaPosition();

            // get horizontal cell position and size (use a default position, if column is not available)
            if (editSettings.colPosition || (editSettings.colPosition = colCollection.getEntryPosition(address[0]))) {
                position.left = editSettings.colPosition.offset;
                // use at least 20 pixels width, enlarge width by one pixel to fit exactly into the selection border
                position.width = Math.max(20, editSettings.colPosition.size + 1);
            } else {
                position.width = 85;
                // if active cell is left of visible area, show the text area left aligned, otherwise right aligned
                position.left = (address[0] < colCollection.getFirstIndex()) ? visiblePosition.left : (visiblePosition.left + visiblePosition.width - position.width);
            }

            // get vertical cell position and size (use a default position, if row is not available)
            if (editSettings.rowPosition || (editSettings.rowPosition = rowCollection.getEntryPosition(address[1]))) {
                position.top = editSettings.rowPosition.offset;
                // enlarge height by one pixel to fit exactly into the selection border
                position.height = editSettings.rowPosition.size + 1;
            } else {
                position.height = 3;
                // if active cell is above the visible area, show the text area top aligned, otherwise bottom aligned
                position.top = (address[1] < rowCollection.getFirstIndex()) ? visiblePosition.top : (visiblePosition.top + visiblePosition.height - position.height);
            }

            // stick to cell width in auto-wrapping mode or block mode
            if (editSettings.wrap || (editSettings.align === 'justify')) {
                textArea.css({ width: position.width });
                leftOffset = position.left;
            } else {

                // get minimum width (cell width, restricted to visible area)
                minWidth = Math.min(position.width, visiblePosition.width);
                maxWidth = getMaxWidth();

                // set initial size of the text area to maximum allowed width and as many text lines as needed
                textArea.css({
                    width: maxWidth,
                    height: (textArea.val().split('\n').length * LineHeight.calculateNormalLineHeight(textArea)) + 'pt'
                });

                // if text fits into a single line, try to collapse width to the actual cell width
                if ((minWidth < maxWidth) && (textArea[0].clientHeight === textArea[0].scrollHeight)) {
                    textArea.css({ width: minWidth });
                    // if text does not fit into the cell width, try to find optimal width via binary search
                    if (textArea[0].clientHeight < textArea[0].scrollHeight) {
                        setOptimalTextAreaWidth();
                    }
                }

                // calculate horizontal position according to alignment
                switch (editSettings.align) {
                case 'right':
                    leftOffset = position.left + position.width - textArea.outerWidth();
                    break;
                case 'center':
                    leftOffset = position.left + Math.round((position.width - textArea.outerWidth()) / 2);
                    break;
                default:
                    leftOffset = position.left;
                }
            }

            // width is optimal, enlarge height in case text still does not fit into one line
            textArea.css({ height: Utils.minMax(textArea[0].scrollHeight, position.height, visiblePosition.height) });

            // move cell into visible area, set effective position
            leftOffset = Utils.minMax(leftOffset, visiblePosition.left, visiblePosition.left + visiblePosition.width - textArea.outerWidth());
            topOffset = Utils.minMax(position.top, visiblePosition.top, visiblePosition.top + visiblePosition.height - textArea.outerHeight());
            textArea.css({ left: leftOffset - visiblePosition.left, top: topOffset - visiblePosition.top });

            // focus back to text area control (lost e.g. while scrolling with scroll bars)
            textArea.focus();
        }

        /**
         * Renders all visible cells according to the current pane layout data
         * received from a view layout update notification.
         */
        function renderCells() {

            var // the HTML mark-up for all visible cells
                markup = '',
                // the attribute parser functions
                parserFunctions = {},
                // the style attribute for the <div>
                divStyleAttr = '',
                // the style attribute for the <span>
                spanStyleAttr = '',
                // handle line-through and underline text-decoration
                spanTextDecoration = null,
                // horizontal alignment. Will be preset in case of 'auto' styling according to the cell value
                horizontalAlign = '',
                // collecting the properties of the cell border attributes
                allBorders = {},
                // the text color
                textColor = '',
                // whether to wrap text automatically
                wrapText = false;

            // collecting the border attributes
            function collectBorderAttributes(cellData, family, attr) {
                if (cellData.attrs[family].hasOwnProperty(attr)) {
                    allBorders[attr] = cellData.attrs[family][attr];
                }
            }

            // returns the CSS text color depending on the fill color of the cell
            function getTextColor(cellData) {
                // calculate effective text color, according to the cell fill color
                var fillColor = cellData.attrs && cellData.attrs.cell && cellData.attrs.cell.fillColor || Color.AUTO,
                    textColor = cellData.attrs && cellData.attrs.character && cellData.attrs.character.color || Color.AUTO;
                return documentStyles.getCssTextColor(textColor, [fillColor]);
            }

            // Character attribute handler
            parserFunctions.bold = function (cellData, family) {
                // add 'font-weight' attribute only if 'bold' is set to true
                if (Utils.getBooleanOption(cellData.attrs[family], 'bold', false)) {
                    spanStyleAttr += 'font-weight:bold;';
                }
            };

            parserFunctions.italic = function (cellData, family) {
                // add 'font-style' attribute only if 'italic' is set to true
                if (Utils.getBooleanOption(cellData.attrs[family], 'italic', false)) {
                    spanStyleAttr += 'font-style:italic;';
                }
            };

            parserFunctions.underline = function (cellData, family) {
                var underline = Utils.getBooleanOption(cellData.attrs[family], 'underline', false);
                spanTextDecoration = Utils.toggleToken(spanTextDecoration, 'underline', underline, 'none');
            };

            parserFunctions.strike = function (cellData, family) {
                var strike = cellData.attrs[family].strike === 'single' || cellData.attrs[family].strike === 'double';
                spanTextDecoration = Utils.toggleToken(spanTextDecoration, 'line-through', strike, 'none');
            };

            parserFunctions.color = function (cellData, family) {
                textColor = getTextColor(cellData);
            };

            parserFunctions.fontName = function (cellData, family) {
                var fontName = documentStyles.getCssFontFamily(cellData.attrs[family].fontName || 'Arial,sans-serif;');

                // If a font name contains white-space, it must be quoted.
                // Apostrophes must be used because the 'style' attribute is generated with quote characters.
                fontName = fontName.replace(/"/g, '\'');
                spanStyleAttr += 'font-family:' + fontName + ';';
            };

            parserFunctions.fontSize = function (cellData, family) {
                var fontSize = cellData.attrs[family].fontSize;

                // font-size: 10pt is defaulted via CSS
                if (fontSize && fontSize !== 10) {
                    spanStyleAttr += 'font-size:' + fontSize + 'pt;';
                }
            };

            // Cell attribute handler
            parserFunctions.alignHor = function (cellData, family) {
                var attrValue = cellData.attrs[family].alignHor;

                // 'auto' style: aligns the cell contents depending on the result type of the cell
                horizontalAlign = (attrValue === 'auto') ? resolveAutoAlignment(cellData.result) : attrValue;

                // justified alignment wraps the text automatically
                wrapText = wrapText || (attrValue === 'justify');
            };

            parserFunctions.alignVert = function (cellData, family) {
                var attrValue = cellData.attrs[family].alignVert;

                // vertical-align: bottom is defaulted via CSS
                if (_(['top', 'middle', 'justify']).contains(attrValue)) {
                    spanStyleAttr += 'vertical-align:' + attrValue + ';';
                }
            };

            parserFunctions.wrapText = function (cellData, family) {
                var attrValue = cellData.attrs[family].wrapText;

                // text wrapping may have been set from 'justified' alignment too
                wrapText = wrapText || attrValue;
            };

            parserFunctions.fillColor = function (cellData, family) {
                var fillColor = documentStyles.getCssColor(cellData.attrs[family].fillColor, 'fill');

                // background-color: transparent is defaulted via CSS
                if (fillColor && fillColor !== 'transparent') {
                    divStyleAttr += 'background-color:' + fillColor + ';';
                }
            };

            parserFunctions.borderTop = collectBorderAttributes;
            parserFunctions.borderBottom = collectBorderAttributes;
            parserFunctions.borderLeft = collectBorderAttributes;
            parserFunctions.borderRight = collectBorderAttributes;
            parserFunctions.borderInsideHor = collectBorderAttributes;
            parserFunctions.borderInsideVert = collectBorderAttributes;

            function createBorderMarkup(position) {

                // Example: Border width '5px' for all borders
                // width increased by 5px
                // height increased by 5px
                // left reduced by 2px
                // top reduced by 2px

                var markup = '<div class="border" style="';

                function processBorder(border, position, roundUp) {

                    var // the effective CSS attributes
                        attributes = documentStyles.getCssBorderAttributes(border),
                        // the number of pixels the border will be moved out of the cell
                        offset = roundUp ? Math.ceil(attributes.width / 2) : Math.floor(attributes.width / 2);

                    // add offset outside of cell to the style
                    markup += position + ':-' + offset + 'px;';
                    // add border style if it is visible
                    if (attributes.width > 0) {
                        markup += 'border-' + position + ':' + attributes.style + ' ' + attributes.width + 'px ' + attributes.color + ';';
                    }
                }

                // offset to left/top cell border: round down (keep middle line for odd line widths on left/top pixel of the OWN cell)
                processBorder(allBorders.borderLeft, 'left', false);
                processBorder(allBorders.borderTop, 'top', false);
                // offset to right/bottom cell border: round up (keep middle line for odd line widths on left/top pixel of the NEXT cell)
                processBorder(allBorders.borderRight, 'right', true);
                processBorder(allBorders.borderBottom, 'bottom', true);

                markup += '"></div>';
                return markup;
            }

            function mergeStylesFromCellData(cellData) {

                var attrFamilies = Utils.getObjectOption(cellData, 'attrs', {});

                spanTextDecoration = 'none';

                // preset horizontal alignment
                horizontalAlign = resolveAutoAlignment(cellData.result);

                // preset text color
                textColor = getTextColor(cellData);

                // handle attribute families
                _(attrFamilies).each(function (attributes, family) {
                    // handle attributes
                    _(attributes).each(function (value, attr) {
                        if (_.isFunction(parserFunctions[attr])) {
                            parserFunctions[attr](cellData, family, attr);
                        } else {
                            //Utils.warn('No attribute parser for attribute: "' + attr + '" of attribute family: "' + family + '"');
                        }
                    });
                });

                if (_.isString(spanTextDecoration) && spanTextDecoration !== 'none') {
                    // text-decoration: none is defaulted via CSS
                    spanStyleAttr += 'text-decoration:' + spanTextDecoration + ';';
                }

                if (horizontalAlign.length) {
                    divStyleAttr += 'text-align:' + horizontalAlign + ';';
                }

                if (textColor.length) {
                    spanStyleAttr += 'color:' + textColor + ';';
                }

                // automatic text wrapping: 'pre' is defaulted via CSS (no wrapping at all)
                if (wrapText) {
                    // TODO: make this actually work
                    //divStyleAttr += 'white-space:pre-wrap;word-wrap:break-word;';
                }
            }

            // generate HTML mark-up text for all cells (also empty cells not contained in the cell collection)
            rowCollection.iterateEntries(function (rowEntry) {
                colCollection.iterateEntries(function (colEntry) {

                    var // the position of the cell in the content node, in pixels
                        position = {
                            left: colEntry.offset - colCollection.getOffset(),
                            top: rowEntry.offset - rowCollection.getOffset(),
                            width: colEntry.size,
                            height: rowEntry.size
                        },
                        // find the cell data object in the map
                        cellData = cellCollection.getEntry(colEntry.index, rowEntry.index),
                        // the data attributes
                        dataAttrs = ' data-address="' + colEntry.index + ',' + rowEntry.index + '"',
                        // the text to be shown in the cell
                        displayText = Utils.getStringOption(cellData, 'display', '');

                    // prepare CSS styling of the cell container and the text span
                    divStyleAttr = ' style="left:' + position.left + 'px;top:' + position.top + 'px;width:' + position.width + 'px;height:' + position.height + 'px;line-height:' + position.height + 'px;';
                    spanStyleAttr = ' style="';
                    mergeStylesFromCellData(cellData);
                    divStyleAttr += '"';
                    spanStyleAttr += '"';

                    // generate the HTML mark-up for the cell
                    markup += '<div class="cell"' + dataAttrs + divStyleAttr + '>';
                    if (!_.isEmpty(allBorders)) { markup += createBorderMarkup(position); }
                    markup += '<span' + spanStyleAttr + '>';
                    if (displayText.length > 0) { markup += Utils.escapeHTML(displayText); }
                    markup += '</span></div>';

                    allBorders = {};
                });
            });

            // insert entire HTML mark-up into the cell container node
            gridNode[0].innerHTML = markup;
        }

        /**
         * Renders all visible selection ranges according to the current pane
         * layout data received from a view layout update notification.
         */
        function renderSelection(selection) {

            var // the HTML mark-up for all selection ranges
                markup = '',
                // the address of the entire cell range contained in the pane layout data
                layoutRange = null,
                // the address and position of the active cell
                activeCell = view.getActiveCell(),
                activeCellPos = getCellPosition(activeCell);

            // do nothing, if no layout information available for columns or rows
            if (colCollection.isEmpty() || rowCollection.isEmpty()) { return; }

            // get the entire cell range contained in the pane layout data
            layoutRange = {
                start: [colCollection.getFirstIndex(), rowCollection.getFirstIndex()],
                end: [colCollection.getLastIndex(), rowCollection.getLastIndex()]
            };

            _(selection.ranges).each(function (range, index) {

                var // the part of the selection range that intersects the layout area (may be null!)
                    intersection = SheetUtils.getIntersectionRange(layoutRange, range),
                    // whether the range is the active range
                    active = index === selection.activeRange,
                    // position and size of the node
                    colsPos = null, rowsPos = null,
                    // position of active cell, relative to current range
                    activePos = null,
                    // CSS classes
                    classAttr = ' class="range' + (active ? ' active' : '') + '"',
                    // the style attribute
                    styleAttr = ' style=',
                    // mark-up for the fill elements
                    fillMarkup = '';

                // range outside the visible range of this grid pane
                if (!intersection) { return; }

                // calculate the screen offsets and sizes of the range
                colsPos = colCollection.getRangePosition(intersection.start[0], intersection.end[0]);
                rowsPos = rowCollection.getRangePosition(intersection.start[1], intersection.end[1]);

                // insert the semi-transparent fill elements (never cover the active cell with the fill elements)
                if (activeCellPos && SheetUtils.rangeContainsCell(intersection, activeCell)) {

                    // initialize position of active cell, relative to selection range
                    activePos = _.clone(activeCellPos);
                    activePos.left -= colsPos.offset;
                    activePos.top -= rowsPos.offset;

                    // insert fill element above active cell
                    if (activePos.top > 0) {
                        fillMarkup += '<div class="fill" style="left:0;right:0;top:0;height:' + activePos.top + 'px;"></div>';
                    }

                    // insert fill element left of active cell
                    if (activePos.left > 0) {
                        fillMarkup += '<div class="fill" style="left:0;width:' + activePos.left + 'px;top:' + activePos.top + 'px;height:' + activePos.height + 'px;"></div>';
                    }

                    // insert fill element right of active cell
                    if (activePos.left + activePos.width < colsPos.size) {
                        fillMarkup += '<div class="fill" style="left:' + (activePos.left + activePos.width) + 'px;right:0;top:' + activePos.top + 'px;height:' + activePos.height + 'px;"></div>';
                    }

                    // insert fill element below active cell
                    if (activePos.top + activePos.height < rowsPos.size) {
                        fillMarkup += '<div class="fill" style="left:0;right:0;top:' + (activePos.top + activePos.height) + 'px;bottom:0;"></div>';
                    }

                } else {
                    fillMarkup = '<div class="abs fill"></div>';
                }

                // adjust offsets according to range type
                if (active) {
                    colsPos.offset -= 2;
                    colsPos.size += 5;
                    rowsPos.offset -= 2;
                    rowsPos.size += 5;
                } else {
                    colsPos.size += 1;
                    rowsPos.size += 1;
                }

                // build the style attribute
                styleAttr += '"left:' + (colsPos.offset - colCollection.getOffset()) + 'px;top:' + (rowsPos.offset - rowCollection.getOffset()) + 'px;width:' + colsPos.size + 'px;height:' + rowsPos.size + 'px;"';

                // generate the HTML mark-up for the selection range
                markup += '<div' + classAttr + styleAttr + '>' + fillMarkup + '</div>';
            });

            // insert entire HTML mark-up into the selection container node
            selectionNode[0].innerHTML = markup;
        }

        /**
         * Resets all internal states after the active sheet has been changed.
         */
        function activeSheetHandler() {
            updatedRectangle = null;
        }

        /**
         * Renders the selected ranges after the selection has been changed.
         */
        function selectionHandler(event, selection) {
            leaveEditMode('cell');
            renderSelection(selection);
        }

        /**
         * Updates this grid pane with the passed cell contents and layout
         * data, as received from a server update request.
         */
        function updateGridHandler(event, changedFlags) {

            // do nothing if the layout data has not been changed, or is not initialized
            if (!changedFlags[panePos] || rowCollection.isEmpty() || colCollection.isEmpty()) { return; }

            // update the size of the scrollable area, render all visible cells and the selection
            Utils.takeTime('GridPane.updateGridHandler(): panePos=' + panePos, function () {
                busyNode.detach();
                updateScrollAreaSize();
                updateContentPosition();
                updateTextAreaPosition();
                renderCells();
                renderSelection(view.getCellSelection());
            });
        }

        /**
         * Handles 'scroll' events of the scrollable node and synchronizes the
         * scroll position of the other grid panes.
         */
        var scrollHandler = app.createDebouncedMethod(function () {

            // calculate absolute scroll position from current scroll bar positions
            if (!scrollManually) {
                scrollLeft = Math.round(scrollNode.scrollLeft() * scrollLeftRatio);
                scrollTop = Math.round(scrollNode.scrollTop() * scrollTopRatio);
            }

            // check if the own cell area needs to be updated
            requestUpdate();

            // update scroll position of content node immediately
            updateContentPosition();
            updateTextAreaPosition();

            // update other grid panes and header panes
            view.syncGridPaneScroll(panePos, scrollLeft, scrollTop);

        }, updateScrollAreaSize, { delay: 250 });

        /**
         * Returns the current absolute sheet position in the grid pane for the
         * passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event.
         *
         * @returns {Object}
         *  The absolute sheet position in the properties 'left' and 'top', in
         *  pixels.
         */
        function getTrackingOffset(event) {

            var // the absolute position of the content node
                contentOffset = contentNode.offset();

            return {
                left: event.pageX - contentOffset.left + colCollection.getOffset(),
                top: event.pageY - contentOffset.top + rowCollection.getOffset()
            };
        }

        /**
         * Returns the column and row entries matching the position in the
         * passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event.
         *
         * @returns {Object|Null}
         *  The column and row entries in the properties 'col' and 'row'; or
         *  null, if at least one entry is missing.
         */
        function getTrackingEntries(event) {

            var // the absolute position of the content node
                offset = getTrackingOffset(event),
                // the entry of the column currently hovered
                colEntry = colCollection.getNearestEntryByOffset(offset.left),
                // the entry of the row currently hovered
                rowEntry = rowCollection.getNearestEntryByOffset(offset.top);

            return (colEntry && rowEntry) ? { col: colEntry, row: rowEntry } : null;
        }

        /**
         * Handles all tracking events for cell selection.
         */
        var selectionTrackingHandler = (function () {

            var // the cell address where tracking has started
                startAddress = null,
                // current address (prevent updates while tracking over the same cell)
                currentAddress = null;

            // makes an adjusted range from the start address to the current address
            function makeRange() {
                return SheetUtils.adjustRange({ start: _.clone(startAddress), end: _.clone(currentAddress) });
            }

            // initializes the selection according to the passed tracking event
            function initializeTracking(event) {

                var // initial column/row entries
                    entries = getTrackingEntries(event),
                    // whether to append a new range to the selection
                    append = !event.shiftKey && (event.ctrlKey || event.metaKey),
                    // whether to extend the active range
                    extend = event.shiftKey && !event.ctrlKey && !event.metaKey;

                if (!entries) {
                    $.cancelTracking();
                    return;
                }

                currentAddress = [entries.col.index, entries.row.index];
                if (extend) {
                    // pressed SHIFT key: modify the active range (from active cell to tracking start position)
                    startAddress = view.getActiveCell();
                    view.changeActiveRange(makeRange());
                } else {
                    // no SHIFT key: create a new selection range
                    startAddress = currentAddress;
                    view.selectCell(startAddress, { append: append });
                }

                self.scrollToCell(currentAddress);
            }

            // updates the selection according to the passed tracking event
            function updateSelection(event) {

                var // the column/row entries currently hovered
                    entries = getTrackingEntries(event);

                if (entries && currentAddress && ((entries.col.index !== currentAddress[0]) || (entries.row.index !== currentAddress[1]))) {
                    currentAddress = [entries.col.index, entries.row.index];
                    view.changeActiveRange(makeRange());
                }
            }

            // updates the scroll position while tracking
            function updateScroll(event) {
                self.scroll(scrollLeft + event.scrollX, scrollTop + event.scrollY);
            }

            // finalizes the selection tracking
            function finalizeTracking(event) {
                if (currentAddress) {
                    self.scrollToCell(currentAddress);
                    currentAddress = null;
                }
                trackingType = null;
                $(event.delegateTarget).off(TRACKING_EVENT_NAMES);
                view.grabFocus();
            }

            // return the actual trackingHandler() method
            return function (event) {

                switch (event.type) {
                case 'tracking:start':
                    initializeTracking(event);
                    break;
                case 'tracking:move':
                    updateSelection(event);
                    break;
                case 'tracking:scroll':
                    updateScroll(event);
                    updateSelection(event);
                    break;
                case 'tracking:end':
                case 'tracking:cancel': // cancel tracking: keep current selection
                    updateSelection(event);
                    finalizeTracking(event);
                    break;
                }
            };

        }()); // end of selectionTrackingHandler() local scope

        /**
         * Handles all tracking events for cell selection.
         */
        function trackingStartHandler(event) {
            trackingType = 'select';
            $(event.delegateTarget).on(TRACKING_EVENT_NAMES, selectionTrackingHandler);
            selectionTrackingHandler.call(this, event);
        }

        /**
         * Handles double clicks and starts the in-place edit mode.
         */
        function doubleClickHandler() {
            if (!editSettings) {
                enterEditMode();
            }
        }

        /**
         * Tries to find the previous/next visible column/row index in the
         * passed collection, relative to the specified index.
         *
         * @param {ColRowCollection} collection
         *  The collection to be searched for a visible entry.
         *
         * @param {Number} index
         *  The zero-based index of the start column/row.
         *
         * @param {Number} diff
         *  The number of columns/rows to be moved. If the target column/row is
         *  available but hidden, looks for the next visible column or row.
         *
         * @returns {Number|Null}
         *  The zero-based index of the found visible column or row; or null,
         *  if the target index is outside the current index interval of this
         *  collection.
         */
        function findCollectionIndex(collection, index, diff) {
            var entry = (diff < 0) ? collection.findPrevEntryByIndex(index + diff) : (diff > 0) ? collection.findNextEntryByIndex(index + diff) : null;
            return entry ? entry.index : (diff === 0) ? index : null;
        }

        /**
         * Clears the current selection and moves the active cell into the
         * specified direction. If the target cell is hidden, moves the cursor
         * to the next available visible cell.
         *
         * @param {Number} cols
         *  The number of columns the active cell will be moved. Negative
         *  values will move the cursor to the left.
         *
         * @param {Number} rows
         *  The number of rows the active cell will be moved. Negative values
         *  will move the cursor up.
         */
        function selectNextCell(cols, rows) {

            var // current selection in the active sheet
                selection = view.getCellSelection(),
                // the target address
                address = [
                    findCollectionIndex(colCollection, selection.activeCell[0], cols),
                    findCollectionIndex(rowCollection, selection.activeCell[1], rows)
                ];

            // collapse selection to current active cell, if no other cell has been found
            if (!_.isNumber(address[0]) || !_.isNumber(address[1])) {
                address = selection.activeCell;
            }

            // set selection to the new active cell
            view.selectCell(address);
            self.scrollToCell(address);
        }

        /**
         * Clears the current selection and moves the active cell up or down by
         * one page.
         *
         * @param {Boolean} down
         *  If set to true, moves the active cell down, otherwise up.
         */
        function selectNextPage(down) {
        }

        /**
         * Extends the active range in the current selection into the specified
         * direction. If the target column/row is hidden, extends the active
         * range to the next available visible column/row.
         *
         * @param {Number} cols
         *  The number of columns to extend the active range. Negative values
         *  will select previous columns.
         *
         * @param {Number} rows
         *  The number of rows to extend the active range. Negative values will
         *  select upper rows.
         */
        function extendToNextCell(cols, rows) {

            var // current selection in the active sheet
                selection = view.getCellSelection(),
                // the current active range address
                activeRange = selection.ranges[selection.activeRange],
                // the new active range address
                newRange = _.copy(activeRange, true),
                // reference to the start or end address in the new active range
                address = null,
                // the new index inside the active range
                index = null;

            // find previous or next column index
            if (cols < 0) {
                address = (selection.activeCell[0] === activeRange.start[0]) ? newRange.end: newRange.start;
            } else if (cols > 0) {
                address = (selection.activeCell[0] === activeRange.end[0]) ? newRange.start : newRange.end;
            } else {
                address = null;
            }
            if (address && _.isNumber(index = findCollectionIndex(colCollection, address[0], cols))) {
                address[0] = index;
            }

            // find previous or next row index
            if (rows < 0) {
                address = (selection.activeCell[1] === activeRange.start[1]) ? newRange.end: newRange.start;
            } else if (rows > 0) {
                address = (selection.activeCell[1] === activeRange.end[1]) ? newRange.start : newRange.end;
            } else {
                address = null;
            }
            if (address && _.isNumber(index = findCollectionIndex(rowCollection, address[1], rows))) {
                address[1] = index;
            }

            // change the active range
            if (!_.isEqual(newRange, activeRange)) {
                view.changeActiveRange(newRange);
                self.scrollToCell([
                    (newRange.start[0] < activeRange.start[0]) ? newRange.start[0] : newRange.end[0],
                    (newRange.start[1] < activeRange.start[1]) ? newRange.start[1] : newRange.end[1]
                ]);
            }
        }

        /**
         * Moves the active cell in the current selection into the specified
         * direction. Cycles through all ranges contained in the selection. In
         * case the selection is a single cell, moves it in the entire sheet.
         *
         * @param {String} direction
         *  The direction the active cell will be moved to. Supported values
         *  are 'left', 'right', 'up', and 'down'.
         */
        function moveActiveCell(direction) {

            var // current selection in the active sheet
                selection = view.getCellSelection(),
                // whether the selection consists of a single cell
                singleCell = (selection.ranges.length === 1) && (SheetUtils.getCellCount(selection.ranges[0]) === 1),
                // the index of the cell address array element in primary direction
                index1 = ((direction === 'left') || (direction === 'right')) ? 0 : 1,
                // the index of the cell address array element in secondary direction
                index2 = 1 - index1,
                // the current active range address
                activeRange = null;

            // single cell selection: use entire sheet as 'selection'
            if (singleCell) {
                selection.ranges = [view.getSheetRange()];
                selection.activeRange = 0;
            }

            // get reference to the active range in the selection
            activeRange = selection.ranges[selection.activeRange];

            // move active cell forward
            // TODO: skip protected cells (horizontal only)
            if ((direction === 'right') || (direction === 'bottom')) {

                if (selection.activeCell[index1] < activeRange.end[index1]) {
                    selection.activeCell[index1] += 1;
                } else if (selection.activeCell[index2] < activeRange.end[index2]) {
                    selection.activeCell[index1] = activeRange.start[index1];
                    selection.activeCell[index2] += 1;
                } else {
                    selection.activeRange = (selection.activeRange + 1) % selection.ranges.length;
                    selection.activeCell = selection.ranges[selection.activeRange].start;
                }

            // move active cell backward
            // TODO: skip protected cells (horizontal only)
            } else {

                if (selection.activeCell[index1] > activeRange.start[index1]) {
                    selection.activeCell[index1] -= 1;
                } else if (selection.activeCell[index2] > activeRange.start[index2]) {
                    selection.activeCell[index1] = activeRange.end[index1];
                    selection.activeCell[index2] -= 1;
                } else {
                    selection.activeRange = (selection.activeRange + selection.ranges.length - 1) % selection.ranges.length;
                    selection.activeCell = selection.ranges[selection.activeRange].end;
                }
            }

            // select the cell or set the new selection, scroll to active cell
            if (singleCell) {
                view.selectCell(selection.activeCell);
            } else {
                view.setCellSelection(selection);
            }
            self.scrollToCell(selection.activeCell);
        }

        /**
         * Expands the selected ranges to entire columns or rows.
         *
         * @param {String} direction
         *  If set to 'columns', the current selection will be expanded to full
         *  columns. If set to 'rows', the current selection will be expanded
         *  to full rows.
         */
        function extendSelectionToLimits(direction) {

            var // current selection in the active sheet
                selection = view.getCellSelection(),
                // the range address of the entire sheet
                sheetRange = view.getSheetRange(),
                // the index of the cell address array element to be modified
                index = (direction === 'columns') ? 1 : 0;

            _(selection.ranges).each(function (range) {
                range.start[index] = 0;
                range.end[index] = sheetRange.end[index];
            });
            view.setCellSelection(selection);
        }

        /**
         * Starts the in-place edit mode for the active cell.
         *
         * @param {String} [text]
         *  The initial text to insert into the edit area. If omitted, the
         *  current value of the active cell will be inserted.
         */
        function enterEditMode(text) {

            var // the contents (display string, result, formula) of the acive cell
                contents = view.getActiveCellContents(),
                // the formatting attributes of the active cell
                attributes = view.getActiveCellAttributes();

            // first, check document edit mode
            if (!app.getModel().getEditMode()) {
                app.rejectEditAttempt();
                return;
            }

            // cancel current tracking
            $.cancelTracking();

            // check whether in-place edit mode is already active
            if (editSettings) { return; }

            // fade the sheet contents out
            scrollNode.addClass('fade-out');

            // first, scroll to the active cell
            self.scrollToCell(view.getActiveCell());

            // initialize edit mode settings
            editSettings = {
                sheet: view.getActiveSheet(),
                selection: view.getCellSelection(),
                value: _.isString(text) ? text : _.isString(contents.formula) ? contents.formula : contents.display,
                align: (attributes.cell.alignHor === 'auto') ? resolveAutoAlignment(contents.result) : attributes.cell.alignHor,
                wrap: attributes.cell.wrapText
            };

            // copy formatting and editing text from active cell to the text area
            textArea.css({
                fontFamily: documentStyles.getCssFontFamily(attributes.character.fontName),
                fontSize: attributes.character.fontSize + 'pt',
                textAlign: editSettings.align
            }).val(editSettings.value);

            // insert and activate the text area
            textArea.appendTo(rootNode).focus();

            // IE9 does not trigger 'input' events when deleting characters or
            // pasting text, use a timer interval as a workaround
            if (Utils.IE9) {
                textArea.data('lastValue', textArea.val());
                textArea.data('updateTimer', app.repeatDelayed(function () {
                    var value = textArea.val();
                    if (value !== textArea.data('lastValue')) {
                        textArea.data('lastValue', value);
                        updateTextAreaPosition();
                    }
                }, { delay: 250 }));
            }

            // set initial position, size, and selection of the text area
            updateTextAreaPosition();
            Utils.setTextFieldSelection(textArea, editSettings.value.length);
        }

        /**
         * Leaves the in-place edit mode for the active cell.
         *
         * @param {Boolean} [commitMode='discard']
         *  If set to 'cell', the current value will be committed to the active
         *  cell of the selection. If set to 'fill', the current value will be
         *  filled to all cells of the selection. If set to 'array', the
         *  current value will be inserted as array formula. Otherwise, the
         *  cell in the document will not be changed.
         */
        function leaveEditMode(commitMode) {

            // nothing to do, if in-place edit mode is not active
            if (!editSettings) { return; }

            // fade the sheet contents in
            scrollNode.removeClass('fade-out');

            // IE9 does not trigger 'input' events when deleting characters or pasting text,
            // remove the timer interval that has been started as a workaround
            if (Utils.IE9) {
                textArea.data('updateTimer').abort();
                textArea.data('updateTimer', null);
            }

            // send current value to document model
            switch (commitMode) {
            case 'cell':
                view.setCellContents(editSettings.sheet, editSettings.selection.activeCell, textArea.val());
                break;
            case 'fill':
                view.fillCellContents(editSettings.sheet, editSettings.selection, textArea.val());
                break;
            case 'array':
                // TODO
                break;
            }

            // back to cell selection mode
            editSettings = null;
            rootNode.focus();
            textArea.detach();
        }

        /**
         * Handles 'keydown' events of the root node. Adjusts the selection, or
         * performs other actions according to the current selection.
         */
        function keyDownHandler(event) {

            // do nothing while in-place edit mode is active
            if (editSettings) { return; }

            // no modifier keys: delete selection, move active cell
            // SHIFT key: extend active range
            if (KeyCodes.matchModifierKeys(event, { shift: null })) {

                switch (event.keyCode) {
                case KeyCodes.LEFT_ARROW:
                    (event.shiftKey ? extendToNextCell : selectNextCell)(-1, 0);
                    return false;
                case KeyCodes.RIGHT_ARROW:
                    (event.shiftKey ? extendToNextCell : selectNextCell)(1, 0);
                    return false;
                case KeyCodes.UP_ARROW:
                    (event.shiftKey ? extendToNextCell : selectNextCell)(0, -1);
                    return false;
                case KeyCodes.DOWN_ARROW:
                    (event.shiftKey ? extendToNextCell : selectNextCell)(0, 1);
                    return false;
                case KeyCodes.TAB:
                    moveActiveCell(event.shiftKey ? 'left' : 'right');
                    return false;
                case KeyCodes.ENTER:
                    moveActiveCell(event.shiftKey ? 'up' : 'down');
                    return false;
                }
            }

            // additional actions (without any modifier keys)
            if (!event.shiftKey && !event.altKey && !event.ctrlKey && !event.metaKey) {
                switch (event.keyCode) {
                case KeyCodes.PAGE_UP:
                    selectNextPage(false);
                    return false;
                case KeyCodes.PAGE_DOWN:
                    selectNextPage(true);
                    return false;
                case KeyCodes.F2:
                    enterEditMode();
                    return false;
                }
            }

            // convert selection to entire columns
            if (KeyCodes.matchKeyCode(event, 'SPACE', { ctrlOrMeta: true })) {
                extendSelectionToLimits('columns');
                return false;
            }

            // convert selection to entire rows
            if (KeyCodes.matchKeyCode(event, 'SPACE', { shift: true })) {
                extendSelectionToLimits('rows');
                return false;
            }
        }

        /**
         * Handles 'keypress' events of the root node. Starts the in-place edit
         * mode on-the-fly for valid Unicode characters.
         */
        function keyPressHandler(event) {
            // do not handle 'keypress' events bubbled up from the active in-place text
            // area, ignore key events where either CTRL/META or ALT is pressed, ignore
            // SPACE keys with any control keys (used as shortcuts for column/row selection)
            if (!editSettings && ((event.charCode > 32) && ((event.ctrlKey || event.metaKey) === event.altKey) || ((event.charCode === 32) && KeyCodes.matchModifierKeys(event)))) {
                enterEditMode(String.fromCharCode(event.charCode));
                return false;
            }
        }

        /**
         * Handles 'keydown' events in the text input control, while in-place
         * editing is active.
         */
        function editKeyDownHandler(event) {

            // quit in-place edit mode on ESCAPE key regardless of any modifier keys
            if (event.keyCode === KeyCodes.ESCAPE) {
                leaveEditMode();
                return;
            }

            // ENTER without modifier keys (but ignoring SHIFT): set value, move cell down/up
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null })) {
                leaveEditMode('cell');
                // let key event bubble up to move the cell cursor
                return;
            }

            // ENTER with CTRL or META key: fill value to all cells (no SHIFT), or insert array formula (SHIFT)
            if (KeyCodes.matchKeyCode(event, 'ENTER', { shift: null, ctrlOrMeta: true })) {
                leaveEditMode(event.shiftKey ? 'array' : 'fill');
                // do not let key event bubble up (no change of selection)
                return false;
            }

            // ENTER with ALT key without any other modifier keys: insert line break
            if (KeyCodes.matchKeyCode(event, 'ENTER', { alt: true })) {
                Utils.replaceTextInTextFieldSelection(textArea, '\n');
                updateTextAreaPosition();
                return false;
            }

            // ENTER with any other modifier keys: discard the entire event
            if (event.keyCode === KeyCodes.ENTER) {
                return false;
            }

            // TAB key without modifier keys (but ignoring SHIFT): set value, move cell to next/previous cell
            if (KeyCodes.matchKeyCode(event, 'TAB', { shift: null })) {
                leaveEditMode('cell');
                // let key event bubble up to move the cell cursor
                return;
            }

            // all other events will be processed by the text field
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the root DOM node of this grid pane.
         *
         * @returns {jQuery}
         *  The root node of this grid pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Sets the browser focus into this grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            (editSettings ? textArea : rootNode).focus();
            return this;
        };

        /**
         * Initializes the settings and layout of this grid pane.
         *
         * @param {Object} horSettings
         *  The view settings of the horizontal pane side (left or right).
         *
         * @param {Object} vertSettings
         *  The view settings of the vertical pane side (top or bottom).
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.initialize = function (horSettings, vertSettings, options) {

            // visibility
            rootNode.toggle(horSettings.visible && vertSettings.visible);

            // position and size
            rootNode.css({ left: horSettings.offset, top: vertSettings.offset, width: horSettings.size, height: vertSettings.size });

            // initialize the scroll node containing the scroll bars
            scrollNode.css({
                overflowX: horSettings.scrollable ? '' : 'hidden',
                overflowY: vertSettings.scrollable ? '' : 'hidden',
                bottom: (horSettings.scrollable && !vertSettings.showOppositeScroll) ? -Math.max(6, Utils.SCROLLBAR_HEIGHT) : 0,
                right: (vertSettings.scrollable && !horSettings.showOppositeScroll) ? -Math.max(6, Utils.SCROLLBAR_WIDTH) : 0
            });

            // initialize auto-scrolling
            scrollSizeNode.enableTracking({
                autoScroll: horSettings.scrollable ? (vertSettings.scrollable ? true : 'horizontal') : (vertSettings.scrollable ? 'vertical' : false),
                borderNode: rootNode,
                borderMargin: -30,
                borderSize: 60,
                minSpeed: 2,
                maxSpeed: 250
            });

            // other layout options
            hiddenWidth = horSettings.hiddenSize;
            hiddenHeight = vertSettings.hiddenSize;

            // update the grid contents
            requestUpdate();
            updateContentPosition();
            updateScrollAreaSize();

            return this;
        };

        /**
         * Requests an update for this grid pane from the view.
         *
         * @param {Array} [cells]
         *  A list of cell addresses with changed contents. Can be used to
         *  update the cell grid partly.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.requestUpdate = function (cells) {
            requestUpdate({ force: true, cells: cells });
            return this;
        };

        /**
         * Returns the position of the first pixel in the entire sheet visible
         * in this grid pane.
         *
         * @returns {Object}
         *  The position of the first visible pixel, in the properties 'left'
         *  and 'top', in pixels.
         */
        this.getTopLeftPosition = function () {
            return {
                left: view.getHorizontalHeaderPane(panePos).getFirstVisibleOffset(),
                top: view.getVerticalHeaderPane(panePos).getFirstVisibleIndex()
            };
        };

        /**
         * Returns the logical address of the top-left cell visible in this
         * grid pane. The cell is considered visible, it it's column and row
         * are at least half visible.
         *
         * @returns {Number[]|Null}
         *  The logical address of the top-left cell if available in the
         *  current layout data, otherwise null.
         */
        this.getTopLeftAddress = function () {

            var // get first visible column from horizontal header pane
                col = view.getHorizontalHeaderPane(panePos).getFirstVisibleIndex(),
                // get first visible row from vertical header pane
                row = view.getVerticalHeaderPane(panePos).getFirstVisibleIndex();

            // check that both indexes are valid (not negative)
            return ((col >= 0) && (row >= 0)) ? [col, row] : null;
        };

        /**
         * Changes the scroll position of this grid pane.
         *
         * @param {Number} left
         *  The new horizontal scroll position, in pixels.
         *
         * @param {Number} top
         *  The new vertical scroll position, in pixels.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scroll = function (left, top) {

            var // validate passed scroll position
                newScrollLeft = Math.max(0, Math.min(left, scrollSizeNode.width() - scrollNode[0].clientWidth)),
                newScrollTop = Math.max(0, Math.min(top, scrollSizeNode.height() - scrollNode[0].clientHeight));

            // Only set scroll position if it differs from current position.
            // This test MUST be done, otherwise Firefox scrolls extremely slow
            // when using the mouse wheel!
            if ((newScrollLeft !== scrollLeft) || (newScrollTop !== scrollTop)) {
                scrollLeft = newScrollLeft;
                scrollTop = newScrollTop;
                updateScrollPosition();
            }

            return this;
        };

        /**
         * Changes the horizontal scroll position of this grid pane.
         *
         * @param {Number} left
         *  The new horizontal scroll position, in pixels.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollLeft = function (left) {
            return this.scroll(left, scrollTop);
        };

        /**
         * Changes the vertical scroll position of this grid pane.
         *
         * @param {Number} top
         *  The new vertical scroll position, in pixels.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollTop = function (top) {
            return this.scroll(scrollLeft, top);
        };

        /**
         * Scrolls this grid pane to make the specified cell visible.
         *
         * @param {Number[]} address
         *  The logical cell position to be made visible.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToCell = function (address) {
            scrollToCell(address[0], address[1]);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified column visible.
         *
         * @param {Number} col
         *  The zero-based index of the column to be made visible.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToColumn = function (col) {
            scrollToCell(col, null);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified row visible.
         *
         * @param {Number} row
         *  The zero-based index of the row to be made visible.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToRow = function (row) {
            scrollToCell(null, row);
            return this;
        };

        // initialization -----------------------------------------------------

        // listen to layout update events from the view
        view.on({
            'change:activesheet': activeSheetHandler,
            'change:selection': selectionHandler,
            'update:grid': updateGridHandler,
            'resize:start': function () { leaveEditMode('cell'); }
        });

        // listen to scroll events and synchronize scroll position of other grid panes
        scrollNode.on('scroll', scrollHandler);

        // tracking for cell selection with mouse/touch
        scrollSizeNode.on({
            'tracking:start': trackingStartHandler,
            dblclick: doubleClickHandler
        });

        // keyboard events in grid pane (scrolling etc.)
        rootNode.on({
            keydown: keyDownHandler,
            keypress: keyPressHandler
        });

        // events for in-place edit mode
        textArea.on({
            keydown: editKeyDownHandler,
            input: updateTextAreaPosition
        });

        // leave in-place edit mode, if document goes into read-only mode
        app.getModel().on('change:editmode', function (event, editMode) {
            if (!editMode) {
                leaveEditMode();
            }
        });

    } // class GridPane

    // exports ================================================================

    return GridPane;

});
