/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/cellcollection', ['io.ox/office/tk/utils'], function (Utils) {

    'use strict';

    // class CellCollection ===================================================

    /**
     * Collects layout information about a range of visible cells.
     *
     * Each collection entry is an object with the following properties:
     * - {Number[]} address
     *      The logical cell address.
     * - {String} display
     *      The formatted display string of the cell value.
     * - {Number|String|Boolean|Null} result
     *      The cell value, or the result of a formula cell. Will be null for
     *      empty cells.
     * - {String} [formula]
     *      The formula definition.
     * - {Object} [attrs]
     *      The cell attribute set.
     *
     * @constructor
     */
    function CellCollection() {

        var // all cell entries, as map, mapped by column/row index (joined by a colon)
            map = null;

        // methods ------------------------------------------------------------

        /**
         * Clears this collection.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.clear = function () {
            map = {};
            return this;
        };

        /**
         * Initializes this collection with the cell data received from a
         * 'docs:update' event notification.
         *
         * @param {Object} gridPaneData
         *  The original layout data of a grid pane, as received from the
         *  'docs:update' event notification. Expects the property 'cells'.
         *
         * @returns {CellCollection}
         *  A reference to this instance.
         */
        this.initialize = function (gridPaneData) {

            var // the data array containing cell entries
                dataArray = Utils.getArrayOption(gridPaneData, 'cells', []);

            // initialize class members
            this.clear();

            // map all existing cells by their position
            _(dataArray).each(function (data) {
                map[data.pos[0] + ':' + data.pos[1]] = data;
            });

            return this;
        };

        /**
         * Calls the passed iterator function for all collection entries.
         *
         * @param {Function} iterator
         *  The iterator function called for all cell entries in this
         *  collection. Receives the cell entry as first parameter. If the
         *  iterator returns the Utils.BREAK object, the iteration process will
         *  be stopped immediately.
         *
         * @param {Object} [options]
         *  A map with options controlling the behavior of this method. The
         *  following options are supported:
         *  @param {Object} [options.context]
         *      If specified, the iterator will be called with this context
         *      (the symbol 'this' will be bound to the context inside the
         *      iterator function).
         *
         * @returns {Utils.BREAK|Undefined}
         *  A reference to the Utils.BREAK object, if the iterator has returned
         *  Utils.BREAK to stop the iteration process, otherwise undefined.
         */
        this.iterateEntries = function (iterator, options) {

            var context = Utils.getOption(options, 'context');

            return _(map).any(function (entry) {
                return iterator.call(context, entry) === Utils.BREAK;
            }) ? Utils.BREAK : undefined;
        };

        /**
         * Returns whether this collection is empty.
         *
         * @returns {Boolean}
         *  Whether this collection is empty.
         */
        this.isEmpty = function () {
            return _.isEmpty(map);
        };

        /**
         * Returns the cell entry with the specified cell address from the
         * collection.
         *
         * @param {Number} col
         *  The zero-based column index.
         *
         * @param {Number} row
         *  The zero-based row index.
         *
         * @returns {Object|Null}
         *  The cell entry; or null, if the entry does not exist.
         */
        this.getEntry = function (col, row) {
            return map[col + ':' + row] || null;
        };

        // initialization -----------------------------------------------------

        // initialize class members
        this.clear();

    } // class CellCollection

    // exports ================================================================

    return CellCollection;

});
