/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/sheetmodel',
    ['io.ox/core/event',
     'io.ox/office/tk/utils'
    ], function (Events, Utils) {

    'use strict';

    // class SheetModel =======================================================

    /**
     * Represents a single sheet in the spreadsheet document.
     *
     * @constructor
     *
     * @extends Events
     *
     * @param {SpreadsheetApplication} app
     *  The application instance with the document model containing this sheet.
     */
    function SheetModel(app) {

        var // the cell selection
            selection = {
                ranges: [{ start: [0, 0], end: [0, 0] }],
                activeRange: 0,
                activeCell: [0, 0]
            };

        // base constructor ---------------------------------------------------

        Events.extend(this);

        // methods ------------------------------------------------------------

        /**
         * Returns an object containing all information about the cell
         * selection in this sheet.
         *
         * @returns {Object}
         *  The cell selection data, in the properties 'ranges' (array of range
         *  addresses), 'activeRange' (array index of the active range in the
         *  'ranges' property), and 'activeCell' (logical address of the active
         *  cell in the active range).
         */
        this.getCellSelection = function () {
            return _.copy(selection, true);
        };

        /**
         * Changes the selection properties of this sheet, and triggers a
         * 'change:selection' event.
         *
         * @param {Object} selection
         *  The cell selection data, in the properties 'ranges' (array of
         *  logical range addresses), 'activeRange' (array index of the active
         *  range in the 'ranges' property), and 'activeCell' (logical address
         *  of the active cell in the active range).
         *
         * @returns {SheetModel}
         *  A reference to this instance.
         */
        this.setCellSelection = function (newSelection) {
            if (!_.isEqual(selection, newSelection)) {
                selection = _.copy(newSelection, true);
                this.trigger('change:selection', this, newSelection);
            }
            return this;
        };

        this.destroy = function () {
            this.events.destroy();
        };

    } // class SheetModel

    // exports ================================================================

    return SheetModel;

});
