/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/model',
    ['io.ox/office/tk/utils',
     'io.ox/office/framework/model/editmodel',
     'io.ox/office/spreadsheet/model/operations',
     'io.ox/office/spreadsheet/model/sheetmodel',
     'io.ox/office/spreadsheet/model/format/documentstyles'
    ], function (Utils, EditModel, Operations, SheetModel, SpreadsheetDocumentStyles) {

    'use strict';

    // class SpreadsheetModel =================================================

    /**
     * Represents a 'workbook', the spreadsheet document model. Implements
     * execution of all supported operations.
     *
     * Triggers the events supported by the base class EditModel, and the
     * following additional events:
     * - 'sheet:insert': After a new sheet has been inserted into the document.
     *      The event handler receives the zero-based index of the new sheet in
     *      the collection of all sheets.
     * - 'sheet:delete': After a sheet has been removed from the document. The
     *      event handler receives the zero-based index of the sheet in the
     *      collection of all sheets before it has been removed.
     * - 'sheet:move': After a sheet has been moved to a new position. The
     *      event handler receives the zero-based index of the new position in
     *      the collection of all sheets, and the zero-based index of the old
     *      position.
     *
     * @constructor
     *
     * @extends EditModel
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this document model.
     */
    function SpreadsheetModel(app) {

        var // self reference
            self = this,

            // all existing sheets and their names, by zero-based index
            sheetsCollection = [],

            // container for all style sheets of all attribute families
            documentStyles = new SpreadsheetDocumentStyles(app);

        // base constructor ---------------------------------------------------

        EditModel.call(this, app, Operations, documentStyles);

        // private methods ----------------------------------------------------

        /**
         * Returns the sheet info structure at the specified index.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {Object|Null}
         *  The info structure of the sheet at the passed index or with the
         *  passed name, containing the sheet name and the reference to the
         *  sheet model instance, or null if the passed index or name is
         *  invalid.
         */
        function getSheetInfo(sheet) {

            var // the resulting sheet info
                sheetInfo = null;

            if (_.isNumber(sheet)) {
                // get sheet info structure by index
                sheetInfo = sheetsCollection[sheet];
            } else if (_.isString(sheet)) {
                // get sheet info structure by name
                sheet = sheet.toLowerCase();
                sheetInfo = _(sheetsCollection).find(function (sheetInfo) { return sheetInfo.name.toLowerCase() === sheet; });
            }

            return _.isObject(sheetInfo) ? sheetInfo : null;
        }

        /**
         * Returns whether the passed sheet name is valid (non-empty and not
         * existing in the document).
         *
         * @param {String} sheetName
         *  The desired name of the sheet.
         *
         * @param {Number} [sheet]
         *  If specified, the zero-based index of the sheet that is allowed to
         *  contain the passed name (used while renaming an existing sheet).
         *
         * @returns {Boolean}
         *  Whether the sheet could be renamed to the specified name. The name
         *  must not be empty, and must not be equal to the name of any other
         *  sheet (case-insensitive).
         */
        function isSheetNameValid(sheetName, sheet) {

            var // an existing sheet with the passed name
                sheetModelByName = self.getSheetModel(sheetName),
                // the model of the specified sheet
                sheetModelByIndex = _.isNumber(sheet) ? self.getSheetModel(sheet) : null;

            // check name (no *other* sheet must contain this name)
            return (sheetName.length > 0) && (!_.isObject(sheetModelByName) || !_.isObject(sheetModelByIndex) || (sheetModelByName === sheetModelByIndex));
        }

        // methods ------------------------------------------------------------

        /**
         * Returns the number of sheets in the spreadsheet document.
         *
         * @returns {Number}
         *  The number of sheets in the document.
         */
        this.getSheetCount = function () {
            return sheetsCollection.length;
        };

        /**
         * Returns the names of all sheets in the spreadsheet document.
         *
         * @returns {String[]}
         *  The names of all sheets in the document.
         */
        this.getSheetNames = function () {
            return _(sheetsCollection).pluck('name');
        };

        /**
         * Returns whether a sheet at the specified index or with the specified
         * case-insensitive name exists in the spreadsheet document.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {Boolean}
         *  Whether a sheet exists at the passed index or with the passed name.
         */
        this.hasSheet = function (sheet) {
            return _.isObject(getSheetInfo(sheet));
        };

        /**
         * Returns the exact name of the sheet at the specified index or with
         * the specified case-insensitive name.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {String|Null}
         *  The name of the sheet at the passed index or with the passed name,
         *  or null if the passed index or name is invalid.
         */
        this.getSheetName = function (sheet) {
            var sheetInfo = getSheetInfo(sheet);
            return _.isObject(sheetInfo) ? sheetInfo.name : null;
        };

        /**
         * Returns the model instance for the sheet at the specified index or
         * with the specified name.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {SheetModel|Null}
         *  The model instance for the sheet at the passed index or with the
         *  specified name, or null if the passed index or name is invalid.
         */
        this.getSheetModel = function (sheet) {
            var sheetInfo = getSheetInfo(sheet);
            return _.isObject(sheetInfo) ? sheetInfo.model : null;
        };

        /**
         * Creates a new sheet in this document.
         *
         * @param {Number} sheet
         *  The zero-based insertion index of the sheet.
         *
         * @param {String} sheetName
         *  The name of the new sheet. Must not be empty. Must not be equal to
         *  the name of any existing sheet.
         *
         * @param {Object} [attrs]
         *  Attribute set containing initial formatting attributes for the new
         *  sheet.
         *
         * @returns {SheetModel|Null}
         *  The model instance of the new sheet.
         */
        this.insertSheet = function (sheet, sheetName, attrs) {
            // remove all NPCs from passed sheet name
            sheetName = Utils.cleanString(sheetName);
            // check that the sheet can be named as specified (prevent internal application error on failing operation)
            if (!isSheetNameValid(sheetName)) { return null; }
            // generate and apply the operation
            return this.applyOperations({ name: Operations.INSERT_SHEET, sheet: sheet, sheetName: sheetName, attrs: attrs }) ? this.getSheetModel(sheet) : null;
        };

        /**
         * Removes a sheet from this document.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @returns {Boolean}
         *  Whether deleting the sheet was successful.
         */
        this.deleteSheet = function (sheet) {
            // check that the sheet can be removed (prevent internal application error on failing operation)
            return (this.getSheetCount() > 1) && this.applyOperations({ name: Operations.DELETE_SHEET, sheet: sheet });
        };

        /**
         * Renames a sheet in the spreadsheet document.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet to be renamed.
         *
         * @param {String} sheetName
         *  The new name of the sheet. Must not be empty. Must not be equal to
         *  the name of any other sheet (case-insensitive).
         *
         * @returns {Boolean}
         *  Whether renaming the sheet was successful.
         */
        this.setSheetName = function (sheet, sheetName) {
            // remove all NPCs from passed sheet name
            sheetName = Utils.trimAndCleanString(sheetName);
            // check that the sheet can be renamed (prevent internal application error on failing operation)
            if (!isSheetNameValid(sheetName, sheet)) { return false; }
            // do not generate the operation, if passed name matches old name
            if (this.getSheetName(sheet) === sheetName) { return true; }
            // generate and apply the operation
            return this.applyOperations({ name: Operations.SET_SHEET_NAME, sheet: sheet, sheetName: sheetName });
        };

        /**
         * Adding cells, rows or columns.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @param {Number[]} start
         *  The logical cell position of the upper-left cell in the range.
         *
         * @param {Number[]} [end]
         *  The logical cell position of the bottom-right cell in the range. If
         *  omitted, the operation addresses a single cell.
         *
         * @param {String} direction
         *  'row or 'column', determines if rows or columns are inserted.
         *
         * @param {Boolean} full
         *  If 'true', complete rows or columns are inserted. Otherwise the
         *  addressed cell range is moved.
         *
         * @param {Number} [count]
         *  Number of rows or columns (optional) that will be inserted.
         *
         * @returns {Boolean}
         *  Whether the operation was successful.
         */
        this.insertCells = function (sheet, start, end, direction, full, count) {
            return this.applyOperations({ name: Operations.INSERT_CELLS, sheet: sheet, start: start, end: end, direction: direction, full: full, count: count });
        };

        /**
         * Deleting cells, rows or columns.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @param {Number[]} start
         *  The logical cell position of the upper-left cell in the range.
         *
         * @param {Number[]} [end]
         *  The logical cell position of the bottom-right cell in the range. If
         *  omitted, the operation addresses a single cell.
         *
         * @param {String} direction
         *  'row or 'column', determines if rows or columns are deleted.
         *
         * @param {Boolean} full
         *  If 'true', complete rows or columns are deleted. Otherwise the
         *  addressed cell range is filled with moved cells from right/bottom.
         *
         * @param {Number} [count]
         *  Number of rows or columns (optional) that will be deleted.
         *
         * @returns {Boolean}
         *  Whether the operation was successful.
         */
        this.deleteCells = function (sheet, start, end, direction, full, count) {
            return this.applyOperations({ name: Operations.DELETE_CELLS, sheet: sheet, start: start, end: end, direction: direction, full: full, count: count });
        };

        /**
         * Fills a range of cells with the same value and/or formatting
         * attributes.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @param {Object} range
         *  The logical address of the cell range, with 'start' and 'end'
         *  properties containing logical cell addresses.
         *
         * @param {CellValue} [value]
         *  The value used to fill the specified cell range. The value null
         *  will clear the cell range. If omitted, the current values will not
         *  be changed.
         *
         * @param {Object} [attrs]
         *  The attribute set containing all attributes to be changed.
         *  Attributes with a value of 'null' will be removed from the cell
         *  range. Attributes not contained in the set remain unmodified.
         *
         * @param {Object} [options]
         *  Additional properties that will be inserted into the generated
         *  'fillCellRange' operation. May contain any additional property
         *  supported by this operation.
         *
         * @returns {Boolean}
         *  Whether the operation was successful.
         */
        this.fillCellRange = function (sheet, range, value, attrs, options) {
            var operation = { name: Operations.FILL_CELL_RANGE, sheet: sheet, start: range.start };
            if (!_.isEqual(range.start, range.end)) { operation.end = range.end; }
            if (!_.isUndefined(value)) { operation.value = value; }
            if (_.isObject(attrs)) { operation.attrs = attrs; }
            return this.applyOperations(_(operation).extend(options));
        };

        /**
         * Overwrites a single cell, or a range of cells with the specified
         * values and/or formatting attributes.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet.
         *
         * @param {Number[]} start
         *  The logical address of a single cell, or the upper-left cell in a
         *  cell range.
         *
         * @param {Object|Object[][]} contents
         *  The values and attribute sets to be written into the cell or range
         *  of cells. If this parameter is an object, a single cell will be
         *  modified. Otherwise, this parameter must be a 2-dimensional array.
         *  The outer array contains rows of cell contents, and the inner row
         *  arrays contain the cell contents for each single row. The lengths
         *  of the inner arrays may be different. Cells not covered by a row
         *  array will not be modified.
         *
         * @param {Object} [options]
         *  Additional properties that will be inserted into the generated
         *  'setCellContents' operation. May contain any additional property
         *  supported by this operation.
         *
         * @returns {Boolean}
         *  Whether the operation was successful.
         */
        this.setCellContents = function (sheet, start, contents, options) {
            return this.applyOperations(_.extend({
                name: Operations.SET_CELL_CONTENTS,
                sheet: sheet,
                start: start,
                // convert single object to a 1x1 array
                contents: _.isArray(contents) ? contents : [[contents]]
            }, options));
        };

        /**
         * Changes the formatting attributes of a range of rows in a sheet.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet containing the rows.
         *
         * @param {Number} start
         *  The zero-based index of the first row in the row range.
         *
         * @param {Number} [end]
         *  The zero-based index of the last row in the row range. If omitted,
         *  the operation addresses a single row.
         *
         * @param {Object} attrs
         *  The attribute set for the rows, containing all attributes to be
         *  changed. Attributes with a value of 'null' will be removed from the
         *  rows. Attributes not contained in the set remain unmodified.
         *
         * @returns {Boolean}
         *  Whether the operation was successful.
         */
        this.setRowAttributes = function (sheet, start, end, attrs) {
            return this.applyOperations({ name: Operations.SET_ROW_ATTRIBUTES, sheet: sheet, start: start, end: end, attrs: attrs });
        };

        /**
         * Changes the formatting attributes of a range of columns in a sheet.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet containing the columns.
         *
         * @param {Number} start
         *  The zero-based index of the first column in the column range.
         *
         * @param {Number} [end]
         *  The zero-based index of the last column in the column range. If
         *  omitted, the operation addresses a single column.
         *
         * @param {Object} attrs
         *  The attribute set for the columns, containing all attributes to be
         *  changed. Attributes with a value of 'null' will be removed from the
         *  columns. Attributes not contained in the set remain unmodified.
         *
         * @returns {Boolean}
         *  Whether the operation was successful.
         */
        this.setColumnAttributes = function (sheet, start, end, attrs) {
            return this.applyOperations({ name: Operations.SET_COLUMN_ATTRIBUTES, sheet: sheet, start: start, end: end, attrs: attrs });
        };

        // initialization -----------------------------------------------------

        this.registerOperationHandler(Operations.INSERT_SHEET, function (operation) {

            var // the sheet index passed in the operation
                sheet = Utils.getIntegerOption(operation, 'sheet', -1),
                // the sheet name passed in the operation
                sheetName = Utils.getStringOption(operation, 'sheetName', ''),
                // the sheet attributes passed in the operation
                attrs = Utils.getObjectOption(operation, 'attrs'),
                // the model of the new sheet
                sheetModel = null;

            // check the sheet index and name (must not exist yet)
            if ((sheet < 0) || (sheet > this.getSheetCount()) || !isSheetNameValid(sheetName)) {
                return false;
            }

            // create the corresponding 'deleteSheet' undo operation
            if (this.isUndoEnabled()) {
                this.addUndo({ name: Operations.DELETE_SHEET, sheet: sheet }, operation);
            }

            // create and insert the new sheet
            sheetModel = new SheetModel(app);
            sheetsCollection.splice(sheet, 0, { model: sheetModel, name: sheetName });
            self.trigger('sheet:insert', sheet);
            return true;
        });

        this.registerOperationHandler(Operations.DELETE_SHEET, function (operation) {

            var // the sheet index passed in the operation
                sheet = Utils.getIntegerOption(operation, 'sheet', -1),
                // the sheet object
                sheetModel = this.getSheetModel(sheet);

            // the sheet must exist, there must remain at least one sheet in the document
            if (!_.isObject(sheetModel) || (this.getSheetCount() === 1)) {
                return false;
            }

            // create the corresponding 'insertSheet' undo operation
            if (this.isUndoEnabled()) {
                this.addUndo({ name: Operations.INSERT_SHEET, sheet: sheet, sheetName: this.getSheetName(sheet) }, operation);
            }

            // finally, remove the sheet from the document model
            sheetModel.destroy();
            sheetsCollection.splice(sheet, 1);
            this.trigger('sheet:delete', sheet);
            return true;
        });

        this.registerOperationHandler(Operations.MOVE_SHEET, function (operation) {
        });

        this.registerOperationHandler(Operations.SET_SHEET_NAME, function (operation) {

            var // the sheet index passed in the operation
                sheet = Utils.getIntegerOption(operation, 'sheet', -1),
                // the sheet object
                sheetInfo = getSheetInfo(sheet),
                // the sheet name passed in the operation
                sheetName = Utils.getStringOption(operation, 'sheetName', '');

            // the sheet must exist, check name (no other sheet must contain this name)
            if (!_.isObject(sheetInfo) || !isSheetNameValid(sheetName, sheet)) {
                return false;
            }

            // no undo action and no events, if passed name matches old name
            if (sheetInfo.name === sheetName) {
                return true;
            }

            // create the corresponding 'setSheetName' undo operation
            if (this.isUndoEnabled()) {
                this.addUndo({ name: Operations.SET_SHEET_NAME, sheet: sheet, sheetName: sheetInfo.name }, operation);
            }

            // rename the sheet
            sheetInfo.name = sheetName;
            this.trigger('sheet:name', sheet, sheetName);
            return true;
        });

        this.registerOperationHandler(Operations.SET_SHEET_ATTRIBUTES, function (operation) {
        });

        // operations to be ignored silently
        this.registerOperationHandler(Operations.SET_ROW_ATTRIBUTES, $.noop);
        this.registerOperationHandler(Operations.SET_COLUMN_ATTRIBUTES, $.noop);
        this.registerOperationHandler(Operations.INSERT_CELLS, $.noop);
        this.registerOperationHandler(Operations.DELETE_CELLS, $.noop);
        this.registerOperationHandler(Operations.FILL_CELL_RANGE, $.noop);
        this.registerOperationHandler(Operations.SET_CELL_CONTENTS, $.noop);

    } // class SpreadsheetModel

    // exports ================================================================

    // derive this class from class EditModel
    return EditModel.extend({ constructor: SpreadsheetModel });

});
