/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/app/controller',
    ['io.ox/office/tk/utils',
     'io.ox/office/framework/model/format/border',
     'io.ox/office/framework/app/editcontroller',
     'io.ox/office/spreadsheet/utils/sheetutils'
    ], function (Utils, Border, EditController, SheetUtils) {

    'use strict';

    // class SpreadsheetController ============================================

    /**
     * The controller of a OX Spreadsheet application.
     *
     * @constructor
     *
     * @extends EditController
     */
    function SpreadsheetController(app) {

        var // self reference
            self = this,

            // the spreadsheet model
            model = null,

            // the spreadsheet view
            view = null,

            // all the little controller items
            items = {

                // view -------------------------------------------------------

                'view/sheet/active': {
                    get: function () { return view.getActiveSheet(); },
                    set: function (sheet) { view.setActiveSheet(sheet); }
                },

                'view/sheet/prev': {
                    enable: function () { return view.getActiveSheet() > 0; },
                    set: function (sheet) { view.setActiveSheet(view.getActiveSheet() - 1); },
                    shortcut: { keyCode: 'PAGE_UP', ctrlOrMeta: true, alt: true }
                },

                'view/sheet/next': {
                    enable: function () { return view.getActiveSheet() + 1 < model.getSheetCount(); },
                    set: function (sheet) { view.setActiveSheet(view.getActiveSheet() + 1); },
                    shortcut: { keyCode: 'PAGE_DOWN', ctrlOrMeta: true, alt: true }
                },

                'view/attributes': {
                    parent: 'document/editable',
                    get: function () { return view.getActiveCellAttributes(); }
                },

                'view/subtotals/sum': {
                    get: function () { return 0; }
                },

                'view/split': {
                    get: function () { return view.getSplitMode(); },
                    set: function (state) { view.setSplitMode(state); }
                },

                'view/freeze': {
                    get: function () { return view.getFreezeMode(); },
                    set: function (state) { view.setFreezeMode(state); }
                },

                // sheet operations -------------------------------------------

                'sheet/insert': {
                    parent: 'document/editable',
                    set: function () { view.insertSheet(); }
                },

                'sheet/delete': {
                    parent: 'document/editable',
                    enable: function () { return model.getSheetCount() > 1; },
                    set: function () { view.deleteSheet(); }
                },

                'sheet/name': {
                    parent: 'document/editable',
                    get: function () { return model.getSheetName(view.getActiveSheet()); },
                    set: function (sheetName) { view.renameSheet(sheetName); }
                },

                // column operations ------------------------------------------

                'column/insert': {
                    parent: 'document/editable',
                    set: function () { view.insertColumns(); }
                },

                'column/delete': {
                    parent: 'document/editable',
                    set: function () { view.deleteColumns(); }
                },

                'column/attributes': {
                    parent: 'view/attributes',
                    get: function (attributes) { return attributes.column; }
                },

                'column/visible': {
                    parent: 'column/attributes',
                    get: function (attributes) { return attributes.visible; }
                },

                'column/width': {
                    parent: 'column/attributes',
                    get: function (attributes) { return attributes.width; },
                    set: function (value) { view.setColumnWidthToSelection(value); }
                },

                // row operations ---------------------------------------------

                'row/insert': {
                    parent: 'document/editable',
                    set: function () { view.insertRows(); }
                },

                'row/delete': {
                    parent: 'document/editable',
                    set: function () { view.deleteRows(); }
                },

                'row/attributes': {
                    parent: 'view/attributes',
                    get: function (attributes) { return attributes.row; }
                },

                'row/visible': {
                    parent: 'row/attributes',
                    get: function (attributes) { return attributes.visible; }
                },

                'row/height': {
                    parent: 'row/attributes',
                    get: function (attributes) { return attributes.height; },
                    set: function (value) { view.setRowHeightToSelection(value); }
                },

                // cell attribute operations ----------------------------------

                'cell/attributes': {
                    parent: 'view/attributes',
                    get: function (attributes) { return attributes.cell; }
                },

                'cell/fillcolor': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return attributes.fillColor; },
                    set: function (color) { view.setCellAttribute('fillColor', color); }
                },

                'cell/borders': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return Border.getCellBorderModeFromAttributes(attributes); },
                    options: function () {
                        var ranges = view.getSelectedRanges();
                        return {
                            showInsideHor: _(ranges).any(function (range) { return SheetUtils.getRowCount(range) > 1; }),
                            showInsideVert: _(ranges).any(function (range) { return SheetUtils.getColCount(range) > 1; })
                        };
                    },
                    set: function (cellBorderMode) { view.setCellAttributes({ cell: Border.getAttributesFromCellBorderMode(cellBorderMode, view.getActiveCellAttributes().cell) }, { rangeBorders: true }); }
                },

                'cell/alignhor': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return attributes.alignHor; },
                    set: function (alignment) { view.setCellAttribute('alignHor', alignment); }
                },

                'cell/alignvert': {
                    parent: 'cell/attributes',
                    get: function (attributes) { return attributes.alignVert; },
                    set: function (alignment) { view.setCellAttribute('alignVert', alignment); }
                },

                'cell/resetAttributes': {
                    parent: 'cell/attributes',
                    set: function () { view.clearCellAttributes(); }
                },

                // cell border attribute operations ---------------------------

                'cell/border/enabled': {
                    parent: 'cell/attributes',
                    enable: function () {
                        var attributes = this.get();
                        return Border.isVisibleBorder(attributes.borderTop) ||
                            Border.isVisibleBorder(attributes.borderBottom) ||
                            Border.isVisibleBorder(attributes.borderLeft) ||
                            Border.isVisibleBorder(attributes.borderRight) ||
                            Border.isVisibleBorder(attributes.borderInsideHor) ||
                            Border.isVisibleBorder(attributes.borderInsideVert);
                    }
                },

                'cell/borderwidth': {
                    parent: 'cell/border/enabled',
                    get: function (attributes) { return view.getCellBorderWidthFromAttributes(attributes); },
                    set: function (value) { view.setCellAttributes({ cell: view.getAttributesFromCellBorderWidth(value) }); }
                },

                'cell/bordercolor': {
                    parent: 'cell/border/enabled',
                    get: function (attributes) { return view.getCellBorderColorFromAttributes(attributes); },
                    set: function (value) { view.setCellAttributes({ cell: view.getAttributesFromCellBorderColor(value) }); }
                },

                'cell/borderstyle': {
                    parent: 'cell/border/enabled',
                    get: function (attributes) { return view.getCellBorderStyleFromAttributes(attributes); },
                    set: function (value) { view.setCellAttributes({ cell: view.getAttributesFromCellBorderStyle(value) }, { visibleBorders: true }); }
                },

                // character attribute operations -----------------------------

                'character/attributes': {
                    parent: 'view/attributes',
                    get: function (attributes) { return attributes.character; }
                },

                'character/bold': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.bold; },
                    set: function (state) { view.setCharacterAttribute('bold', state); }
                },

                'character/italic': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.italic; },
                    set: function (state) { view.setCharacterAttribute('italic', state); }
                },

                'character/underline': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.underline; },
                    set: function (state) { view.setCharacterAttribute('underline', state); }
                },

                'character/strike': {
                    parent: 'character/attributes',
                    get: function (attributes) { return _.isString(attributes.strike) ? (attributes.strike === 'single' || attributes.strike === 'double') : null; },
                    set: function (state) { view.setCharacterAttribute('strike', state ? 'single' : 'none'); }
                },

                'character/color': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.color; },
                    set: function (color) { view.setCharacterAttribute('color', color); }
                },

                'character/fontname': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.fontName; },
                    set: function (fontName) { view.setCharacterAttribute('fontName', fontName); }
                },

                'character/fontsize': {
                    parent: 'character/attributes',
                    get: function (attributes) { return attributes.fontSize; },
                    set: function (fontSize) { view.setCharacterAttribute('fontSize', fontSize); }
                }

                // debug ------------------------------------------------------

            };

        // base constructor ---------------------------------------------------

        EditController.call(this, app);

        // initialization -----------------------------------------------------

        // register item definitions
        this.registerDefinitions(items);

        // initialization after construction
        app.on('docs:init', function () {

            // model and view are not available at construction time
            model = app.getModel();
            view = app.getView();

            // update GUI after changed selection, and after receiving view updates
            view.on('change:selection', function () { self.update(); })
                .on('update:grid', function (event, changedFlags) { if (changedFlags.selection) { self.update(); } });
        });

    } // class SpreadsheetController

    // exports ================================================================

    // derive this class from class EditController
    return EditController.extend({ constructor: SpreadsheetController });

});
