/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) Open-Xchange Inc., 2006-2012
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/framework/model/format/border',
    ['io.ox/office/tk/utils',
     'io.ox/office/framework/model/format/color'
    ], function (Utils, Color) {

    'use strict';

    var // map operation line styles to CSS line styles (anything else maps to 'solid')
        CSS_LINE_STYLES = {
            none: 'none',
            double: 'double',
            triple: 'double',
            dashed: 'dashed',
            dashSmallGap: 'dashed',
            dotted: 'dotted',
            dotDash: 'dotted',
            dotDotDash: 'dotted',
            dashDotStroked: 'dotted'
        },

        // maximum line width for borders in GUI preview elements, in pixels
        MAX_PREVIEW_LINEWIDTH = 2,

        // minimum line width for borders with style 'double', in pixels
        MIN_DOUBLE_LINEWIDTH = 3,

        // the names of all border properties in cells, mapped by border mode properties
        CELL_BORDER_ATTRIBUTES = { left: 'borderLeft', right: 'borderRight', top: 'borderTop', bottom: 'borderBottom', insideh: 'borderInsideHor', insidev: 'borderInsideVert' };

    // static class Border ====================================================

    /**
     * Predefined border objects.
     */
    var Border = {
            NONE: { style: 'none' },
            SINGLE: { style: 'single', width: 17, space: 140, color: Color.AUTO }
        };

    /**
     * Returns the best CSS border style that can represent the passed line
     * style used in border attributes.
     *
     * @param {String} style
     *  The line style used in border attributes in operations.
     *
     * @returns {String}
     *  The best matching CSS border style.
     */
    Border.getCssBorderStyle = function (style) {
        return (style in CSS_LINE_STYLES) ? CSS_LINE_STYLES[style] : 'solid';
    };

    /**
     * Returns the effective CSS attributes for the passed border value.
     *
     * @param {Object} border
     *  The border object as used in operations.
     *
     * @param {Theme} theme
     *  The theme object used to map scheme color names to color values.
     *
     * @param {Object} [options]
     *  A map of options controlling the operation. Supports the following
     *  options:
     *  @param {Boolean} [options.preview=false]
     *      If set to true, the border will be rendered for a preview element
     *      in the GUI. The border width will be restricted to two pixels in
     *      that case.
     *
     * @returns {Object}
     *  A map with CSS border attributes. Contains the property 'style' with
     *  the effective CSS border style as string; the property 'width' with the
     *  effective border width in pixels (as number, rounded to entire pixels);
     *  and the property 'color' with the effective CSS color (as string with
     *  leading hash sign).
     */
    Border.getCssBorderAttributes = function (border, theme, options) {

        var style = Utils.getStringOption(border, 'style', 'none'),
            width = Utils.getIntegerOption(border, 'width', 0),
            color = Utils.getObjectOption(border, 'color', Color.AUTO),
            preview = Utils.getBooleanOption(options, 'preview', false);

        // convert operation line styles to CSS styles
        style = Border.getCssBorderStyle(style);

        // convert 1/100mm to pixels (round to one digit)
        width = Utils.convertHmmToLength(width, 'px', 0.1);

        // minimum width for borders with style 'double', maximum width for preview borders
        if ((style === 'double') && (width > 0)) {
            width = preview ? MIN_DOUBLE_LINEWIDTH : Math.max(width, MIN_DOUBLE_LINEWIDTH);
        } else if (preview) {
            width = Math.min(width, MAX_PREVIEW_LINEWIDTH);
        }

        // round to entire pixels
        if (width > 0) {
            width = Math.max(Math.round(width), 1);
        }

        // convert color object to CSS color
        color = Color.getCssColor(color, 'line', theme);

        return { style: style, width: width, color: color };
    };

    /**
     * Converts the passed border attribute object to a CSS border value.
     *
     * @param {Object} border
     *  The border object as used in operations.
     *
     * @param {Theme} theme
     *  The theme object used to map scheme color names to color values.
     *
     * @param {Object} [options]
     *  A map of options controlling the operation. Supports the following
     *  options:
     *  @param {Boolean} [options.clearNone=false]
     *      If set to true, the return value for invisible borders will be the
     *      empty string instead of the keyword 'none'.
     *  @param {Boolean} [options.preview=false]
     *      If set to true, the border will be rendered for a preview element
     *      in the GUI. The border width will be restricted to two pixels in
     *      that case.
     *
     * @returns {String}
     *  The CSS border value converted from the passed border object.
     */
    Border.getCssBorder = function (border, theme, options) {
        var visible = Border.isVisibleBorder(border),
            attributes = visible ? Border.getCssBorderAttributes(border, theme, options) : null;
        return visible ? (attributes.style + ' ' + attributes.width + 'px ' + attributes.color) :
            Utils.getBooleanOption(options, 'clearNone', false) ? '' : 'none';
    };

    /**
     * Returns whether the passed border object is a valid and visible border
     * style (line style different to 'none', and line width greater than 0).
     *
     * @param {Object} border
     *  The border object as used in operations.
     */
    Border.isVisibleBorder = function (border) {
        return _.isObject(border) && _.isString(border.style) && (border.style !== 'none') && _.isNumber(border.width) && (border.width > 0);
    };

    /**
     * Converts the passed cell attributes to an object describing the
     * visibility states of all border attributes.
     *
     * @param {Object} attributes
     *  The table/cell attributes.
     *
     * @returns {Object}
     *  An object with position keys ('left', 'top', etc.), and Boolean values
     *  or Null values specifying whether the respective borders are visible.
     *  If a border attribute in the passed attribute map is ambiguous (value
     *  null or missing), the map value will be null.
     */
    Border.getCellBorderModeFromAttributes = function (attributes) {

        var // result mapping border attribute names to boolean values
            borderMode = {};

        _(CELL_BORDER_ATTRIBUTES).each(function (attrName, propName) {
            borderMode[propName] = _.isObject(attributes[attrName]) ? Border.isVisibleBorder(attributes[attrName]) : null;
        });

        return borderMode;
    };

    /**
     * Converts the passed object describing the visibility states of cell
     * border attributes to a table attribute map containing all border
     * attributes.
     *
     * @param {Object} borderMode
     *  An object with position keys ('left', 'top', etc.), and Boolean values
     *  specifying whether the respective borders are visible. Attributes not
     *  contained in this object will not occur in the returned attribute set.
     *  If the passed object contains a single property set to true, the
     *  current visibility state of the respective border attribute will be
     *  toggled.
     *
     * @param {Object} [origAttributes]
     *  The original attributes of the target object (table, cell, ...).
     *
     * @returns {Object}
     *  The table/cell attributes containing all border attributes that are
     *  mentioned in the passed object.
     */
    Border.getAttributesFromCellBorderMode = function (borderMode, origAttributes) {

        var // the resulting table attributes
            attributes = {},
            // whether to toggle the visibility of a single border
            toggle = _.size(borderMode) === 1;

        _(borderMode).each(function (visible, propName) {
            var borderAttrName = CELL_BORDER_ATTRIBUTES[propName];
            visible = toggle ? !Border.isVisibleBorder(origAttributes[borderAttrName]) : visible;
            attributes[borderAttrName] = visible ? Border.SINGLE : Border.NONE;

            // saving existing values for color and width and style
            if (origAttributes[borderAttrName].color) { attributes[borderAttrName].color = origAttributes[borderAttrName].color; }
            if (origAttributes[borderAttrName].width) { attributes[borderAttrName].width = origAttributes[borderAttrName].width; }

            if (visible) {
                if ((origAttributes[borderAttrName].style && origAttributes[borderAttrName].style !== 'none')) { attributes[borderAttrName].style = origAttributes[borderAttrName].style; }
            }
        });

        return attributes;
    };


    // exports ================================================================

    return Border;

});
