/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the Open-Xchange, Inc. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2004-2010 Open-Xchange, Inc.
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.mimemail;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import javax.activation.DataHandler;
import javax.mail.Header;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Part;
import javax.mail.Session;
import javax.mail.internet.AddressException;
import javax.mail.internet.InternetHeaders;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimePart;
import javax.mail.internet.MimeUtility;
import javax.mail.util.ByteArrayDataSource;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.exception.OXException;
import com.openexchange.mail.mime.ContentDisposition;
import com.openexchange.mail.mime.ContentType;
import com.openexchange.mail.mime.QuotedInternetAddress;
import com.openexchange.usm.json.ConnectorBundleErrorCodes;
import com.openexchange.usm.json.USMJSONAPIException;
import com.openexchange.usm.json.response.ResponseStatusCode;
import com.openexchange.usm.util.JSONToolkit;
import com.openexchange.usm.util.Toolkit;

/**
 * This class converts the JSON structure with information about the mail to the mime mail. 
 * The JAVA Mail API is used for this aim.
 * @author ibr
 *
 */
public class MimeMailBuilder {

	private static final String X_ORIGINAL_HEADERS = "x-original-headers";

	private static final String READ_DATE = "read-date";

	private static final String UTF_8 = "UTF-8";

	private static final String PERSONAL = "personal";

	private static final String DISPOSITION_NOTIFICATION_TO_ADDRESSLIST_HEADER = "disposition-notification-to";
	private static final String RESENT_SENDER_ADDRESSLIST_HEADER = "resent-sender";
	private static final String RESENT_TO_ADDRESSLIST_HEADER = "resent-to";
	private static final String RESENT_FROM_ADDRESSLIST_HEADER = "resent-from";
	private static final String RESENT_CC_ADDRESSLIST_HEADER = "resent-cc";
	private static final String RESENT_BCC_ADDRESSLIST_HEADER = "resent-bcc";
	private static final String ERRORS_TO_ADDRESSLIST_HEADER = "errors-to";
	private static final String SENDER_ADDRESSLIST_HEADER = "sender";
	private static final String REPLY_TO_ADDRESSLIST_HEADER = "reply-to";

	private static final String ADDRESS = "address";

	private static final String CONTENTTYPE_PARAMS = "params";

	private static final String CONTENTTYPE_TYPE = "type";

	private static final String terminalSeq = "\r\n";

	private static final String BODY = "body";
	private static final String DATA = "data";

	private static final String SMIME_BODY_DATA = "smime_body_data";

	private static final String EMPTY_STR = "";

	private static final char SEMICOLON = ';';

	private static final String CONTENT_TYPE_UPPER = "Content-Type";
	private static final String CONTENT_TRANSFER_ENCODING_UPPER = "Content-Transfer-Encoding";

	private static final String CONTENT_TYPE = "content-type";
	private static final String CONTENT_TRANSFER_ENCODING = "content-transfer-encoding";
	private static final String CONTENT_DISPOSITION = "content-disposition";
	private static final String BASE64_ENCODING = "base64";
	//private static final String SEVENBIT_ENCODING = "7bit";

	private static final String MIME_VERSION = "mime-version";
	private static final String RECEIVED = "received";
	private static final String RETURN_PATH = "return-path";
	private static final String X_MAILER = "x-mailer";
	private static final String X_SIEVE = "x-sieve";

	private static final String FROM_ADDRESSLIST_HEADER = "from";
	private static final String TO_ADDRESSLIST_HEADER = "to";
	private static final String CC_ADDRESSLIST_HEADER = "cc";
	private static final String BCC_ADDRESSLIST_HEADER = "bcc";

	private static final String HEADERS_FIELDNAME = "headers";

	private static final String CONTENTTYPE_MULTIPART_MIXED = "multipart/mixed";
	private static final String CONTENTTYPE_MULTIPART_ALTER = "multipart/alternative";
	private static final String MULTIPART_ALTERNATIVE = "alternative";
	private static final String MULTIPART_MIXED = "mixed";
	private static final String MULTIPART = "multipart/";
	//private static final String CONTENTTYPE_TEXT_PLAIN = "text/plain";
	//private static final String CONTENTTYPE_TEXT_HTML = "text/html";
	private static final String CONTENTTYPE_TEXT_PREFIX = "text/";
	private static final String CONTENTTYPE_TEXT_PLAIN = "plain";
	private static final String CONTENTTYPE_RFC822 = "message/rfc822";

	private static final String CHARSET = "charset";

	private static final String CHARSET_PARAMETER = CHARSET + "=";

	private final SimpleDateFormat _dateFormat = new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss Z", Locale.ENGLISH);

	private USMMimeMessage _message;

	private boolean _isReplyOrForward;

	//----------------------------------------------------------------------------------------------------
	// Begin: with JAVA Mail API

	public String writeTo() throws USMJSONAPIException {
		ByteArrayOutputStream outputStream = new ByteArrayOutputStream();

		try {
			_message.writeTo(outputStream);
		} catch (IOException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_WRITE_OUT_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "error by output of mime mail: " + e.getMessage(), e);

		} catch (MessagingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_WRITE_OUT_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "messaging error by output of mime mail: " + e.getMessage(), e);
		} finally {
			if (outputStream != null)
				try {
					outputStream.close();
				} catch (IOException e) {
					throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_IO_ERROR,
							ResponseStatusCode.INTERNAL_ERROR, "IO error by output of mime mail: " + e.getMessage(), e);
				}
		}
		return outputStream.toString();
	}

	public void buildMimeMailFromJSONObject(JSONObject data) throws USMJSONAPIException {
		_message = new USMMimeMessage(Session.getInstance(new Properties()));
		buildPart(_message, data);
		try {
			_message.saveChanges();
		} catch (MessagingException e1) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_SAVE_CHANGE_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "error by save change of mime mail: " + e1.getMessage());
		}

	}

	public String convertJSONObjectToMimeMail(JSONObject data, boolean isReplyOrForward) throws USMJSONAPIException {
		_isReplyOrForward = isReplyOrForward;
		if (data.has(SMIME_BODY_DATA)) {
			if (data.has(BODY))
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_BODY_FIELDS,
						ResponseStatusCode.BAD_REQUEST, "Message contains smime body field and other body fields!");
			return buildSmimeMailFromJSONObject(data);
		} else {
			buildMimeMailFromJSONObject(data);
			return writeTo();
		}
	}

	private String buildSmimeMailFromJSONObject(JSONObject data) throws USMJSONAPIException {
		JSONObject headers = data.optJSONObject(HEADERS_FIELDNAME);
		StringBuilder sb = new StringBuilder(1000);
		try {
			Object contentType;
			contentType = (headers != null && headers.has(CONTENT_TYPE)) ? headers.get(CONTENT_TYPE) : null;
			String contentTypeOfPart = buildContentTypeAsString(contentType);
			//add headers to message
			MimePart part = new MimeBodyPart();
			appendHeadersToPart(part, headers, contentTypeOfPart);
			for (Enumeration<?> e = part.getAllHeaderLines(); e.hasMoreElements();) {
				sb.append((String) e.nextElement());
				sb.append("\r\n");
			}
			sb.append(new String(Toolkit.decodeBase64(data.getString(SMIME_BODY_DATA))));
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.SMIMEMAIL_JSONEXCEPTION_ERROR,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "JSONException: " + e.getMessage());
		} catch (MessagingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.SMIMEMAIL_JAVAPI_MESSAGING_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "error by building of body part: " + e.getMessage());
		}
		return sb.toString();
	}

	private MimeMultipart buildMimePartForMultipart(JSONObject data, String subtype) throws MessagingException,
			JSONException, USMJSONAPIException {

		MimeMultipart multiPart = new MimeMultipart(subtype);
		if (MULTIPART_MIXED.equals(subtype))
			multiPart = new MixedMimeMultipart();

		// create and add parts
		JSONArray parts = data.getJSONArray(BODY);
		if (parts == null)
			return multiPart;

		int length = parts.length();
		for (int i = 0; i < length; i++) {
			JSONObject jsonpart = parts.getJSONObject(i);
			MimeBodyPart part = new MimeBodyPart();
			buildPart(part, jsonpart);
			multiPart.addBodyPart(part);
		}

		return multiPart;
	}

	private void buildPart(MimePart part, JSONObject json) throws USMJSONAPIException {
		try {
			JSONObject headers = json.optJSONObject(HEADERS_FIELDNAME);
			Object contentType = (headers != null && headers.has(CONTENT_TYPE)) ? headers.get(CONTENT_TYPE)
					: CONTENTTYPE_TEXT_PREFIX + CONTENTTYPE_TEXT_PLAIN + "; " + CHARSET_PARAMETER + UTF_8;
			String contentTypeOfPart = buildContentTypeAsString(contentType);
			if (contentTypeOfPart.startsWith(CONTENTTYPE_MULTIPART_MIXED)) {
				//add headers to message
				appendHeadersToPart(part, headers, CONTENTTYPE_MULTIPART_MIXED);
				// set multi part content
				part.setContent(buildMimePartForMultipart(json, MULTIPART_MIXED), CONTENTTYPE_MULTIPART_MIXED);
			} else if (contentTypeOfPart.startsWith(CONTENTTYPE_MULTIPART_ALTER)) {
				//add headers
				appendHeadersToPart(part, json.getJSONObject(HEADERS_FIELDNAME), CONTENTTYPE_MULTIPART_ALTER);
				// set multi part content
				part.setContent(buildMimePartForAlternative(json), CONTENTTYPE_MULTIPART_ALTER);
			} else if (contentTypeOfPart.startsWith(MULTIPART)) {
				//add headers to message
				appendHeadersToPart(part, headers, contentTypeOfPart);
				// set multi part content
				String subtype = contentTypeOfPart.replaceFirst(MULTIPART, "");
				part.setContent(buildMimePartForMultipart(json, subtype), contentTypeOfPart);

			} else {
				JSONObject body = json.has(BODY) ? json.getJSONObject(BODY) : null;
				if (contentTypeOfPart.startsWith(CONTENTTYPE_TEXT_PREFIX)) {
					try {
						//add headers
						String encoding = getValue(headers, CONTENT_TRANSFER_ENCODING, 0, null);
						appendHeaderConditional(part, CONTENT_TRANSFER_ENCODING_UPPER, encoding);
						appendHeadersToPart(part, headers, contentTypeOfPart);
						//build text part
						String content = (body != null && body.has(DATA)) ? body.getString(DATA) : "";
						buildTextPart(part, contentTypeOfPart, content);
					} catch (UnsupportedEncodingException e) {
						throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_ENCODING_ERROR_3,
								ResponseStatusCode.INTERNAL_ERROR, "unsupported encoding of text content: "
										+ e.getMessage());

					}
				} else if (body != null && body.has(DATA)) {
					String mailcontentTypeAttachment = extractPlainContentType(contentTypeOfPart).trim();
					//add headers
					appendHeadersToPart(part, headers, contentTypeOfPart);
					// set content
					String content = body.getString(DATA);
					String encoding = getValue(headers, CONTENT_TRANSFER_ENCODING, 0, null);
					if (BASE64_ENCODING.equals(encoding)) {
						appendHeaderConditional(part, CONTENT_TRANSFER_ENCODING_UPPER, encoding);
						part.setDataHandler(new DataHandler(new ByteArrayDataSource(Toolkit.decodeBase64(content),
								mailcontentTypeAttachment)));
					} else {
						part.setContent(content, mailcontentTypeAttachment);
					}
					part.setHeader(CONTENT_TYPE_UPPER, contentTypeOfPart);
				} else if (contentTypeOfPart.startsWith(CONTENTTYPE_RFC822)) {
					appendHeadersToPart(part, headers, contentTypeOfPart);
					USMMimeMessage nestedPart = new USMMimeMessage(Session.getInstance(new Properties()));
					buildPart(nestedPart, body);
					part.setContent(nestedPart, CONTENTTYPE_RFC822);
				}
			}
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_JSONEXCEPTION_ERROR,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "JSONException: " + e.getMessage());

		} catch (MessagingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_JAVAPI_MESSAGING_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "error by building of body part: " + e.getMessage());
		}

	}

	private String extractPlainContentType(String contentTypeOfPart) {
		int len = contentTypeOfPart.length();
		for (int i = 0; i < len; i++) {
			char c = contentTypeOfPart.charAt(i);
			if (c == SEMICOLON || Character.isWhitespace(c))
				return contentTypeOfPart.substring(0, i);
		}
		return contentTypeOfPart;
	}

	private String buildContentTypeAsString(Object header) throws JSONException, USMJSONAPIException {
		if (header == null)
			return null;
		ContentType contentType = new ContentType();
		try {
			if (header instanceof String) {
				contentType = new ContentType((String) header);
			} else if (header instanceof JSONObject) {
				String type = ((JSONObject) header).getString(CONTENTTYPE_TYPE);
				contentType.setBaseType(type);
				JSONObject params = (((JSONObject) header).has(CONTENTTYPE_PARAMS)) ? ((JSONObject) header)
						.getJSONObject(CONTENTTYPE_PARAMS) : new JSONObject();
				for (Iterator<?> iterator = params.keys(); iterator.hasNext();) {
					String key = (String) iterator.next();
					String value = "";
					if (READ_DATE.equals(key)) {
						value = extractDateFromJSONObject(params.getJSONObject(READ_DATE));
					} else {
						value = params.getString(key);
					}
					contentType.addParameter(key, value);
				}
			}
		} catch (OXException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_CONTENT_TYPE_MAIL_EXC,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Mail Exception by add content-type header "
							+ e.getMessage());
		}
		return contentType.toString();
	}

	private String buildContentDispositionAsString(Object header) throws JSONException, USMJSONAPIException {
		ContentDisposition contentDisp = new ContentDisposition();
		try {
			if (header instanceof String)
				contentDisp = new ContentDisposition((String) header);
			else if (header instanceof JSONObject) {
				String type = ((JSONObject) header).getString(CONTENTTYPE_TYPE);
				contentDisp.setDisposition(type);
				JSONObject params = (((JSONObject) header).has(CONTENTTYPE_PARAMS)) ? ((JSONObject) header)
						.getJSONObject(CONTENTTYPE_PARAMS) : new JSONObject();
				for (Iterator<?> iterator = params.keys(); iterator.hasNext();) {
					String key = (String) iterator.next();
					String value = "";
					if (READ_DATE.equals(key)) {
						value = extractDateFromJSONObject(params.getJSONObject(READ_DATE));
					} else {
						value = params.getString(key);
					}
					contentDisp.addParameter(key, value);
				}
			}
		} catch (OXException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_CONTENT_DISP_MAIL_EXC,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "Mail Exception by add content-disposition header "
							+ e.getMessage());
		}
		return contentDisp.toString();
	}

	private void buildTextPart(MimePart part, String contentTypeOfPart, String decodedContent) throws JSONException,
			MessagingException, UnsupportedEncodingException {
		String mailcontentTypeText = extractPlainContentType(contentTypeOfPart).trim();
		int posCharset = contentTypeOfPart.indexOf(CHARSET);
		int charSetBeginPos = posCharset + CHARSET.length() + 1;
		int charSetEndPos = contentTypeOfPart.indexOf(';', charSetBeginPos);
		String charset = ((charSetEndPos >= 0) ? contentTypeOfPart.substring(charSetBeginPos, charSetEndPos)
				: contentTypeOfPart.substring(charSetBeginPos)).trim();
		// set content
		int posSubtype = mailcontentTypeText.indexOf('/');
		String subtype = CONTENTTYPE_TEXT_PLAIN;
		if (posSubtype > 0)
			subtype = mailcontentTypeText.substring(posSubtype + 1);
		// set content
		byte[] contentBytes;
		//decode if Content-Transfer-Encoding = Base64
		if (BASE64_ENCODING.equals(part.getEncoding()))
			contentBytes = Toolkit.decodeBase64(decodedContent);
		else
			contentBytes = decodedContent.getBytes(charset);
	    part.setDataHandler(new DataHandler(new ByteArrayDataSource(contentBytes, contentTypeOfPart)));
	    part.setHeader("Content-Type", contentTypeOfPart);
	    // part.setText(new String(contentBytes, charset), charset, subtype);
	}

	private MimeMultipart buildMimePartForAlternative(JSONObject data) throws MessagingException, JSONException,
			USMJSONAPIException {

		MimeMultipart multiPart = new MimeMultipart(MULTIPART_ALTERNATIVE);
		// create and add parts
		JSONArray parts = data.getJSONArray(BODY);
		if (parts == null)
			return multiPart;
		// build simple text parts: plain, html
		int length = parts.length();
		for (int i = 0; i < length; i++) {
			JSONObject jsonpart = parts.getJSONObject(i);
			MimeBodyPart part = new MimeBodyPart();
			JSONObject headers = jsonpart.getJSONObject(HEADERS_FIELDNAME);
			//build text part
			Object contentType = headers.get(CONTENT_TYPE);
			String contentTypeOfPart = buildContentTypeAsString(contentType);

			try {
				//add headers
				part.setHeader(CONTENT_TYPE_UPPER, contentTypeOfPart);
				appendHeadersToSimplePart(part, headers);
				//build text
				buildTextPart(part, contentTypeOfPart, jsonpart.getJSONObject(BODY).getString(DATA));
			} catch (UnsupportedEncodingException e) {
				throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_ENCODING_ERROR,
						ResponseStatusCode.INTERNAL_ERROR, "unsupported encoding of text content: " + e.getMessage());
			}
			// set content
			multiPart.addBodyPart(part);
		}
		return multiPart;
	}

	private void appendHeadersToSimplePart(Part part, JSONObject headers) throws USMJSONAPIException {
		try {
			if (headers != null && headers.has(MIME_VERSION))
				part.setHeader(MIME_VERSION, getValue(headers, MIME_VERSION, 0, null));

			if (headers != null && headers.has(CONTENT_DISPOSITION)) {
				String contDispAsString = buildContentDispositionAsString(headers.get(CONTENT_DISPOSITION));
				part.setDisposition(contDispAsString);
			}
			part.setHeader(CONTENT_TRANSFER_ENCODING_UPPER, getValue(headers, CONTENT_TRANSFER_ENCODING, 0, null));
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_ADD_HEADER_JSON_ERROR,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "JSONException by add header: " + e.getMessage());

		} catch (MessagingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_ADD_HEADER_JAVAPI_MESSAGING_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "MessagingException by add header: " + e.getMessage());
		}

	}

	static Set<String> specialHeaderNamesList = new HashSet<String>();
	static {
		specialHeaderNamesList.add(CONTENT_TYPE);
		specialHeaderNamesList.add(CONTENT_DISPOSITION);
		specialHeaderNamesList.add(CONTENT_TRANSFER_ENCODING);
		specialHeaderNamesList.add(RECEIVED);
		specialHeaderNamesList.add(RETURN_PATH);
		specialHeaderNamesList.add(X_SIEVE);
		specialHeaderNamesList.add(MIME_VERSION);
		specialHeaderNamesList.add(X_MAILER);
	}

	static Set<String> knownAddressHeaderNamesList = new HashSet<String>();
	static {
		knownAddressHeaderNamesList.add(FROM_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(REPLY_TO_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(SENDER_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(ERRORS_TO_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(RESENT_BCC_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(RESENT_CC_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(RESENT_FROM_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(RESENT_TO_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(RESENT_SENDER_ADDRESSLIST_HEADER);
		knownAddressHeaderNamesList.add(DISPOSITION_NOTIFICATION_TO_ADDRESSLIST_HEADER);
	}

    static Map<String, Message.RecipientType> _RECIPIENT_ADDRESS_HEADERS = new HashMap<String, Message.RecipientType>();
    static {
        _RECIPIENT_ADDRESS_HEADERS.put(TO_ADDRESSLIST_HEADER, Message.RecipientType.TO);
        _RECIPIENT_ADDRESS_HEADERS.put(CC_ADDRESSLIST_HEADER, Message.RecipientType.CC);
        _RECIPIENT_ADDRESS_HEADERS.put(BCC_ADDRESSLIST_HEADER, Message.RecipientType.BCC);
    }

	private Enumeration<Header> parseXOriginalHeaders(String xOriginalHeaders) throws USMJSONAPIException {
		ByteArrayInputStream is = new ByteArrayInputStream(Toolkit.decodeBase64(xOriginalHeaders));
		try {
			InternetHeaders ih = new InternetHeaders(is);
			return ih.getAllHeaders();
		} catch (MessagingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_JAVAPI_MESSAGING_ERROR_PARSE_ORIGINAL_HEADERS,
					ResponseStatusCode.INTERNAL_ERROR, "MessagingException by parsing x_original headers: " + e.getMessage());
		}
	}

	private void appendHeadersToPart(Part part, JSONObject headers, String contentType) throws USMJSONAPIException {
		try {

			if (contentType != null)
				part.setHeader(CONTENT_TYPE_UPPER, contentType);
			if (headers != null && headers.has(CONTENT_DISPOSITION)) {
				String contDispAsString = buildContentDispositionAsString(headers.get(CONTENT_DISPOSITION));
				part.setDisposition(contDispAsString);
			}
			appendHeaderConditional(part, CONTENT_TRANSFER_ENCODING_UPPER, getValue(headers, CONTENT_TRANSFER_ENCODING,
					0, null));
			//add (or replace with) the original headers
			if (headers != null && headers.has(X_ORIGINAL_HEADERS)) {
				Enumeration<Header> originalHeadersMap = parseXOriginalHeaders(headers.getString(X_ORIGINAL_HEADERS));
				while (originalHeadersMap.hasMoreElements()) {
					Header nextElement = originalHeadersMap.nextElement();
					String headerName = nextElement.getName().toLowerCase();
					if (RECEIVED.equalsIgnoreCase(headerName)) {
						JSONArray receivedArray = headers.optJSONArray(RECEIVED);
						if (receivedArray == null)
							receivedArray = new JSONArray();
						receivedArray.put(nextElement.getValue().replaceAll(terminalSeq, String.valueOf(SEMICOLON)));
						headers.put(RECEIVED, receivedArray);
					} else {
						if (_isReplyOrForward) {
							headers.put(headerName, nextElement.getValue());
						} else {
							if (!headers.has(headerName))
								headers.put(headerName, nextElement.getValue());

						}
					}
				}
			} 
			//add special headers
			String returnPath = getValue(headers, RETURN_PATH, 0, null);
		    String received = getValue(headers, RECEIVED, 0, null);
			String received2 = getValue(headers, RECEIVED, 1, null);
			String xSieve = getValue(headers, X_SIEVE, 0, null);

			setHeaderConditional(part, RETURN_PATH, returnPath);

			if (headers != null && headers.has(RECEIVED))
				appendHeaderConditional(part, RECEIVED, formatReceived(received));

			setHeaderConditional(part, X_SIEVE, xSieve);

			if (headers != null && headers.has(RECEIVED))
				appendHeaderConditional(part, RECEIVED, formatReceived(received2));

			if (headers != null && headers.has(MIME_VERSION))
				setHeaderConditional(part, MIME_VERSION, getValue(headers, MIME_VERSION, 0, EMPTY_STR));

			if (headers != null && headers.has(X_MAILER))
				setHeaderConditional(part, X_MAILER, getValue(headers, X_MAILER, 0, EMPTY_STR));

			//add additional headers
			if (headers != null) {
				Iterable<String> headerNames = JSONToolkit.keys(headers);
				for (String headerName : headerNames) {
				    Message.RecipientType recipientType = _RECIPIENT_ADDRESS_HEADERS.get(headerName);
				    if(recipientType != null) {
				        appendRecipientHeader(part, headers, recipientType, headerName);
				    } else if (knownAddressHeaderNamesList.contains(headerName)) {
						setHeaderConditional(part, headerName, getAddress(headers, headerName, null));
					} else if (!specialHeaderNamesList.contains(headerName)) {
						setHeaderConditional(part, headerName, getValue(headers, headerName, 0, null));
					}
				}
			}
			
		} catch (JSONException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_ADD_HEADER_JSON_ERROR,
					ResponseStatusCode.WRONG_MISSING_PARAMETERS, "JSONException by add header: " + e.getMessage());

		} catch (MessagingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_ADD_HEADER_JAVAPI_MESSAGING_ERROR,
					ResponseStatusCode.INTERNAL_ERROR, "MessagingException by add header: " + e.getMessage());
		}

	}

    private String getValue(JSONObject headers, String key, int i, String defaultVal) throws JSONException {
		if (headers == null || !headers.has(key))
			return defaultVal;
		Object value = headers.get(key);
		if (value == null)
			return defaultVal;
		if (value instanceof String)
			return (String) value;
		else if (value instanceof JSONArray) {
			JSONArray array = (JSONArray) value;
			if (i >= array.length())
				return defaultVal;
			try {
				return array.getString(i);
			} catch (JSONException e) {
				return defaultVal;
			}
		} else if (value instanceof JSONObject) {
			String v = extractDateFromJSONObject((JSONObject) value);
			return (v == null) ? value.toString() : v;
		}
		return defaultVal;
	}

    private void appendRecipientHeader(Part part, JSONObject headers, javax.mail.Message.RecipientType recipientType, String key) throws USMJSONAPIException {
        if (headers == null || !headers.has(key))
            return;
        try {
            if (!(part instanceof MimeMessage)) {
                setHeaderConditional(part, key, getAddress(headers, key, null));
                return;
            }
            MimeMessage message = (MimeMessage) part;
            Object value = headers.get(key);
            if (value instanceof String) {
                message.setRecipients(recipientType, (String) value);
            } else if (value instanceof JSONObject) {
                message.addRecipient(recipientType, extractInternetAddress((JSONObject) value));
            } else if (value instanceof JSONArray) {
                JSONArray array = (JSONArray) value;
                int length = array.length();
                for (int i = 0; i < length; i++)
                    message.addRecipient(recipientType, extractInternetAddress(array.getJSONObject(i)));
            } else {
                throw new IllegalArgumentException("header of unknown type" + value);
            }
        } catch (UnsupportedEncodingException e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_ENCODING_ERROR_4,
                ResponseStatusCode.INTERNAL_ERROR,
                "unsupported encoding of text content: " + e.getMessage(),
                e);
        } catch (Exception e) {
            throw new USMJSONAPIException(
                ConnectorBundleErrorCodes.MIMEMAIL_RECIPIENT_EXCEPTION,
                ResponseStatusCode.INTERNAL_ERROR,
                "illegal mail address: " + e.getMessage(),
                e);
        }
    }

	private String getAddress(JSONObject headers, String key, String defaultVal) throws JSONException,
			USMJSONAPIException {
		if (headers == null || !headers.has(key))
			return defaultVal;
		Object value = headers.get(key);
		if (value == null)
			return defaultVal;
		if (value instanceof String)
			return value.toString();
		try {
			if (value instanceof JSONObject)
				return extractAddress((JSONObject) value);
			if (value instanceof JSONArray) {
				JSONArray array = (JSONArray) value;
				int length = array.length();
				if (0 >= length)
					return defaultVal;
				StringBuilder address = new StringBuilder(128);
				for (int j = 0; j < length; j++) {
					if (j > 0)
						address.append(',');
					address.append(extractAddress(array.getJSONObject(j)));
				}
				return address.toString();
			}
		} catch (JSONException e) {
			return defaultVal;
		} catch (UnsupportedEncodingException e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_UNSUPPORTED_ENCODING_ERROR_2,
					ResponseStatusCode.INTERNAL_ERROR, "unsupported encoding of text content: " + e.getMessage(), e);
		} catch (Exception e) {
			throw new USMJSONAPIException(ConnectorBundleErrorCodes.MIMEMAIL_ADDRESS_EXCEPTION,
					ResponseStatusCode.INTERNAL_ERROR, "illegal mail address: " + e.getMessage(), e);
		}
		return defaultVal;
	}

    private QuotedInternetAddress extractInternetAddress(JSONObject address) throws JSONException, UnsupportedEncodingException, AddressException {
        String addressPart = address.getString(ADDRESS);
        String personalPart = address.optString(PERSONAL, null);
        return new QuotedInternetAddress(addressPart, personalPart, UTF_8);
    }

	private String extractAddress(JSONObject address) throws JSONException, UnsupportedEncodingException {
		try {
		    return extractInternetAddress(address).toString();
		} catch (AddressException e) {
	        String addressPart = address.getString(ADDRESS);
	        String personalPart = address.optString(PERSONAL, null);
			// In case of invalid address data, use simple composition of the raw (illegal) data
			if (personalPart == null)
				return addressPart;
			return personalPart + " <" + addressPart + ">";
		}
	}

	private void appendHeaderConditional(Part part, String header, String value) throws MessagingException {
		if (value != null && !"null".equals(value)) {
			try {
				part.addHeader(header, MimeUtility.encodeText(value, UTF_8, null));
			} catch (UnsupportedEncodingException e) {
				part.addHeader(header, value);
			}
		}
	}
	
	private void setHeaderConditional(Part part, String header, String value) throws MessagingException {
		if (value != null && !"null".equals(value)) {
			try {
				part.setHeader(header, MimeUtility.encodeText(value, UTF_8, null));
			} catch (UnsupportedEncodingException e) {
				part.setHeader(header, value);
			}
		}
	}

	private String extractDateFromJSONObject(JSONObject dateObject) {
		try {
			if (dateObject.has("date"))
				return dateObject.getString("date");
			if (dateObject.has("utc"))
				return _dateFormat.format(new Date(dateObject.getLong("utc")));
		} catch (JSONException ignored) {
		}
		return null;
	}

	private String formatReceived(String rec) {
		if (rec == null || rec.length() < 2)
			return rec;
		StringBuilder buffer = new StringBuilder(60);
		int posBy = rec.indexOf("by ");
		if(posBy < 0) posBy = rec.indexOf("by" + terminalSeq);
		int posFor = rec.indexOf("for ");
		if (posFor > 0 && posBy > 0) {
			buffer.append(rec.substring(0, posBy - 1)).append(terminalSeq);
			buffer.append('	').append(rec.substring(posBy, posFor - 1)).append(terminalSeq);
			buffer.append('	').append(rec.substring(posFor, rec.length()));
		} else if (posBy > 0) {
			int posWith = rec.indexOf("with ", posBy);
			if (posWith < 0)
				return rec;
			int pos = rec.indexOf(SEMICOLON, posWith);
			if (pos < 0 || (pos + 1) >= rec.length())
				return rec;
			buffer.append(rec.substring(0, posBy - 1)).append(terminalSeq);
			buffer.append('	').append(rec.substring(posBy, pos + 1)).append(terminalSeq);
			buffer.append('	').append(rec.substring(pos + 2, rec.length()));
		}
		return buffer.toString();
	}

	//----------------------------------------------------------------------------------------------------
	//End	

}
